// -*- Mode: C++; -*-
//                              File      : RDIChannelUtil.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of several utility functions used by EventChannel_i
//
 
/*
$Log: RDIChannelUtil.cc,v $
Revision 1.33  2000/10/04 15:15:06  alcfp
more small updates to get rid of compiler warnings

Revision 1.32  2000/08/22 18:23:55  alcfp
added description to each file

Revision 1.31  2000/08/16 20:19:45  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include "RDIDebug.h"
#include "omnithread.h"
#include "RDIBitmap.h"
#include "RDIThreadPriority.h"
#include "CosNotifyChannelAdmin_i.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//                         Various Utility Functions                     //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

const char* RDI_PRX_TYPE(const CosNA_ProxyType& type)
{
  switch ( type ) {
    case CosNA_PUSH_ANY:        return (const char*)"PUSH_ANY";
    case CosNA_PULL_ANY:        return (const char*)"PULL_ANY";
    case CosNA_PUSH_STRUCTURED: return (const char*)"PUSH_STR";
    case CosNA_PULL_STRUCTURED: return (const char*)"PULL_STR";
    case CosNA_PUSH_SEQUENCE:   return (const char*)"PUSH_SEQ";
    case CosNA_PULL_SEQUENCE:   return (const char*)"PULL_SEQ";
    case CosNA_PUSH_TYPED:      return (const char*)"PUSH_TYP";
    case CosNA_PULL_TYPED:      return (const char*)"PULL_TYP";
  }
  return (const char *)"INVALID";
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//        ChannelAdminGroup and ChannelProxyGroup implementations        //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

#define RDI_GROUP_LIMIT         9999999

ChannelAdminGroup::ChannelAdminGroup() :
        _oplock(), _calist(), _cacurs(), _length(RDI_GROUP_LIMIT)
{;}

ChannelAdminGroup::~ChannelAdminGroup()
{ 
  // NOTE: we do not acquire the mutex to avoid EBUSY errors
  //       during their destruction
  // _oplock.lock();
  _cacurs.clear(); 
  _calist.drain();
}

void ChannelAdminGroup::insert(ConsumerAdmin_i* entry)
{
  _oplock.lock();
  (void) _calist.insert_tail(entry);
  _oplock.unlock();
}

void ChannelAdminGroup::remove(ConsumerAdmin_i* entry)
{
  RDI_ListCursor<ConsumerAdmin_i *> curs;
  _oplock.lock();
  curs = _calist.cursor();
  for (unsigned int ix = 0; ix < _calist.length(); ix++) {
        if ( entry == *curs ) {
                _calist.remove(curs);
                break;
        }
        curs++;
  }
  _oplock.unlock();
}

ConsumerAdmin_i* ChannelAdminGroup::next()
{
  // If we are starting a new iteration, initialize the cursor
  if ( _length == RDI_GROUP_LIMIT ) {
        _length = 0;
        _cacurs = _calist.cursor();
        _oplock.lock();
  }
  if ( _length < _calist.length() ) {
        ConsumerAdmin_i* admin = *_cacurs;
        _cacurs++;
        _length++;
        return admin;
  }
  // The iteration is over: clear the cursor, set the '_length' to
  // RDI_GROUP_LIMIT so that the cursor is initialized at a future
  // invocation of this method.
  _cacurs.clear();
  _length = RDI_GROUP_LIMIT;
  _oplock.unlock();
  return 0;
}

ostream& operator << (ostream& out, ChannelAdminGroup& agr)
{
  RDI_ListCursor<ConsumerAdmin_i *> curs;
  ConsumerAdmin_i* admin;
  unsigned int length = 0;
  out << "ChannelAdminGroup: " << & agr << endl;
  for ( curs = agr._calist.cursor(); length < agr._calist.length(); length++) {
        admin = *curs;
        out << "\tConsumerAdmin_i " << admin << " ID " << setw(4) <<
               admin->MyID() << " #Proxies " << admin->NumProxies() << endl;
        ++curs;
  }
  return out;
}

// --------------------------------------------------------------------- //

ChannelAdminGroup_m::ChannelAdminGroup_m(unsigned int numGroups,
                                         unsigned int numThreds) :
                        _oplock(), _groups(0), _length(numGroups),
                        _nextgr(0), _numthr(numThreds),
                        _rmgrps(numGroups), _rmthrs(numThreds)
{ _groups = new ChannelAdminGroup [_length]; }

ChannelAdminGroup_m::~ChannelAdminGroup_m()
{
  // NOTE: we do not acquire the mutex to avoid EBUSY errors
  //       during their destruction
  // _oplock.lock();
  delete [] _groups;
  _groups = 0;
}

void ChannelAdminGroup_m::insert(ConsumerAdmin_i* entry)
{
  unsigned int indx = (unsigned int) (entry->MyID() % _length);
  RDI_DUMP("Add ConsumerAdmin "<<entry->MyID()<<" to group "<<&_groups[indx]);
  _groups[ indx ].insert(entry);
}

void ChannelAdminGroup_m::remove(ConsumerAdmin_i* entry)
{
  unsigned int indx = (unsigned int) (entry->MyID() % _length);
  RDI_DUMP("Rem ConsumerAdmin "<<entry->MyID()<<" from group "<<&_groups[indx]);
  _groups[ indx ].remove(entry);
}

CORBA::Boolean ChannelAdminGroup_m::allocate_range(unsigned int& lo,
                                                   unsigned int& hi)
{
  unsigned int slice = 0;
  lo = hi = 0;
  _oplock.lock();

  if ( _nextgr == _length ) {   // Everything has been allocated
        _oplock.unlock();
        return 0;
  }
  if ( _rmthrs == 0 ) {		// Internal error!!!!!
	_oplock.unlock();
	return 0;
  }
  slice    = _rmgrps / _rmthrs;
  lo       = _nextgr;
  _nextgr += slice;
  hi       = _nextgr - 1;
  _rmgrps -= slice;
  _rmthrs -= 1;
  _oplock.unlock();
  return 1;
}

ostream& operator << (ostream& out, ChannelAdminGroup_m& agrm)
{
  agrm._oplock.lock();
  out << "ChannelAdminGroup_m" << endl << "-------------------" << endl;
  for ( unsigned int ix=0; ix < agrm._length; ix++ )
        out << "    " << agrm._groups[ix];
  agrm._oplock.unlock();
  return out;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//             Support classes used for event dispatching                //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

EventChannelDispatch::EventChannelDispatch(EventChannel_i*    evnt_channel,
                                           unsigned int       num_athreads,
                                           EventChannelMethod admin_method,
                                           unsigned int       num_pthreads,
                                           EventChannelMethod proxy_method) :
                _num_athreads(num_athreads), _admin_thread(0),
                _num_pthreads(num_pthreads), _proxy_thread(0),
                _evnt_channel(evnt_channel)
{
  unsigned int ix;
  if ( _num_athreads ) {
  	_admin_thread = new omni_thread * [ _num_athreads ];
  	RDI_Assert(_admin_thread, "Memory allocation failed -- omni_thread");
  }
  for ( ix=0; ix < _num_athreads; ix++) {
     // _admin_thread[ix]=new EventChannelWorker(_evnt_channel, admin_method);
     _admin_thread[ix]=new EventChannelBoundWorker(_evnt_channel, admin_method);
     RDI_Assert(_admin_thread[ix], "Failed to create new admin thread");
     RDI_DUMP("New admin filtering thread: " << _admin_thread[ix]->id());
  }
  if ( _num_pthreads ) {
  	_proxy_thread = new omni_thread * [ _num_pthreads ];
  	RDI_Assert(_proxy_thread, "Memory allocation failed -- omni_thread");
  }
  for ( ix=0; ix < _num_pthreads; ix++) {
     // _proxy_thread[ix]=new EventChannelWorker(_evnt_channel, proxy_method);
     _proxy_thread[ix]=new EventChannelBoundWorker(_evnt_channel, proxy_method);
     RDI_Assert(_proxy_thread[ix], "Failed to create new proxy thread");
     RDI_DUMP("New proxy filtering thread: " << _proxy_thread[ix]->id());
  }
  // The following is needed if we create unbound threads to start them
  // for ( ix=0; ix < _num_athreads; ix++)
  //    _admin_thread[ix]->start();
  // for ( ix=0; ix < _num_pthreads; ix++)
  //    _proxy_thread[ix]->start();
}

EventChannelDispatch::~EventChannelDispatch()
{
  // If we have bound threads, we should wait till all of them exit to 
  // reclaim their space -- this is not needed for unbound threads ...
  unsigned int ix;
  for ( ix=0; ix < _num_athreads; ix++) {
	_admin_thread[ix]->join(0);
	_admin_thread[ix] = 0;
  }
  RDI_DUMP("\t" << _num_athreads << " admin filtering threads terminated");
  for ( ix=0; ix < _num_pthreads; ix++) {
	_proxy_thread[ix]->join(0);
	_proxy_thread[ix] = 0;
  }
  if ( _num_pthreads )
  	RDI_DUMP("\t"<<_num_pthreads<<" proxy filtering threads terminated");
  if ( _admin_thread ) delete [] _admin_thread; _admin_thread = 0;
  if ( _proxy_thread ) delete [] _proxy_thread; _proxy_thread = 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//    Thread Pool Implementations for Pull Supplier and Push Consumers   //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

typedef void (RDI_PullSupplier::*PullSupplierMethod)(void);

class RDI_PullWorker : public omni_thread {
public:
  RDI_PullWorker(RDI_PullSupplier* object, PullSupplierMethod method, 
		 priority_t priority = RDI_PULL_WORKER_PRIORITY);
  void run(void *);
private:
  RDI_PullSupplier*  _object;
  PullSupplierMethod _method;
  RDI_PullWorker() {;}
};

RDI_PullWorker::RDI_PullWorker(RDI_PullSupplier*  object,
			       PullSupplierMethod method,
			       priority_t         thprio) :
                omni_thread(NULL, thprio), _object(object), _method(method) 
{;}

void RDI_PullWorker::run(void *)
{ (_object->*_method)(); }

// --------------------------------------------------------------------- //

class RDI_PullBoundWorker : public omni_thread {
public:
  RDI_PullBoundWorker(RDI_PullSupplier* object, PullSupplierMethod method,
                      priority_t prio=RDI_PULL_WORKER_PRIORITY);
  void* run_undetached(void *);
private:
  RDI_PullSupplier*  _object;
  PullSupplierMethod _method;
  RDI_PullBoundWorker() {;}
};

RDI_PullBoundWorker::RDI_PullBoundWorker(RDI_PullSupplier*  object, 
					 PullSupplierMethod method,
                      			 priority_t         thprio) :
                omni_thread(NULL, thprio), _object(object), _method(method) 
{ start_undetached(); }

void* RDI_PullBoundWorker::run_undetached(void *)
{ (_object->*_method)(); return 0; }

// --------------------------------------------------------------------- //

RDI_PullSupplier::RDI_PullSupplier(unsigned int numThreads, 
				   unsigned int periodMSec) :
			_oplock(), _nonempty(&_oplock), _period(periodMSec),
			_threads(0), _nactive(0),  _ndeleted(0), _terminate(0),
			_entries(0), _lastone(0)
{
  _oplock.lock();
  if ( numThreads ) {
  	_threads = new omni_thread * [ numThreads ];
	RDI_Assert(_threads, "Memory allocation failed -- omni_thread");
  }
  for (unsigned int i = 0; i < numThreads; i++) {
     // _threads[i] = new RDI_PullWorker(this, &RDI_PullSupplier::pull_event);
     _threads[i] = new RDI_PullBoundWorker(this, &RDI_PullSupplier::pull_event);
     RDI_Assert(_threads[i], "Failed to create new thread");
     // If we use unbound threads, we should start the thread using
     // _threads[i]->start();
     RDI_DUMP("New pull supplier thread: " << _threads[i]->id());
  }
  _nactive = numThreads;
  _oplock.unlock();
}

void RDI_PullSupplier::destroy()
{
  _oplock.lock();
  _terminate = 1;
  _nonempty.broadcast();
  _oplock.unlock();

  for (unsigned int ix = 0; ix < _nactive; ix++ ) {
	_threads[ix]->join(0);
	_threads[ix] = 0;
  }
  // When using unbound threads, we should wait till all of them exit
  // to avoid getting core dumps due to invalid memory accesses. Note
  // that in this case _nactive is decremented by the exiting thread.
  //
  // while ( _nactive && (ix++ < 120) ) {
  // 	_nonempty.broadcast();
  // 	omni_thread::sleep(1);
  // }
  // 
  // NOTE: we do not acquire the mutex to avoid EBUSY errors during 
  //       their destruction
  // _oplock.lock();

  RDI_DUMP("\t" << _nactive << " pull supplier threads terminated");
  if ( _threads ) delete [] _threads; _threads = 0;
  while ( _entries ) {
	ProxyEntry_t* node = _entries;
	_entries = _entries->_next;
	delete node;
  }
}

// Insert a new entry into the list of ProxyConsumer proxies that are
// handled by the threads of this object. The insertion has no effect
// if this object is being destroyed,  the provided reference is NIL,
// or the proxy does not correspond to a pull supplier.

void RDI_PullSupplier::insert_proxy(RDIProxyPullConsumer* proxy)
{
  _oplock.lock();
  if ( _terminate || ! proxy ) {
	_oplock.unlock();
	return;
  }
  ProxyEntry_t* node = new ProxyEntry_t(proxy);
  RDI_Assert(node, "Memory allocation failed -- ProxyEntry_t");
  node->_next = _entries;
  _entries    = node;
  _nonempty.signal();
  RDI_DUMP("PullConsumer proxy " << proxy << " to be monitored");
  _oplock.unlock();
}

// Remove an entry from the list of ProxyConsumer proxies handled by
// the threads of this object.  The deletion has no effect when this
// object is being destroyed, the provided reference is NIL, or such
// entry is not found. Otherwise, the entry is marked as deleted.

void RDI_PullSupplier::remove_proxy(RDIProxyPullConsumer* proxy)
{
  ProxyEntry_t* node = 0;
  unsigned int  iter = 0;

  _oplock.lock();
  if ( ! proxy || _terminate ) {
	_oplock.unlock();
	return;
  }
  for ( node = _entries; node != (ProxyEntry_t *) 0; node = node->_next ) {
	if ( node->_proxy == proxy )
		break;
  }
  if ( node == (ProxyEntry_t *) 0 ) {
	_oplock.unlock();
	return;
  }
  // Since this method may be called when a supplier is disconnecting
  // from the channel, we should wait till a pending pull by a thread
  // is completed to avoid exceptions and/or core dumps.
  while ( node->_inuse && (iter++ < 20) ) {
	_oplock.unlock();
	omni_thread::sleep(1);
	_oplock.lock();
	if ( _terminate ) {
		_oplock.unlock();
		return;
	}
  }
  node->_deled = 1;
  if ( ++_ndeleted > 5 ) 
	_gcollect();
  RDI_DUMP("PullConsumer proxy " << proxy << " not monitored");
  _oplock.unlock();
}

// This is the code that gets executed by each created thread for as
// long as the object is not being destroyed.

void RDI_PullSupplier::pull_event()
{
  ProxyEntry_t*         node= 0;
  CORBA::Boolean        invalid;

  while ( 1 ) {
	_oplock.lock();
	while ( ! _terminate && ! (node = _next_available()) )
		_nonempty.wait();
	if ( _terminate ) {
                // _nactive -= 1;    --> When using unbound threads
		_oplock.unlock();
		omni_thread::exit();
	}
	node->_inuse = 1;
        _oplock.unlock();

	// NOTE: we update 'node' and may update '_ndeleted' in the 
	//       following segment without holding the lock.  While
 	//       this is not safe in general, it does not affect us
	//       here since 'node->_deled' and 'node->_inuse' are 1
	//       byte each (atomic assignment) and '_ndeleted' does
	//       not have to be 100% accurate

	if ( _period != 0 ) {
		const RDI_TimeValue& ptms = node->_proxy->last_pull_tms();
		      RDI_TimeValue  tnow = RDI_TimeValue::timeofday();
		      RDI_TimeValue  twin(_period/1000, (_period % 1000)*1000);
		if ( RDI_TimeValue::delta(tnow, ptms) < twin ) {
			node->_inuse = 0;
			continue;
		}
	}
	node->_proxy->pull_event(invalid);
	if ( invalid ) {
		node->_deled = 1;
		_ndeleted   += 1;
	}
	node->_inuse = 0;
  }
}

// Locate an entry that is not being 'used; by another thread and is
// still vaild, i.e., has not been deleted.  If no such entry can be
// found, return NULL

RDI_PullSupplier::ProxyEntry_t* RDI_PullSupplier::_next_available()
{
  unsigned int numIter = 1;
  if ( ! _entries )
        return 0;
  while ( numIter < 3 ) {
	if ( ! _lastone || ! _lastone->_next ) {
		numIter += 1;
		_lastone = _entries;
	} else {
		_lastone = _lastone->_next;	
	}
        if ( ! _lastone->_inuse && ! _lastone->_deled )
                return _lastone;
  }
  return 0;
}

// Iterate over all entries in the list of ProxyConsumer proxies and
// delete those that marked as deleted and are not being used by any
// of the created threads.

void RDI_PullSupplier::_gcollect()
{
  ProxyEntry_t* node= _entries;
  ProxyEntry_t* prev= 0;

  while ( node ) {
        if ( node->_deled && ! node->_inuse ) {
                if ( node == _lastone )
                        _lastone = prev;
                if ( ! prev ) {
                        _entries = node->_next;
                        delete node;
                        node = _entries;
                } else {
                        prev->_next = node->_next;
                        delete node;
                        node = prev->_next;
                }
                _ndeleted -= 1;
        } else {
                prev = node;
                node = node->_next;
        }
  }
}

// --------------------------------------------------------------------- //

typedef void (RDI_NotifyConsumer::*NotifyConsumerMethod)(void);

class RDI_NotifyWorker : public omni_thread {
public:
  RDI_NotifyWorker(RDI_NotifyConsumer* object, NotifyConsumerMethod method, 
		   priority_t priority = RDI_NOTIFY_WORKER_PRIORITY);
  void run(void *);
private:
  RDI_NotifyConsumer*  _object;
  NotifyConsumerMethod _method;
  RDI_NotifyWorker() {;}
};

RDI_NotifyWorker::RDI_NotifyWorker(RDI_NotifyConsumer*  object,
				   NotifyConsumerMethod method,
				   priority_t           thprio) :
                omni_thread(NULL, thprio), _object(object), _method(method) 
{;}

void RDI_NotifyWorker::run(void *)
{ (_object->*_method)(); }

// --------------------------------------------------------------------- //

class RDI_NotifyBoundWorker : public omni_thread {
public:
  RDI_NotifyBoundWorker(RDI_NotifyConsumer* object,NotifyConsumerMethod method,
		        priority_t priority = RDI_NOTIFY_WORKER_PRIORITY);
  void* run_undetached(void *);
private:
  RDI_NotifyConsumer*  _object;
  NotifyConsumerMethod _method;
  RDI_NotifyBoundWorker() {;}
};

RDI_NotifyBoundWorker::RDI_NotifyBoundWorker(RDI_NotifyConsumer*  object,
					     NotifyConsumerMethod method,
				   	     priority_t           thprio) :
                omni_thread(NULL, thprio), _object(object), _method(method) 
{ start_undetached(); }

void* RDI_NotifyBoundWorker::run_undetached(void *)
{ (_object->*_method)(); return 0; }

// --------------------------------------------------------------------- //

RDI_NotifyConsumer::RDI_NotifyConsumer(unsigned int numThreads) :
			_oplock(),  _nonempty(&_oplock), _threads(0), 
			_nactive(0), _ndeleted(0), _terminate(0), 
			_entries(0), _lastone(0)
{
  _oplock.lock();
  if ( numThreads ) {
	_threads = new omni_thread * [ numThreads ];
	RDI_Assert(_threads, "Memory allocation failed -- omni_thread");
  }
  for (unsigned int i = 0; i < numThreads; i++) {
     // _threads[i] = new RDI_NotifyWorker(this, &RDI_NotifyConsumer::notify);
     _threads[i] = new RDI_NotifyBoundWorker(this, &RDI_NotifyConsumer::notify);
     RDI_Assert(_threads[i], "Failed to create new thread");
     // If we use unbound threads, we should start the thread using
     // _threads[i]->start();
     RDI_DUMP("New push consumer thread: " << _threads[i]->id());
  }
  _nactive = numThreads;
  _oplock.unlock();
}

void RDI_NotifyConsumer::destroy()
{
  _oplock.lock();
  _terminate = 1;
  _nonempty.broadcast();
  _oplock.unlock();
  for (unsigned int ix = 0; ix < _nactive; ix++ ) {
	_threads[ix]->join(0);
	_threads[ix] = 0;
  }
  // When using unbound threads, we should wait till all of them exit
  // to avoid getting core dumps due to invalid memory accesses. Note
  // that in this case _nactive is decremented by the exiting thread.
  // 
  // while ( _nactive && (ix++ < 120) ) {
  //    _nonempty.broadcast();
  //    omni_thread::sleep(1);
  // }
  // 
  // NOTE: we do not acquire the mutex to avoid EBUSY errors during 
  //       their destruction
  // _oplock.lock();

  RDI_DUMP("\t" << _nactive << " push consumer threads terminated");
  if ( _threads ) delete [] _threads; _threads = 0;
  while ( _entries ) {
	ProxyEntry_t* node = _entries;
	_entries = _entries->_next;
	delete node;
  }
}

// Insert a new entry into the list of ProxySupplier proxies that are
// handled by the threads of this object. The insertion has no effect
// if this object is being destroyed,  the provided reference is NIL,
// or the proxy does not correspond to a push consumer.

void RDI_NotifyConsumer::insert_proxy(RDIProxyPushSupplier* proxy)
{
  _oplock.lock();
  if ( _terminate || ! proxy ) {
	_oplock.unlock();
	return;
  }
  ProxyEntry_t* node = new ProxyEntry_t(proxy);
  RDI_Assert(node, "Memory allocation failed -- ProxyEntry_t");
  node->_next = _entries;
  _entries    = node;
  _nonempty.signal();
  RDI_DUMP("PushSupplier proxy " << proxy << " to be monitored");
  _oplock.unlock();
}

// Remove an entry from the list of ProxySupplier proxies handled by
// the threads of this object.  The deletion has no effect when this
// object is being destroyed, the provided reference is NIL, or such
// entry is not found. Otherwise, the entry is marked as deleted.

void RDI_NotifyConsumer::remove_proxy(RDIProxyPushSupplier* proxy)
{
  ProxyEntry_t* node = 0;

  _oplock.lock();
  if ( ! proxy || _terminate ) {
	_oplock.unlock();
	return;
  }
  for ( node = _entries; node != (ProxyEntry_t *) 0; node = node->_next ) {
	if ( node->_proxy == proxy )
		break;
  }
  if ( node ) {
	node->_deled = 1;
	if ( ++_ndeleted > 5 )
		_gcollect();
	RDI_DUMP("PushSupplier proxy " << proxy << " not monitored");
  }
  _oplock.unlock();
}

// This is the code that gets executed by each created thread for as
// long as the object is not being destroyed.

void RDI_NotifyConsumer::notify()
{
  ProxyEntry_t*  node= 0;
  CORBA::Boolean invalid;

  while ( 1 ) {
	_oplock.lock();
	while ( ! _terminate && ! (node = _next_available()) ) {
		_nonempty.wait();
 	}
	if ( _terminate ) {
		// _nactive -= 1;    --> When using unbound threads
		_oplock.unlock();
		omni_thread::exit();
	}
	node->_inuse = 1;
	_oplock.unlock();

	// NOTE: we update 'node' and may update '_ndeleted' in the
        //       following segment without holding the lock.  While
        //       this is not safe in general, it does not affect us
        //       here since 'node->_deled' and 'node->_inuse' are 1
        //       byte each (atomic assignment) and '_ndeleted' does
        //       not have to be 100% accurate

	if ( ! node->_deled ) {
		node->_proxy->push_event(invalid);
		if ( invalid ) {
			node->_deled = 1;
			_ndeleted   += 1;
		}
	}
	node->_inuse = 0;
	omni_thread::yield();
  }
}

// Locate an entry that is not being 'used' by another thread and is
// still vaild, i.e., has not been deleted.  If no such entry can be
// found, return NULL

RDI_NotifyConsumer::ProxyEntry_t* RDI_NotifyConsumer::_next_available()
{
  unsigned int numIter = 1;
  if ( ! _entries )
        return 0;
  while ( numIter < 3 ) {
	if ( ! _lastone || ! _lastone->_next ) {
		numIter += 1;
		_lastone = _entries;
	} else {
		_lastone = _lastone->_next;
	}
        if ( ! _lastone->_inuse && 
	     ! _lastone->_deled && _lastone->_proxy->has_events() )
                return _lastone;
  }
  return 0;
}

// Iterate over all entries in the list of ProxySupplier proxies and
// delete those that marked as deleted and are not being used by any
// of the created threads.

void RDI_NotifyConsumer::_gcollect()
{
  ProxyEntry_t* node = _entries;
  ProxyEntry_t* prev = 0;
  unsigned int  dsav = 0;

  while ( node ) {
	if ( ! node->_deled ) {
		prev = node;
		node = node->_next;
		continue;
	} 
	// Node is marked as being deleted. If the node is in use, 
	// we just update the counter of deleted but still needed
	// nodes. Else, we physically destroy the node
	if ( node->_inuse ) {
		dsav += 1;
		prev  = node;
		node  = node->_next;
	} else {
		if ( node == _lastone )
			_lastone = prev;
		if ( ! prev ) {
			_entries = node->_next;
			delete node;
			node = _entries;
		} else {
			prev->_next = node->_next;
			delete node;
			node = prev->_next;
		}
	}
  }
  _ndeleted = dsav;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

EventChannelWorker::EventChannelWorker(EventChannel_i*    evchan,
				       EventChannelMethod method,
				       priority_t         trprio) :
		 omni_thread(NULL, trprio), _channel(evchan), _method(method)
{;}

void EventChannelWorker::run(void *)
{ (_channel->*_method)(); }

EventChannelBoundWorker::EventChannelBoundWorker(EventChannel_i*    evchan,
						 EventChannelMethod method,
						 priority_t         trprio) :
		omni_thread(NULL, trprio), _channel(evchan), _method(method)
{ start_undetached(); }

void* EventChannelBoundWorker::run_undetached(void *)
{ (_channel->*_method)(); return 0; }

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//              Optimized Code That Uses Internal omniORB APIs           //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

#ifdef __FASTPATH__

void RDI_EventCallDescr::initialize(cdrStream& giop_client)
{
  giopStream& s = (giopStream&) giop_client;
  giopStream& t = _giop_s;
  _fast_c = s.prepareCopyMessageBodyFrom(t);
}

void RDI_EventCallDescr::marshalArguments(cdrStream& giop_client)
{
  giopStream& s1 = (giopStream&) giop_client;
  giopStream& s2 = _giop_s;

  if ( _fast_c ) {
	s1.copyMessageBodyFrom(s2);
  } else {
	RDI_DUMP("Need to unmarshal the event from s2 and marshal it in s1");
  }
  _giop_s.RequestReceived(); 
}

void RDI_EventCallDescr::unmarshalReturnedValues(cdrStream& giop_client)
{
  RDI_DUMP("Should not have been here since we send events to clients");
}

#endif
