// -*- Mode: C++; -*-
//                              File      : PushSupplier_i.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of PushSupplier proxies
//
 
/*
$Log: PushSupplier_i.cc,v $
Revision 1.51  2000/10/01 13:39:53  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.50.2.1  2000/09/30 15:39:52  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.50  2000/08/22 18:23:54  alcfp
added description to each file

Revision 1.49  2000/08/16 20:19:42  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <iomanip.h>
#include "RDI.h"
#include "RDIEvent.h"
#include "RDIDebug.h"
#include "CosNotifyChannelAdmin_i.h"

extern const char* RDI_PRX_TYPE(const CosNA_ProxyType& type);

// ------------------------------------------------------------- //
// Support classes needed for implementing  the thread-per-proxy //
// notification approach                                         //
// ------------------------------------------------------------- //

typedef void (ProxyPushSupplier_i::*AnyPushSupplierMethod)(void);

class AnyProxyWorker : public omni_thread {
public:
  AnyProxyWorker(ProxyPushSupplier_i* p, AnyPushSupplierMethod m) 
		: omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  ProxyPushSupplier_i*  _proxy;
  AnyPushSupplierMethod _method;
  AnyProxyWorker()  {;}
};

class AnyProxyBoundWorker : public omni_thread {
public:
  AnyProxyBoundWorker(ProxyPushSupplier_i* p, AnyPushSupplierMethod m) 
		: omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) 
				{ start_undetached();   }
  void* run_undetached(void *)	{ (_proxy->*_method)(); return 0; }
private:
  ProxyPushSupplier_i*  _proxy;
  AnyPushSupplierMethod _method;
  AnyProxyBoundWorker()  {;}
};

typedef void (StructuredProxyPushSupplier_i::*StrPushSupplierMethod)(void);

class StrProxyWorker : public omni_thread {
public:
  StrProxyWorker(StructuredProxyPushSupplier_i* p, StrPushSupplierMethod m) 
		: omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  StructuredProxyPushSupplier_i* _proxy;
  StrPushSupplierMethod          _method;
  StrProxyWorker()  {;}
};

class StrProxyBoundWorker : public omni_thread {
public:
  StrProxyBoundWorker(StructuredProxyPushSupplier_i* p, StrPushSupplierMethod m)
	 	: omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) 
				{ start_undetached();   }
  void* run_undetached(void *)  { (_proxy->*_method)(); return 0; }
private:
  StructuredProxyPushSupplier_i* _proxy;
  StrPushSupplierMethod          _method;
  StrProxyBoundWorker()  {;}
};

typedef void (SequenceProxyPushSupplier_i::*SeqPushSupplierMethod)(void);

class SeqProxyWorker : public omni_thread {
public:
  SeqProxyWorker(SequenceProxyPushSupplier_i* p, SeqPushSupplierMethod m) 
		: omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  SequenceProxyPushSupplier_i* _proxy;
  SeqPushSupplierMethod        _method;
  SeqProxyWorker()  {;}
};

class SeqProxyBoundWorker : public omni_thread {
public:
  SeqProxyBoundWorker(SequenceProxyPushSupplier_i* p, SeqPushSupplierMethod m) 
		: omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) 
				{ start_undetached();   }
  void* run_undetached(void *)  { (_proxy->*_method)(); return 0; }
private:
  SequenceProxyPushSupplier_i* _proxy;
  SeqPushSupplierMethod        _method;
  SeqProxyBoundWorker()  {;}
};

// ------------------------------------------------------------- //
// ProxyPushSupplier_i                                           //
// ------------------------------------------------------------- //

ProxyPushSupplier_i::ProxyPushSupplier_i(ConsumerAdmin_i* admin, 
					 EventChannel_i*  chann,
					 const CosNA_ProxyID&  prxid) : 
                RDINotifySubscribe(),
		_oplock(), _fa_helper(), _channel(chann), _myadmin(admin), 
		_prxtype(CosNA_PUSH_ANY),
		_pserial(prxid), _cosevnt(0), _nevents(0),
		_pxstate(RDI_NotConnected), _active(0), 
		_qempty(0), _worker(0), _qosprop(0), _offauto(1), 
		_rqstypes(), _ntfqueue(32, 128)
{
  _rqstypes.length(0);
  _pfilter  = CosNF_MappingFilter::_nil();
  _lfilter  = CosNF_MappingFilter::_nil();
  _consumer = CosEventComm::PushConsumer::_nil();

  // When the number of push threads allocated at the channel level
  // is 0, each proxy uses its own thread for pushing events 
  if ( _channel->push_threads() == 0 ) {
     _qempty = new omni_condition(&_oplock);
     RDI_Assert(_qempty, "Memory allocation failed -- omni_condition");
     // _worker = new AnyProxyWorker(this, &ProxyPushSupplier_i::_push_event);
     _worker = new AnyProxyBoundWorker(this, &ProxyPushSupplier_i::_push_event);
     RDI_Assert(_worker, "Memory allocation failed -- omni_thread");
     // If we use unbound threads, we should start the thread using
     // _worker->start();
     RDI_DUMP("Push thread for proxy " << this << " -- " << _worker->id());
  }
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_ConsumerAdmin_ptr ProxyPushSupplier_i::MyAdmin( WRAPPED_IMPLARG_VOID )
{ return WRAPPED_IMPL2OREF(CosNA_ConsumerAdmin, _myadmin); }

CosNF_MappingFilter_ptr ProxyPushSupplier_i::priority_filter( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_pfilter) )
  	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _pfilter);
}

void ProxyPushSupplier_i::priority_filter(CosNF_MappingFilter_ptr prio_filter
					  WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(prio_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _pfilter = WRAPPED_DUPLICATE(CosNF_MappingFilter, prio_filter);
}

CosNF_MappingFilter_ptr ProxyPushSupplier_i::lifetime_filter( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_lfilter) )
	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _lfilter);
}

void ProxyPushSupplier_i::lifetime_filter(CosNF_MappingFilter_ptr life_filter
				          WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(life_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _lfilter = WRAPPED_DUPLICATE(CosNF_MappingFilter, life_filter); 
}

CosN_EventTypeSeq* 
ProxyPushSupplier_i::obtain_offered_types(CosNA_ObtainInfoMode mode WRAPPED_IMPLARG)
{
  CosN_EventTypeSeq* etypeseq=0;
  if ( mode == CosNA_ALL_NOW_UPDATES_OFF ) {
	etypeseq = _channel->obtain_offered_types();
	_oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_ALL_NOW_UPDATES_ON ) {
	etypeseq = _channel->obtain_offered_types();
	_oplock.lock(); _offauto = 0; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_OFF ) {
	etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
	_oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_ON ) {
	etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
	_oplock.lock(); _offauto = 0; _oplock.unlock();
  }
  return etypeseq;
}

void ProxyPushSupplier_i::subscription_change(
				const CosN_EventTypeSeq& added,
                        	const CosN_EventTypeSeq& deled   WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq nadded;
  CosN_EventTypeSeq ndeled;
  CORBA::Boolean i_am_consumer = 1;
  CORBA::ULong   ix=0;
  if ( ! RDI_EventType::valid_sequence(added, ix, i_am_consumer) )
	throw CosNC_InvalidEventType(added[ix]);
  if ( ! RDI_EventType::valid_sequence(deled, ix, i_am_consumer) )
	throw CosNC_InvalidEventType(deled[ix]);
  // The consumer can invoke this operation as long as it does not
  // use filter objects -- CORBA Notification Specification 2.6.4.
  if ( _fa_helper.has_filters() ) 
	throw CORBA::BAD_OPERATION(0, CORBA::COMPLETED_NO);
  // If this is not the first time this operation gets invoked, we
  // need to compute the actual added and deleted event types that
  // will be used to update the TypeMap
  if ( _rqstypes.length() ) {
	RDI_EventType::compute_diff(_rqstypes, added, deled, nadded, ndeled);
	if ( nadded.length() || ndeled.length() ) {
		(void) _channel->update_mapping(nadded, ndeled, this, 0);
		RDI_EventType::update(_rqstypes, added, deled);
	}
  } else {
        ndeled.length(0); _rqstypes = added; 
        RDI_EventType::update(_rqstypes, added, deled);
        (void) _channel->update_mapping(_rqstypes, ndeled, this, 0);
  }
}

CosNF_FilterID 
ProxyPushSupplier_i::add_filter(CosNF_Filter_ptr filter
				WRAPPED_IMPLARG )
{
  _oplock.lock();
  // If we have registered interest in specific event types, using
  // 'subscription_change()', we have to cancel it at this point..
  if ( _rqstypes.length() != 0 ) {
	CosN_EventTypeSeq added; added.length(0);
	_channel->update_mapping(added, _rqstypes, this, 0);
	_rqstypes.length(0);
  }
  _oplock.unlock();
  // The FilterAdmin_i object maintains its own latch and,  hence,
  // do not need to hold the latch during the following operation.
  return _fa_helper.add_filter_i(filter, (RDINotifySubscribe_ptr) this);
}
 
void ProxyPushSupplier_i::subscription_change_i(
				const CosN_EventTypeSeq& added,
				const CosN_EventTypeSeq& deled,
				      Filter_i*          flter)
{ (void) _channel->update_mapping(added, deled, this, flter); }

void ProxyPushSupplier_i::filter_destroy_i(Filter_i* filter)
{ _fa_helper.rem_filter_i(filter); }

CosN_QoSProperties* ProxyPushSupplier_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_S_AnyPRX);
}

void ProxyPushSupplier_i::set_qos(const CosN_QoSProperties& r_qos
				  WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_S_AnyPRX,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos);
}

void ProxyPushSupplier_i::validate_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_S_AnyPRX,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void ProxyPushSupplier_i::validate_event_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
	WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void ProxyPushSupplier_i::disconnect_push_supplier( WRAPPED_IMPLARG_VOID )
{
  // Need to remove from admin hash table first to avoid potential
  // deadlock between this call and some add_event() call from one
  // of the event matching/dispatch threads
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void ProxyPushSupplier_i::connect_any_push_consumer(
		CosEventComm::PushConsumer_ptr consumer WRAPPED_IMPLARG )
{
  CosEventComm::PushConsumer_var evcons;
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(consumer) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  if ( _pxstate != RDI_NotConnected ) 
	throw CosEventChannelAdmin::AlreadyConnected();

  // Find if the given reference corresponds to a CosEventComm
  // consumer -- this consumer does not support offer_change()

  evcons    = CosEventComm::PushConsumer::_narrow(consumer);
  _cosevnt  = CORBA::is_nil(evcons) ? 0 : 1;
  _consumer = WRAPPED_DUPLICATE(CosEventComm::PushConsumer, consumer);
  _active   = 1;
  _pxstate  = RDI_Connected;
}

void ProxyPushSupplier_i::suspend_connection( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected )
	throw CosNA_NotConnected();
  if ( ! _active ) 
	throw CosNA_ConnectionAlreadyInactive();
  _active = 0;
}

void ProxyPushSupplier_i::resume_connection( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected )
	throw CosNA_NotConnected();
  if ( _active ) 
	throw CosNA_ConnectionAlreadyActive();
  _active = 1;
  if ( _worker && _qempty ) 	// Notify worker thread
	_qempty->signal();
}

void ProxyPushSupplier_i::_push_event()
{
  RDI_StructuredEvent* event=0;
  unsigned int         qsize=0;

  while ( 1 ) {
        _oplock.lock();
	while ( (_pxstate==RDI_NotConnected) ||
		((_pxstate==RDI_Connected)&&(!_active || !_ntfqueue.length())) )
		_qempty->wait();
        if ( _pxstate != RDI_Connected ) {
                _oplock.unlock();
		RDI_DUMP("Push thread for proxy " << this << " exits");
		omni_thread::exit();
        }
	event = _ntfqueue.get_head();
	qsize = _ntfqueue.length() - 1;
        _ntfqueue.remove_head();
	_nevents += 1;
	// We are not holding the lock while pushing events to a consumer
	// to avoid blocking the thread adding new events to to our queue
        _oplock.unlock();

	if ( RDI::lockEvent() ) event->mylock().lock();
        try {
		const CosN_StructuredEvent& sev = event->get_cos_event();
		if ( strcmp(event->get_type_name(), "%ANY") == 0 ) 
			_consumer->push(sev.remainder_of_body);
		else {
			CORBA::Any any_event;
			any_event <<= sev;
			_consumer->push(any_event);
		}
                _channel->incr_num_notifications(qsize);
        } catch (...) {
		_pxstate = RDI_Exception;
        }
	if ( RDI::lockEvent() ) event->mylock().unlock();
	event->decr_ref_counter(RDI::lockEvent());
	omni_thread::yield();
  }
}

void ProxyPushSupplier_i::push_event(CORBA::Boolean& invalid)
{
  RDI_StructuredEvent* event=0;
  unsigned int         qsize=0;

  invalid = 0;
  _oplock.lock();
  if ( (_pxstate != RDI_Connected) || !_active || !_ntfqueue.length() ) {
	_oplock.unlock();
	return;
  }
  event = _ntfqueue.get_head();
  qsize = _ntfqueue.length() - 1;
  _ntfqueue.remove_head();
  _nevents += 1;
  // We are not holding the lock while pushing events to a consumer
  // to avoid blocking the thread adding new events to to our queue
  _oplock.unlock();

  if ( RDI::lockEvent() ) event->mylock().lock();
  try {
	const CosN_StructuredEvent& sev = event->get_cos_event();
	if ( strcmp(event->get_type_name(), "%ANY") == 0 ) 
		_consumer->push(sev.remainder_of_body);
	else {
		CORBA::Any any_event;
                any_event <<= sev;
                _consumer->push(any_event);
        }
	_channel->incr_num_notifications(qsize);
  } catch (...) {
	invalid  = 1;
	_pxstate = RDI_Exception;
  }
  if ( RDI::lockEvent() ) event->mylock().unlock();
  event->decr_ref_counter(RDI::lockEvent());
}

void ProxyPushSupplier_i::add_event(RDI_StructuredEvent* entry WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( (_pxstate == RDI_Connected) && entry ) {
	// Check if the 'MaxEventsPerConsumer' threshold has been reached.
	// In this case, we discard events based on 'DiscardPolicy' -- the
	// current implementation assumes FIFO 
	RDI_NotifQoS* qospr = (_qosprop?_qosprop:_myadmin->qos_properties());	
	unsigned int  msize = (unsigned int) qospr->maxEventsPerConsumer;
	if ( msize && (_ntfqueue.length() >= msize) ) {
		RDI_StructuredEvent* evnt = _ntfqueue.get_head();
		_ntfqueue.remove_head();
		evnt->decr_ref_counter(RDI::lockEvent());
	}
	entry->incr_ref_counter(RDI::lockEvent());
        _ntfqueue.insert_tail(entry);
	if ( _worker && _qempty ) 
		_qempty->signal();
  }
}

ostream& operator << (ostream& out, const ProxyPushSupplier_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype) <<
	 " ID " << setw(3) << prx._pserial;
  if ( prx._cosevnt ) out << " CosEventComm Consumer";
  if ( ! CORBA::is_nil(prx._pfilter) ) out << " PFilter " << prx._pfilter;
  if ( ! CORBA::is_nil(prx._lfilter) ) out << " LFilter " << prx._lfilter;
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " NotConnected "; break;
     case RDI_Connected:    out << " Connected    "; break;
     case RDI_Disconnected: out << " Disconnected "; break;
     case RDI_Exception:    out << " Exception    "; break;
  }
  out << "QSize "<< prx._ntfqueue.length() << " #Push "<< prx._nevents;
  if ( prx._rqstypes.length() != 0 ) {
     for (CORBA::ULong ix=0; ix < prx._rqstypes.length(); ix++) {
	out << endl << "\t" << prx._rqstypes[ix].domain_name;
	out << "::" << prx._rqstypes[ix].type_name;
     }
  }
  return out;
}

void ProxyPushSupplier_i::disconnect_client_and_dispose()
{
  RDI_StructuredEvent* event=0;
  _oplock.lock();
  _pxstate = RDI_Disconnected;
  _active  = 0;

  if ( _rqstypes.length() == 0 ) { 
        remove_all_filters();
  } else {
        CosN_EventTypeSeq added; added.length(0);
        (void) _channel->update_mapping(added, _rqstypes, this, 0);
  }

  // If we are using our own thread for dispatching, signal the
  // thread that it should terminate itself
  if ( _worker && _qempty ) {
        _qempty->signal();
        _oplock.unlock();
	_worker->join(0);
	_worker = 0;
	// When using an unbound thread,  we should wait till it exits
	// to avoid getting core dumps due to invalid memory accesses.
        _oplock.lock();
        delete _qempty;
	_qempty = 0;
  }
  if ( ! CORBA::is_nil(_consumer) ) {
        // WRAPPED_RELEASE(CosEventComm::PushConsumer, _consumer);
        _consumer = CosEventComm::PushConsumer::_nil();
  }
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
	_ntfqueue.remove_head();
	event->decr_ref_counter(RDI::lockEvent());
  }
  if ( _qosprop ) 
	delete _qosprop;

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}

// ------------------------------------------------------------- //
// StructuredProxyPushSupplier_i                                 //
// ------------------------------------------------------------- //

StructuredProxyPushSupplier_i::StructuredProxyPushSupplier_i(
					ConsumerAdmin_i* admin, 
					EventChannel_i*  chann,
					const CosNA_ProxyID&  prxid) : 
                RDINotifySubscribe(),
		_oplock(), _fa_helper(), _channel(chann), _myadmin(admin), 
                _prxtype(CosNA_PUSH_STRUCTURED),
                _pserial(prxid), _nevents(0), _pxstate(RDI_NotConnected), 
		_active(0), _qempty(0), _worker(0), _qosprop(0),
		_offauto(1), _rqstypes(),  _ntfqueue(128, 128)
{
  _rqstypes.length(0);
  _pfilter  = CosNF_MappingFilter::_nil();
  _lfilter  = CosNF_MappingFilter::_nil();
  _consumer = CosNC_StructuredPushConsumer::_nil();
  // When the number of push threads allocated at the channel level is
  // 0, each proxy uses its own thread to push events to its consumer.
  if (  _channel->push_threads() == 0 ) {
     _qempty = new omni_condition(&_oplock);
     // _worker = new StrProxyWorker(this,&StructuredProxyPushSupplier_i::_push_event);
     _worker = new StrProxyBoundWorker(this,&StructuredProxyPushSupplier_i::_push_event);
     // If we use unbound threads, we should start the thread using
     // _worker->start();
     RDI_DUMP("PushSupplier Proxy Thread " << _worker->id());
  }
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_ConsumerAdmin_ptr StructuredProxyPushSupplier_i::MyAdmin(WRAPPED_IMPLARG_VOID)
{ return WRAPPED_IMPL2OREF(CosNA_ConsumerAdmin, _myadmin); }

CosNF_MappingFilter_ptr 
StructuredProxyPushSupplier_i::priority_filter( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_pfilter) )
  	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _pfilter);
}

void StructuredProxyPushSupplier_i::priority_filter( 
			CosNF_MappingFilter_ptr prio_filter WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(prio_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _pfilter = WRAPPED_DUPLICATE(CosNF_MappingFilter, prio_filter);
}

CosNF_MappingFilter_ptr 
StructuredProxyPushSupplier_i::lifetime_filter( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_lfilter) )
	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _lfilter);
}

void StructuredProxyPushSupplier_i::lifetime_filter(
			CosNF_MappingFilter_ptr life_filter WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(life_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _lfilter = WRAPPED_DUPLICATE(CosNF_MappingFilter, life_filter); 
}

CosN_EventTypeSeq* 
StructuredProxyPushSupplier_i::obtain_offered_types(CosNA_ObtainInfoMode mode
						    WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq* etypeseq=0;
  if ( mode == CosNA_ALL_NOW_UPDATES_OFF ) {
        etypeseq = _channel->obtain_offered_types();
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_ALL_NOW_UPDATES_ON ) {
        etypeseq = _channel->obtain_offered_types();
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_OFF ) {
	etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_ON ) {
	etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  }
  return etypeseq;
}

void StructuredProxyPushSupplier_i::subscription_change(
                                const CosN_EventTypeSeq& added,
                                const CosN_EventTypeSeq& deled
			        WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq nadded;
  CosN_EventTypeSeq ndeled;
  CORBA::Boolean i_am_consumer = 1;
  CORBA::ULong   ix=0;
  if ( ! RDI_EventType::valid_sequence(added, ix, i_am_consumer) )
        throw CosNC_InvalidEventType(added[ix]);
  if ( ! RDI_EventType::valid_sequence(deled, ix, i_am_consumer) )
        throw CosNC_InvalidEventType(deled[ix]);
  // The consumer can invoke this operation as long as it does not
  // use filter objects -- CORBA Notification Specification 2.6.4.
  if ( _fa_helper.has_filters() )
        throw CORBA::BAD_OPERATION(0, CORBA::COMPLETED_NO);
  // If this is not the first time this operation gets invoked, we
  // need to compute the actual added and deleted event types that
  // will be used to update the TypeMap
  if ( _rqstypes.length() ) {
        RDI_EventType::compute_diff(_rqstypes, added, deled, nadded, ndeled);
        if ( nadded.length() || ndeled.length() ) {
                (void) _channel->update_mapping(nadded, ndeled, this, 0);
                RDI_EventType::update(_rqstypes, added, deled);
        }
  } else {
        ndeled.length(0); _rqstypes = added; 
        RDI_EventType::update(_rqstypes, added, deled);
        (void) _channel->update_mapping(_rqstypes, ndeled, this, 0);
  }
}

CosNF_FilterID 
StructuredProxyPushSupplier_i::add_filter(CosNF_Filter_ptr filter
					  WRAPPED_IMPLARG )
{
  _oplock.lock();
  // If we have registered interest in specific event types, using
  // 'subscription_change()', we have to cancel it at this point..
  if ( _rqstypes.length() != 0 ) { 
        CosN_EventTypeSeq added; added.length(0);
        _channel->update_mapping(added, _rqstypes, this, 0);        
	_rqstypes.length(0);
  }
  _oplock.unlock();
  // The FilterAdmin_i object maintains its own latch and,  hence, 
  // do not need to hold the latch during the following operation.
  return _fa_helper.add_filter_i(filter, (RDINotifySubscribe_ptr) this);
}

void StructuredProxyPushSupplier_i::subscription_change_i(
				const CosN_EventTypeSeq& added,
                                const CosN_EventTypeSeq& deled,
				      Filter_i*                      flter)
{ (void) _channel->update_mapping(added, deled, this, flter); }

void StructuredProxyPushSupplier_i::filter_destroy_i(Filter_i* filter)
{ _fa_helper.rem_filter_i(filter); }

CosN_QoSProperties* 
StructuredProxyPushSupplier_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_S_StrPRX);
}

void StructuredProxyPushSupplier_i::set_qos(
		const CosN_QoSProperties& r_qos WRAPPED_IMPLARG)
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_S_StrPRX,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos);
}

void StructuredProxyPushSupplier_i::validate_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_S_StrPRX,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void StructuredProxyPushSupplier_i::validate_event_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void StructuredProxyPushSupplier_i::disconnect_structured_push_supplier( WRAPPED_IMPLARG_VOID )
{
  // Need to remove from admin hash table first to avoid potential 
  // deadlock between this call and some add_event() call from one
  // of the event matching/dispatch threads
  if ( _pxstate == RDI_Disconnected )
   	return;
  RDI_DUMP("Consumer requested to be disconnected from proxy " << this);
  _myadmin->remove_proxy(this);
  RDI_DUMP("  - cleanup and dispose proxy " << this);
  disconnect_client_and_dispose();
}

void StructuredProxyPushSupplier_i::connect_structured_push_consumer(
	CosNC_StructuredPushConsumer_ptr consumer WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(consumer) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  if ( _pxstate != RDI_NotConnected ) 
	throw CosEventChannelAdmin::AlreadyConnected();
  _active  = 1;
  _pxstate = RDI_Connected;
  _consumer= WRAPPED_DUPLICATE(CosNC_StructuredPushConsumer, consumer);
}

void StructuredProxyPushSupplier_i::suspend_connection( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected )
        throw CosNA_NotConnected();
  if ( ! _active ) 
        throw CosNA_ConnectionAlreadyInactive();
  _active = 0;
}

void StructuredProxyPushSupplier_i::resume_connection( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected )
        throw CosNA_NotConnected();
  if ( _active ) 
        throw CosNA_ConnectionAlreadyActive();
  _active = 1;
  if ( _worker && _qempty ) 
	_qempty->signal();
}

void StructuredProxyPushSupplier_i::_push_event()
{
  RDI_StructuredEvent* event=0;
  unsigned int         qsize=0;

  while ( 1 ) {
        _oplock.lock();
	while ( (_pxstate==RDI_NotConnected) ||
		((_pxstate==RDI_Connected)&&(!_active || !_ntfqueue.length())) )
		_qempty->wait();
	if ( _pxstate != RDI_Connected ) {
		_oplock.unlock();
  		RDI_DUMP("     - Push thread for proxy " << this << " exits");
		omni_thread::exit();
	}
	event = _ntfqueue.get_head();
	qsize = _ntfqueue.length() - 1;
	_ntfqueue.remove_head();
	_nevents += 1;
	// We are not holding the lock while pushing events to a consumer
	// to avoid blocking the thread adding new events to to our queue
	_oplock.unlock();

	if ( RDI::lockEvent() ) event->mylock().lock();
	try {
		_consumer->push_structured_event(event->get_cos_event());
		_channel->incr_num_notifications(qsize);
	} catch ( CORBA::INV_OBJREF& e ) {
            	RDI_DUMP("Invalid object reference");
            	_pxstate = RDI_Exception;
  	} catch ( CORBA::OBJECT_NOT_EXIST& e ) {
           	 RDI_DUMP("Consumer object does not exist!!!!");
            	_pxstate = RDI_Exception;
  	} catch ( CORBA::COMM_FAILURE& e ) {
            	RDI_DUMP("Communication Failure");
            	_pxstate = RDI_Exception;
  	} catch (...) {
           	 RDI_DUMP("Exception while pushing event to consumer");
            	_pxstate = RDI_Exception;
  	}

	if ( RDI::lockEvent() ) event->mylock().unlock();
	event->decr_ref_counter(RDI::lockEvent());
	omni_thread::yield();
  }
}

void StructuredProxyPushSupplier_i::push_event(CORBA::Boolean& invalid)
{
  RDI_StructuredEvent* event=0;;
  unsigned int         qsize=0;

  invalid = 0;
  _oplock.lock();
  if ( (_pxstate != RDI_Connected) || !_active || !_ntfqueue.length() ) {
	_oplock.unlock();
	return;
  }
  event = _ntfqueue.get_head();
  qsize = _ntfqueue.length() - 1;
  _ntfqueue.remove_head();
  _nevents += 1;
  // We are not holding the lock while pushing events to a consumer
  // to avoid blocking the thread adding new events to to our queue
  _oplock.unlock();
  
  if ( RDI::lockEvent() ) event->mylock().lock();
  try {
      	_consumer->push_structured_event(event->get_cos_event());
	_channel->incr_num_notifications(qsize);
  } catch ( CORBA::INV_OBJREF& e ) {
	RDI_DUMP("Invalid object reference");
	invalid = 1; _pxstate = RDI_Exception;
  } catch ( CORBA::OBJECT_NOT_EXIST& e ) {
	RDI_DUMP("Consumer object does not exist!!!!");
	invalid = 1; _pxstate = RDI_Exception;
  } catch ( CORBA::COMM_FAILURE& e ) {
	RDI_DUMP("Communication Failure");
	invalid = 1; _pxstate = RDI_Exception;
  } catch (...) {
	RDI_DUMP("Exception while pushing event to consumer");
	invalid = 1; _pxstate = RDI_Exception;
  }
  if ( RDI::lockEvent() ) event->mylock().unlock();
  event->decr_ref_counter(RDI::lockEvent());
}

void StructuredProxyPushSupplier_i::add_event(RDI_StructuredEvent* entry
					      WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( (_pxstate == RDI_Connected) && entry ) {
	// Check if the 'MaxEventsPerConsumer' threshold has been reached.
	// In this case, we discard events based on 'DiscardPolicy' -- the
	// current implementation assumes FIFO 
	RDI_NotifQoS* qospr = (_qosprop?_qosprop:_myadmin->qos_properties());	
	unsigned int  msize = (unsigned int) qospr->maxEventsPerConsumer;
	if ( msize && (_ntfqueue.length() >= msize) ) {
		RDI_StructuredEvent* evnt = _ntfqueue.get_head();
		_ntfqueue.remove_head();
		evnt->decr_ref_counter(RDI::lockEvent());
	}
	entry->incr_ref_counter(RDI::lockEvent());
        _ntfqueue.insert_tail(entry);
	if ( _worker && _qempty ) 
		_qempty->signal();
  }
}

ostream& operator << (ostream& out, const StructuredProxyPushSupplier_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype) <<
         " ID " << setw(3) << prx._pserial;
  if ( ! CORBA::is_nil(prx._pfilter) ) out << " PFilter " << prx._pfilter;
  if ( ! CORBA::is_nil(prx._lfilter) ) out << " LFilter " << prx._lfilter;
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " NotConnected "; break;
     case RDI_Connected:    out << " Connected    "; break;
     case RDI_Disconnected: out << " Disconnected "; break;
     case RDI_Exception:    out << " Exception    "; break;
  }
  out << "QSize "<< prx._ntfqueue.length() << " #Push "<< prx._nevents;
  if ( prx._rqstypes.length() != 0 ) {
     for (CORBA::ULong ix=0; ix < prx._rqstypes.length(); ix++) {
        out << endl << "\t" << prx._rqstypes[ix].domain_name; 
        out << "::" << prx._rqstypes[ix].type_name;
     }
  }
  return out;
}

void StructuredProxyPushSupplier_i::disconnect_client_and_dispose()
{
  RDI_StructuredEvent* event=0;

  _pxstate = RDI_Disconnected;
  _active  = 0;
  _oplock.lock();
  if ( _rqstypes.length() == 0 ) { 
        remove_all_filters();
  } else {
        CosN_EventTypeSeq added; added.length(0);
        (void) _channel->update_mapping(added, _rqstypes, this, 0);
  }

  // If we are using our own thread for dispatching, signal the
  // thread that it should terminate itself and wait for 5 secs
  if ( _worker && _qempty ) {
  	RDI_DUMP("  - proxy " << this << " signal push thread to exit");
        _qempty->signal();
        _oplock.unlock();
	_worker->join(0);
        _worker = 0;
        // When using an unbound thread,  we should wait till it exits
        // to avoid getting core dumps due to invalid memory accesses.
        _oplock.lock();
        delete _qempty;
	_qempty = 0;
  }
  if ( ! CORBA::is_nil(_consumer) ) {
        // WRAPPED_RELEASE(CosNC_StructuredPushConsumer, _consumer);
        _consumer =  CosNC_StructuredPushConsumer::_nil();
  }
  RDI_DUMP("  - proxy " << this << " empty notification queue");
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
        _ntfqueue.remove_head();
        event->decr_ref_counter(RDI::lockEvent());
  }
  if ( _qosprop )
	delete _qosprop;

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  RDI_DUMP("  - proxy " << this << " is being disposed");
  WRAPPED_DISPOSE_IMPL(this);
}

// ------------------------------------------------------------- //
// SequenceProxyPushSupplier_i                                   //
// ------------------------------------------------------------- //

SequenceProxyPushSupplier_i::SequenceProxyPushSupplier_i(
					ConsumerAdmin_i* admin, 
				 	EventChannel_i*  chann,
					const CosNA_ProxyID&  prxid) : 
                RDINotifySubscribe(),
		_oplock(), _fa_helper(),_channel(chann), _myadmin(admin), 
                _prxtype(CosNA_PUSH_SEQUENCE),
                _pserial(prxid), _nevents(0), _pxstate(RDI_NotConnected), 
		_active(0), _qempty(0), _worker(0), _qosprop(0),
		_offauto(1), _rqstypes(), _ntfqueue(128, 128)
{
  _rqstypes.length(0);
  _pfilter  = CosNF_MappingFilter::_nil();
  _lfilter  = CosNF_MappingFilter::_nil();
  _consumer = CosNC_SequencePushConsumer::_nil();
  // When the number of push threads allocated at the channel level is
  // 0, each proxy uses its own thread to push events to its consumer.
  if (  _channel->push_threads() == 0 ) {
     _qempty = new omni_condition(&_oplock);
     // _worker = new SeqProxyWorker(this,&SequenceProxyPushSupplier_i::_push_event);
     _worker = new SeqProxyBoundWorker(this,&SequenceProxyPushSupplier_i::_push_event);
     // If we use unbound threads, we should start the thread using
     // _worker->start();
     RDI_DUMP("PushSupplier Proxy Thread " << _worker->id());
  }
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_ConsumerAdmin_ptr SequenceProxyPushSupplier_i::MyAdmin( WRAPPED_IMPLARG_VOID )
{ return WRAPPED_IMPL2OREF(CosNA_ConsumerAdmin, _myadmin); }

CosNF_MappingFilter_ptr 
SequenceProxyPushSupplier_i::priority_filter( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_pfilter) )
  	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _pfilter);
}

void SequenceProxyPushSupplier_i::priority_filter(
			CosNF_MappingFilter_ptr prio_filter WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(prio_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _pfilter = WRAPPED_DUPLICATE(CosNF_MappingFilter, prio_filter);
}

CosNF_MappingFilter_ptr 
SequenceProxyPushSupplier_i::lifetime_filter( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_lfilter) )
	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _lfilter);
}

void SequenceProxyPushSupplier_i::lifetime_filter(
			CosNF_MappingFilter_ptr life_filter WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(life_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _lfilter = WRAPPED_DUPLICATE(CosNF_MappingFilter, life_filter); 
}

CosN_EventTypeSeq* 
SequenceProxyPushSupplier_i::obtain_offered_types(CosNA_ObtainInfoMode mode
						  WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq* etypeseq=0;
  if ( mode == CosNA_ALL_NOW_UPDATES_OFF ) {
        etypeseq = _channel->obtain_offered_types();
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_ALL_NOW_UPDATES_ON ) {
        etypeseq = _channel->obtain_offered_types();
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_OFF ) {
	etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_ON ) {
	etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  }
  return etypeseq;
}

void SequenceProxyPushSupplier_i::subscription_change(
                                const CosN_EventTypeSeq& added,
                                const CosN_EventTypeSeq& deled
				WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq nadded;
  CosN_EventTypeSeq ndeled;
  CORBA::Boolean i_am_consumer = 1;
  CORBA::ULong   ix=0;
  if ( ! RDI_EventType::valid_sequence(added, ix, i_am_consumer) )
        throw CosNC_InvalidEventType(added[ix]);
  if ( ! RDI_EventType::valid_sequence(deled, ix, i_am_consumer) )
        throw CosNC_InvalidEventType(deled[ix]);
  // The consumer can invoke this operation as long as it does not
  // use filter objects -- CORBA Notification Specification 2.6.4.
  if ( _fa_helper.has_filters() )
        throw CORBA::BAD_OPERATION(0, CORBA::COMPLETED_NO);
  // If this is not the first time this operation gets invoked, we
  // need to compute the actual added and deleted event types that
  // will be used to update the TypeMap
  if ( _rqstypes.length() ) {
        RDI_EventType::compute_diff(_rqstypes, added, deled, nadded, ndeled);
        if ( nadded.length() || ndeled.length() ) {
                (void) _channel->update_mapping(nadded, ndeled, this, 0);
                RDI_EventType::update(_rqstypes, added, deled);
        }
  } else {
        ndeled.length(0); _rqstypes = added; 
        RDI_EventType::update(_rqstypes, added, deled);
        (void) _channel->update_mapping(_rqstypes, ndeled, this, 0);
  }
}

CosNF_FilterID 
SequenceProxyPushSupplier_i::add_filter(CosNF_Filter_ptr filter
					WRAPPED_IMPLARG )
{
  _oplock.lock();
  // If we have registered interest in specific event types, using
  // 'subscription_change()', we have to cancel it at this point..
  if ( _rqstypes.length() != 0 ) { 
        CosN_EventTypeSeq added; added.length(0);
        _channel->update_mapping(added, _rqstypes, this, 0);        
	_rqstypes.length(0);
  }
  _oplock.unlock();
  // The FilterAdmin_i object maintains its own latch and,  hence, 
  // do not need to hold the latch during the following operation.
  return _fa_helper.add_filter_i(filter, (RDINotifySubscribe_ptr) this);
}

void SequenceProxyPushSupplier_i::subscription_change_i(
				const CosN_EventTypeSeq& added,
                                const CosN_EventTypeSeq& deled,
				      Filter_i*                      flter)
{ (void) _channel->update_mapping(added, deled, this, flter); }

void SequenceProxyPushSupplier_i::filter_destroy_i(Filter_i* filter)
{ _fa_helper.rem_filter_i(filter); }

CosN_QoSProperties* 
SequenceProxyPushSupplier_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_S_SeqPRX);
}

void SequenceProxyPushSupplier_i::set_qos(
		const CosN_QoSProperties& r_qos WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_S_SeqPRX,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos);
}

void SequenceProxyPushSupplier_i::validate_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_S_SeqPRX,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void SequenceProxyPushSupplier_i::validate_event_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void SequenceProxyPushSupplier_i::disconnect_sequence_push_supplier( WRAPPED_IMPLARG_VOID )
{
  // Need to remove from admin hash table first to avoid potential
  // deadlock between this call and some add_event() call from one 
  // of the event matching/dispatch threads
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void SequenceProxyPushSupplier_i::connect_sequence_push_consumer(
	CosNC_SequencePushConsumer_ptr consumer WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(consumer) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  if ( _pxstate != RDI_NotConnected )
	throw CosEventChannelAdmin::AlreadyConnected();
  _active   = 1;
  _pxstate  = RDI_Connected;
  _consumer = WRAPPED_DUPLICATE(CosNC_SequencePushConsumer, consumer);
}

void SequenceProxyPushSupplier_i::suspend_connection( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected )
        throw CosNA_NotConnected();
  if ( ! _active ) 
        throw CosNA_ConnectionAlreadyInactive();
  _active = 0;
}

void SequenceProxyPushSupplier_i::resume_connection( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected )
        throw CosNA_NotConnected();
  if ( _active ) 
        throw CosNA_ConnectionAlreadyActive();
  _active = 1;
  if ( _worker && _qempty ) 
	_qempty->signal();
}

void SequenceProxyPushSupplier_i::_push_event()
{
  RDI_NotifQoS* qospr = (_qosprop ? _qosprop : _myadmin->qos_properties());
  CORBA::Long   i, bsize = qospr->maximumBatchSize;
  CosN_EventBatch notif(bsize);
  RDI_StructuredEvent** event = new RDI_StructuredEvent*[bsize];
  unsigned int qsize;
  RDI_Assert(event, "Memory allocation failed -- RDI_StructuredEvent");

  while ( 1 ) {
	_oplock.lock();
	while ( (_pxstate==RDI_NotConnected) ||
	        ((_pxstate==RDI_Connected)&&(!_active || !_ntfqueue.length())) )
		_qempty->wait();
	if ( _pxstate != RDI_Connected ) {
		_oplock.unlock();
		delete [] event;
		RDI_DUMP("Push thread for proxy " << this << " exits");
		omni_thread::exit();
	}
	qsize = _ntfqueue.length() - 1;
	for ( i=0; i < bsize; i++ ) {
		event[i] = _ntfqueue.get_head();
		_ntfqueue.remove_head();
		notif[i] = event[i]->get_cos_event();
	}
	_nevents += bsize;
	_oplock.unlock();
	// We are not holding the lock while pushing events to a consumer
	// to avoid blocking the thread adding new events to to our queue
	try {
        	_consumer->push_structured_events(notif);
		for ( i=0; i < bsize; i++ ) {
			_channel->incr_num_notifications(qsize--);
			event[i]->decr_ref_counter(RDI::lockEvent());
			event[i] = 0;
		}
		omni_thread::yield();
	} catch ( ... ) {
		for ( i=0; i < bsize; i++ )
			event[i]->decr_ref_counter(RDI::lockEvent());
		_pxstate = RDI_Exception;
	}
  }
}

void SequenceProxyPushSupplier_i::push_event(CORBA::Boolean& invalid)
{
  RDI_NotifQoS* qospr = (_qosprop ? _qosprop : _myadmin->qos_properties());
  CORBA::Long   i, bsize = qospr->maximumBatchSize;
  CosN_EventBatch notif(bsize);
  RDI_StructuredEvent** event = 0;
  unsigned int qsize;

  invalid = 0;
  _oplock.lock();

  if ( (_pxstate != RDI_Connected) || 
       !_active || ((CORBA::Long)_ntfqueue.length() < bsize) ) {
	_oplock.unlock();
	return;
  }

  event = new RDI_StructuredEvent * [bsize];
  RDI_Assert(event, "Memory allocation failed -- RDI_StructuredEvent");
  qsize = _ntfqueue.length() - 1;
  notif.length(bsize);

  for ( i=0; i < bsize; i++ ) {
	event[i] = _ntfqueue.get_head();
	_ntfqueue.remove_head();
	notif[i] = event[i]->get_cos_event();
  }
  _nevents += bsize;
  _oplock.unlock();
  // We are not holding the lock while pushing events to a consumer
  // to avoid blocking the thread adding new events to to our queue
  try {
	_consumer->push_structured_events(notif);
	for ( i=0; i < bsize; i++ ) {
		_channel->incr_num_notifications(qsize--);
		event[i]->decr_ref_counter(RDI::lockEvent());
	}
	delete [] event;
  } catch ( ... ) {
	for ( i=0; i < bsize; i++ )
		event[i]->decr_ref_counter(RDI::lockEvent());
	delete [] event;
	invalid  = 1;
	_pxstate = RDI_Exception;
  }
}

void SequenceProxyPushSupplier_i::add_event(RDI_StructuredEvent* entry
					    WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( (_pxstate == RDI_Connected) && entry ) {
	// Check if the 'MaxEventsPerConsumer' threshold has been reached.
	// In this case, we discard events based on 'DiscardPolicy' -- the
	// current implementation assumes FIFO 
	RDI_NotifQoS* qospr = (_qosprop?_qosprop:_myadmin->qos_properties());	
	unsigned int  msize = (unsigned int) qospr->maxEventsPerConsumer;
	if ( msize && (_ntfqueue.length() >= msize) ) {
		RDI_StructuredEvent* evnt = _ntfqueue.get_head();
		_ntfqueue.remove_head();
		evnt->decr_ref_counter(RDI::lockEvent());
	}
        entry->incr_ref_counter(RDI::lockEvent());
        _ntfqueue.insert_tail(entry);
  	if ( _worker && _qempty ) 
		_qempty->signal();
  }
}

ostream& operator << (ostream& out, const SequenceProxyPushSupplier_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype) <<
         " ID " << setw(3) << prx._pserial;
  if ( ! CORBA::is_nil(prx._pfilter) ) out << " PFilter " << prx._pfilter;
  if ( ! CORBA::is_nil(prx._lfilter) ) out << " LFilter " << prx._lfilter;
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " NotConnected "; break;
     case RDI_Connected:    out << " Connected    "; break;
     case RDI_Disconnected: out << " Disconnected "; break;
     case RDI_Exception:    out << " Exception    "; break;
  }
  out << "QSize "<< prx._ntfqueue.length() << " #Push "<< prx._nevents;
  if ( prx._rqstypes.length() != 0 ) {
     for (CORBA::ULong ix=0; ix < prx._rqstypes.length(); ix++) {
        out << endl << "\t" << prx._rqstypes[ix].domain_name; 
        out << "::" << prx._rqstypes[ix].type_name;
     }
  }
  return out;
}

void SequenceProxyPushSupplier_i::disconnect_client_and_dispose()
{
  RDI_StructuredEvent* event = 0;
  _oplock.lock();
  _pxstate = RDI_Disconnected;
  _active  = 0;

  if ( _rqstypes.length() == 0 ) { 
        remove_all_filters();
  } else {
        CosN_EventTypeSeq added; added.length(0);
        (void) _channel->update_mapping(added, _rqstypes, this, 0);
  }

  // If we are using our own thread for dispatching, signal the
  // thread that it should terminate itself and wait for 5 secs
  if ( _worker && _qempty ) {
        _qempty->signal();
        _oplock.unlock();
	_worker->join(0);
        _worker = 0;
        // When using an unbound thread,  we should wait till it exits
        // to avoid getting core dumps due to invalid memory accesses.
        _oplock.lock();
        delete _qempty;
	_qempty = 0;
  }
  if ( ! CORBA::is_nil(_consumer) ) {
        // WRAPPED_RELEASE(CosNC_SequencePushConsumer, _consumer);
        _consumer =  CosNC_SequencePushConsumer::_nil();
  }
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
        _ntfqueue.remove_head();
        event->decr_ref_counter(RDI::lockEvent());
  }
  if ( _qosprop )
	delete _qosprop;

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}
