// -*- Mode: C++; -*-
//                              File      : PullSupplier_i.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of PullSupplier proxies
//
 
/*
$Log: PullSupplier_i.cc,v $
Revision 1.39  2000/10/01 13:39:53  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.38.2.1  2000/09/30 15:39:52  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.38  2000/08/22 18:23:54  alcfp
added description to each file

Revision 1.37  2000/08/16 20:19:35  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <iomanip.h>
#include "RDI.h"
#include "RDIDebug.h"
#include "RDIEvent.h"

#include "CosNotify.h"

#include "CosNotifyChannelAdmin_i.h"

extern const char* RDI_PRX_TYPE(const CosNA_ProxyType& type);

// ------------------------------------------------------------- //
// ProxyPullSupplier_i                                           //
// ------------------------------------------------------------- //

ProxyPullSupplier_i::ProxyPullSupplier_i(ConsumerAdmin_i* admin, 
					 EventChannel_i*  chann,
					 const CosNA_ProxyID&  prxid) : 
		RDINotifySubscribe(), 
		_oplock(), _fa_helper(), _noempty(&_oplock), _channel(chann),
		_myadmin(admin), _prxtype(CosNA_PULL_ANY),
		_pserial(prxid), _cosevnt(0), _nevents(0), _blkpull(0),
		_pxstate(RDI_NotConnected), _qosprop(0), 
		_offauto(1), _rqstypes(), _ntfqueue(32,32)
{
  _rqstypes.length(0);
  _pfilter  = CosNF_MappingFilter::_nil();
  _lfilter  = CosNF_MappingFilter::_nil();
  _consumer = CosEventComm::PullConsumer::_nil();
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_ConsumerAdmin_ptr ProxyPullSupplier_i::MyAdmin( WRAPPED_IMPLARG_VOID )
{ 
  omni_mutex_lock lock(_oplock);
  return WRAPPED_IMPL2OREF(CosNA_ConsumerAdmin, _myadmin); 
}

CosNF_MappingFilter_ptr ProxyPullSupplier_i::priority_filter( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_pfilter) )
  	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _pfilter);
}

void ProxyPullSupplier_i::priority_filter(CosNF_MappingFilter_ptr prio_filter
				          WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(prio_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _pfilter = WRAPPED_DUPLICATE(CosNF_MappingFilter, prio_filter);
}

CosNF_MappingFilter_ptr ProxyPullSupplier_i::lifetime_filter( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_lfilter) )
	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _lfilter);
}

void ProxyPullSupplier_i::lifetime_filter(CosNF_MappingFilter_ptr life_filter
					  WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(life_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _lfilter = WRAPPED_DUPLICATE(CosNF_MappingFilter, life_filter); 
}

CosN_EventTypeSeq* 
ProxyPullSupplier_i::obtain_offered_types(CosNA_ObtainInfoMode mode WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq* etypeseq=0;
  if ( mode == CosNA_ALL_NOW_UPDATES_OFF ) {
        etypeseq = _channel->obtain_offered_types();
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_ALL_NOW_UPDATES_ON ) {
        etypeseq = _channel->obtain_offered_types();
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_OFF ) {
	etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_ON ) {
	etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  }
  return etypeseq;
}

CosN_QoSProperties* 
ProxyPullSupplier_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_S_AnyPRX);
}

void ProxyPullSupplier_i::set_qos(const CosN_QoSProperties& r_qos
			          WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_S_AnyPRX,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos);
}

void ProxyPullSupplier_i::validate_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_S_AnyPRX,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void ProxyPullSupplier_i::validate_event_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

CORBA::Any* ProxyPullSupplier_i::pull( WRAPPED_IMPLARG_VOID )
{
  _oplock.lock();
  _blkpull += 1;
  while ( (_pxstate == RDI_Connected) && (_ntfqueue.length() == 0) )
	_noempty.wait();
  if ( _pxstate != RDI_Connected ) {
	_blkpull -= 1;
	_oplock.unlock();
	throw CosEventComm::Disconnected();
  }

  RDI_StructuredEvent* event = _ntfqueue.get_head();
  const CosN_StructuredEvent& cosev = event->get_cos_event();
  _ntfqueue.remove_head();
  _nevents += 1;
  _blkpull -= 1;
  _oplock.unlock();

  if ( strcmp(event->get_type_name(), "%ANY") == 0 ) {
	return new CORBA::Any(cosev.remainder_of_body);
  } else {
	CORBA::Any* anyev = new CORBA::Any;
	(*anyev) <<= cosev;
	return anyev;
  }
}

CORBA::Any* 
ProxyPullSupplier_i::try_pull(CORBA::Boolean& has_event WRAPPED_IMPLARG)
{
  _oplock.lock();
  if ( _pxstate != RDI_Connected ) {
	_oplock.unlock();
	throw CosEventComm::Disconnected();
  } else if ( _ntfqueue.length() == 0 ) {
	_oplock.unlock();
	has_event = 0;
	return new CORBA::Any();
  } else {
	RDI_StructuredEvent* event = _ntfqueue.get_head();
        const CosN_StructuredEvent&      cosev = event->get_cos_event();
	_ntfqueue.remove_head();
	_nevents += 1;
	_oplock.unlock();
	has_event = 1;
	if ( strcmp(event->get_type_name(), "%ANY") == 0 ) {
		return new CORBA::Any(cosev.remainder_of_body);
	} else {
		CORBA::Any* anyev = new CORBA::Any;
		(*anyev) <<= cosev;
                return anyev;
	}
  }
}
  
void ProxyPullSupplier_i::disconnect_pull_supplier( WRAPPED_IMPLARG_VOID )
{
  // Need to remove from admin hash table first to avoid potential
  // deadlock between this call and some add_event() call from one
  // of the event matching/dispatch threads
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void ProxyPullSupplier_i::subscription_change(
                                const CosN_EventTypeSeq& added,
                                const CosN_EventTypeSeq& deled
				WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq nadded;
  CosN_EventTypeSeq ndeled;
  CORBA::Boolean i_am_consumer = 1;
  CORBA::ULong   ix=0;
  if ( ! RDI_EventType::valid_sequence(added, ix, i_am_consumer) )
        throw CosNC_InvalidEventType(added[ix]);
  if ( ! RDI_EventType::valid_sequence(deled, ix, i_am_consumer) )
        throw CosNC_InvalidEventType(deled[ix]);
  // The consumer can invoke this operation as long as it does not
  // use filter objects -- CORBA Notification Specification 2.6.4.
  if ( _fa_helper.has_filters() )
        throw CORBA::BAD_OPERATION(0, CORBA::COMPLETED_NO);
  // If this is not the first time this operation gets invoked, we
  // need to compute the actual added and deleted event types that
  // will be used to update the TypeMap
  if ( _rqstypes.length() ) {
        RDI_EventType::compute_diff(_rqstypes, added, deled, nadded, ndeled);
        if ( nadded.length() || ndeled.length() ) {
                (void) _channel->update_mapping(nadded, ndeled, this, 0);
                RDI_EventType::update(_rqstypes, added, deled);
        }
  } else {
	ndeled.length(0); _rqstypes = added; 
	RDI_EventType::update(_rqstypes, added, deled);
        (void) _channel->update_mapping(_rqstypes, ndeled, this, 0);
  }
}

CosNF_FilterID
ProxyPullSupplier_i::add_filter(CosNF_Filter_ptr filter
			        WRAPPED_IMPLARG )
{
  _oplock.lock();
  // If we have registered interest in specific event types, using
  // 'subscription_change()', we have to cancel it at this point..
  if ( _rqstypes.length() != 0 ) {
        CosN_EventTypeSeq added; added.length(0);
        _channel->update_mapping(added, _rqstypes, this, 0);
        _rqstypes.length(0);
  }
  _oplock.unlock();
  // The FAdminHelper object maintains its own latch and,  hence,
  // do not need to hold the latch during the following operation.
  return _fa_helper.add_filter_i(filter, (RDINotifySubscribe_ptr) this);
}

void ProxyPullSupplier_i::subscription_change_i(
				const CosN_EventTypeSeq& added,
				const CosN_EventTypeSeq& deled,
				      Filter_i*                      flter)
{ (void) _channel->update_mapping(added, deled, this, flter); }

void ProxyPullSupplier_i::filter_destroy_i(Filter_i* filter)
{ _fa_helper.rem_filter_i(filter); }

void ProxyPullSupplier_i::connect_any_pull_consumer(
			CosEventComm::PullConsumer_ptr consumer WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate == RDI_Connected )
	throw CosEventChannelAdmin::AlreadyConnected();
  // According to the CORBA Notification Service specification, pull
  // consumers can provide a _nil reference [2.6.1].  If this is the 
  // case, we will treat the consumer as CosEventComm::PullConsumer
  if ( ! CORBA::is_nil(consumer) ) {
  	CosEventComm::PullConsumer_var coscons;
  	coscons   = CosEventComm::PullConsumer::_narrow(consumer);
  	_cosevnt  = CORBA::is_nil(coscons) ? 0 : 1;
  	_consumer = WRAPPED_DUPLICATE(CosEventComm::PullConsumer, consumer);
  } else {
	_cosevnt = 1;
  }
  _pxstate = RDI_Connected;
}

void ProxyPullSupplier_i::add_event(RDI_StructuredEvent* entry WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( (_pxstate == RDI_Connected) && entry ) {
	// Check if the 'MaxEventsPerConsumer' threshold has been reached.
        // In this case, we discard events based on 'DiscardPolicy' -- the
        // current implementation assumes FIFO
        RDI_NotifQoS* qospr = (_qosprop?_qosprop:_myadmin->qos_properties());
        unsigned int  msize = (unsigned int) qospr->maxEventsPerConsumer;
        if ( msize && (_ntfqueue.length() >= msize) ) {
                RDI_StructuredEvent* evnt = _ntfqueue.get_head();
                _ntfqueue.remove_head();
                evnt->decr_ref_counter( RDI::lockEvent() );
        }
	entry->incr_ref_counter( RDI::lockEvent() );
	_ntfqueue.insert_tail(entry);
	_noempty.signal();
  }
}

ostream& operator << (ostream& out, const ProxyPullSupplier_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype);
  if ( prx._cosevnt ) out << " CosEventComm Consumer";
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " Not Connected "; break;
     case RDI_Connected:    out << " Connected     "; break;
     case RDI_Disconnected: out << " Disconnected  "; break;
     case RDI_Exception:    out << " Exception     "; break;
  }
  out << " QSize "<< prx._ntfqueue.length()<< " #Pull "<< prx._nevents;
  if ( prx._rqstypes.length() != 0 ) {
     for (CORBA::ULong ix=0; ix < prx._rqstypes.length(); ix++) {
        out << endl << "\t" << prx._rqstypes[ix].domain_name; 
        out << "::" << prx._rqstypes[ix].type_name;
     }
  }
  return out;
}

void ProxyPullSupplier_i::disconnect_client_and_dispose()
{
  RDI_StructuredEvent* event=0;
  _oplock.lock();
  _pxstate = RDI_Disconnected;

  if ( _rqstypes.length() == 0 ) {
	remove_all_filters();
  } else {
	CosN_EventTypeSeq added; added.length(0);
	(void) _channel->update_mapping(added, _rqstypes, this, 0);
  }

  // The consumer may be blocked on a pull() at this point(?)
  // and, thus, we should unblock it before self-destructing.
  _noempty.broadcast();
  _oplock.unlock();
  while ( _blkpull != 0 ) {
	_noempty.signal();
	omni_thread::yield();
  }
  _oplock.lock();

  if ( ! CORBA::is_nil(_consumer) ) {
        // WRAPPED_RELEASE(CosEventComm::PullConsumer, _consumer);
        _consumer = CosEventComm::PullConsumer::_nil();
  }
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
        _ntfqueue.remove_head();
	event->decr_ref_counter( RDI::lockEvent() );
  }
  if ( ! CORBA::is_nil(_pfilter) ) WRAPPED_RELEASE(_Filter, _pfilter);
  if ( ! CORBA::is_nil(_lfilter) ) WRAPPED_RELEASE(_FIlter, _lfilter);
  if ( ! _qosprop ) delete _qosprop;

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}

// ------------------------------------------------------------- //
// StructuredProxyPullSupplier_i                                 //
// ------------------------------------------------------------- //

StructuredProxyPullSupplier_i::StructuredProxyPullSupplier_i(
					ConsumerAdmin_i* admin, 
					EventChannel_i*  chann,
					const CosNA_ProxyID&  prxid) : 
		RDINotifySubscribe(), _oplock(), _fa_helper(),
		_noempty(&_oplock), _channel(chann), _myadmin(admin),
		_prxtype(CosNA_PULL_STRUCTURED), _pserial(prxid), 
		_nevents(0), _blkpull(0), _pxstate(RDI_NotConnected),
		_qosprop(0), _offauto(1), _rqstypes(), _ntfqueue(32,32)
{
  _rqstypes.length(0);
  _pfilter  = CosNF_MappingFilter::_nil();
  _lfilter  = CosNF_MappingFilter::_nil();
  _consumer = CosNC_StructuredPullConsumer::_nil();
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_ConsumerAdmin_ptr StructuredProxyPullSupplier_i::MyAdmin(WRAPPED_IMPLARG_VOID)
{ 
  omni_mutex_lock lock(_oplock);
  return WRAPPED_IMPL2OREF(CosNA_ConsumerAdmin, _myadmin); 
}

CosNF_MappingFilter_ptr 
StructuredProxyPullSupplier_i::priority_filter( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_pfilter) )
  	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _pfilter);
}

void StructuredProxyPullSupplier_i::priority_filter(
			CosNF_MappingFilter_ptr prio_filter WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(prio_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _pfilter = WRAPPED_DUPLICATE(CosNF_MappingFilter, prio_filter);
}

CosNF_MappingFilter_ptr 
StructuredProxyPullSupplier_i::lifetime_filter(WRAPPED_IMPLARG_VOID)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_lfilter) )
	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _lfilter);
}

void StructuredProxyPullSupplier_i::lifetime_filter(
			CosNF_MappingFilter_ptr life_filter WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(life_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _lfilter = WRAPPED_DUPLICATE(CosNF_MappingFilter, life_filter); 
}

CosN_EventTypeSeq* 
StructuredProxyPullSupplier_i::obtain_offered_types(CosNA_ObtainInfoMode mode
						    WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq* etypeseq=0;
  if ( mode == CosNA_ALL_NOW_UPDATES_OFF ) {
        etypeseq = _channel->obtain_offered_types();
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_ALL_NOW_UPDATES_ON ) {
        etypeseq = _channel->obtain_offered_types();
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_OFF ) {
	etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_ON ) {
	etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  }
  return etypeseq;
}

CosN_QoSProperties* 
StructuredProxyPullSupplier_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_S_StrPRX);
}

void StructuredProxyPullSupplier_i::set_qos(
		const CosN_QoSProperties& r_qos WRAPPED_IMPLARG)
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_S_StrPRX,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos);
}

void StructuredProxyPullSupplier_i::validate_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_S_StrPRX,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void StructuredProxyPullSupplier_i::validate_event_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

CosN_StructuredEvent* 
StructuredProxyPullSupplier_i::pull_structured_event( WRAPPED_IMPLARG_VOID )
{
  CosN_StructuredEvent* cosse=0;
  RDI_StructuredEvent*  entry=0;

  _oplock.lock();
  _blkpull += 1;
  while ( (_pxstate == RDI_Connected) && (_ntfqueue.length() == 0) )
	_noempty.wait();
  if ( _pxstate != RDI_Connected ) {
  	_blkpull -= 1;
	_oplock.unlock();
	throw CosEventComm::Disconnected();
  } 

  entry = _ntfqueue.get_head();
  _ntfqueue.remove_head();
  _nevents += 1;
  _blkpull -= 1;
  _oplock.unlock();
  entry->decr_ref_counter( RDI::lockEvent() );
  cosse = new CosN_StructuredEvent( entry->get_cos_event());
  entry->decr_ref_counter( RDI::lockEvent() );
  return cosse;
}

CosN_StructuredEvent* StructuredProxyPullSupplier_i::try_pull_structured_event(
				CORBA::Boolean& has_event   WRAPPED_IMPLARG )
{
  _oplock.lock();
  if ( _pxstate != RDI_Connected ) {
	_oplock.unlock();
	throw CosEventComm::Disconnected();
  } else if ( _ntfqueue.length() == 0 ) {
	_oplock.unlock();
	has_event = 0;
	return new CosN_StructuredEvent();
  } else {
	CosN_StructuredEvent*    cosse=0; 
	RDI_StructuredEvent* entry=_ntfqueue.get_head();
	_ntfqueue.remove_head();
	_nevents += 1;
	_oplock.unlock();
	has_event = 1;
	cosse = new CosN_StructuredEvent( entry->get_cos_event());
	entry->decr_ref_counter( RDI::lockEvent() );
	return cosse;
  }
}	

void StructuredProxyPullSupplier_i::disconnect_structured_pull_supplier( WRAPPED_IMPLARG_VOID )
{
  // Need to remove from admin hash table first to avoid potential
  // deadlock between this call and some add_event() call from one
  // of the event matching/dispatch threads
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void StructuredProxyPullSupplier_i::subscription_change(
                                const CosN_EventTypeSeq& added,
                                const CosN_EventTypeSeq& deled 
				WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq nadded;
  CosN_EventTypeSeq ndeled;
  CORBA::Boolean i_am_consumer = 1;
  CORBA::ULong   ix=0;
  if ( ! RDI_EventType::valid_sequence(added, ix, i_am_consumer) )
        throw CosNC_InvalidEventType(added[ix]);
  if ( ! RDI_EventType::valid_sequence(deled, ix, i_am_consumer) )
        throw CosNC_InvalidEventType(deled[ix]);
  // The consumer can invoke this operation as long as it does not
  // use filter objects -- CORBA Notification Specification 2.6.4.
  if ( _fa_helper.has_filters() )
        throw CORBA::BAD_OPERATION(0, CORBA::COMPLETED_NO);
  // If this is not the first time this operation gets invoked, we
  // need to compute the actual added and deleted event types that
  // will be used to update the TypeMap
  if ( _rqstypes.length() ) {
        RDI_EventType::compute_diff(_rqstypes, added, deled, nadded, ndeled);
        if ( nadded.length() || ndeled.length() ) {
                (void) _channel->update_mapping(nadded, ndeled, this, 0);
                RDI_EventType::update(_rqstypes, added, deled);
        }
  } else {
	ndeled.length(0); _rqstypes = added; 
        RDI_EventType::update(_rqstypes, added, deled);
        (void) _channel->update_mapping(_rqstypes, ndeled, this, 0);
  }
}

CosNF_FilterID
StructuredProxyPullSupplier_i::add_filter(CosNF_Filter_ptr filter
					  WRAPPED_IMPLARG )
{
  _oplock.lock();
  // If we have registered interest in specific event types, using
  // 'subscription_change()', we have to cancel it at this point..
  if ( _rqstypes.length() != 0 ) {
        CosN_EventTypeSeq added; added.length(0);
        _channel->update_mapping(added, _rqstypes, this, 0);
        _rqstypes.length(0);
  }
  _oplock.unlock();
  // The FAdminHelper object maintains its own latch and,  hence,
  // do not need to hold the latch during the following operation.
  return _fa_helper.add_filter_i(filter, (RDINotifySubscribe_ptr) this);
}

void StructuredProxyPullSupplier_i::subscription_change_i(
				const CosN_EventTypeSeq& added,
				const CosN_EventTypeSeq& deled,
				      Filter_i*                      flter)
{ (void) _channel->update_mapping(added, deled, this, flter); }

void StructuredProxyPullSupplier_i::filter_destroy_i(Filter_i* filter)
{ _fa_helper.rem_filter_i(filter); }

void StructuredProxyPullSupplier_i::connect_structured_pull_consumer(
	CosNC_StructuredPullConsumer_ptr consumer WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate == RDI_Connected ) {
	RDI_DUMP("Consumer already connected to proxy ....");
	throw CosEventChannelAdmin::AlreadyConnected();
  }
  // According to the CORBA Notification Service specification, pull
  // consumers can provide a _nil reference [2.6.1] ................
  if ( ! CORBA::is_nil(consumer) ) {
  	_consumer = WRAPPED_DUPLICATE(CosNC_StructuredPullConsumer, consumer);
  }
  _pxstate  = RDI_Connected;
}

void StructuredProxyPullSupplier_i::add_event(RDI_StructuredEvent* entry 
					      WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( (_pxstate == RDI_Connected) && entry ) {
	// Check if the 'MaxEventsPerConsumer' threshold has been reached.
        // In this case, we discard events based on 'DiscardPolicy' -- the
        // current implementation assumes FIFO
        RDI_NotifQoS* qospr = (_qosprop?_qosprop:_myadmin->qos_properties());
        unsigned int  msize = (unsigned int) qospr->maxEventsPerConsumer;
        if ( msize && (_ntfqueue.length() >= msize) ) {
                RDI_StructuredEvent* evnt = _ntfqueue.get_head();
                _ntfqueue.remove_head();
                evnt->decr_ref_counter( RDI::lockEvent() );
        }
        entry->incr_ref_counter( RDI::lockEvent() );      
        _ntfqueue.insert_tail(entry);
       	_noempty.signal();
  }
}

ostream& operator << (ostream& out, const StructuredProxyPullSupplier_i& prx)
{
  out << & prx  << " -- "  << RDI_PRX_TYPE(prx._prxtype) << 
	 " ID " << setw(3) << prx._pserial;
  if ( ! CORBA::is_nil(prx._pfilter ) )
	out << " PFilter " << prx._pfilter;
  if ( ! CORBA::is_nil(prx._lfilter ) )
        out << " LFilter " << prx._lfilter;
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " Not Connected "; break;
     case RDI_Connected:    out << " Connected     "; break;
     case RDI_Disconnected: out << " Disconnected  "; break;
     case RDI_Exception:    out << " Exception     "; break;
  }
  out << " QSize "<< prx._ntfqueue.length()<< " #Pull "<< prx._nevents;
  if ( prx._rqstypes.length() != 0 ) {
     for (CORBA::ULong ix=0; ix < prx._rqstypes.length(); ix++) {
        out << endl << "\t" << prx._rqstypes[ix].domain_name; 
        out << "::" << prx._rqstypes[ix].type_name;
     }
  }
  return out;
}

void StructuredProxyPullSupplier_i::disconnect_client_and_dispose()
{
  RDI_StructuredEvent* event = 0;
  _oplock.lock();
  _pxstate = RDI_Disconnected;

  if ( _rqstypes.length() == 0 ) { 
        remove_all_filters();
  } else {
        CosN_EventTypeSeq added; added.length(0);
        (void) _channel->update_mapping(added, _rqstypes, this, 0);
  }

  // The consumer may be blocked on a pull() at this point(?)
  // and, thus, we should unblock it before self-destructing.
  _noempty.broadcast();
  _oplock.unlock();
  while ( _blkpull != 0 ) {
        _noempty.signal();
        omni_thread::yield();
  }
  _oplock.lock();

  if ( ! CORBA::is_nil(_consumer) ) {
        // WRAPPED_RELEASE(CosNC_StructuredPullConsumer, _consumer);
        _consumer = CosNC_StructuredPullConsumer::_nil();
  }
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
	_ntfqueue.remove_head();
	event->decr_ref_counter( RDI::lockEvent() );
  }
  if ( ! CORBA::is_nil(_pfilter) ) WRAPPED_RELEASE(_Filter, _pfilter);
  if ( ! CORBA::is_nil(_lfilter) ) WRAPPED_RELEASE(_Filter, _lfilter);
  if ( _qosprop ) delete _qosprop;

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}

// ------------------------------------------------------------- //
// SequenceProxyPullSupplier_i                                   //
// ------------------------------------------------------------- //

SequenceProxyPullSupplier_i::SequenceProxyPullSupplier_i(
					ConsumerAdmin_i* admin, 
				 	EventChannel_i*  chann,
					const CosNA_ProxyID&  prxid) : 
		RDINotifySubscribe(), _oplock(), _fa_helper(),
		_noempty(&_oplock), _channel(chann), _myadmin(admin), 
		_prxtype(CosNA_PULL_SEQUENCE), _pserial(prxid), 
		_nevents(0), _blkpull(0), _pxstate(RDI_NotConnected),
		_qosprop(0), _offauto(1), _rqstypes(), _ntfqueue(32,32)
{
  _rqstypes.length(0);
  _pfilter  = CosNF_MappingFilter::_nil();
  _lfilter  = CosNF_MappingFilter::_nil();
  _consumer = CosNC_SequencePullConsumer::_nil();
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_ConsumerAdmin_ptr SequenceProxyPullSupplier_i::MyAdmin( WRAPPED_IMPLARG_VOID )
{ 
  omni_mutex_lock lock(_oplock);
  return WRAPPED_IMPL2OREF(CosNA_ConsumerAdmin, _myadmin); 
}

CosNF_MappingFilter_ptr 
SequenceProxyPullSupplier_i::priority_filter( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_pfilter) )
  	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _pfilter);
}

void SequenceProxyPullSupplier_i::priority_filter(
			CosNF_MappingFilter_ptr prio_filter WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(prio_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _pfilter = WRAPPED_DUPLICATE(CosNF_MappingFilter, prio_filter);
}

CosNF_MappingFilter_ptr 
SequenceProxyPullSupplier_i::lifetime_filter( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_lfilter) )
	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _lfilter);
}

void SequenceProxyPullSupplier_i::lifetime_filter(
			CosNF_MappingFilter_ptr life_filter WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(life_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _lfilter = WRAPPED_DUPLICATE(CosNF_MappingFilter, life_filter); 
}

CosN_EventTypeSeq* 
SequenceProxyPullSupplier_i::obtain_offered_types(CosNA_ObtainInfoMode mode
						  WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq* etypeseq=0;
  if ( mode == CosNA_ALL_NOW_UPDATES_OFF ) {
        etypeseq = _channel->obtain_offered_types();
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_ALL_NOW_UPDATES_ON ) {
        etypeseq = _channel->obtain_offered_types();
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_OFF ) {
	etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_ON ) {
	etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  }
  return etypeseq;
}

CosN_QoSProperties* 
SequenceProxyPullSupplier_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_S_SeqPRX);
}

void SequenceProxyPullSupplier_i::set_qos(
		const CosN_QoSProperties& r_qos WRAPPED_IMPLARG)
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_S_SeqPRX,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos);
}

void SequenceProxyPullSupplier_i::validate_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_S_SeqPRX,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void SequenceProxyPullSupplier_i::validate_event_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

CosN_EventBatch* 
SequenceProxyPullSupplier_i::pull_structured_events(CORBA::Long maxnum
						    WRAPPED_IMPLARG)
{
  CosN_EventBatch* notif=0;
  RDI_StructuredEvent* event=0;
  RDI_NotifQoS* qospr = (_qosprop ? _qosprop : _myadmin->qos_properties());
  CORBA::ULong  bsize = (CORBA::ULong) ((maxnum < 0) ? 5 : maxnum);

  if ( (CORBA::Long)maxnum > qospr->maximumBatchSize )
	maxnum = qospr->maximumBatchSize;

  _oplock.lock();
  _blkpull += 1;

  while ( (_pxstate == RDI_Connected) && (_ntfqueue.length() < bsize) )
	_noempty.wait();
  if ( _pxstate != RDI_Connected ) {
	_blkpull -= 1;
	_oplock.unlock();
	throw CosEventComm::Disconnected();
  }
  if ( ! (notif = new CosN_EventBatch) ) {
	_blkpull -= 1;
	_oplock.unlock();
	RDI_DUMP("Memory allocation failed -- CosN_EventBatch");
	return 0;
  }
  notif->length(bsize);
  for (CORBA::ULong i=0; i < bsize; i++) {
	event = _ntfqueue.get_head();
	_ntfqueue.remove_head();
	(*notif)[i] = event->get_cos_event();
	_nevents += 1;
	event->decr_ref_counter( RDI::lockEvent() );
  }
  _blkpull -= 1;
  _oplock.unlock();
  return notif;
}
  
CosN_EventBatch* 
SequenceProxyPullSupplier_i::try_pull_structured_events(
		CORBA::Long maxnum, CORBA::Boolean& has_event WRAPPED_IMPLARG)
{
  CosN_EventBatch* notif=0;
  RDI_StructuredEvent* event=0;
  RDI_NotifQoS* qospr = (_qosprop ? _qosprop : _myadmin->qos_properties());
  CORBA::ULong  bsize = (CORBA::ULong) ((maxnum < 0) ? 5 : maxnum);

  if ( (CORBA::Long)maxnum > qospr->maximumBatchSize )
        maxnum = qospr->maximumBatchSize;

  _oplock.lock();
  if ( _pxstate != RDI_Connected ) {
	_oplock.unlock();
	throw CosEventComm::Disconnected();
  }

  if ( ! (notif = new CosN_EventBatch) ) {
	_oplock.unlock();
	RDI_DUMP("Memory allocation failed -- CosN_EventBatch");
        return 0;
  }

  if ( _ntfqueue.length() < bsize ) {
	_oplock.unlock();
	has_event = 0;
	notif->length(0);
  } else {
	for (CORBA::ULong i=0; i < bsize; i++) {
		event = _ntfqueue.get_head();
		_ntfqueue.remove_head();
		(*notif)[i] = event->get_cos_event();
		_nevents += 1;
		event->decr_ref_counter( RDI::lockEvent() );
	}
	_oplock.unlock();
	has_event = 1;
  }
  return notif;
}

void SequenceProxyPullSupplier_i::disconnect_sequence_pull_supplier( WRAPPED_IMPLARG_VOID )
{
  // Need to remove from admin hash table first to avoid potential
  // deadlock between this call and some add_event() call from one
  // of the event matching/dispatch threads
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void SequenceProxyPullSupplier_i::subscription_change(
                                const CosN_EventTypeSeq& added,
                                const CosN_EventTypeSeq& deled
				WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq nadded;
  CosN_EventTypeSeq ndeled;
  CORBA::Boolean i_am_consumer = 1;
  CORBA::ULong   ix=0;
  if ( ! RDI_EventType::valid_sequence(added, ix, i_am_consumer) )
        throw CosNC_InvalidEventType(added[ix]);
  if ( ! RDI_EventType::valid_sequence(deled, ix, i_am_consumer) )
        throw CosNC_InvalidEventType(deled[ix]);
  // The consumer can invoke this operation as long as it does not
  // use filter objects -- CORBA Notification Specification 2.6.4.
  if ( _fa_helper.has_filters() )
        throw CORBA::BAD_OPERATION(0, CORBA::COMPLETED_NO);
  // If this is not the first time this operation gets invoked, we
  // need to compute the actual added and deleted event types that
  // will be used to update the TypeMap
  if ( _rqstypes.length() ) {
        RDI_EventType::compute_diff(_rqstypes, added, deled, nadded, ndeled);
        if ( nadded.length() || ndeled.length() ) {
                (void) _channel->update_mapping(nadded, ndeled, this, 0);
                RDI_EventType::update(_rqstypes, added, deled);
        }
  } else {
        ndeled.length(0); _rqstypes = added; 
        RDI_EventType::update(_rqstypes, added, deled);
        (void) _channel->update_mapping(_rqstypes, ndeled, this, 0);
  }
}

CosNF_FilterID
SequenceProxyPullSupplier_i::add_filter(CosNF_Filter_ptr filter
				        WRAPPED_IMPLARG )
{
  _oplock.lock();
  // If we have registered interest in specific event types, using
  // 'subscription_change()', we have to cancel it at this point..
  if ( _rqstypes.length() != 0 ) {
        CosN_EventTypeSeq added; added.length(0);
        _channel->update_mapping(added, _rqstypes, this, 0);
        _rqstypes.length(0);
  }
  _oplock.unlock();
  // The FAdminHelper object maintains its own latch and,  hence,
  // do not need to hold the latch during the following operation.
  return _fa_helper.add_filter_i(filter, (RDINotifySubscribe_ptr) this);
}

void SequenceProxyPullSupplier_i::subscription_change_i(
				const CosN_EventTypeSeq& added,
				const CosN_EventTypeSeq& deled,
				      Filter_i*                      flter)
{ (void) _channel->update_mapping(added, deled, this, flter); }

void SequenceProxyPullSupplier_i::filter_destroy_i(Filter_i* filter)
{ _fa_helper.rem_filter_i(filter); }

void SequenceProxyPullSupplier_i::connect_sequence_pull_consumer(
	CosNC_SequencePullConsumer_ptr consumer WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate == RDI_Connected )
	throw CosEventChannelAdmin::AlreadyConnected();
  // According to the CORBA Notification Service specification, pull
  // consumers can provide a _nil reference [2.6.1] ................
  if ( ! CORBA::is_nil(consumer) ) {
  	_consumer = WRAPPED_DUPLICATE(CosNC_SequencePullConsumer, consumer);
  }
  _pxstate  = RDI_Connected;
}

void SequenceProxyPullSupplier_i::add_event(RDI_StructuredEvent* entry
					    WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( (_pxstate == RDI_Connected) && entry ) {
	// Check if the 'MaxEventsPerConsumer' threshold has been reached.
        // In this case, we discard events based on 'DiscardPolicy' -- the
        // current implementation assumes FIFO
        RDI_NotifQoS* qospr = (_qosprop?_qosprop:_myadmin->qos_properties());
        unsigned int  msize = (unsigned int) qospr->maxEventsPerConsumer;
        if ( msize && (_ntfqueue.length() >= msize) ) {
                RDI_StructuredEvent* evnt = _ntfqueue.get_head();
                _ntfqueue.remove_head();
                evnt->decr_ref_counter( RDI::lockEvent() );
        }
        entry->incr_ref_counter( RDI::lockEvent() );
        _ntfqueue.insert_tail(entry);
        _noempty.signal();
  }
}

ostream& operator << (ostream& out, const SequenceProxyPullSupplier_i& prx)
{
  out << & prx  << " -- "  << RDI_PRX_TYPE(prx._prxtype) << 
	 " ID " << setw(3) << prx._pserial;
  if ( ! CORBA::is_nil(prx._pfilter) ) out << " PFilter " << prx._pfilter;
  if ( ! CORBA::is_nil(prx._lfilter) ) out << " LFilter " << prx._lfilter;
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " Not Connected "; break;
     case RDI_Connected:    out << " Connected     "; break;
     case RDI_Disconnected: out << " Disconnected  "; break;
     case RDI_Exception:    out << " Exception     "; break;
  }
  out << " QSize "<< prx._ntfqueue.length()<< " #Pull "<< prx._nevents;
  if ( prx._rqstypes.length() != 0 ) {
     for (CORBA::ULong ix=0; ix < prx._rqstypes.length(); ix++) {
        out << endl << "\t" << prx._rqstypes[ix].domain_name; 
        out << "::" << prx._rqstypes[ix].type_name;
     }
  }
  return out;
}

void SequenceProxyPullSupplier_i::disconnect_client_and_dispose()
{
  RDI_StructuredEvent* event=0;
  _oplock.lock();
  _pxstate = RDI_Disconnected;

  if ( _rqstypes.length() == 0 ) { 
        remove_all_filters();
  } else {
        CosN_EventTypeSeq added; added.length(0);
        (void) _channel->update_mapping(added, _rqstypes, this, 0);
  }

  // The consumer may be blocked on a pull() at this point(?)
  // and, thus, we should unblock it before self-destructing.
  _noempty.broadcast();
  _oplock.unlock();
  while ( _blkpull != 0 ) {
        _noempty.signal();
        omni_thread::yield();
  }
  _oplock.lock();

  if ( ! CORBA::is_nil(_consumer) ) {
        // WRAPPED_RELEASE(CosNC_SequencePullConsumer, _consumer);
        _consumer = CosNC_SequencePullConsumer::_nil();
  }
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
	_ntfqueue.remove_head();
	event->decr_ref_counter( RDI::lockEvent() );
  }
  if ( ! CORBA::is_nil(_pfilter) ) WRAPPED_RELEASE(_Filter, _pfilter);
  if ( ! CORBA::is_nil(_lfilter) ) WRAPPED_RELEASE(_Filter, _lfilter);
  if ( ! _qosprop ) delete _qosprop;

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}
