// -*- Mode: C++; -*-
//                              File      : ChannelAdmin_i.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of ConsumerAdmin_i and SupplierAdmin_i
//
 
/*
$Log: ChannelAdmin_i.cc,v $
Revision 1.55  2000/08/22 18:23:52  alcfp
added description to each file

Revision 1.54  2000/08/16 20:19:20  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <iomanip.h>
#include "corba_wrappers.h"
#include "CosNotify.h"
#include "RDIDebug.h"
#include "RDITypeMap.h"
#include "CosEventChannelAdmin_i.h"
#include "CosNotifyChannelAdmin_i.h"

// ------------------------------------------------------------- //
// ConsumerAdmin_i implementation                                //
// ------------------------------------------------------------- //

ConsumerAdmin_i::ConsumerAdmin_i(EventChannel_i* channel,
				 CosNA_InterFilterGroupOperator op,
				 const CosNA_AdminID& serial) :
		RDINotifySubscribe(),
		_oplock(), _fa_helper(), _channel(channel), _qosprop(0),
		_serial(serial), _and_or_oper(op), _prx_serial(1),
		_num_proxies(0), _cosevent_push(), _cosevent_pull(),
		_prx_any_push(RDI_ULongHash, RDI_ULongRank),
		_prx_any_pull(RDI_ULongHash, RDI_ULongRank),
		_prx_struc_push(RDI_ULongHash, RDI_ULongRank),
		_prx_struc_pull(RDI_ULongHash, RDI_ULongRank),
		_prx_batch_push(RDI_ULongHash, RDI_ULongRank),
		_prx_batch_pull(RDI_ULongHash, RDI_ULongRank)
{
  _prio_filter = CosNF_MappingFilter::_nil();
  _life_filter = CosNF_MappingFilter::_nil();
  _time_born   = RDI_TimeValue::timeofday();
  _prxy_exit   = _time_born;
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_EventChannel_ptr ConsumerAdmin_i::MyChannel( WRAPPED_IMPLARG_VOID )
{ return WRAPPED_IMPL2OREF(CosNA_EventChannel, _channel); }

CosNF_MappingFilter_ptr ConsumerAdmin_i::priority_filter( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_prio_filter) )
	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _prio_filter);
}
 
void ConsumerAdmin_i::priority_filter(CosNF_MappingFilter_ptr prio_filter   
                                      WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(prio_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _prio_filter = WRAPPED_DUPLICATE(CosNF_MappingFilter, prio_filter);
}

CosNF_MappingFilter_ptr ConsumerAdmin_i::lifetime_filter( WRAPPED_IMPLARG_VOID )
{ 
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(_life_filter) )
	return CosNF_MappingFilter::_nil();
  return WRAPPED_DUPLICATE(CosNF_MappingFilter, _life_filter); 
}

void ConsumerAdmin_i::lifetime_filter(CosNF_MappingFilter_ptr life_filter   
				      WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(life_filter) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  _life_filter = WRAPPED_DUPLICATE(CosNF_MappingFilter, life_filter);
}

CosEventChannelAdmin::ProxyPushSupplier_ptr 
ConsumerAdmin_i::obtain_push_supplier( WRAPPED_IMPLARG_VOID )
{
  EventProxyPushSupplier_i* prx=0;
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("CosEvent ProxyPushSupplier creation requested");
  if ( ! _channel->incr_consumers() )
	return CosEventChannelAdmin::ProxyPushSupplier::_nil();
  if ( ! (prx = new EventProxyPushSupplier_i(this, _channel)) ) {
	_channel->decr_consumers();
	return CosEventChannelAdmin::ProxyPushSupplier::_nil();
  }
  if ( _cosevent_push.insert_tail(prx) != 0 ) {
	_channel->decr_consumers();
	delete prx;
	return CosEventChannelAdmin::ProxyPushSupplier::_nil();
  }
  if ( _channel->_push_consumer )         /* Register Push Proxy */
	_channel->_push_consumer->insert_proxy(prx);
  _num_proxies += 1;
  RDI_DUMP("CosEvent ProxyPushSupplier creation completed");
  return WRAPPED_IMPL2OREF(CosEventChannelAdmin::ProxyPushSupplier, prx);
}

CosEventChannelAdmin::ProxyPullSupplier_ptr
ConsumerAdmin_i::obtain_pull_supplier( WRAPPED_IMPLARG_VOID )
{
  EventProxyPullSupplier_i* prx=0;
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("CosEvent ProxyPullSupplier creation requested");
  if ( ! _channel->incr_consumers() )
	return CosEventChannelAdmin::ProxyPullSupplier::_nil();
  if ( ! (prx = new EventProxyPullSupplier_i(this, _channel)) ) {
	_channel->decr_consumers();
	return CosEventChannelAdmin::ProxyPullSupplier::_nil();
  }
  if ( _cosevent_pull.insert_tail(prx) != 0 ) {
	_channel->decr_consumers();
        delete prx;
	return CosEventChannelAdmin::ProxyPullSupplier::_nil();
  }
  _num_proxies += 1;
  RDI_DUMP("CosEvent ProxyPullSupplier creation completed");
  return WRAPPED_IMPL2OREF(CosEventChannelAdmin::ProxyPullSupplier, prx);
}

void ConsumerAdmin_i::subscription_change(const CosN_EventTypeSeq& added,
					  const CosN_EventTypeSeq& deled   
					  WRAPPED_IMPLARG )
{
  if ( ! _fa_helper.has_filters() ) {
	(void) _channel->update_mapping(added, deled, this, 0);
  } else {
	RDI_DUMP("Invalid operation -- object has filters already");
  }
}

void ConsumerAdmin_i::subscription_change_i(const CosN_EventTypeSeq& added,
					    const CosN_EventTypeSeq& deled,
					    Filter_i*            flter)
{ (void) _channel->update_mapping(added, deled, this, flter); }

void ConsumerAdmin_i::filter_destroy_i(Filter_i* filter)
{ _fa_helper.rem_filter_i(filter); }

CosNA_ProxyIDSeq* ConsumerAdmin_i::pull_suppliers( WRAPPED_IMPLARG_VOID )
{
  RDI_HashCursor<CosNA_ProxyID, ProxyPullSupplier_i *>           ac;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullSupplier_i *> sc;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullSupplier_i * >  bc;
  unsigned long num;
  CosNA_ProxyIDSeq*  seq = new CosNA_ProxyIDSeq();
  if ( ! seq ) {
	throw CORBA::NO_MEMORY(0, CORBA::COMPLETED_MAYBE);
  }
  _oplock.lock();
  num = _prx_any_pull.length() +
	_prx_struc_pull.length() + _prx_batch_pull.length();
  seq->length(num);
  for ( num=0, ac = _prx_any_pull.cursor(); ac.is_valid(); ++ac, ++num )
	(*seq)[num] = ac.key();
  for ( sc = _prx_struc_pull.cursor(); sc.is_valid(); ++sc, ++num )
	(*seq)[num] = sc.key();
  for ( bc = _prx_batch_pull.cursor(); bc.is_valid(); ++bc, ++num )
	(*seq)[num] = bc.key();
  _oplock.unlock();
  return seq;
}

CosNA_ProxyIDSeq* ConsumerAdmin_i::push_suppliers( WRAPPED_IMPLARG_VOID )
{
  RDI_HashCursor<CosNA_ProxyID, ProxyPushSupplier_i *>           ac;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushSupplier_i *> sc;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushSupplier_i * >  bc;
  unsigned long num;
  CosNA_ProxyIDSeq*  seq = new CosNA_ProxyIDSeq();
  if ( ! seq ) {
	throw CORBA::NO_MEMORY(0, CORBA::COMPLETED_MAYBE);
  }
  _oplock.lock();
  num = _prx_any_push.length() +
	_prx_struc_push.length() + _prx_batch_push.length();
  seq->length(num);
   for ( num=0, ac = _prx_any_push.cursor(); ac.is_valid(); ++ac, ++num )
        (*seq)[num] = ac.key();
  for ( sc = _prx_struc_push.cursor(); sc.is_valid(); ++sc, ++num )
        (*seq)[num] = sc.key();        
  for ( bc = _prx_batch_push.cursor(); bc.is_valid(); ++bc, ++num )
        (*seq)[num] = bc.key();
  _oplock.unlock();
  return seq;
}

CosNA_ProxySupplier_ptr 
ConsumerAdmin_i::get_proxy_supplier(CosNA_ProxyID proxy_id WRAPPED_IMPLARG )
{
  ProxyPushSupplier_i*           apush=0;
  ProxyPullSupplier_i*           apull=0;
  StructuredProxyPushSupplier_i* spush=0;
  StructuredProxyPullSupplier_i* spull=0;
  SequenceProxyPushSupplier_i*   bpush=0;
  SequenceProxyPullSupplier_i*   bpull=0;
  omni_mutex_lock lock(_oplock); 

  if ( proxy_id > _prx_serial ) 
	throw CosNA_ProxyNotFound();
  if ( _prx_any_push.lookup(proxy_id, apush) ) 
	return WRAPPED_IMPL2OREF(CosNA_ProxySupplier, apush);
  if ( _prx_any_pull.lookup(proxy_id, apull) ) 
	return WRAPPED_IMPL2OREF(CosNA_ProxySupplier, apull);
  if ( _prx_struc_push.lookup(proxy_id, spush) )
	return WRAPPED_IMPL2OREF(CosNA_ProxySupplier, spush);
  if ( _prx_struc_pull.lookup(proxy_id, spull) )
	return WRAPPED_IMPL2OREF(CosNA_ProxySupplier, spull);
  if ( _prx_batch_push.lookup(proxy_id, bpush) )
	return WRAPPED_IMPL2OREF(CosNA_ProxySupplier, bpush);
  if ( _prx_batch_pull.lookup(proxy_id, bpull) )
	return WRAPPED_IMPL2OREF(CosNA_ProxySupplier, bpull);
   throw CosNA_ProxyNotFound();
}

CosNA_ProxySupplier_ptr 
ConsumerAdmin_i::obtain_notification_pull_supplier(CosNA_ClientType ctype, 
						   CosNA_ProxyID&   proxy_id
						   WRAPPED_IMPLARG )
{
  CosNA_AdminLimit limit;
  omni_mutex_lock lock(_oplock);

  // Check with  the channel if we are about to exceed the maximum
  // number of consumers supported by the channel

  RDI_DUMP("Creation of a new ProxySupplier -- type " << ctype);
  if ( ! _channel->incr_consumers() ) {
  	RDI_DUMP("Consumer limit has been reached creation failed");
	limit.name    = (const char *) "MaxConsumers";
	limit.value <<= _channel->max_consumers();
	throw CosNA_AdminLimitExceeded(limit);
  }

  if ( ctype == CosNA_ANY_EVENT ) {
	ProxyPullSupplier_i* prx = 0;
	if ( ! (prx = new ProxyPullSupplier_i(this, _channel,_prx_serial)) ) {
		_channel->decr_consumers();
		return CosNA_ProxySupplier::_nil();
	}
	proxy_id = _prx_serial++;
	if ( _prx_any_pull.insert(proxy_id, prx) != 0 ) {
		delete prx;
		_channel->decr_consumers(); 
		return CosNA_ProxySupplier::_nil();
	}
	_num_proxies += 1;
	return WRAPPED_IMPL2OREF(CosNA_ProxySupplier, prx);
  } else if ( ctype == CosNA_STRUCTURED_EVENT ) {
	StructuredProxyPullSupplier_i* prx = 
		new StructuredProxyPullSupplier_i(this,_channel,_prx_serial);
	if ( ! prx ) {
		_channel->decr_consumers();
		return CosNA_ProxySupplier::_nil();
	}
	proxy_id = _prx_serial++;
	if ( _prx_struc_pull.insert(proxy_id, prx) != 0 ) {
		delete prx;
		_channel->decr_consumers();
		return CosNA_ProxySupplier::_nil();
	}
	_num_proxies += 1;
	return WRAPPED_IMPL2OREF(CosNA_ProxySupplier, prx);
  } else if ( ctype == CosNA_SEQUENCE_EVENT ) {
	SequenceProxyPullSupplier_i* prx =
		new SequenceProxyPullSupplier_i(this,_channel,_prx_serial);
	if ( ! prx ) {
		_channel->decr_consumers();
		return CosNA_ProxySupplier::_nil();
	}
	proxy_id = _prx_serial++;
	if ( _prx_batch_pull.insert(proxy_id, prx) != 0 ) {
               	delete prx;
		_channel->decr_consumers();
		return CosNA_ProxySupplier::_nil();
	} 
	_num_proxies += 1;
	return WRAPPED_IMPL2OREF(CosNA_ProxySupplier, prx);
  }
  return CosNA_ProxySupplier::_nil();
}

CosNA_ProxySupplier_ptr 
ConsumerAdmin_i::obtain_notification_push_supplier(CosNA_ClientType ctype, 
						   CosNA_ProxyID&   proxy_id
						   WRAPPED_IMPLARG )
{
  CosNA_AdminLimit limit;
  omni_mutex_lock lock(_oplock);

  // Check with  the channel if we are about to exceed the maximum
  // number of consumers supported by the channel

  RDI_DUMP("Creation of a new ProxySupplier -- type " << ctype);
  if ( ! _channel->incr_consumers() ) {
  	RDI_DUMP("Consumer limit has been reached creation failed");
        limit.name    = (const char *) "MaxConsumers";
        limit.value <<= _channel->max_consumers();
        throw CosNA_AdminLimitExceeded(limit);
  }

  if ( ctype == CosNA_ANY_EVENT ) {
	ProxyPushSupplier_i* prx = 0;
	if ( ! (prx = new ProxyPushSupplier_i(this, _channel, _prx_serial)) ) {
		_channel->decr_consumers();
		return CosNA_ProxySupplier::_nil();
	}
	proxy_id = _prx_serial++;
	if ( _prx_any_push.insert(proxy_id, prx) != 0 ) {
		delete prx;
		_channel->decr_consumers(); 
		return CosNA_ProxySupplier::_nil();
	}
	if ( _channel->_push_consumer )		/* Register Push Proxy */
		_channel->_push_consumer->insert_proxy(prx);
	_num_proxies += 1;
	return WRAPPED_IMPL2OREF(CosNA_ProxySupplier, prx);
  } else if ( ctype == CosNA_STRUCTURED_EVENT ) {
	StructuredProxyPushSupplier_i* prx = 
		new StructuredProxyPushSupplier_i(this,_channel,_prx_serial);
	if ( ! prx ) {
		RDI_DUMP("\tfailed to create proxy");
		_channel->decr_consumers();
		return CosNA_ProxySupplier::_nil();
	}
	proxy_id = _prx_serial++;
	if ( _prx_struc_push.insert(proxy_id, prx) != 0 ) {
		RDI_DUMP("\tfailed to register proxy with id " << proxy_id);
		delete prx;
		_channel->decr_consumers();
		return CosNA_ProxySupplier::_nil();
	}
	if ( _channel->_push_consumer )		/* Register Push Proxy */
		_channel->_push_consumer->insert_proxy(prx);
	_num_proxies += 1;
	return WRAPPED_IMPL2OREF(CosNA_ProxySupplier, prx);
  } else if ( ctype == CosNA_SEQUENCE_EVENT ) {
	SequenceProxyPushSupplier_i* prx = 
		new SequenceProxyPushSupplier_i(this,_channel,_prx_serial);
	if ( ! prx ) {
		_channel->decr_consumers();
		return CosNA_ProxySupplier::_nil();
	}
	proxy_id = _prx_serial++;
	if ( _prx_batch_push.insert(proxy_id, prx) != 0 ) {
               	delete prx;
		_channel->decr_consumers();
		return CosNA_ProxySupplier::_nil();
	} 
	if ( _channel->_push_consumer )		/* Register Push Proxy */
		_channel->_push_consumer->insert_proxy(prx);
	_num_proxies += 1;
	return WRAPPED_IMPL2OREF(CosNA_ProxySupplier, prx);
  }
  return CosNA_ProxySupplier::_nil();
}

CosN_QoSProperties* ConsumerAdmin_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = qos_properties();
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_C_ADMIN);
}

void ConsumerAdmin_i::set_qos(const CosN_QoSProperties& r_qos   
			      WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = (_num_proxies ? 1 : 0);
  RDI_NotifQoS* p_qos = _channel->qos_properties();
  RDI_NotifQoS* a_qos = (_qosprop ? _qosprop : p_qos);
  
  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_ADMIN,eseq,rseq,child,p_qos))
	throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
	_qosprop = new RDI_NotifQoS();
	RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos); 
}

void ConsumerAdmin_i::validate_qos(
	      const CosN_QoSProperties& r_qos,
	      WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
	      WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = (_num_proxies ? 1 : 0);
  RDI_NotifQoS*   p_qos = _channel->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_ADMIN,eseq,*rseq,child,p_qos))
	throw CosN_UnsupportedQoS(eseq);
}

void ConsumerAdmin_i::destroy( WRAPPED_IMPLARG_VOID )
{
  if ( _serial == 0 ) {
	RDI_DUMP("Default ConsumerAdmin cannot be destroyed!");
	return;
  }
  // Unregister yourself from the channel -- need to be done first
  // to avoid deadlock when channel is being destroyed
  RDI_DUMP("Destroy ConsumerAdmin @ " << this << " [" << _serial << "]");
  _channel->unregister(this);
  disconnect_clients_and_dispose();
}

void ConsumerAdmin_i::offer_change(const CosN_EventTypeSeq& added,
                                   const CosN_EventTypeSeq& deled
                                   WRAPPED_IMPLARG )
{
  RDI_HashCursor<CosNA_ProxyID, ProxyPushSupplier_i *>           apushc;
  RDI_HashCursor<CosNA_ProxyID, ProxyPullSupplier_i *>           apullc;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushSupplier_i *> spushc;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullSupplier_i *> spullc;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushSupplier_i *>   bpushc;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullSupplier_i *>   bpullc;
  omni_mutex_lock lock(_oplock);

#if (0)
  /** apushc.val()->_consumer inherits from CosEventComm::PushConsumer
    * and, thus, does not support offer_change()
    */
  for ( apushc = _prx_any_push.cursor(); apushc.is_valid(); ++apushc ) {
     apushc.val()->_oplock.lock();
     if ( (apushc.val()->_pxstate==RDI_Connected) && !apushc.val()->_offauto &&
	  apushc.val()->_active && !CORBA::is_nil(apushc.val()->_consumer) ) {
     	try         { apushc.val()->_consumer->offer_change(added, deled); }
     	catch (...) { /* Consumer doesn't support offer_change() method */ }
     }
     apushc.val()->_oplock.unlock();
  }
  /** apullc.val()->_consumer inherits from CosEventComm::PullConsumer
    * and, thus, does not support offer_change() 
    */
  for ( apullc = _prx_any_pull.cursor(); apullc.is_valid(); ++apullc ) {
     apullc.val()->_oplock.lock();
     if ( (apullc.val()->_pxstate==RDI_Connected) && 
	  !apullc.val()->_offauto && !CORBA::is_nil(apullc.val()->_consumer) ) {
        try         { apullc.val()->_consumer->offer_change(added, deled); }
        catch (...) { /* Consumer doesn't support offer_change() method */ }
     }
     apullc.val()->_oplock.unlock();
  }
#endif
  for ( spushc = _prx_struc_push.cursor(); spushc.is_valid(); ++spushc ) {
     spushc.val()->_oplock.lock();
     if ( (spushc.val()->_pxstate==RDI_Connected) && !spushc.val()->_offauto &&
          spushc.val()->_active && ! CORBA::is_nil(spushc.val()->_consumer) ) {
        try         { spushc.val()->_consumer->offer_change(added, deled); }
        catch (...) { /* Consumer doesn't support offer_change() method */ }
     }
     spushc.val()->_oplock.unlock();
  }
  for ( spullc = _prx_struc_pull.cursor(); spullc.is_valid(); ++spullc ) {
     spullc.val()->_oplock.lock();
     if ( (spullc.val()->_pxstate==RDI_Connected) && !spullc.val()->_offauto &&
          ! CORBA::is_nil(spullc.val()->_consumer) ) {
        try         { spullc.val()->_consumer->offer_change(added, deled); }
        catch (...) { /* Consumer doesn't support offer_change() method */ }
     }
     spullc.val()->_oplock.unlock();
  }
  for ( bpushc = _prx_batch_push.cursor(); bpushc.is_valid(); ++bpushc ) {
     bpushc.val()->_oplock.lock();
     if ( (bpushc.val()->_pxstate==RDI_Connected) && !bpushc.val()->_offauto &&
          bpushc.val()->_active && ! CORBA::is_nil(bpushc.val()->_consumer) ) {
        try         { bpushc.val()->_consumer->offer_change(added, deled); }
        catch (...) { /* Consumer doesn't support offer_change() method */ }
     }
     bpushc.val()->_oplock.unlock();
  }
  for ( bpullc = _prx_batch_pull.cursor(); bpullc.is_valid(); ++bpullc ) {
     bpullc.val()->_oplock.lock();
     if ( (bpullc.val()->_pxstate==RDI_Connected) && !bpullc.val()->_offauto &&
          ! CORBA::is_nil(bpullc.val()->_consumer) ) {
        try         { bpullc.val()->_consumer->offer_change(added, deled); }
        catch (...) { /* Consumer doesn't support offer_change() method */ }
     }
     bpullc.val()->_oplock.unlock();
  }
}

void ConsumerAdmin_i::disconnect_clients_and_dispose()
{
  unsigned int i;
  RDI_ListCursor<EventProxyPushSupplier_i *>                cpushcur;
  RDI_ListCursor<EventProxyPullSupplier_i *>                cpullcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPushSupplier_i *>           apushcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPullSupplier_i *>           apullcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushSupplier_i *> spushcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullSupplier_i *> spullcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushSupplier_i *>   bpushcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullSupplier_i *>   bpullcur;

  RDI_DUMP("Destruction of ConsumerAdmin " << this << " [" << _serial << "]");
  _oplock.lock();
  cpushcur = _cosevent_push.cursor();
  for (i=0; i < _cosevent_push.length(); ++i, ++cpushcur)
        (*cpushcur)->disconnect_client_and_dispose();
  cpullcur = _cosevent_pull.cursor();
  for (i=0; i < _cosevent_pull.length(); ++i, ++cpullcur)
        (*cpullcur)->disconnect_client_and_dispose();
  for ( apushcur = _prx_any_push.cursor(); apushcur.is_valid(); ++apushcur )
	apushcur.val()->disconnect_client_and_dispose();
  for ( apullcur = _prx_any_pull.cursor(); apullcur.is_valid(); ++apullcur )
	apullcur.val()->disconnect_client_and_dispose();
  for ( spushcur = _prx_struc_push.cursor(); spushcur.is_valid(); ++spushcur )
        spushcur.val()->disconnect_client_and_dispose();
  for ( spullcur = _prx_struc_pull.cursor(); spullcur.is_valid(); ++spullcur )
        spullcur.val()->disconnect_client_and_dispose();
  for ( bpushcur = _prx_batch_push.cursor(); bpushcur.is_valid(); ++bpushcur )
        bpushcur.val()->disconnect_client_and_dispose();
  for ( bpullcur = _prx_batch_pull.cursor(); bpullcur.is_valid(); ++bpullcur )
        bpullcur.val()->disconnect_client_and_dispose();

  _cosevent_push.drain();  _cosevent_pull.drain();
  _prx_any_push.clear();   _prx_any_pull.clear();
  _prx_struc_push.clear(); _prx_struc_pull.clear();
  _prx_batch_push.clear(); _prx_batch_pull.clear();

  if ( ! CORBA::is_nil(_prio_filter) ) WRAPPED_RELEASE(CosNF_Filter, _prio_filter);
  if ( ! CORBA::is_nil(_life_filter) ) WRAPPED_RELEASE(CosNF_Filter, _life_filter);
  if ( _fa_helper.has_filters() ) _fa_helper.remove_all_filters();
  if ( _qosprop )      delete _qosprop;

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  RDI_DUMP("ConsumerAdmin @ " << this << " [" << _serial << "] destroyed");
  WRAPPED_DISPOSE_IMPL(this);
}

void ConsumerAdmin_i::remove_proxy(EventProxyPushSupplier_i* prx)
{
  RDI_ListCursor<EventProxyPushSupplier_i *> lc;
  unsigned int    indx = 0;
  omni_mutex_lock lock(_oplock);

  for ( lc = _cosevent_push.cursor(); indx < _cosevent_push.length(); indx++ ) {
        EventProxyPushSupplier_i* proxy = *lc;
        if ( proxy == prx ) {
                _cosevent_push.remove(lc);
                _num_proxies -= 1;
                _prxy_exit    = RDI_TimeValue::timeofday();
                _channel->decr_consumers();
                if ( _channel->_push_consumer )         /* Deregister Proxy */
                        _channel->_push_consumer->remove_proxy(prx);
                break;
        }
        ++lc;
  }
}

void ConsumerAdmin_i::remove_proxy(EventProxyPullSupplier_i* prx)
{
  RDI_ListCursor<EventProxyPullSupplier_i *> lc;
  unsigned int    indx = 0;
  omni_mutex_lock lock(_oplock);

  for ( lc = _cosevent_pull.cursor(); indx < _cosevent_pull.length(); indx++ ) {
        EventProxyPullSupplier_i* proxy = *lc;
        if (  proxy == prx ) {
                _cosevent_pull.remove(lc);
                _num_proxies -= 1;
                _prxy_exit    = RDI_TimeValue::timeofday();
                _channel->decr_consumers();
                break;
        }
        ++lc;
  }
}

void ConsumerAdmin_i::remove_proxy(ProxyPushSupplier_i* prx)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("Remove proxy [" << prx->MyID() << "] from admin " << this);
  if ( _prx_any_push.exists(prx->MyID()) ) {
	_prx_any_push.remove(prx->MyID());
	_num_proxies -= 1;
	_prxy_exit    = RDI_TimeValue::timeofday();
	 _channel->decr_consumers();
	if ( _channel->_push_consumer )         /* Deregister Proxy */
		_channel->_push_consumer->remove_proxy(prx);
  } else {
	RDI_DUMP("Invalid proxy [" << prx->MyID() << "] provided");
  }
}

void ConsumerAdmin_i::remove_proxy(ProxyPullSupplier_i* prx)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("Remove proxy [" << prx->MyID() << "] from admin " << this);
  if ( _prx_any_pull.exists(prx->MyID()) ) {
        _prx_any_pull.remove(prx->MyID());
        _num_proxies -= 1;
        _prxy_exit    = RDI_TimeValue::timeofday();
        _channel->decr_consumers();
  } else {
        RDI_DUMP("Invalid proxy [" << prx->MyID() << "] provided");
  }
}

void ConsumerAdmin_i::remove_proxy(StructuredProxyPushSupplier_i* prx)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("Remove proxy [" << prx->MyID() << "] from admin " << this);
  if ( _prx_struc_push.exists(prx->MyID()) ) {
  	_prx_struc_push.remove(prx->MyID());
	_num_proxies -= 1;
	_prxy_exit    = RDI_TimeValue::timeofday();
	_channel->decr_consumers();
	if ( _channel->_push_consumer ) 	/* Deregister Proxy */
  		_channel->_push_consumer->remove_proxy(prx);
  } else {
	RDI_DUMP("Invalid proxy [" << prx->MyID() << "] provided");
  }
}

void ConsumerAdmin_i::remove_proxy(StructuredProxyPullSupplier_i* prx)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("Remove proxy [" << prx->MyID() << "] from admin " << this);
  if ( _prx_struc_pull.exists(prx->MyID()) ) {
        _prx_struc_pull.remove(prx->MyID());
        _num_proxies -= 1;
        _prxy_exit    = RDI_TimeValue::timeofday();
        _channel->decr_consumers();
  } else {
        RDI_DUMP("Invalid proxy [" << prx->MyID() << "] provided");
  }
}

void ConsumerAdmin_i::remove_proxy(SequenceProxyPushSupplier_i* prx)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("Remove proxy [" << prx->MyID() << "] from admin " << this);
  if ( _prx_batch_push.exists(prx->MyID()) ) {
  	_prx_batch_push.remove(prx->MyID());
	_num_proxies -= 1;
	_prxy_exit    = RDI_TimeValue::timeofday();
	_channel->decr_consumers();
	if ( _channel->_push_consumer )		/* Deregister Proxy */
  		_channel->_push_consumer->remove_proxy(prx);
  } else {
        RDI_DUMP("Invalid proxy [" << prx->MyID() << "] provided");
  }
}

void ConsumerAdmin_i::remove_proxy(SequenceProxyPullSupplier_i* prx)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("Remove proxy [" << prx->MyID() << "] from admin " << this);
  if ( _prx_batch_pull.exists(prx->MyID()) ) {
  	_prx_batch_pull.remove(prx->MyID());
	_num_proxies -= 1;
	_prxy_exit    = RDI_TimeValue::timeofday();
	_channel->decr_consumers();
  } else {
        RDI_DUMP("Invalid proxy [" << prx->MyID() << "] provided");
  }
}

void ConsumerAdmin_i::dispatch_event(RDI_StructuredEvent* event)
{
  RDI_ListCursor<EventProxyPushSupplier_i *> psc;
  RDI_ListCursor<EventProxyPullSupplier_i *> plc;
  unsigned int i=0;
  // Lock is acquired by caller [EventChannel_i::admin_dispatch()]
  for (i=0,psc=_cosevent_push.cursor(); i<_cosevent_push.length(); ++i,++psc)
	(*psc)->add_event(event);
  for (i=0,plc=_cosevent_pull.cursor(); i<_cosevent_pull.length(); ++i,++plc)
	(*plc)->add_event(event);
  // If a thread pool is used for pushing events to consumers,
  // make sure that any blocked push threads are notified ....
  if ( _cosevent_push.length() && _channel->_push_consumer )
	_channel->_push_consumer->signal_push_threads();
}

void ConsumerAdmin_i::dispatch_event(RDI_StructuredEvent* event,
				     RDI_FilterState_t    astat, 
				     RDI_TypeMap*         tmap)
{
  RDI_HashCursor<CosNA_ProxyID, ProxyPushSupplier_i *>           apushcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPullSupplier_i *>           apullcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushSupplier_i *> spushcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullSupplier_i *> spullcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushSupplier_i *>   bpushcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullSupplier_i *>   bpullcur;
  RDI_TypeMap::FNode_t* fnode=0;
  unsigned char match_found=0;
  RDI_TypeMap::FList_t flist;
  CORBA::Boolean evntadded=0;
 
  // The lock has already been acquired by the caller, which can be either 
  // EventChannel_i::proxy_dispatch()  or EventChannel_i::admin_dispatch()
  // In addition,  when consumers use 'subscription_change()'  to register 
  // interest in specific event types, we have 'has_filters()' return true
  // so that we execute the filter evaluation logic. However, there are no
  // filters registered and, hence, we need the '!fnode->_fltr' statement.

  if ( _num_proxies == 0 )
     return;

  const char* dname = event->get_domain_name();
  const char* tname = event->get_type_name();

  /*** Evaluate Filters for consumers using ProxyPushSupplier_i ***/

  for (apushcur = _prx_any_push.cursor(); apushcur.is_valid(); ++apushcur ) {
     if ( astat == OrMatch ) {
	apushcur.val()->add_event(event);
	evntadded = 1;
     } else if ( ! apushcur.val()->has_filters() ) {
	if ( (astat == NoFilters) || (astat == AndMatch) ) {
	   apushcur.val()->add_event(event);
	   evntadded = 1;
        }
     } else {
	tmap->lookup_begin(dname, tname, apushcur.val(), flist);
	if ( ! flist._star_star && ! flist._domn_star &&
	     ! flist._star_type && ! flist._domn_type ) {
	   if ( astat == OrMatch ) {
	      apushcur.val()->add_event(event);
	      evntadded = 1;
	   }
        } else {
	   match_found = 0;
	   for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
              if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                 match_found = 1;
                 break;
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }

           if ( match_found ) {
	      apushcur.val()->add_event(event);
	      evntadded = 1;
	   }
        }
        tmap->lookup_end();
     }
  }

  /*** Evaluate Filters for consumers using ProxyPullSupplier_i ***/

  for (apullcur = _prx_any_pull.cursor(); apullcur.is_valid(); ++apullcur ) {
     if ( astat == OrMatch ) {
	apullcur.val()->add_event(event);
     } else if ( ! apullcur.val()->has_filters() ) {
	if ( (astat == NoFilters) || (astat == AndMatch) ) {
	   apullcur.val()->add_event(event);
        }
     } else {
	tmap->lookup_begin(dname, tname, apullcur.val(), flist);
	if ( ! flist._star_star && ! flist._domn_star &&
	     ! flist._star_type && ! flist._domn_type ) {
	   if ( astat == OrMatch ) {
	      apullcur.val()->add_event(event);
	   }
        } else {
	   match_found = 0;
	   for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
              if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                 match_found = 1;
                 break;
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }

           if ( match_found ) {
	      apullcur.val()->add_event(event);
	   }
        }
        tmap->lookup_end();
     }
  }

  /*** Evaluate Filters for consumers using StructuredProxyPushSupplier_i ***/

  for ( spushcur=_prx_struc_push.cursor(); spushcur.is_valid(); ++spushcur ) {
     if ( astat == OrMatch ) {
	spushcur.val()->add_event(event);
	evntadded = 1;
     } else if ( ! spushcur.val()->has_filters() ) {
	if ( (astat == NoFilters) || (astat == AndMatch) ) {
	   spushcur.val()->add_event(event);
	   evntadded = 1;
        }
     } else {
	tmap->lookup_begin(dname, tname, spushcur.val(), flist);
	if ( ! flist._star_star && ! flist._domn_star &&
	     ! flist._star_type && ! flist._domn_type ) {
	   if ( astat == OrMatch ) {
	      spushcur.val()->add_event(event);
	      evntadded = 1;
	   }
        } else {
	   match_found = 0;
	   for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
              if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                 match_found = 1;
                 break;
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }

           if ( match_found ) {
	      spushcur.val()->add_event(event);
	      evntadded = 1;
	   }
        }
        tmap->lookup_end();
     }
  }

  /*** Evaluate Filters for consumers using StructuredProxyPullSupplier_i ***/

  for ( spullcur=_prx_struc_pull.cursor(); spullcur.is_valid(); ++spullcur ) {
     if ( astat == OrMatch ) {
        spullcur.val()->add_event(event);
     } else if ( ! spullcur.val()->has_filters() ) {
        if ( (astat == NoFilters) || (astat == AndMatch) ) {
           spullcur.val()->add_event(event);
	}
     } else {
        tmap->lookup_begin(dname, tname, spullcur.val(), flist);
        if ( ! flist._star_star && ! flist._domn_star &&
             ! flist._star_type && ! flist._domn_type ) {
           if ( astat == OrMatch ) {
              spullcur.val()->add_event(event);
	   }
        } else {
           match_found = 0;
           for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
              if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                 match_found = 1;
                 break;
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }

           if ( match_found ) {
              spullcur.val()->add_event(event);
	   }
        }
        tmap->lookup_end();
     }
  }

  /*** Evaluate Filters for consumers using SequenceProxyPushSupplier_i ***/

   for ( bpushcur=_prx_batch_push.cursor(); bpushcur.is_valid(); ++bpushcur ) {
     if ( astat == OrMatch ) {
        bpushcur.val()->add_event(event);
	evntadded = 1;
     } else if ( ! bpushcur.val()->has_filters() ) {
        if ( (astat == NoFilters) || (astat == AndMatch) ) {
           bpushcur.val()->add_event(event);
	   evntadded = 1;
 	}
     } else {
        tmap->lookup_begin(dname, tname, bpushcur.val(), flist);
        if ( ! flist._star_star && ! flist._domn_star &&
             ! flist._star_type && ! flist._domn_type ) {
           if ( astat == OrMatch ) {
              bpushcur.val()->add_event(event);
	      evntadded = 1;
	   }
        } else {
           match_found = 0;
           for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
              if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                 match_found = 1;
                 break;
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }

           if ( match_found ) {
              bpushcur.val()->add_event(event);
	      evntadded = 1;
	   }
        }
        tmap->lookup_end();
     }
  }

  /*** Evaluate Filters for consumers using SequenceProxyPullSupplier_i ***/

  for ( bpullcur=_prx_batch_pull.cursor(); bpullcur.is_valid(); ++bpullcur ) {
     if ( astat == OrMatch ) {
        bpullcur.val()->add_event(event);
     } else if ( ! bpullcur.val()->has_filters() ) {
        if ( (astat == NoFilters) || (astat == AndMatch) ) {
           bpullcur.val()->add_event(event);
	}
     } else {
        tmap->lookup_begin(dname, tname, bpullcur.val(), flist);
        if ( ! flist._star_star && ! flist._domn_star &&
             ! flist._star_type && ! flist._domn_type ) {
           if ( astat == OrMatch ) {
              bpullcur.val()->add_event(event);
	   }
        } else {
           match_found = 0;
           for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
              if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                 match_found = 1;
                 break;
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }
           if ( ! match_found ) {
              for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
                 if (!fnode->_fltr || fnode->_fltr->rdi_match(event,_channel)) {
                    match_found = 1;
                    break;
                 }
              }
           }

           if ( match_found ) {
              bpullcur.val()->add_event(event);
	   }
        }
        tmap->lookup_end();
     }
  }

  // If a thread pool is used for pushing events to consumers, 
  // make sure that any blocked push threads are notified ....
  if ( evntadded && _channel->_push_consumer )
        _channel->_push_consumer->signal_push_threads();
}

RDI_NotifQoS* ConsumerAdmin_i::qos_properties()
{ return _qosprop ? _qosprop : _channel->qos_properties(); }

ostream& operator << (ostream& out, ConsumerAdmin_i& adm)
{
  RDI_ListCursor<EventProxyPushSupplier_i *>                cpushcur;
  RDI_ListCursor<EventProxyPullSupplier_i *>                cpullcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPushSupplier_i *>           apushcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPullSupplier_i *>           apullcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushSupplier_i *> spushcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullSupplier_i *> spullcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushSupplier_i *>   bpushcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullSupplier_i *>   bpullcur;
  unsigned int cnt = 0;

  adm._oplock.lock();
  out << &adm <<" ID "  << setw(3) << adm._serial << " Born " << 
	 adm._time_born << " Last Proxy "  << adm._prxy_exit  << endl;
  if ( adm._qosprop )
  	out << adm._qosprop << endl;
  else
	out << "\tSame as QoSAdmin settings of Event Channel" << endl;
  cpushcur = adm._cosevent_push.cursor();
  for (cnt=0; cnt < adm._cosevent_push.length(); ++cnt, ++cpushcur)
	out << setw(4) << cnt << ": " << *(*cpushcur) << endl;
  cpullcur = adm._cosevent_pull.cursor();
  for (cnt=0; cnt < adm._cosevent_pull.length(); ++cnt, ++cpullcur)
	out << setw(4) << cnt << ": " << *(*cpullcur) << endl;
  cnt = 0;
  for (apushcur = adm._prx_any_push.cursor(); apushcur.is_valid(); ++apushcur)
        out << setw(4) << ++cnt << ": " << *(apushcur.val()) << endl;
  cnt = 0;
  for (apullcur = adm._prx_any_pull.cursor(); apullcur.is_valid(); ++apullcur)
        out << setw(4) << ++cnt << ": " << *(apullcur.val()) << endl;
  cnt = 0;
  for (spushcur = adm._prx_struc_push.cursor(); spushcur.is_valid(); ++spushcur)
        out << setw(4) << ++cnt << ": " << *(spushcur.val()) << endl;
  cnt = 0;
  for (spullcur = adm._prx_struc_pull.cursor(); spullcur.is_valid(); ++spullcur)
	out << setw(4) << ++cnt << ": " << *(spullcur.val()) << endl;
  cnt = 0;
  for (bpushcur = adm._prx_batch_push.cursor(); bpushcur.is_valid(); ++bpushcur)
        out << setw(4) << ++cnt << ": " << *(bpushcur.val()) << endl;
  cnt = 0;
  for (bpullcur = adm._prx_batch_pull.cursor(); bpullcur.is_valid(); ++bpullcur)
        out << setw(4) << ++cnt << ": " << *(bpullcur.val()) << endl;
  adm._oplock.unlock();
  return out;
}

// ------------------------------------------------------------- //
// SupplierAdmin_i implementation                                //
// ------------------------------------------------------------- //

SupplierAdmin_i::SupplierAdmin_i(EventChannel_i* channel,
				 CosNA_InterFilterGroupOperator op,
				 const CosNA_AdminID& serial) :
		_oplock(), _fa_helper(), _channel(channel), _qosprop(0), 
		_serial(serial), _and_or_oper(op), _prx_serial(1), 
		_num_proxies(0), 
		_evtypes(RDI_EventType::hash, RDI_EventType::rank),
		_cosevent_push(), _cosevent_pull(),
		_prx_any_push(RDI_ULongHash, RDI_ULongRank),
		_prx_any_pull(RDI_ULongHash, RDI_ULongRank),
		_prx_struc_push(RDI_ULongHash, RDI_ULongRank),
		_prx_struc_pull(RDI_ULongHash, RDI_ULongRank),
		_prx_batch_push(RDI_ULongHash, RDI_ULongRank),
		_prx_batch_pull(RDI_ULongHash, RDI_ULongRank)
{
  _time_born = RDI_TimeValue::timeofday();
  _prxy_exit = _time_born;
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_EventChannel_ptr SupplierAdmin_i::MyChannel( WRAPPED_IMPLARG_VOID )
{ return WRAPPED_IMPL2OREF(CosNA_EventChannel, _channel); }

CosEventChannelAdmin::ProxyPushConsumer_ptr 
SupplierAdmin_i::obtain_push_consumer( WRAPPED_IMPLARG_VOID )
{
  EventProxyPushConsumer_i* prx=0;
  RDI_DUMP("CosEvent ProxyPushConsumer creation requested");
  omni_mutex_lock lock(_oplock);
  if ( ! _channel->incr_suppliers() )
        return CosEventChannelAdmin::ProxyPushConsumer::_nil();
  if ( ! (prx = new EventProxyPushConsumer_i(this, _channel)) ) {
        _channel->decr_suppliers();
        return CosEventChannelAdmin::ProxyPushConsumer::_nil();
  }
  if ( _cosevent_push.insert_tail(prx) != 0 ) {
        _channel->decr_suppliers();
        delete prx;
        return CosEventChannelAdmin::ProxyPushConsumer::_nil();
  }
  _num_proxies += 1;
  return WRAPPED_IMPL2OREF(CosEventChannelAdmin::ProxyPushConsumer, prx);
}

CosEventChannelAdmin::ProxyPullConsumer_ptr
SupplierAdmin_i::obtain_pull_consumer( WRAPPED_IMPLARG_VOID )
{
  EventProxyPullConsumer_i* prx=0;
  RDI_DUMP("CosEvent ProxyPullConsumer creation requested");
  omni_mutex_lock lock(_oplock);
  if ( ! _channel->incr_suppliers() )
        return CosEventChannelAdmin::ProxyPullConsumer::_nil();
  if ( ! (prx = new EventProxyPullConsumer_i(this, _channel)) ) {
        _channel->decr_suppliers();
        return CosEventChannelAdmin::ProxyPullConsumer::_nil();
  }
  if ( _cosevent_pull.insert_tail(prx) != 0 ) {
        _channel->decr_suppliers();
        delete prx;
        return CosEventChannelAdmin::ProxyPullConsumer::_nil();
  }
   if ( _channel->_pull_supplier )       /* Register Pull Proxy */
	_channel->_pull_supplier->insert_proxy(prx);
  _num_proxies += 1;
  return WRAPPED_IMPL2OREF(CosEventChannelAdmin::ProxyPullConsumer, prx);
}

// Update the local hash table and if new entries are created or
// deleted, update the channel as well, after releasing the lock
// to avoid deadlocks due to reverse order of lock acquisitions.

void SupplierAdmin_i::offer_change(const CosN_EventTypeSeq& added,
                           	   const CosN_EventTypeSeq& deled   WRAPPED_IMPLARG )
{
  CORBA::ULong ix=0, sz=0, vl=0;
  CosN_EventTypeSeq new_added;
  CosN_EventTypeSeq old_deled;

  // First, validate the entries in the provided lists to make sure
  // that correct values are provided for these event types .......

  if ( ! RDI_EventType::valid_sequence(added, vl) ) 
     throw CosNC_InvalidEventType(added[vl]);
  if ( ! RDI_EventType::valid_sequence(deled, vl) ) 
     throw CosNC_InvalidEventType(deled[vl]);

  _oplock.lock();
  for (sz=0, ix = 0; ix < added.length(); ix++) {
     if ( _evtypes.lookup(added[ix], vl) ) {
	vl += 1; _evtypes.replace(added[ix], vl);
     } else {
	vl  = 1; _evtypes.insert(added[ix], vl);
	new_added.length(sz + 1);
	new_added[sz++] = added[ix];
     }
  }
  for (sz=0, ix = 0; ix < deled.length(); ix++) {
     if ( _evtypes.lookup(deled[ix], vl) ) {
	if ( vl == 1 ) {
	   _evtypes.remove(deled[ix]);
	   old_deled.length(sz + 1);
	   old_deled[sz++] = deled[ix];
	} else {
	   vl -= 1; _evtypes.replace(deled[ix], vl);
	}
     } else {
	RDI_DUMP("Invalid "<<deled[ix].domain_name<<"::"<<deled[ix].type_name);
     }
  }
  _oplock.unlock();
  if ( new_added.length() || old_deled.length() )
     _channel->offer_change(new_added, old_deled);
}

CosNA_ProxyIDSeq* SupplierAdmin_i::pull_consumers( WRAPPED_IMPLARG_VOID )
{
  RDI_HashCursor<CosNA_ProxyID, ProxyPullConsumer_i *>           ac;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullConsumer_i *> sc;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullConsumer_i *>   bc;
  unsigned long num;
  CosNA_ProxyIDSeq*  seq = new CosNA_ProxyIDSeq();
  RDI_Assert(seq, "Memory allocation failed -- ProxyIDSeq");
  _oplock.lock();
  num = _prx_any_pull.length() +
	_prx_struc_pull.length() + _prx_batch_pull.length();
  seq->length(num);
  for ( num=0, ac = _prx_any_pull.cursor(); ac.is_valid(); ++ac, ++num )
	(*seq)[num] = ac.key();
  for ( sc = _prx_struc_pull.cursor(); sc.is_valid(); ++sc, ++num )
	(*seq)[num] = sc.key();
  for ( bc = _prx_batch_pull.cursor(); bc.is_valid(); ++bc, ++num )
	(*seq)[num] = bc.key();
  _oplock.unlock();
  return seq;
}

CosNA_ProxyIDSeq* SupplierAdmin_i::push_consumers( WRAPPED_IMPLARG_VOID )
{
  RDI_HashCursor<CosNA_ProxyID, ProxyPushConsumer_i *>           ac;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushConsumer_i *> sc;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushConsumer_i *>  bc;
  unsigned long num;
  CosNA_ProxyIDSeq*  seq = new CosNA_ProxyIDSeq();
  RDI_Assert(seq, "Memory allocation failed -- ProxyIDSeq");
  _oplock.lock();
  num = _prx_any_push.length() +
	_prx_struc_push.length() + _prx_batch_push.length();
  seq->length(num);
  for ( num=0, ac = _prx_any_push.cursor(); ac.is_valid(); ++ac, ++num )
        (*seq)[num] = ac.key();
  for ( sc = _prx_struc_push.cursor(); sc.is_valid(); ++sc, ++num )
        (*seq)[num] = sc.key();        
  for ( bc = _prx_batch_push.cursor(); bc.is_valid(); ++bc, ++num )
        (*seq)[num] = bc.key();
  _oplock.unlock();
  return seq;
}

CosNA_ProxyConsumer_ptr SupplierAdmin_i::get_proxy_consumer(CosNA_ProxyID proxy_id   WRAPPED_IMPLARG )
{
  ProxyPushConsumer_i*           apush=0;
  ProxyPullConsumer_i*           apull=0;
  StructuredProxyPushConsumer_i* spush=0;
  SequenceProxyPushConsumer_i*   bpush=0;
  StructuredProxyPullConsumer_i* spull=0;
  SequenceProxyPullConsumer_i*   bpull=0;
  omni_mutex_lock lock(_oplock); 

  if ( proxy_id > _prx_serial ) 
	throw CosNA_ProxyNotFound();
  if ( _prx_any_push.lookup(proxy_id, apush) ) 
	return WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, apush);
  if ( _prx_any_pull.lookup(proxy_id, apull) ) 
	return WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, apull);
  if ( _prx_struc_push.lookup(proxy_id, spush) )
	return WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, spush);
  if ( _prx_struc_pull.lookup(proxy_id, spull) )
	return WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, spull);
  if ( _prx_batch_push.lookup(proxy_id, bpush) )
	return WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, bpush);
  if ( _prx_batch_pull.lookup(proxy_id, bpull) )
	return WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, bpull);
  throw CosNA_ProxyNotFound();
}

CosNA_ProxyConsumer_ptr SupplierAdmin_i::obtain_notification_pull_consumer(
                               CosNA_ClientType ctype, CosNA_ProxyID& proxy_id   WRAPPED_IMPLARG )
{
  CosNA_AdminLimit limit;
  omni_mutex_lock lock(_oplock);

  // Check with  the channel if we are about to exceed the maximum
  // number of suppliers supported by the channel

  RDI_DUMP("Creation of a new ProxyConsumer -- type " << ctype);
  if ( ! _channel->incr_suppliers() ) {
  	RDI_DUMP("Supplier limit has been reached creation failed");
        limit.name    = (const char *) "MaxSuppliers";
        limit.value <<= _channel->max_suppliers();
        throw CosNA_AdminLimitExceeded(limit);
  }

  if ( ctype == CosNA_ANY_EVENT ) {
	ProxyPullConsumer_i* prx = 0;
	if ( ! (prx = new ProxyPullConsumer_i(this, _channel, _prx_serial)) ) {
		_channel->decr_suppliers();
		return CosNA_ProxyConsumer::_nil();
	}
	proxy_id = _prx_serial++;
	if ( _prx_any_pull.insert(proxy_id, prx) != 0 ) {
		delete prx;
		_channel->decr_suppliers(); 
		return CosNA_ProxyConsumer::_nil();
	}
	if ( _channel->_pull_supplier )       /* Register Pull Proxy */
		_channel->_pull_supplier->insert_proxy(prx);
	_num_proxies += 1;
	return WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, prx);
  } else if ( ctype == CosNA_STRUCTURED_EVENT ) {
	StructuredProxyPullConsumer_i* prx = 
		new StructuredProxyPullConsumer_i(this,_channel,_prx_serial);
	if ( ! prx ) {
		_channel->decr_suppliers();
		return CosNA_ProxyConsumer::_nil();
	}
	proxy_id = _prx_serial++;
	if ( _prx_struc_pull.insert(proxy_id, prx) != 0 ) {
		delete prx;
		_channel->decr_suppliers();
		return CosNA_ProxyConsumer::_nil();
	}
	if ( _channel->_pull_supplier )        /* Register Pull Proxy */
		_channel->_pull_supplier->insert_proxy(prx);
	_num_proxies += 1;
	return WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, prx);
  } else if ( ctype == CosNA_SEQUENCE_EVENT ) {
	SequenceProxyPullConsumer_i* prx =
		new SequenceProxyPullConsumer_i(this,_channel,_prx_serial);
	if ( ! prx ) {
		_channel->decr_suppliers();
		return CosNA_ProxyConsumer::_nil();
	}
	proxy_id = _prx_serial++;
	if ( _prx_batch_pull.insert(proxy_id, prx) != 0 ) {
               	delete prx;
		_channel->decr_suppliers();
		return CosNA_ProxyConsumer::_nil();
	} 
	if ( _channel->_pull_supplier )       /* Register Pull Proxy */
		_channel->_pull_supplier->insert_proxy(prx);
	_num_proxies += 1;
	return WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, prx);
  }
  return CosNA_ProxyConsumer::_nil();
}

CosNA_ProxyConsumer_ptr SupplierAdmin_i::obtain_notification_push_consumer(
                                        CosNA_ClientType ctype, CosNA_ProxyID& proxy_id   WRAPPED_IMPLARG )
{
  CosNA_AdminLimit limit;
  omni_mutex_lock lock(_oplock);

  // Check with  the channel if we are about to exceed the maximum
  // number of suppliers supported by the channel

  RDI_DUMP("Creation of a new ProxyConsumer -- type " << ctype);
  if ( ! _channel->incr_suppliers() ) {
  	RDI_DUMP("Supplier limit has been reached creation failed");
        limit.name    = (const char *) "MaxSuppliers";
        limit.value <<= _channel->max_suppliers();
        throw CosNA_AdminLimitExceeded(limit);
  }

  if ( ctype == CosNA_ANY_EVENT ) {
	ProxyPushConsumer_i* prx = 0;
	if ( ! (prx = new ProxyPushConsumer_i(this, _channel, _prx_serial)) ) {
		_channel->decr_suppliers();
		return CosNA_ProxyConsumer::_nil();
	}
	proxy_id = _prx_serial++;
	if ( _prx_any_push.insert(proxy_id, prx) != 0 ) {
		delete prx;
		_channel->decr_suppliers(); 
		return CosNA_ProxyConsumer::_nil();
	}
	_num_proxies += 1;
	return WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, prx);
  } else if ( ctype == CosNA_STRUCTURED_EVENT ) {
	StructuredProxyPushConsumer_i* prx =
		new StructuredProxyPushConsumer_i(this,_channel,_prx_serial);
	if ( ! prx ) {
		_channel->decr_suppliers();
		return CosNA_ProxyConsumer::_nil();
	}
	proxy_id = _prx_serial++;
	if ( _prx_struc_push.insert(proxy_id, prx) != 0 ) {
		delete prx;
		_channel->decr_suppliers();
		return CosNA_ProxyConsumer::_nil();
	}
	_num_proxies += 1;
	return WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, prx);
  } else if ( ctype == CosNA_SEQUENCE_EVENT ) {
	SequenceProxyPushConsumer_i* prx = 
		new SequenceProxyPushConsumer_i(this,_channel,_prx_serial);
	if ( ! prx ) {
		_channel->decr_suppliers();
		return CosNA_ProxyConsumer::_nil();
	}
	proxy_id = _prx_serial++;
	if ( _prx_batch_push.insert(proxy_id, prx) != 0 ) {
               	delete prx;
		_channel->decr_suppliers();
		return CosNA_ProxyConsumer::_nil();
	} 
	_num_proxies += 1;
	return WRAPPED_IMPL2OREF(CosNA_ProxyConsumer, prx);
  }
  return CosNA_ProxyConsumer::_nil();
}

CosN_QoSProperties* SupplierAdmin_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = qos_properties();
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_C_ADMIN);
}

void SupplierAdmin_i::set_qos(const CosN_QoSProperties& r_qos   WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = (_num_proxies ? 1 : 0);
  RDI_NotifQoS*   p_qos = _channel->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);
  
  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_ADMIN,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos); 
}

void SupplierAdmin_i::validate_qos(
        const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq 
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = (_num_proxies ? 1 : 0);
  RDI_NotifQoS*   p_qos = _channel->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_ADMIN,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void SupplierAdmin_i::destroy( WRAPPED_IMPLARG_VOID )
{
  CosN_EventTypeSeq added;
  CosN_EventTypeSeq deled;

  if ( _serial == 0 ) {
	RDI_DUMP("Default SupplierAdmin is not allowed to be destroyed!");
	return;
  }
  // If there are any event types registered with this administrative
  // object, update the channel's aggregate information about them.
  // NOTE: we release the lock before we inkoke channel operations so
  //      that we avoid potential deadlocks when the channel is being
  //      destroyed at the same time
  _oplock.lock();
  if ( _evtypes.length() ) {
     RDI_HashCursor<CosN_EventType, CORBA::ULong> curs;
     CORBA::ULong ix=0;
     deled.length( _evtypes.length() );
     for ( ix=0, curs = _evtypes.cursor(); curs.is_valid(); ++ix, ++curs )
	deled[ix] = curs.key();
  }
  _oplock.unlock();
  RDI_DUMP("Destroy SupplierAdmin @ " << this << " [" << _serial << "]");
  _channel->unregister(this);
  _channel->offer_change(added, deled);
  disconnect_clients_and_dispose();
}

void SupplierAdmin_i::disconnect_clients_and_dispose()
{
  unsigned int i;
  RDI_ListCursor<EventProxyPushConsumer_i *>                cpushcur;
  RDI_ListCursor<EventProxyPullConsumer_i *>                cpullcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPushConsumer_i *>           apushcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPullConsumer_i *>           apullcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushConsumer_i *> spushcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullConsumer_i *> spullcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushConsumer_i *>   bpushcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullConsumer_i *>   bpullcur;

  _oplock.lock();
  RDI_DUMP("Destruction of SupplierAdmin @ " << this << " [" << _serial << "]");
  cpushcur = _cosevent_push.cursor();
  for (i=0; i < _cosevent_push.length(); ++i, ++cpushcur)
        (*cpushcur)->disconnect_client_and_dispose();
  cpullcur = _cosevent_pull.cursor();
  for (i=0; i < _cosevent_pull.length(); ++i, ++cpullcur)
        (*cpullcur)->disconnect_client_and_dispose();
  for ( apushcur = _prx_any_push.cursor(); apushcur.is_valid(); ++apushcur )
        apushcur.val()->disconnect_client_and_dispose();
  for ( apullcur = _prx_any_pull.cursor(); apullcur.is_valid(); ++apullcur )
        apullcur.val()->disconnect_client_and_dispose();
  for ( spushcur = _prx_struc_push.cursor(); spushcur.is_valid(); ++spushcur )
        spushcur.val()->disconnect_client_and_dispose();
  for ( spullcur = _prx_struc_pull.cursor(); spullcur.is_valid(); ++spullcur )
        spullcur.val()->disconnect_client_and_dispose();
  for ( bpushcur = _prx_batch_push.cursor(); bpushcur.is_valid(); ++bpushcur )
        bpushcur.val()->disconnect_client_and_dispose();
  for ( bpullcur = _prx_batch_pull.cursor(); bpullcur.is_valid(); ++bpullcur )
        bpullcur.val()->disconnect_client_and_dispose();

  _cosevent_push.drain();  _cosevent_pull.drain();
  _prx_any_push.clear();   _prx_any_pull.clear();
  _prx_struc_push.clear(); _prx_struc_pull.clear();
  _prx_batch_push.clear(); _prx_batch_pull.clear();

  if ( _fa_helper.has_filters() ) _fa_helper.remove_all_filters();
  if ( _qosprop )      delete _qosprop;

  // NOTE: we have to release the mutex in order to avoid EBUSY errors
  _oplock.unlock();
  RDI_DUMP("SupplierAdmin @ " << this << " [" << _serial << "] destroyed");
  WRAPPED_DISPOSE_IMPL(this);
}

void SupplierAdmin_i::remove_proxy(EventProxyPushConsumer_i* prx)
{
  RDI_ListCursor<EventProxyPushConsumer_i *> lc;
  unsigned int    indx = 0;
  omni_mutex_lock lock(_oplock);
 
  for ( lc = _cosevent_push.cursor(); indx < _cosevent_push.length(); indx++ ) {
        EventProxyPushConsumer_i* proxy = *lc;
        if ( proxy == prx ) {
                _cosevent_push.remove(lc);
                _num_proxies -= 1;
                _prxy_exit    = RDI_TimeValue::timeofday();
  		_channel->decr_suppliers();
                break;
        }
        ++lc;
  }
}

void SupplierAdmin_i::remove_proxy(EventProxyPullConsumer_i* prx)
{
  RDI_ListCursor<EventProxyPullConsumer_i *> lc;
  unsigned int    indx = 0;
  omni_mutex_lock lock(_oplock);
 
  for ( lc = _cosevent_pull.cursor(); indx < _cosevent_pull.length(); indx++ ) {
        EventProxyPullConsumer_i* proxy = *lc;
        if ( proxy == prx ) {
                _cosevent_pull.remove(lc);
                _num_proxies -= 1;
                _prxy_exit    = RDI_TimeValue::timeofday();
		_channel->decr_suppliers();
		if ( _channel->_pull_supplier )	/* Deregister Pull Proxy */
			_channel->_pull_supplier->remove_proxy(prx);
                break;
        }
        ++lc;
  }
}

void SupplierAdmin_i::remove_proxy(ProxyPushConsumer_i* prx)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("Remove proxy [" << prx->MyID() << "] from admin " << this);
  if ( _prx_any_push.exists(prx->MyID()) ) {
        _prx_any_push.remove(prx->MyID());
        _num_proxies -= 1;
        _prxy_exit    = RDI_TimeValue::timeofday();
        _channel->decr_suppliers();
  } else {
        RDI_DUMP("Invalid proxy [" << prx->MyID() << "] provided");
  }
}

void SupplierAdmin_i::remove_proxy(ProxyPullConsumer_i* prx)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("Remove proxy [" << prx->MyID() << "] from admin " << this);
  if ( _prx_any_pull.exists(prx->MyID()) ) {
        _prx_any_pull.remove(prx->MyID());
        _num_proxies -= 1;
        _prxy_exit    = RDI_TimeValue::timeofday();
        _channel->decr_suppliers();
	if ( _channel->_pull_supplier )       /* Deregister Pull Proxy */
		_channel->_pull_supplier->remove_proxy(prx);
  } else {
        RDI_DUMP("Invalid proxy [" << prx->MyID() << "] provided");
  }
}

void SupplierAdmin_i::remove_proxy(StructuredProxyPushConsumer_i* prx)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("Remove proxy [" << prx->MyID() << "] from admin " << this);
  if ( _prx_struc_push.exists(prx->MyID()) ) {
  	_prx_struc_push.remove(prx->MyID());
	_num_proxies -= 1;
	_prxy_exit    = RDI_TimeValue::timeofday();
	_channel->decr_suppliers();
  } else {
        RDI_DUMP("Invalid proxy [" << prx->MyID() << "] provided");
  }
}

void SupplierAdmin_i::remove_proxy(StructuredProxyPullConsumer_i* prx)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("Remove proxy [" << prx->MyID() << "] from admin " << this);
  if ( _prx_struc_pull.exists(prx->MyID()) ) {
  	_prx_struc_pull.remove(prx->MyID());
	_num_proxies -= 1;
	_prxy_exit    = RDI_TimeValue::timeofday();
	_channel->decr_suppliers();
	if ( _channel->_pull_supplier )       /* Deregister Pull Proxy */
		_channel->_pull_supplier->remove_proxy(prx);
  } else {
        RDI_DUMP("Invalid proxy [" << prx->MyID() << "] provided");
  }
}

void SupplierAdmin_i::remove_proxy(SequenceProxyPushConsumer_i* prx)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("Remove proxy [" << prx->MyID() << "] from admin " << this);
  if ( _prx_batch_push.exists(prx->MyID()) ) {
  	_prx_batch_push.remove(prx->MyID());
	_num_proxies -= 1;
	_prxy_exit    = RDI_TimeValue::timeofday();
	_channel->decr_suppliers();
  } else {
        RDI_DUMP("Invalid proxy [" << prx->MyID() << "] provided");
  }
}

void SupplierAdmin_i::remove_proxy(SequenceProxyPullConsumer_i* prx)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("Remove proxy [" << prx->MyID() << "] from admin " << this);
  if ( _prx_batch_pull.exists(prx->MyID()) ) {
  	_prx_batch_pull.remove(prx->MyID());
	_num_proxies -= 1;
	_prxy_exit    = RDI_TimeValue::timeofday();
	_channel->decr_suppliers();
	if ( _channel->_pull_supplier )       /* Deregister Pull Proxy */
		_channel->_pull_supplier->remove_proxy(prx);
  } else {
        RDI_DUMP("Invalid proxy [" << prx->MyID() << "] provided");
  }
}

// ****************************************************************** //
// While this module does not support CosNC_NotifySubscribe, //
// we added the following operation to propagate changes in the event //
// types referenced in consumer filters to the actual suppliers.      //
//                                                                    //
// NOTE: Suppliers using the  CORBA Event Specification interfaces do //
//       not support 'subscription_change()' and, hence, they are not //
//       included in this operation                                   //
// ****************************************************************** //

void SupplierAdmin_i::subscription_change(
			const CosN_EventTypeSeq& added,
			const CosN_EventTypeSeq& deled)
{
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushConsumer_i *> spushcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullConsumer_i *> spullcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushConsumer_i *>   bpushcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullConsumer_i *>   bpullcur;
  omni_mutex_lock lock(_oplock);

  for ( spushcur = _prx_struc_push.cursor(); spushcur.is_valid(); ++spushcur ) {
     if ( ! spushcur.val()->_offauto &&
	  (spushcur.val()->_pxstate == RDI_Connected) ) {
	try { spushcur.val()->_supplier->subscription_change(added, deled); }
  	catch (...) { /* We assume that the supplier does not support it */ }
     }
  }
  for ( spullcur = _prx_struc_pull.cursor(); spullcur.is_valid(); ++spullcur ) {
     if ( ! spullcur.val()->_offauto && 
          (spullcur.val()->_pxstate == RDI_Connected) ) {
	try { spullcur.val()->_supplier->subscription_change(added, deled); }
        catch (...) { /* We assume that the supplier does not support it */ }
     }  
  }
  for ( bpushcur = _prx_batch_push.cursor(); bpushcur.is_valid(); ++bpushcur ) {
     if ( ! bpushcur.val()->_offauto && 
          (bpushcur.val()->_pxstate == RDI_Connected) ) {
        try { bpushcur.val()->_supplier->subscription_change(added, deled); }
        catch (...) { /* We assume that the supplier does not support it */ }
     } 
  }
  for ( bpullcur = _prx_batch_pull.cursor(); bpullcur.is_valid(); ++bpullcur ) {
     if ( ! bpullcur.val()->_offauto && 
          (bpullcur.val()->_pxstate == RDI_Connected) ) {
        try { bpullcur.val()->_supplier->subscription_change(added, deled); }
        catch (...) { /* We assume that the supplier does not support it */ }
     } 
  }
}

CORBA::Boolean SupplierAdmin_i::match_event(const CORBA::Any& event)
{
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean matched = 1;
  if ( _fa_helper.has_filters() ) {
        CosNF_FilterIDSeq* fltrseq = get_all_filters();
        CosNF_Filter_ptr   fltr = CosNF_Filter::_nil();
        matched = 0;
        for (CORBA::ULong ix=0; ix < fltrseq->length(); ix++) {
                fltr = get_filter((*fltrseq)[ix]);
                if ( fltr->match(event) ) {
                        matched = 1;
                        break;
                }
        }
        delete fltrseq;
  }
  return matched;
}

CORBA::Boolean SupplierAdmin_i::match_event(RDI_StructuredEvent* event)
{
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean matched = 1;
  if ( _fa_helper.has_filters() ) {
	CosNF_FilterIDSeq* fltrseq = get_all_filters();
	CosNF_Filter_ptr filter = CosNF_Filter::_nil();
	Filter_i* rdfltr = 0;
	matched = 0;
	for (CORBA::ULong ix=0; ix < fltrseq->length(); ix++) {
		filter = get_filter((*fltrseq)[ix]);
		rdfltr = Filter_i::Filter2Filter_i(filter);
		RDI_Assert(rdfltr, "Filter was not created by READY");
		if ( rdfltr->rdi_match(event, _channel) ) { 
			matched = 1;
			break;
		}
	}
	delete fltrseq;
  }
  return matched;
}

RDI_NotifQoS* SupplierAdmin_i::qos_properties()
{ return _qosprop ? _qosprop : _channel->qos_properties(); }

ostream& operator << (ostream& out, SupplierAdmin_i& adm)
{
  RDI_ListCursor<EventProxyPushConsumer_i *>                cpushcur;
  RDI_ListCursor<EventProxyPullConsumer_i *>                cpullcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPushConsumer_i *>           apushcur;
  RDI_HashCursor<CosNA_ProxyID, ProxyPullConsumer_i *>           apullcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPushConsumer_i *> spushcur;
  RDI_HashCursor<CosNA_ProxyID, StructuredProxyPullConsumer_i *> spullcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPushConsumer_i *>   bpushcur;
  RDI_HashCursor<CosNA_ProxyID, SequenceProxyPullConsumer_i *>   bpullcur;
  unsigned int cnt = 0;

  adm._oplock.lock();
  out << &adm <<" ID "  << setw(3) << adm._serial << " Born " << 
	 adm._time_born << " Last Proxy "  << adm._prxy_exit  << endl;
   if ( adm._qosprop )
        out << adm._qosprop << endl;
  else  
        out << "\tSame as QoSAdmin settings of Event Channel" << endl;
  cpushcur = adm._cosevent_push.cursor();
  for (cnt=0; cnt < adm._cosevent_push.length(); ++cnt, ++cpushcur)
        out << setw(4) << cnt << ": " << *(*cpushcur) << endl;
  cpullcur = adm._cosevent_pull.cursor();
  for (cnt=0; cnt < adm._cosevent_pull.length(); ++cnt, ++cpullcur)
        out << setw(4) << cnt << ": " << *(*cpullcur) << endl;
  cnt = 0;
  for (apushcur = adm._prx_any_push.cursor(); apushcur.is_valid(); ++apushcur)
        out << setw(4) << ++cnt << ": " << *(apushcur.val()) << endl;
  cnt = 0;
  for (apullcur = adm._prx_any_pull.cursor(); apullcur.is_valid(); ++apullcur)
        out << setw(4) << ++cnt << ": " << *(apullcur.val()) << endl;
  cnt = 0;
  for (spushcur = adm._prx_struc_push.cursor(); spushcur.is_valid(); ++spushcur)
	out << setw(4) << ++cnt << ": " << *(spushcur.val()) << endl;
  cnt = 0;
  for (spullcur = adm._prx_struc_pull.cursor(); spullcur.is_valid(); ++spullcur)
	out << setw(4) << ++cnt << ": " << *(spullcur.val()) << endl;
  cnt = 0;
  for (bpushcur = adm._prx_batch_push.cursor(); bpushcur.is_valid(); ++bpushcur)
	out << setw(4) << ++cnt << ": " << *(bpushcur.val()) << endl;
  cnt = 0;
  for (bpullcur = adm._prx_batch_pull.cursor(); bpullcur.is_valid(); ++bpullcur)
	out << setw(4) << ++cnt << ": " << *(bpullcur.val()) << endl;
  adm._oplock.unlock();
  return out;
}

