// -*- Mode: C++; -*-
//                              File      : CosNotification_i.h
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//     set of C++ definitions for the CosNotification module
//
 
/*
$Log: CosNotification_i.h,v $
Revision 1.15  2000/08/22 18:23:47  alcfp
added description to each file

Revision 1.14  2000/08/16 20:18:54  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#ifndef _COS_NOTIFICATION_I_H_
#define _COS_NOTIFICATION_I_H_

#include <iostream.h>
#include "omnithread.h"
#include "CosNotify.h"

////////////////////////////////////////////////////////////////////////////
// QoS properties can be set at multiple levels: proxy and admin objects, //
// event channels, and individual event messages.                         //
//                                                                        //
//   eventReliability      : BestEffort or Persistent - can be set on the //
//                         : event message or the event channel           //
//   connectionReliability : BestEffort or Persistent - can be set on the //
//                         : event channel only                           //
//   priority              : The priority used to dispatch matced events  //
//   timeout               : Expiration time assigned to event messages   //
//   startTimeSupported    : Scheduled future delivery of events starts   //
//   stopTimeSupported     : Scheduled future delivery of events ends     //
//   orderPolicy           : Ordering policy used for dispatching events  //
//   discardPolicy         : Discarding policy used when queues overflow  //
//   pacingInterval        : Time period for collecting events before the //
//                         : delivery to a consumer                       // 
//   maxEventsPerConsumer  : Maximum number of queued events for a given  //
//                         : consumer -- 0 does not impose any limits     //
//   maximumBatchSize      : Maximum number of events in a batch; used by //
//                         : consumers that receive sequences of events   //
////////////////////////////////////////////////////////////////////////////

enum RDI_ObjectKind { RDI_EMESSAGE, RDI_ECHANNEL, RDI_C_ADMIN,  RDI_S_ADMIN,
                      RDI_S_AnyPRX, RDI_S_StrPRX, RDI_S_SeqPRX, RDI_C_AnyPRX,
		      RDI_C_StrPRX, RDI_C_SeqPRX };

class RDI_NotifQoS {
public:
  RDI_NotifQoS();
  RDI_NotifQoS(const RDI_NotifQoS& qos)	{ *this = qos; }
  
  RDI_NotifQoS& operator = (const RDI_NotifQoS& qos);

  CosN_QoSProperties* get_qos(RDI_ObjectKind otype) const;

  // Set some/all values of the QoS properties based on the values given
  // in 'r_qos'.  We assume that validation has already taken place

  void set_qos(const CosN_QoSProperties& r_qos);
  
  // Validate the requested QoS properties, 'r_qos', for a given object.
  // The current QoS properties of this object are 'a_qos', and its kind
  // is 'otype'.  In case the object has a parent, the QoS properties of
  // its parent, 'p_qos', are used to resolve conflicts.  Since some QoS
  // properties cannot be changed once set for objects that manage other
  // objects, 'child' is TRUE when the current object has sub-objects

  static CORBA::Boolean validate(const CosN_QoSProperties& r_qos,
				 const RDI_NotifQoS& a_qos,
				 RDI_ObjectKind otype,
				 CosN_PropertyErrorSeq& error,
				 CosN_NamedPropertyRangeSeq& range,
				 CORBA::Boolean child=0,
				 RDI_NotifQoS* p_qos=0);

  friend ostream& operator << (ostream& out, const RDI_NotifQoS& qos);
public:
  enum { NUM_PROPERTIES = 11 };

  CORBA::Short    eventReliability;
  CORBA::Short    connectionReliability;
  CORBA::Short    priority;
  TimeBase::TimeT timeout;
  CORBA::Boolean  startTimeSupported;
  CORBA::Boolean  stopTimeSupported;
  CORBA::Short    orderPolicy;
  CORBA::Short    discardPolicy;
  TimeBase::TimeT pacingInterval;
  CORBA::Long     maxEventsPerConsumer;
  CORBA::Long     maximumBatchSize;
};

////////////////////////////////////////////////////////////////////////////
// The following are the Administrative QoS properties that can be set on //
// a per-channel basis. The first four are part of the CORBA Notification //
// specification. The rest are supported by READY for performance tuning. //
//                                                                        //
//   maxQueueLength  : Max size of internal event queues - 0 for no limit //
//   maxConsumers    : Max number of connected consumers - 0 for no limit //
//   maxSuppliers    : Max number of connected suppliers - 0 for no limit //
//   rejectNewEvents : If TRUE, new events are rejected on queue overflow //
//   numAdminGroups  : # administrative groups - should be at least 1     //
//   numAdminThreads : # threads for filter evaluation at the admin level //
//                   : These threads perfrom proxy filter evaluation when //
//                   : 'numProxyThreads' is 0 - should be at least 1      //
//   numProxyThreads : # threads used for proxy filter evaluation         //
//   numPushThreads  : # threads used for dispatching events to consumers //
//                   : If 0, each push consumer has its own thread        //
//   numPullThreads  : # threads used for 'pulling' events from suppliers //
//                   : If 0, each pull suppliers has its own thread       //
//   pullEventPeriod : Period, in milliseconds, when pulling events. When //
//                   : set to 0, we ask for new events without any delays //
////////////////////////////////////////////////////////////////////////////

class RDI_AdminQoS {
public:
  RDI_AdminQoS() : maxQueueLength(0),  maxConsumers(0),   maxSuppliers(0),
		   rejectNewEvents(1), numAdminGroups(2), numAdminThreads(2),
		   numProxyThreads(0), numPushThreads(4), numPullThreads(0),
		   pullEventPeriod(100)	{;}
  RDI_AdminQoS(const RDI_AdminQoS& qos)	{ *this = qos; }

  // Generate the 'CosN_AdminProperties' list from the
  // existing values for the various administrative QoS properties

  CosN_AdminProperties* to_admin() const;

  // Assuming that the provided 'CosN_AdminProperties'
  // list is valid, set the administrative QoS properties 

  void from_admin(const CosN_AdminProperties& a_qos);

  // Validate the provided QoS property list based on the existing
  // settings.  Invalid properties/values are recorded in 'error'.
  // This member can be called by the ChannelFactory or a Channel.
  // In the latter case, some of the existing QoS values cannot be
  // changed once they have been set to a value other than 0

  CORBA::Boolean validate(const CosN_AdminProperties& a_qos,
			  CosN_PropertyErrorSeq& error,
			  CORBA::Boolean factory=1);

  RDI_AdminQoS& operator = (const RDI_AdminQoS& qos);

  friend ostream& operator << (ostream& out, const RDI_AdminQoS& aqos);
public:
  CORBA::Long    maxQueueLength;
  CORBA::Long    maxConsumers;
  CORBA::Long    maxSuppliers;
  CORBA::Boolean rejectNewEvents;
  CORBA::UShort  numAdminGroups;
  CORBA::UShort  numAdminThreads;
  CORBA::UShort  numProxyThreads;
  CORBA::UShort  numPushThreads;
  CORBA::UShort  numPullThreads;
  CORBA::UShort  pullEventPeriod;
};

#endif
