// This application implements a push consumer that connects
// to a PUSH_STRUCTURED proxy.
// It consumes CosNotification::StructuredEvent events which 
// have a CORBA::ULong value in the remainder_of_body field.
 
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iomanip.h>
#include <iostream.h>
#include "corba_wrappers.h"
#include "CosNotify.h"
#include "CosNotifyComm_i.h"

CORBA::Boolean verbose  = 0;
CORBA::ULong   num_push = 0;
int            numProxy = 1;
unsigned int   remProxy = 1;
omni_mutex     mutex;
omni_condition finish(&mutex);
static CORBA::ULong prevVal = 0;

/////////////////////////////////////////////////////////////////////
// *************************************************************** //
/////////////////////////////////////////////////////////////////////


class MyPushConsumer_i : 
	WRAPPED_SKELETON_SUPER(CosNC::, StructuredPushConsumer) {
public:
  MyPushConsumer_i(int max_numb=0) : _max_numb(max_numb)  {;}

  inline void push_structured_event(const CosN_StructuredEvent& event 
				    WRAPPED_DECLARG );
  void disconnect_structured_push_consumer( WRAPPED_DECLARG_VOID ) {;}
  inline void offer_change(const CosN_EventTypeSeq&, 
			   const CosN_EventTypeSeq& WRAPPED_DECLARG );
protected:
  CORBA::ULong _max_numb;
};

inline void MyPushConsumer_i::push_structured_event(
			const CosN_StructuredEvent& ev WRAPPED_IMPLARG )
{
  CORBA::ULong l;
  ev.remainder_of_body >>= l;
  mutex.lock();
  num_push += 1;

  if ( numProxy == 1 ) {
  	if ( prevVal == 0 ) {
		prevVal = l;
  	} else {
		if ( prevVal + 1 != l && verbose )
			cout << "Gap: prev " << prevVal << " new " << l << endl;
		prevVal = l;
  	}
	if ( verbose ) {
		cout << setw(4) << num_push << " : " <<
			ev.header.fixed_header.event_type.domain_name << "::" <<
			ev.header.fixed_header.event_type.type_name << " -- ";
		cout << l << endl;
	}
  }

  if ( _max_numb && (num_push >= _max_numb) ) {
	if ( remProxy > 0 ) 
		remProxy -= 1;
	finish.signal();
  }
  mutex.unlock();
}

inline void MyPushConsumer_i::offer_change(const CosN_EventTypeSeq& added,
					   const CosN_EventTypeSeq& deled
			  	   	         WRAPPED_IMPLARG )
{
  CORBA::ULong indx;
  for (indx = 0; indx < added.length(); indx++) {
        cout << "\t+ " << added[indx].domain_name << 
		"::" << added[indx].type_name << endl;
  }
  for (indx = 0; indx < deled.length(); indx++) {
        cout << "\t- " << deled[indx].domain_name << 
		"::" << deled[indx].type_name << endl;
  }
}

/////////////////////////////////////////////////////////////////////
// *************************************************************** //
/////////////////////////////////////////////////////////////////////

static void usage(const char* pname)
{
   cout << "Usage: " << pname << " [-d numb] [-n name] [-p numb] [-v]" << endl;
   cout << "       -d numb : disconnect after numb pushes" << endl;
   cout << "       -n name : channel name [\"EventChannel\"]" << endl;
   cout << "       -p numb : number of SupplierProxies to create" << endl;
   cout << "       -v      : verbose output mode" << endl;
}

int main(int argc, char** argv)
{
  CosNA_StructuredProxyPushSupplier_var* supp;
  CosNA_EventChannel_ptr channel;
  MyPushConsumer_i** cons = 0;
  unsigned int numEvent=0;
  char *channelName = (char *) "EventChannel";
  char *channelKind = (char *) "EventChannel";
  int c, i;

  // Parse command line arguments and initialize global variables

  while ( (c = getopt(argc, argv,"d:n:p:v")) != EOF ) {
  	switch (c) {
        case 'd': numEvent = (unsigned int) atoi(optarg);
                  break;
        case 'n': channelName = optarg;
		  channelKind = optarg;
                  break;
	case 'p': numProxy = (unsigned int) atoi(optarg);
		  remProxy = numProxy;
		  break;
 	case 'v': verbose = 1;
		  break;
        default : usage(argv[0]);
                  exit(-1);
     }
  }

  CORBA::ORB_ptr orb = WRAPPED_ORB_INIT(argc, argv);
  CORBA::BOA_ptr boa = WRAPPED_BOA_INIT(orb, argc, argv);

  cons = new MyPushConsumer_i * [numProxy];
  supp = new CosNA_StructuredProxyPushSupplier_var [numProxy];

  char servant_nm[100];
  for (i=0; i < numProxy; i++) {
	if ( ! (cons[i] = new MyPushConsumer_i(numEvent)) ) {
		cerr << "Failed to allocate proxy object" << endl;
		exit(-1);
	}
	sprintf(servant_nm, "struct_push_consumer_%d", i+1);
	WRAPPED_ORB_REGISTER_SERVANT_NAME(orb, servant_nm);
	WRAPPED_BOA_OBJ_IS_READY(boa, cons[i]);
  }

  WRAPPED_IMPL_IS_READY_FORK(boa, "struct_push_consumer_1");

  try {
  	CosNaming::Name                          name;
	CORBA::Object_var                        echannel_ref;
	CORBA::Object_var                        name_service;
  	CosNaming::NamingContext_var             name_context;
	CosNA_AdminID           caID;
  	CosNA_ProxyID           pxID;
	CosNA_ObtainInfoMode    omode;
  	CosNA_ConsumerAdmin_var cadmin;

	if ( verbose ) 
		cout << "Resolving reference to: 'NameService'" << endl;
	name_service = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService"); 
	name_context = CosNaming::NamingContext::_narrow(name_service);
	if ( CORBA::is_nil(name_context) ) {
		cerr << "Failed to obtain context for NameService" << endl;
		exit(1);
	}
	name.length(1);
	name[0].id   = CORBA::string_dup(channelName);
	name[0].kind = CORBA::string_dup(channelKind);
	if ( verbose ) 
		cout << "Resolving reference to: '"<<channelName<<"'" << endl;
	echannel_ref = name_context->resolve(name);
        channel = CosNA_EventChannel::_narrow(echannel_ref);
        if ( CORBA::is_nil(channel) ) {
                cerr << "Failed to narrow Event Channel !" << endl;
		exit(1);
        }

	if ( verbose ) 
		cout << "Releasing reference to: 'NameService'" << endl;
	if ( verbose ) 
		cout << "Creating new ConsumerAdmin object" << endl;
	cadmin = channel->new_for_consumers(CosNA_AND_OP,caID);
	if ( CORBA::is_nil(cadmin) ) {
		cerr << "Failed to find Consumer Admin !" << endl;
		exit(1);
	}
  	if ( verbose ) 
  		cout << "Created ConsumerAdmin with ID " << caID << endl;

  	CosNA_ProxySupplier_var psupp;

	for (i=0; i < numProxy; i++) {
		if ( verbose )
			cout << "Creating proxy for StructuredSupplier" << endl;
		psupp = cadmin->obtain_notification_push_supplier(
				CosNA_STRUCTURED_EVENT, pxID);
		supp[i] = CosNA_StructuredProxyPushSupplier::_narrow(psupp);
		if ( CORBA::is_nil(supp[i]) ) {
			cerr << "Failed to find Proxy Supplier !" << endl;
			exit(-1);
		}
		if ( verbose ) {
			cout << "Created proxy with ID " << pxID << endl;
			cout << "Connecting consumer object to proxy" << endl;
		}
		supp[i]->connect_structured_push_consumer(cons[i]->_this());

		// Request from the very first proxy to be notified about
		// changes in the offered types in the future
		if ( i == 0 ) {
			omode = CosNA_NONE_NOW_UPDATES_ON;
			supp[i]->obtain_offered_types(omode);
		}
	}

	while ( remProxy ) {
		cout << "Waiting for " << remProxy << " proxies to end" << endl;
		mutex.lock();
        	finish.wait();
        	mutex.unlock();
  	}

  	// Suspend connection for all consumers, disconnect them, and
  	// dispose of all the consumer objects created

  	for (i=0; i < numProxy; i++) {
        	supp[i]->suspend_connection();
        	if ( verbose )
               		cout << "Disconnecting consumer from channel" << endl;
        	supp[i]->disconnect_structured_push_supplier();
		WRAPPED_DISPOSE_IMPL(cons[i]);
  	}
 	if ( verbose )
		cout << "Disposing of the consumer admin object" << endl;
	cadmin->destroy();
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (CosEventChannelAdmin::AlreadyConnected& ex) {
	cerr << "Push Consumer already connected !" << endl;
	exit(1);
  } catch (CORBA::BAD_PARAM& ex) {
	cerr << "BAD_PARAM Exception while connecting Push Consumer !" << endl;
	exit (1);
  } catch (...) {
        cerr << "Caught exception while resolving the naming service" << endl;
        exit(1);
  }

  delete [] cons;
  delete [] supp;
  cout << ::getpid() << " got " << num_push << " events" << endl;
  return 0;
}
