// This application implements a pull supplier that is connected
// to a PULL_STRUCTURED proxy.
// It generates CosNotification::StructuredEvent events that have
// Stock::Ticker event type, 5 filterable_data fields, and a
// CORBA::ULong stored in the remainder_of_body field.
 
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iostream.h>

#include "corba_wrappers.h"


#include "CosNotify.h"

#include "CosNotifyComm_i.h"

CORBA::Boolean verbose=0;
omni_mutex     mutex;

/*****************************/

class MyPullSupplier_i : 
	WRAPPED_SKELETON_SUPER(CosNC::, StructuredPullSupplier) {
public:
  MyPullSupplier_i(int max_numb=0) : 
		_num_pull(0), _max_numb(max_numb)  { mutex.lock(); }
  inline CosN_StructuredEvent* pull_structured_event();
  inline CosN_StructuredEvent* try_pull_structured_event(CORBA::Boolean&);
         void              disconnect_structured_pull_supplier()	{;}
  inline void              subscription_change(const CosN_EventTypeSeq& added,
                                               const CosN_EventTypeSeq& deled);
protected:
  CORBA::ULong _num_pull;
  CORBA::ULong _max_numb;
};

inline CosN_StructuredEvent* MyPullSupplier_i::pull_structured_event()
{
  CosN_StructuredEvent* se = new CosN_StructuredEvent();
  se->header.fixed_header.event_type.domain_name = CORBA::string_dup("Stock");
  se->header.fixed_header.event_type.type_name   = CORBA::string_dup("Ticker");
  se->header.variable_header.length( 0 );
  se->filterable_data.length(5);
  se->filterable_data[0].name = (const char*) "Field 0";
  se->filterable_data[0].value <<= (CORBA::Long) 0;
  se->filterable_data[1].name = (const char*) "Field 1";
  se->filterable_data[1].value <<= (CORBA::Long) 1;
  se->filterable_data[2].name = (const char*) "Field 2";
  se->filterable_data[2].value <<= (CORBA::Long) 2;
  se->filterable_data[3].name = (const char*) "Field 3";
  se->filterable_data[3].value <<= (CORBA::Long) 3;
  se->filterable_data[4].name = (const char*) "Priority";
  se->filterable_data[4].value <<= (CORBA::ULong) _num_pull;
  se->remainder_of_body <<= _num_pull;
  if ( verbose )
  	cout << "Channel pulled event: " << _num_pull << endl;
  _num_pull += 1;
  if ( _max_numb && (_num_pull >= _max_numb) )
	mutex.unlock();
  return se;
}

inline CosN_StructuredEvent* MyPullSupplier_i::try_pull_structured_event(
                                        CORBA::Boolean& has_event)
{
  CosN_StructuredEvent* se = new CosN_StructuredEvent();
  if ( _max_numb && (_num_pull >= _max_numb) ) {
	mutex.unlock();
	has_event = 0;
	return se;
  }
  se->header.fixed_header.event_type.domain_name = CORBA::string_dup("Stock");
  se->header.fixed_header.event_type.type_name   = CORBA::string_dup("Ticker");
  se->header.variable_header.length(0);
  se->filterable_data.length(5);
  se->filterable_data[0].name = (const char*) "Field 0";
  se->filterable_data[0].value <<= (CORBA::Long) 0;
  se->filterable_data[1].name = (const char*) "Field 1";
  se->filterable_data[1].value <<= (CORBA::Long) 1;
  se->filterable_data[2].name = (const char*) "Field 2";
  se->filterable_data[2].value <<= (CORBA::Long) 2;
  se->filterable_data[3].name = (const char*) "Field 3";
  se->filterable_data[3].value <<= (CORBA::Long) 3;
  se->filterable_data[4].name = (const char*) "Priority";
  se->filterable_data[4].value <<= (CORBA::ULong) _num_pull;
  se->remainder_of_body <<= (CORBA::ULong) _num_pull;
  has_event = 1;
  if ( verbose )
	cout << "Channel pulled event: " << _num_pull << endl;
  _num_pull += 1;
  return se;
}

inline void MyPullSupplier_i::subscription_change(
				const CosN_EventTypeSeq& added,
                                const CosN_EventTypeSeq& deled)
{
  CORBA::ULong ix;
  if ( verbose ) {
     cout << "MyPullSupplier_i: subscription_change contains: " << endl;
     for (ix = 0; ix < added.length(); ix++)
        cout<<"\t+ "<<added[ix].domain_name<<"::"<<added[ix].type_name<<endl;
     for (ix = 0; ix < deled.length(); ix++)
	cout<<"\t+ "<<deled[ix].domain_name<<"::"<<deled[ix].type_name<<endl;
  }
}

/////////////////////////////////////////////////////////////////////
// *************************************************************** //
/////////////////////////////////////////////////////////////////////

static void usage(const char* pname)
{
   cout << "Usage: " << pname << " [-d numb] [-n name] [-v]" << endl;
   cout << "       -d numb : disconnect after numb pulls" << endl;
   cout << "       -n name : channel name [\"EventChannel\"]" << endl;
   cout << "       -v      : verbose output mode" << endl;
}

int main(int argc, char** argv)
{
  CosNA_StructuredProxyPullConsumer_var prxc;
  CosNA_EventChannel_ptr channel;
  CosN_EventTypeSeq added;
  CosN_EventTypeSeq deled;
  char *channelName = (char *) "EventChannel";
  MyPullSupplier_i* supplier=0;
  unsigned int numEvent=0;
  int c, i;

  // Parse command line arguments and initialize global variables

  while ( (c = getopt(argc, argv,"d:n:v")) != EOF ) {
  	switch (c) {
        case 'd': numEvent = (unsigned int) atoi(optarg);
                  break;
        case 'n': channelName = optarg;
                  break;
	case 'v': verbose = 1;
                  break;
        default : usage(argv[0]);
                  exit(-1);
     }
  }

  CORBA::ORB_ptr orb = WRAPPED_ORB_INIT(argc, argv);
  CORBA::BOA_ptr boa = WRAPPED_BOA_INIT(orb, argc, argv);

  if ( ! (supplier = new MyPullSupplier_i(numEvent)) ) {
	cerr << "Failed to create structured supplier object" << endl;
	exit(-1);
  }
  WRAPPED_ORB_REGISTER_SERVANT_NAME(orb, "struct_pull_supplier");
  WRAPPED_BOA_OBJ_IS_READY(boa, supplier);

  try {
	CosNaming::Name                          name;
        CORBA::Object_var                        echannel_ref;
        CORBA::Object_var                        name_service;
        CosNaming::NamingContext_var             name_context;
        CosNA_AdminID           saID;
        CosNA_ProxyID           pxID;
        CosNA_SupplierAdmin_var sadmin;
	CosNF_ConstraintExpSeq        constr;
        CosNF_FilterFactory_var       flfact;
        CosNF_Filter_var              filter;

	if ( verbose ) 
                cout << "Resolving reference to: 'NameService'" << endl;
	name_service = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService"); 
	name_context = CosNaming::NamingContext::_narrow(name_service);
	if ( CORBA::is_nil(name_context) ) {
		cerr << "Failed to obtain context for NameService" << endl;
                exit(1);
        }
        name.length(1);
        name[0].id   = CORBA::string_dup(channelName);
        name[0].kind = CORBA::string_dup(channelName);
        if ( verbose ) 
                cout << "Resolving reference to: '"<<channelName<<"'" << endl;
        echannel_ref = name_context->resolve(name);
        channel = CosNA_EventChannel::_narrow(echannel_ref);
        if ( CORBA::is_nil(channel) ) {
                cerr << "Failed to narrow Event Channel !" << endl;
                exit(1);
        }

        if ( verbose ) 
                cout << "Releasing reference to: 'NameService'" << endl;
	if ( verbose ) 
                cout << "Creating new SupplierAdmin object" << endl;
        sadmin = channel->new_for_suppliers(CosNA_AND_OP,saID);
        if ( CORBA::is_nil(sadmin) ) {
                cerr << "Failed to create SupplierAdmin object!" << endl;
                exit(1);
        }
        if ( verbose ) 
                cout << "Created SupplierAdmin with ID " << saID << endl;

	CosNA_ProxyConsumer_var cons;
	if ( verbose )
		cout << "Creating proxy for StructuredConsumer" << endl;
	cons=sadmin->obtain_notification_pull_consumer(
				CosNA_STRUCTURED_EVENT, pxID);
	prxc=CosNA_StructuredProxyPullConsumer::_narrow(cons);
	if ( CORBA::is_nil(prxc) ) {
		cerr << "Failed to find Proxy Consumer !" << endl;
                exit(1);
	}
	if ( verbose ) {
		cout << "Created proxy with ID " << pxID << endl;
		cout << "Register with BOA to receive requests" << endl;
	}

	WRAPPED_IMPL_IS_READY_FORK(boa, "struct_pull_supplier");

	if ( verbose )
		cout << "Informing channel about supplied event types" << endl;
	added.length(1);
	deled.length(0);
	added[0].domain_name = CORBA::string_dup("Stock");
	added[0].type_name   = CORBA::string_dup("Ticker");
	prxc->offer_change(added, deled);

	if ( verbose )
                cout << "Creating a filter to attach to proxy" << endl;
        flfact = channel->default_filter_factory();
        filter = flfact->create_filter("EXTENDED_TCL");
        constr.length(1);
        constr[0].event_types.length(1);
        constr[0].event_types[0].domain_name = CORBA::string_dup("Stock");
        constr[0].event_types[0].type_name   = CORBA::string_dup("Ticker");
        constr[0].constraint_expr = CORBA::string_dup("$Priority > 10");
        filter->add_constraints(constr);
        prxc->add_filter(filter);

	if ( verbose )
		cout << "Connecting consumer object to proxy" << endl;
	prxc->connect_structured_pull_supplier(supplier->_this());

	// Wait till the desired number of events has been supplied
	mutex.lock();
	mutex.unlock();
	if ( verbose )
		cout << "Disconnecting supplier from channel" << endl;
	prxc->disconnect_structured_pull_consumer();
	sadmin->destroy();
        WRAPPED_DISPOSE_IMPL(supplier);

  } catch (CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (CosEventChannelAdmin::AlreadyConnected& ex) {
        cerr << "Pull Supplier already connected!" << endl;
        exit(1);
  } catch (CORBA::BAD_PARAM& ex) {
        cerr << "BAD_PARAM Exception while connecting Pull Supplier!" << endl;
        exit (1);
  } catch (...) {
        cerr << "Caught exception while resolving the naming service" << endl;
        exit(1);
  }
  return 0;
}
