// This application implements a push supplier that connects to 
// a PULL_SEQUENCE proxy.
// It generates CosNotification::StructuredEvent event sequences.
// Each event in the sequence has Sensor::Value as its event type
// and the remainder_of_body stores a CORBA::ULong value.
 
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iostream.h>

#include "corba_wrappers.h"

#include "CosNotify.h"
#include "CosNotifyComm_i.h"

static void usage(const char* pname)
{
   cout << "Usage: " << pname << " [-d numb] [-b size] [-n name]" << endl;
   cout << "       -d numb : disconnect after numb batches"   << endl;
   cout << "       -b size : max number of events per batch"  << endl;
   cout << "       -n name : channel name [\"EventChannel\"]" << endl;
}

int main(int argc, char** argv)
{
  CosNA_EventChannel_var channel;
  CosN_StructuredEvent se;
  CosN_EventBatch batch;
  CosNaming::NamingContext_var name_context;
  CosNaming::Name name;
  int c, num_pushes=4, batch_sz=5;
  char pname[128];

  strcpy(pname, argv[0]);

  CORBA::ORB_ptr orb = WRAPPED_ORB_INIT(argc, argv);
  CORBA::BOA_ptr boa = WRAPPED_BOA_INIT(orb, argc, argv);

  SequencePushSupplier_i* supplier = new SequencePushSupplier_i();

  WRAPPED_ORB_REGISTER_SERVANT_NAME(orb, "batch_push_supplier");
  WRAPPED_BOA_OBJ_IS_READY(boa, supplier);

  // Process command line arguments

  char *channelName = (char *) "EventChannel";
  char *channelKind = (char *) "EventChannel";

  while ( (c = getopt(argc, argv,"d:b:n:")) != EOF ) {
  	switch (c) {
        case 'd': num_pushes = atoi(optarg); break;
        case 'b': batch_sz   = atoi(optarg); break;
        case 'n': channelName = optarg; channelKind = optarg;  break;
        default : usage(pname); exit(-1);
     }
  }

  cout << "# event batches to be pushed = " << num_pushes << endl;
  cout << "max # events per each batch  = " << batch_sz   << endl;

  // Locate, using the naming service, the event channel, after
  // obtaining a reference to the nameing service itself.......

  try {
	CORBA::Object_var name_service;
	cout << "** Resolving initial reference to NameService" << endl;
	name_service = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService"); 
	cout << "** Narrowing the retrieved reference" << endl;
	name_context = CosNaming::NamingContext::_narrow(name_service);
	if ( CORBA::is_nil(name_context) ) {
		cerr << "Failed to obtain context for NameService" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the naming service" << endl;
        exit(1);
  }

  name.length(1);
  name[0].id   = CORBA::string_dup(channelName);
  name[0].kind = CORBA::string_dup(channelKind);
  try {
	cout << "** Resolving the reference to the event channel" << endl;
	CORBA::Object_var channel_ref = name_context->resolve(name);
	cout << "** Narrowing the resolved reference" << endl;
	channel = CosNA_EventChannel::_narrow(channel_ref);
	if ( CORBA::is_nil(channel) ) {
		cerr << "Failed to narrow Event Channel !" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the event channel" << endl;
        exit(1);
  }
 

  // Let the BOA know that we are ready -- no blocking since we
  // need to supply events

  cout << "** Call to WRAPPED_IMPL_IS_READY" << endl;
  WRAPPED_IMPL_IS_READY_FORK(boa, "batch_push_supplier");
   
  // Get supplier admin object from the channel

  CosNA_InterFilterGroupOperator ifoper = 
			CosNA_AND_OP;
  CosNA_SupplierAdmin_var supplier_admin;
  CosNA_AdminID suppID;

  try {
	cout << "** Obtaining a supplier admin" << endl;
	supplier_admin = channel->new_for_suppliers(ifoper, suppID);
	if ( CORBA::is_nil(supplier_admin) ) {
		cerr << "Failed to find Supplier Admin !" << endl;
		exit(1);
	}
  } catch (...) {
	cerr << "Failed to obtain Supplier Admin !" << endl;
	exit(1);
  }

  cout << "Obtained SupplierAdmin from the channel" << endl;

  // Get a push proxy using the above admin object

  CosNA_ProxyID prxID;
  CosNA_ClientType ctype = 
				CosNA_SEQUENCE_EVENT;
  CosNA_ProxyConsumer_var cons;
  CosNA_SequenceProxyPushConsumer_var pcons;
  try {
	cons = supplier_admin->obtain_notification_push_consumer(ctype,prxID);
  	pcons=CosNA_SequenceProxyPushConsumer::_narrow(cons);
	if ( CORBA::is_nil(pcons) ) {
		cerr << "Failed to find Proxy Consumer !" << endl;
		exit(1);
	}
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
	cerr << "Failed to obtain Proxy Consumer !" << endl;
	exit(1);
  }

  cout << "Obtained ProxyPushConsumer from SupplierAdmin" << endl;

  // Connect supplier to the proxy

  try {
	pcons->connect_sequence_push_supplier(supplier->_this());
  }
  catch (CORBA::BAD_PARAM& ex) {
	cerr << "BAD_PARAM Exception while connecting Push Supplier !" << endl;
        exit (1);
  }
  catch (CosEventChannelAdmin::AlreadyConnected& ex) {
	cerr << "Pull Supplier already connected !" << endl;
  } catch (...) {
	cerr << "Failed to connect Push Supplier !" << endl;
	exit(1);
  }

  cout << "Connected push supplier to ProxyPushConsumer" << endl;


  se.header.fixed_header.event_type.domain_name = CORBA::string_dup("Sensor");
  se.header.fixed_header.event_type.type_name   = CORBA::string_dup("Value");
  se.header.variable_header.length( 0 );
  se.filterable_data.length( 0 );
  batch.length(batch_sz);

  for (int i=0; (num_pushes == 0) || (i < num_pushes); i++) {
	for (int j=0; j < batch_sz; j++) {
		se.remainder_of_body <<= (CORBA::ULong) (i*1000 + j);
		batch[j] = se;
	}
	try {
		pcons->push_structured_events(batch);
		cout << "Pushed: " <<i*1000<<" to "<<i*1000+batch_sz<< endl;
	} catch (...) {
		cerr << "push() failed" << endl;
	}
	sleep(1);
  }

  // Disconnect supplier from the service

  try {
  	pcons->disconnect_sequence_push_consumer();
	WRAPPED_DISPOSE_IMPL(supplier);
  } catch (...) {
	cerr << "some exception -- perhaps COMM_FAILURE" << endl;
  }

  return 0;
}
