// -*- Mode: C++; -*-
//                              File      : omniNotify.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of notifyServer [main library interface]
//
 
/*
$Log: omniNotify.cc,v $
Revision 1.3  2000/08/22 18:22:03  alcfp
getopt fix

Revision 1.2  2000/08/16 20:20:12  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fstream.h>
#include <iostream.h>
#include "RDI.h"
#include "RDIDebug.h"
#include "RDIConfig.h"
#include "CosNotification_i.h"
#include "CosNotifyChannelAdmin_i.h"
#include "omniNotify.h"

unsigned long QueueGCPeriod = 300;

static int validateSettings(const RDI_Config&, RDI_AdminQoS&, RDI_NotifQoS&);

// ----------------------------------------------------------- //

class rdi_notifyServer : public notifyServer 
{
public:
  rdi_notifyServer() : 
	notifyServer(), _configp(0) {;}
  ~rdi_notifyServer();

  void setConfig(RDI_Config* config)
	{ _configp = config; }
  void setChannelFactory(CosNA_EventChannelFactory_ptr factory)
	{ _factory = factory; }
private:
  RDI_Config* _configp;  
};


rdi_notifyServer::~rdi_notifyServer()
{
  if ( _configp ) {
     char* factoryIORFile = 0;
     char* channelIORFile = 0;
     _configp->get_value("FactoryIORFileName", factoryIORFile);
     _configp->get_value("ChannelIORFileName", channelIORFile);
     if ( factoryIORFile ) 
     	(void) unlink(channelIORFile);
     if ( channelIORFile ) 
     	(void) unlink(channelIORFile);
     delete _configp; 
     _configp = 0;
  }

  if ( ! CORBA::is_nil(_factory) ) {
     CosNA_ChannelIDSeq* cids = _factory->get_all_channels();
     for ( CORBA::ULong ix = 0; ix < cids->length(); ix++ ) {
     	CosNA_EventChannel_ptr chan_ptr;
     	chan_ptr = _factory->get_event_channel( (*cids)[ix] );
     	if ( ! CORBA::is_nil(chan_ptr) ) {
	   RDI_DUMP("Destroying Event Channel " << (*cids)[ix]);
	   chan_ptr->destroy();
  	}
     }
     delete cids;
     WRAPPED_DISPOSE(_factory);
  }
  _factory = CosNotifyChannelAdmin::EventChannelFactory::_nil();
}

// ----------------------------------------------------------- //

notifyServer::notifyServer() 
{ _factory = CosNotifyChannelAdmin::EventChannelFactory::_nil(); }

notifyServer::notifyServer(const notifyServer&)
{ cerr << "notifyServer(const notifyServer&) : invalid" << endl; }

notifyServer& notifyServer::operator = (const notifyServer&)
{ cerr << "notifyServer& operator=(const notifyServer&) : invalid" << endl; 
  return *this; }

CosNA_EventChannelFactory_ptr notifyServer::getChannelFactory() const
{ return WRAPPED_DUPLICATE(CosNA_EventChannelFactory, _factory); }

notifyServer* notifyServer::create(int argc, char** argv, CORBA::ORB_ptr orb)
{
  RDI_AdminQoS  defadm;
  RDI_NotifQoS  defqos;
  RDI_Config*   config = 0;
  char*         theChannelName = 0;
  char*         theFactoryName = 0;
  char*         factoryIORFile = 0;
  char*         channelIORFile = 0;
  char*         configFileName = 0;
  char*         debugFileNames = 0;
  char*         debugLogFile   = 0;
  long          queueGCPeriod  = 300;
  unsigned char useNameService = 1;

  // Remember the ORB and BOA references

  RDI::init(orb, CORBA::BOA::getBOA());

  // Parse command line arguments to locate any supported switches; 
  // They include '-n' to bypass the Naming Service, and '-c fname'
  // to use the configuration file 'fname'
  int indx = 0;
  while ( indx < argc ) {
     if ( strcmp(argv[indx], "-n") == 0 ) {
        useNameService = 1;
     } else if ( strcmp(argv[indx], "-c") == 0 ) {
	if ( ++indx < argc ) {
	   configFileName = argv[indx];
	}
     }
     indx += 1;
  }

  if ( ! (config = new RDI_Config()) ) {
     cerr << "Failed to create RDI_Config object" << endl;
     return 0;
  }
  if ( configFileName && config->import_settings(configFileName) ) {
     cerr << "Failed to import settings from: " << configFileName << endl;
     delete config; return 0;
  }
  config->parse_arguments(argc, argv, 1);
  
  if ( config->get_value("ChannelFactoryName", theFactoryName) != 0 )
     theFactoryName = (char *) "ChannelFactory";
  if ( config->get_value("DefaultChannelName", theChannelName) != 0 )
     theChannelName  = (char *) "EventChannel";
  config->get_value("FactoryIORFileName", factoryIORFile);
  config->get_value("ChannelIORFileName", channelIORFile);
  config->get_value("DebugLogFile",       debugLogFile);
  config->get_value("DebugFileNames",     debugFileNames);
  config->get_value("QueueGCPeriod",      queueGCPeriod);
  QueueGCPeriod = (queueGCPeriod <= 0) ? 300 : queueGCPeriod;

  if ( debugLogFile ) {
     RDI_Debug::init_default_debug(0, debugLogFile);
     if ( debugFileNames && ! RDI_STR_EQ_I(debugFileNames, "none") ) {
     	if ( RDI_STR_EQ_I(debugFileNames, "all") ) {
	   RDI_Debug::default_debug()->enable_all_files();
	   RDI_DUMP("Enabled debugging for all files");
     	} else {
	  RDI_Debug::default_debug()->enable("*/omniNotify.cc");
	  RDI_DUMP("Enabled debugging for: omniNotify.cc");
	  RDI_Debug::default_debug()->enable(debugFileNames);
	  RDI_DUMP("Enabled debugging for: " << debugFileNames << " files");
     	}
     }
  }

  // Validate the QoS and Admin Property values provided
  if ( validateSettings(*config, defadm, defqos) ) {
     if ( debugLogFile ) {
     	RDI_DUMP("Invalid QoS or Admin Properties were provided");
     } else {
     	cerr << "Invalid QoS or Admin Properties were provided" << endl;
     }
     return 0;
  }

  rdi_notifyServer* server = new rdi_notifyServer();
  if ( ! server ) {
     if ( debugLogFile ) {
     	RDI_DUMP("Failed to create notifyServer object");
     } else {
	cerr << "Failed to create notifyServer object" << endl;
     }
     return 0;
  } else {
     server->setConfig(config);
  }
  EventChannelFactory_i* factory = new EventChannelFactory_i(defqos, defadm);
  CosNA_EventChannelFactory_var factref;
  if ( ! factory ) {
     if ( debugLogFile ) {
     	RDI_DUMP("Failed to create event channel factory object");
     } else {
	cerr << "Failed to create event channel factory object" << endl;
     }
     delete server; return 0;
  } else {
     factref = WRAPPED_IMPL2OREF(CosNA_EventChannelFactory, factory);
     server->setChannelFactory(factref);
     if ( factoryIORFile ) {   
        ofstream file(factoryIORFile, ios::out);
        if ( ! file ) {
	   if ( debugLogFile ) {
	      RDI_DUMP("Failed to open IOR file: " << factoryIORFile);
	   } else {
	      cerr << "Failed to open IOR file: " << factoryIORFile << endl;
	   }
        } else {
           char* ior_name = orb->object_to_string(factref);
           file << ior_name;
           file.close();
           delete [] ior_name;
  	}
     }
  }

  CosNA_ChannelID        channID;
  CosNA_EventChannel_var chanref;
  EventChannel_i*        channel = factory->create_channel(channID);

  if ( ! channel ) {
     if ( debugLogFile ) {
        RDI_DUMP("Failed to create event channel object");
     } else {
     	cerr << "Failed to create event channel object" << endl;
     }
     delete server; return 0;
  }
  if ( channelIORFile ) {
     ofstream file(channelIORFile, ios::out);
     if ( ! file ) {
	if ( debugLogFile ) {
           RDI_DUMP("Failed to open IOR file: " << channelIORFile );
	} else {
	   cerr << "Failed to open IOR file: " << channelIORFile << endl;
	}
     } else {
	chanref=WRAPPED_IMPL2OREF(AttNotifyChannelAdmin::EventChannel,channel);
     	char* ior_name = orb->object_to_string(chanref);
     	file << ior_name;
     	file.close();
     	delete [] ior_name;
     }
  }

  try {
     if ( useNameService ) {
    	CosNaming::NamingContext_var nmcx;
	CORBA::Object_var            nmsv;
	CosNaming::Name              name;

	nmsv = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService");
	nmcx = CosNaming::NamingContext::_narrow(nmsv);
	if ( CORBA::is_nil(nmcx) ) {
	   if ( debugLogFile ) {
	      RDI_DUMP("Failed to obtain NameService context");
	   } else {
	      cerr << "Failed to obtain NameService context" << endl;
	   }
	   delete server; return 0;
	}
	RDI_DUMP("Register '" << theFactoryName << "' with NameService");
	name.length(1);
	name[0].id   = CORBA_STRING_DUP(theFactoryName);
	name[0].kind = CORBA_STRING_DUP(theFactoryName);
	nmcx->rebind(name, factref);

	RDI_DUMP("Register '" << theChannelName << "' with NameService");
	name[0].id   = CORBA_STRING_DUP(theChannelName);
	name[0].kind = CORBA_STRING_DUP(theChannelName);
	nmcx->rebind(name, chanref);
     }
  } catch ( CORBA::ORB::InvalidName& ex ) {
     if ( debugLogFile ) {
     	RDI_DUMP("Service required is invalid [does not exist]");
     } else {
	cerr << "Service required is invalid [does not exist]" << endl;
     }
     delete server; return 0;
  } catch ( CORBA::COMM_FAILURE& ex ) {
     if ( debugLogFile ) {
     	RDI_DUMP("Caught system exception COMM_FAILURE");
     } else {
	cerr << "Caught system exception COMM_FAILURE" << endl;
     }
     delete server; return 0;
  } catch ( ... ) {
     if ( debugLogFile ) {
     	RDI_DUMP("Unexpected exception was caught");
     } else {
	cerr << "Unexpected exception was caught" << endl;
     }
     delete server; return 0;
  }

  return server;
}

// ----------------------------------------------------------- //

int validateSettings(const RDI_Config& config, 
		     RDI_AdminQoS& admqos, RDI_NotifQoS& ntfqos)
{
  char* debugLogFile = 0;
  long  lval=0;
  config.get_value("DebugLogFile", debugLogFile);
  if ( config.get_value("EventReliability", lval) == 0 ) {
     if ( lval != 0 ) {
	if ( debugLogFile ) {
	   RDI_DUMP("Invalid value for EventReliability - " << lval);
	   RDI_DUMP("\tonly BestEffort [0] is currently supported");
	} else {
	   cerr << "Invalid value for EventReliability - " << lval << endl;
	   cerr << "\tonly BestEffort [0] is currently supported" << endl;
	}
	return -1;
     }
  }
  lval = 0;
  if ( config.get_value("ConnectionReliability", lval) == 0 ) {
     if ( lval != 0 ) {
	if ( debugLogFile ) {
	   RDI_DUMP("Invalid value for ConnectionReliability - " << lval);
	   RDI_DUMP("\tonly BestEffort [0] is currently supported");
	} else {
	   cerr << "Invalid value for ConnectionReliability - " << lval << endl;
	   cerr << "\tonly BestEffort [0] is currently supported" << endl;
	}
	return -1;
     }
  }
  lval = 0;
  if ( config.get_value("Priority", lval) == 0 ) {
        ntfqos.priority = (CORBA::Short) lval;
  }
  lval = 0;
  if ( config.get_value("StartTimeSupported", lval) == 0 ) {
     if ( lval != 0 ) {
	if ( debugLogFile ) {
	   RDI_DUMP("Invalid value for StartTimeSupported - " << lval);
	   RDI_DUMP("\tonly FALSE [0] is currently supported");
	} else {
	   cerr << "Invalid value for StartTimeSupported - " << lval << endl;
	   cerr << "\tonly FALSE [0] is currently supported" << endl;
	}
	return -1;
     }
  }
  lval = 0;
  if ( config.get_value("StopTimeSupported", lval) == 0 ) {
     if ( lval != 0 ) {
	if ( debugLogFile ) {
	   RDI_DUMP("Invalid value for StopTimeSupported - " << lval);
	   RDI_DUMP("\tonly FALSE [0] is currently supported");
	} else {
	   cerr << "Invalid value for StopTimeSupported - " << lval << endl;
	   cerr << "\tonly FALSE [0] is currently supported" << endl;
	}
	return -1;
     }  
  }
  lval = 0;
  if ( config.get_value("OrderPolicy", lval) == 0 ) {
     if ( lval != 1 ) {
	if ( debugLogFile ) {
	   RDI_DUMP("Invalid value for OrderPolicy - " << lval);
	   RDI_DUMP("\tonly FifoOrder [1] is currently supported");
	} else {
	   cerr << "Invalid value for OrderPolicy - " << lval << endl;
	   cerr << "\tonly FifoOrder [1] is currently supported" << endl;
	}
        return -1;
     }
  }
  lval = 0;
  if ( config.get_value("DiscardPolicy", lval) == 0 ) {
     if ( lval != 1 ) {
	if ( debugLogFile ) {
	   RDI_DUMP("Invalid value for DiscardPolicy - " << lval);
	   RDI_DUMP("\tonly FifoOrder [1] is currently supported");
	} else {
	   cerr << "Invalid value for DiscardPolicy - " << lval << endl;
	   cerr << "\tonly FifoOrder [1] is currently supported" << endl;
	}
	return -1;
     }
  }
  lval = 0;
  if ( config.get_value("MaxEventsPerConsumer", lval) == 0 ) {
        ntfqos.maxEventsPerConsumer = (CORBA::Long) lval;
  }
  lval = 0;
  if ( config.get_value("MaximumBatchSize", lval) == 0 ) {
     if ( lval <= 0 ) {
	if ( debugLogFile ) {
	   RDI_DUMP("Invalid value for MaximumBatchSize - " << lval);
	   RDI_DUMP("\tvalue should be greater or equal to 1");
	} else {
	   cerr << "Invalid value for MaximumBatchSize - " << lval << endl;
	   cerr << "\tvalue should be greater or equal to 1" << endl;
	}
	return -1;
     }
     ntfqos.maximumBatchSize = (CORBA::Long) lval;
  }

  if ( config.get_value("MaxQueueLength", lval) == 0 ) {
     admqos.maxQueueLength = (CORBA::Long) lval;
  }
  if ( config.get_value("MaxConsumers", lval) == 0 ) {
     admqos.maxConsumers = (CORBA::Long) lval;
  }
  if ( config.get_value("MaxSuppliers", lval) == 0 ) {
     admqos.maxSuppliers = (CORBA::Long) lval;
  }
  lval = 0;
  if ( config.get_value("RejectNewEvents", lval) == 0 ) {
     if ( (lval != 0) &&  (lval != 1) ) {
	if ( debugLogFile ) {
	   RDI_DUMP("Invalid value for RejectNewEvents - " << lval);
	   RDI_DUMP("\tvalue should be 0 (FALSE) or 1 (TRUE)");
	} else { 
	   cerr << "Invalid value for RejectNewEvents - " << lval << endl;
	   cerr << "\tvalue should be 0 (FALSE) or 1 (TRUE)" << endl;
	}
        return -1;
     }
     admqos.rejectNewEvents = (CORBA::Boolean) lval;
  }
  lval = 0;
  if ( config.get_value("NumAdminGroups", lval) == 0 ) {
     if ( lval <= 0 ) {
	if ( debugLogFile ) {
	   RDI_DUMP("Invalid value for NumAdminGroups - " << lval);
	   RDI_DUMP("\tshould be at least 1");
	} else {
	   cerr << "Invalid value for NumAdminGroups - " << lval << endl;
	   cerr << "\tshould be at least 1" << endl;
	}
	return -1;
     }
     admqos.numAdminGroups = (CORBA::UShort) lval;
  }
  lval = 0;
  if ( config.get_value("NumAdminThreads", lval) == 0 ) {
     if ( lval <= 0 ) {
	if ( debugLogFile ) {
	   RDI_DUMP("Invalid value for NumAdminThreads - " << lval);
	   RDI_DUMP("\tshould be at least 1");
	} else {
	   cerr << "Invalid value for NumAdminThreads - " << lval << endl;
           cerr << "\tshould be at least 1" << endl;
	}
	return -1;
     }
     admqos.numAdminThreads = (CORBA::UShort) lval;
  }
  if ( config.get_value("NumProxyThreads", lval) == 0 ) {
     admqos.numProxyThreads = (CORBA::UShort) lval;
  }
  if ( config.get_value("NumPushThreads", lval) == 0 ) {
     admqos.numPushThreads = (CORBA::UShort) lval;
  }
  if ( config.get_value("NumPullThreads", lval) == 0 ) {
     admqos.numPullThreads = (CORBA::UShort) lval;
  }
  if ( config.get_value("PullEventPeriod", lval) == 0 ) {
     admqos.pullEventPeriod = (CORBA::UShort) lval;
  }
  if ( admqos.numAdminThreads > admqos.numAdminGroups ) {
     if ( debugLogFile ) {
     	RDI_DUMP("WARNING: NumAdminThreads [" << admqos.numAdminThreads  << "] greater than NumAdminGroups [" << admqos.numAdminGroups << "]");
     	RDI_DUMP("\tSetting NumAdminThreads = NumAdminGroups");
     } else { 
	cerr << "WARNING: NumAdminThreads [" << admqos.numAdminThreads << 
	        "] greater than NumAdminGroups [" << admqos.numAdminGroups <<
		"]" << endl;
	cerr << "\tSetting NumAdminThreads = NumAdminGroups" << endl;
     }
     admqos.numAdminThreads = admqos.numAdminGroups;
  }
  return 0;
}
