// -*- Mode: C++; -*-
//                              File      : RDIFileIO.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of file I/O related functionality
//
 
/*
$Log: RDIFileIO.cc,v $
Revision 1.4  2000/08/22 18:23:56  alcfp
added description to each file

Revision 1.3  2000/08/16 20:19:51  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <netdb.h>
#if defined(__sgi) || defined(sgi)
#   include <limits.h>
#   include <sys/param.h>
#endif
#if defined (SOLARIS) || defined (__sgi) || defined (sgi)
#   include <sys/systeminfo.h>
#endif
#if defined (_AIX)
#   include <sys/select.h>
#endif
#include "RDIFileIO.h"

// ************************************************************* //
// IO-related operations for reading from and writing to regular //
// files and disk raw partictions.                               //
// ************************************************************* //

#ifdef DEV_BSIZE
#  define BLOCK_SIZE DEV_BSIZE
#else
#  define BLOCK_SIZE 512
#endif

#ifndef SEEK_CUR
#   define SEEK_SET 0
#   define SEEK_CUR 1
#   define SEEK_END 2
#endif

static int _RDI_RawRead(int fd, char* to, size_t sz, off_t off)
{
  char   dummy[BLOCK_SIZE];
  size_t left=sz, middle=0, right=0;
  char*  ptr = to;

  off_t l_off = off & ~(BLOCK_SIZE - 1);
  off_t u_off = (off + sz) & ~(BLOCK_SIZE - 1);

  if ( l_off != u_off ) { 	// multi-block read
     	left   = (l_off == off) ? 0 : ((l_off + BLOCK_SIZE) - off);
     	middle = u_off - (l_off + ((left == 0) ? 0 : BLOCK_SIZE));
     	right  = (off + sz) - u_off;
  }
  if ( ::lseek(fd, l_off, SEEK_SET) != l_off )
	return -1;
  if ( left ) {
	if ( RDI_Read(fd, dummy, BLOCK_SIZE) )
     		return -1;
     	::memcpy(ptr, dummy + (off - l_off), left);
     	ptr += left;
  }
  if ( middle ) {
     	if ( RDI_Read(fd, ptr, middle) )
		return -1;
     	ptr += middle;
  }
  if ( right ) {
     	if ( RDI_Read(fd, dummy, BLOCK_SIZE) )
		return -1;
     	::memcpy(ptr, dummy, right);
  }
  return 0;
}

static int _RDI_RawWrite(int fd, const char* from, size_t sz, off_t off)
{
  char   dummy[BLOCK_SIZE];
  size_t left=sz, middle=0, right=0;
  const char* ptr=from;

  off_t l_off = off & ~(BLOCK_SIZE - 1);
  off_t u_off = (off + sz) & ~(BLOCK_SIZE - 1);

  if ( l_off != u_off ) { 	// multi-block write
     	left   = (l_off == off) ? 0 : ((l_off + BLOCK_SIZE) - off);
     	middle = u_off - (l_off + ((left == 0) ? 0 : BLOCK_SIZE));
     	right  = (off + sz) - u_off;
  }
  if ( ::lseek(fd, l_off, SEEK_SET) != l_off )
	return -1;
  if ( left ) {		// have to read the block first
     	if ( RDI_Read(fd, dummy, BLOCK_SIZE) )
		return -1;
     	::memcpy(dummy + (off - l_off), ptr, left);
     	if ( ::lseek(fd, l_off, SEEK_SET) != l_off )
		return -1;
     	if ( RDI_Write(fd, dummy, BLOCK_SIZE) )
		return -1;
     	ptr += left;
  }
  if ( middle ) {
	if ( RDI_Write(fd, ptr, middle) )
		return -1;
     	ptr += middle;
  }
  if ( right ) {	// have to read the block first
	if ( RDI_Read(fd, dummy, BLOCK_SIZE) )
		return -1;
     	::memcpy(dummy, ptr, right);
     	if ( ::lseek(fd, u_off, SEEK_SET) != u_off )
		return -1;
     	if ( RDI_Write(fd, dummy, BLOCK_SIZE) )
		return -1;
  }
  return 0;
}

int RDI_Read(int fd, void* to, size_t sz, off_t off, int raw)
{
  int rbytes=0, offset=0;

  if ( sz == 0 ) 
	return 0;
  if ( raw != 0 )
	return _RDI_RawRead(fd, (char *) to, sz, off);
  if ( (off != -1) && (::lseek(fd, off, SEEK_SET) != off) ) 
	return -1;
  while ( sz != 0 ) {
	if ( (rbytes = ::read(fd, (char *) to + offset, sz)) == -1 ) {
		if ( errno == EINTR ) continue;
		return -1;
	} else if ( rbytes == 0 ) {	// Premature EOF
		return -1;
	}
	sz     -= rbytes;
	offset += rbytes;
  }
  return 0;
}

int RDI_Write(int fd, const void* from, size_t sz, off_t off, int raw)
{
  int wbytes=0, offset=0;

  if ( sz == 0 )
	return 0;
  if ( raw != 0 )
	return _RDI_RawWrite(fd, (const char *) from, sz, off);
  if ( (off != -1) && (::lseek(fd, off, SEEK_SET) != off) )
	return -1;
  while ( sz != 0 ) {
	if ( (wbytes = ::write(fd, (char *) from + offset, sz)) == -1 ) {
		if ( errno == EINTR ) continue;
		return -1;
	}
	sz     -= wbytes;
	offset += wbytes;
  }
  return 0;
}

