// -*- Mode: C++; -*-
//                              File      : PullConsumer_i.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of PullConsumer proxies
//
 
/*
$Log: PullConsumer_i.cc,v $
Revision 1.35  2000/10/01 13:39:53  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.33.2.1  2000/09/30 15:39:51  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.33  2000/08/22 18:23:54  alcfp
added description to each file

Revision 1.32  2000/08/16 20:19:32  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
// -*- Mode: C++; -*-

#include <iomanip.h>

#include "corba_wrappers.h"

#include "RDIDebug.h"
#include "CosNotifyChannelAdmin_i.h"

extern const char* RDI_PRX_TYPE(const CosNA_ProxyType& type);

// ------------------------------------------------------------- //
// Support classes needed for implementing  the thread-per-proxy //
// notification approach                                         //
// ------------------------------------------------------------- //

class AnyPullWorker : public omni_thread {
public:
  typedef void (ProxyPullConsumer_i::*Method)(void);
  AnyPullWorker(ProxyPullConsumer_i* p, Method m) :
                omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  ProxyPullConsumer_i* _proxy;
  Method               _method;
  AnyPullWorker()  {;}
};

class StrPullWorker : public omni_thread {
public:
  typedef void (StructuredProxyPullConsumer_i::*Method)(void);
  StrPullWorker(StructuredProxyPullConsumer_i* p, Method m) :
                omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  StructuredProxyPullConsumer_i* _proxy;
  Method                         _method;
  StrPullWorker()  {;}
};

class SeqPullWorker : public omni_thread {
public:
  typedef void (SequenceProxyPullConsumer_i::*Method)(void);
  SeqPullWorker(SequenceProxyPullConsumer_i* p, Method m) :
                omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  SequenceProxyPullConsumer_i* _proxy;
  Method                       _method;
  SeqPullWorker()  {;}
};

// ------------------------------------------------------------- //
// ProxyPullConsumer_i                                           //
// ------------------------------------------------------------- //

ProxyPullConsumer_i::ProxyPullConsumer_i(SupplierAdmin_i* admin,
					 EventChannel_i*  chann,
					 const CosNA_ProxyID&  prxid) : 
		_oplock(), _fa_helper(), _channel(chann), _myadmin(admin),
		_prxtype(CosNA_PULL_ANY), 
		_pserial(prxid), _cosevnt(0), _nevents(0), 
		_pulltms(), _pxstate(RDI_NotConnected), _active(0), 
		_qempty(0), _worker(0), _thrdone(0), _qosprop(0), _offauto(0), 
		_evtypes(RDI_EventType::hash, RDI_EventType::rank)
{
  _supplier = CosEventComm::PullSupplier::_nil();
  // When the number of pull threads allocated at the channel level is
  // 0, each proxy uses its own thread to pull events from its supplier
  if ( _channel->pull_threads() == 0 ) {
        _qempty = new omni_condition(&_oplock);
	RDI_Assert(_qempty, "Memory allocation failed -- omni_condition");
        _worker = new AnyPullWorker(this, &ProxyPullConsumer_i::_pull_event);
	RDI_Assert(_worker, "Memory allocation failed -- omni_thread");
        _worker->start();
	_thrdone = 0;
	RDI_DUMP("Pull thread for proxy " << this << " -- " << _worker->id());
  }
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_SupplierAdmin_ptr ProxyPullConsumer_i::MyAdmin( WRAPPED_IMPLARG_VOID )
{ return WRAPPED_IMPL2OREF(CosNA_SupplierAdmin, _myadmin); }

CosN_EventTypeSeq*
ProxyPullConsumer_i::obtain_subscription_types(CosNA_ObtainInfoMode mode
					       WRAPPED_IMPLARG)
{
  CosN_EventTypeSeq* etypeseq=0;
  if ( mode == CosNA_ALL_NOW_UPDATES_OFF ) {
        etypeseq = _channel->obtain_subscription_types();
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_ALL_NOW_UPDATES_ON ) {
        etypeseq = _channel->obtain_subscription_types();
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_OFF ) {
        etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_ON ) {
        etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  }
  return etypeseq;
}

CosN_QoSProperties* 
ProxyPullConsumer_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_C_AnyPRX);
}

void ProxyPullConsumer_i::set_qos(const CosN_QoSProperties& r_qos
				  WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);
  
  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_AnyPRX,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos); 
}

void ProxyPullConsumer_i::validate_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_AnyPRX,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void ProxyPullConsumer_i::validate_event_qos(
	const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
	WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos))
	throw CosN_UnsupportedQoS(eseq);
}

void ProxyPullConsumer_i::disconnect_pull_consumer(WRAPPED_IMPLARG_VOID)
{
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void ProxyPullConsumer_i::offer_change(
			const CosN_EventTypeSeq& added, 
		       	const CosN_EventTypeSeq& deled
		        WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq new_added;
  CosN_EventTypeSeq old_deled;
  CORBA::ULong ix=0, sz=0, vl=0;
  if ( ! RDI_EventType::valid_sequence(added, ix) ) 
     throw CosNC_InvalidEventType(added[ix]);
  if ( ! RDI_EventType::valid_sequence(deled, ix) ) 
     throw CosNC_InvalidEventType(deled[ix]);
  new_added.length(0);
  old_deled.length(0);
  _oplock.lock();
  for (sz=0, ix=0; ix < added.length(); ix++) {
     if ( _evtypes.lookup(added[ix], vl) ) {
	vl += 1; _evtypes.replace(added[ix], vl);
     } else {
	vl  = 1; _evtypes.insert(added[ix], vl);
	new_added.length(sz + 1);
	new_added[sz].domain_name = added[ix].domain_name;
	new_added[sz++].type_name = added[ix].type_name;
     }
  }
  for (sz=0, ix=0; ix < deled.length(); ix++) {
     if ( _evtypes.lookup(deled[ix], vl) ) {
	if ( vl == 1 ) {
	   _evtypes.remove(deled[ix]);
	   old_deled.length(sz + 1);
	   old_deled[sz].domain_name = deled[ix].domain_name;
	   old_deled[sz++].type_name = deled[ix].type_name;
	} else {
	   vl -= 1; _evtypes.replace(deled[ix], vl);
	} 
     } else {
	RDI_DUMP("Invalid "<<deled[ix].domain_name<<"::"<<deled[ix].type_name);
     }
  }
  _oplock.unlock();
  // If any new event types were added or existing event types were
  // deleted, notify the parent administrative object
  if ( new_added.length() || old_deled.length() )
     _myadmin->offer_change(new_added, old_deled);
}

void ProxyPullConsumer_i::connect_any_pull_supplier(
		CosEventComm::PullSupplier_ptr supplier WRAPPED_IMPLARG)
{
  CosEventComm::PullSupplier_var evsupl;
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(supplier) || (_pxstate != RDI_NotConnected) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  if ( _pxstate == RDI_Connected )
	throw CosEventChannelAdmin::AlreadyConnected();

  // Need to find if the given reference corresponds to a CosEventComm
  // supplier -- this supplier does not support subscription_change()

  evsupl    = CosEventComm::PullSupplier::_narrow(supplier);
  _cosevnt  = CORBA::is_nil(evsupl) ? 0 : 1;
  _supplier = WRAPPED_DUPLICATE(CosEventComm::PullSupplier, supplier);
  _pxstate  = RDI_Connected;
  _active   = 1;
  if ( _worker && _qempty )     // Notify worker thread
	_qempty->signal();
  if ( _qosprop ) 
	delete _qosprop;
}

void ProxyPullConsumer_i::suspend_connection(WRAPPED_IMPLARG_VOID)
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected )
	throw CosNA_NotConnected();
  if ( ! _active ) 
	throw CosNA_ConnectionAlreadyInactive();
  _active = 0;
}

void ProxyPullConsumer_i::resume_connection(WRAPPED_IMPLARG_VOID)
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected )
	throw CosNA_NotConnected();
  if ( _active ) 
	throw CosNA_ConnectionAlreadyActive();
  _active = 1;
  if ( _worker && _qempty )	// Notify worker thread
	_qempty->signal();
}

void ProxyPullConsumer_i::pull_event(CORBA::Boolean& invalid)
{
  CORBA::Boolean matched=1;
  CORBA::Boolean hasev=0;
  CORBA::Any*    event=0;

  invalid = 0;
  _oplock.lock();
  if ( (_pxstate != RDI_Connected) || !_active ) {
	_oplock.unlock();
	return;
  }
  try {
  	event = _supplier->try_pull(hasev);
	_pulltms = RDI_TimeValue::timeofday();
	if ( ! hasev ) {
	   _oplock.unlock();
	   return;
	}
	_nevents += 1;
	if ( ! _fa_helper.has_filters() && ! _myadmin->has_filters() ) {
           _channel->new_any_event(*event);
	} else {
	   CosNF_FilterIDSeq* fseq = get_all_filters();
	   CosNF_Filter_ptr filter;
	   matched = 0;
	   for (CORBA::ULong ix=0; ix < fseq->length(); ix++) {
	       filter = get_filter((*fseq)[ix]);
	       if ( filter->match(*event) ) {
		  matched = 1;
		  break;
	       }
	   }
	   if ( matched && 
		(_myadmin->MyOperator()==CosNA_AND_OP) )
	      matched = _myadmin->match_event(*event);
	   if ( matched )
	      _channel->new_any_event(*event);
	   delete fseq;
	}
	_oplock.unlock();
	delete event;
  } catch (...) {
	invalid  = 1;
	_pxstate = RDI_Exception;
	_oplock.unlock();
	if ( event ) 
		delete event;
  }
}

void ProxyPullConsumer_i::_pull_event()
{
  CORBA::Boolean hasev=0,matched=0;
  CORBA::UShort  pullp=0;
  CORBA::Any*    event=0;

  while ( 1 ) {
     _oplock.lock();
     while ( (_pxstate == RDI_NotConnected) || 
             ((_pxstate == RDI_Connected) && ! _active) )
	_qempty->wait();
     if ( _pxstate != RDI_Connected ) {
	_thrdone = 1;
	_oplock.unlock();
	RDI_DUMP("Pull thread for proxy " << this << " exits");
	omni_thread::exit();
     }
     try {
	event = _supplier->try_pull(hasev);
	_pulltms = RDI_TimeValue::timeofday();
	if ( hasev && event ) {
	   _nevents += 1;
	   matched   = 1;
	   if ( _fa_helper.has_filters() ) {
	      CosNF_FilterIDSeq* fseq = get_all_filters();
	      CosNF_Filter_ptr filter;
	      matched = 0;
	      for (CORBA::ULong ix=0; ix < fseq->length(); ix++) {
	         filter = get_filter((*fseq)[ix]);
	         if ( filter->match(*event) ) {
		    matched = 1;
		    break;
		 }
	      }
	      delete fseq;
	   }
	   if ( matched && 
	        (_myadmin->MyOperator() == CosNA_AND_OP) )
	      matched = _myadmin->match_event(*event);
	   if ( matched )
	      _channel->new_any_event(*event);
	   delete event;
	   event = 0;
	}
	if ( (pullp = _channel->pull_period()) ) {
	   unsigned long nsec, usec, r_nsec, r_usec;
           r_nsec = pullp / 1000;
           r_usec = (pullp % 1000) * 1000;
           omni_thread::get_time(&nsec, &usec, r_nsec, r_usec);
	   _qempty->timedwait(nsec, usec);
	   _oplock.unlock();
	} else {
	   _oplock.unlock();
	   omni_thread::yield();
	}
     } catch (...) {
	_pxstate = RDI_Exception;
	RDI_DUMP("Pull thread for proxy " << this << " got exception");
	_oplock.unlock();
	if ( hasev && event )
		delete event;
     }
  }
}

ostream& operator << (ostream& out, const ProxyPullConsumer_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype) << 
         " ID " << setw(3) << prx._pserial;
  if ( prx._cosevnt ) out << " CosEventComm Consumer";
   switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " Not Connected "; break;
     case RDI_Connected:    out << " Connected     "; break;
     case RDI_Disconnected: out << " Disconnected  "; break;
     case RDI_Exception:    out << " Exception     "; break;
  }
  out << (prx._active ? "Active " : "Suspended ");
  return out << " #Pull " << prx._nevents;
}

void ProxyPullConsumer_i::disconnect_client_and_dispose()
{
  _oplock.lock();
  _pxstate = RDI_Disconnected;
  _active  = 0;
  remove_all_filters();
  // If we are using our own thread for dispatching, signal the
  // thread that it should terminate itself and wait for 5 secs
  if ( _worker && _qempty ) {
        _qempty->signal();
	_oplock.unlock();
	while ( ! _thrdone ) {
                omni_thread::yield();
        	_qempty->signal();
        }
        _oplock.lock();
        delete _qempty;
	_qempty = 0;
  }
  if ( ! CORBA::is_nil(_supplier) ) {
        // WRAPPED_RELEASE(CosEventComm::PullSupplier, _supplier);
        _supplier   = CosEventComm::PullSupplier::_nil();
  }

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}

// ------------------------------------------------------------- //
// StructuredProxyPullConsumer_i                                 //
// ------------------------------------------------------------- //

StructuredProxyPullConsumer_i::StructuredProxyPullConsumer_i(
					SupplierAdmin_i* admin,	
					EventChannel_i*  chann,
					const CosNA_ProxyID&  prxid) : 
		_oplock(), _fa_helper(), _channel(chann), _myadmin(admin),
		_prxtype(CosNA_PULL_STRUCTURED),
		_pserial(prxid), _nevents(0), _pulltms(), 
		_pxstate(RDI_NotConnected), _active(0), 
		_qempty(0), _worker(0), _thrdone(0), _qosprop(0), _offauto(0), 
		_evtypes(RDI_EventType::hash, RDI_EventType::rank)
{
  _supplier = CosNC_StructuredPullSupplier::_nil();
  // When the number of pull threads allocated at the channel level is
  // 0, each proxy uses its own thread to pull events from its supplier
  if ( _channel->pull_threads() == 0 ) {
        _qempty = new omni_condition(&_oplock);
	RDI_Assert(_qempty, "Memory allocation failed -- omni_condition");
        _worker = new StrPullWorker(this, &StructuredProxyPullConsumer_i::_pull_event);
	RDI_Assert(_worker, "Memory allocation failed -- omni_thread");
        _worker->start();
	_thrdone = 0;
	RDI_DUMP("Pull thread for proxy " << this << " -- " << _worker->id());
  }
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_SupplierAdmin_ptr StructuredProxyPullConsumer_i::MyAdmin(WRAPPED_IMPLARG_VOID)
{ return WRAPPED_IMPL2OREF(CosNA_SupplierAdmin, _myadmin); }

CosN_EventTypeSeq*
StructuredProxyPullConsumer_i::obtain_subscription_types(CosNA_ObtainInfoMode mode
						         WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq* etypeseq=0;
  if ( mode == CosNA_ALL_NOW_UPDATES_OFF ) {
        etypeseq = _channel->obtain_subscription_types();
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_ALL_NOW_UPDATES_ON ) {
        etypeseq = _channel->obtain_subscription_types();
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_OFF ) {
        etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_ON ) {
        etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  }
  return etypeseq;
}

CosN_QoSProperties* 
StructuredProxyPullConsumer_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_C_StrPRX);
}

void StructuredProxyPullConsumer_i::set_qos(
		const CosN_QoSProperties& r_qos WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_StrPRX,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos);
}

void StructuredProxyPullConsumer_i::validate_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_StrPRX,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void StructuredProxyPullConsumer_i::validate_event_qos(
	const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
 CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void StructuredProxyPullConsumer_i::disconnect_structured_pull_consumer( WRAPPED_IMPLARG_VOID )
{
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void StructuredProxyPullConsumer_i::offer_change(
			const CosN_EventTypeSeq& added,
			const CosN_EventTypeSeq& deled 
			WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq new_added;
  CosN_EventTypeSeq old_deled;
  CORBA::ULong ix=0, sz=0, vl=0;
  if ( ! RDI_EventType::valid_sequence(added, ix) )
     throw CosNC_InvalidEventType(added[ix]);
  if ( ! RDI_EventType::valid_sequence(deled, ix) )
     throw CosNC_InvalidEventType(deled[ix]);
  new_added.length(0);
  old_deled.length(0);
  _oplock.lock();
  for (sz=0, ix=0; ix < added.length(); ix++) {
     if ( _evtypes.lookup(added[ix], vl) ) {
        vl += 1; _evtypes.replace(added[ix], vl);
     } else {
        vl  = 1; _evtypes.insert(added[ix], vl);
        new_added.length(sz + 1);
        new_added[sz].domain_name = added[ix].domain_name;
        new_added[sz++].type_name = added[ix].type_name;
     }
  }
  for (sz=0, ix=0; ix < deled.length(); ix++) {
     if ( _evtypes.lookup(deled[ix], vl) ) { 
        if ( vl == 1 ) {
           _evtypes.remove(deled[ix]);
           old_deled.length(sz + 1);
           old_deled[sz].domain_name = deled[ix].domain_name;
           old_deled[sz++].type_name = deled[ix].type_name;
        } else {
           vl -= 1; _evtypes.replace(deled[ix], vl);
        }
     } else {
        RDI_DUMP("Invalid "<<deled[ix].domain_name<<"::"<<deled[ix].type_name);
     }
  }
  _oplock.unlock();
  // If any new event types were added or existing event types were
  // deleted, notify the parent administrative object
  if ( new_added.length() || old_deled.length() ) 
     _myadmin->offer_change(new_added, old_deled);
}

void StructuredProxyPullConsumer_i::connect_structured_pull_supplier(
	CosNC_StructuredPullSupplier_ptr supplier WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(supplier) || (_pxstate != RDI_NotConnected) )
        throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  if ( _pxstate == RDI_Connected )
        throw CosEventChannelAdmin::AlreadyConnected();
  _supplier = WRAPPED_DUPLICATE(CosNC_StructuredPullSupplier, supplier);
  _pxstate  = RDI_Connected;
  _active   = 1;
  if ( _worker && _qempty )     // Notify worker thread
	_qempty->signal();
  if ( _qosprop )
	delete _qosprop;
}

void StructuredProxyPullConsumer_i::suspend_connection( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected )
        throw CosNA_NotConnected();
  if ( ! _active )
        throw CosNA_ConnectionAlreadyInactive();
  _active = 0;
}

void StructuredProxyPullConsumer_i::resume_connection( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected )
        throw CosNA_NotConnected();
  if ( _active )
        throw CosNA_ConnectionAlreadyActive();
  _active = 1;
  if ( _worker && _qempty )	// Notify worker thread
	_qempty->signal();
}

void StructuredProxyPullConsumer_i::pull_event(CORBA::Boolean& invalid)
{
  CORBA::Boolean hasev=0,matched=1;
  CosN_StructuredEvent* event=0;
  RDI_StructuredEvent* sevnt=0;

  invalid = 0;
  _oplock.lock();
  if ( (_pxstate != RDI_Connected) || ! _active ) {
	_oplock.unlock();
	return;
  }
  try {
	event = _supplier->try_pull_structured_event(hasev);
	_pulltms = RDI_TimeValue::timeofday();
	if ( ! hasev ) {
	   _oplock.unlock();
	   return;
	}
	sevnt = new RDI_StructuredEvent(*event);
	RDI_Assert(sevnt, "Memory allocation failure -- RDI_StructuredEvent");
	_nevents += 1;
	if ( _fa_helper.has_filters() ) {
	   CosNF_FilterIDSeq* fltrseq = get_all_filters();
	   CosNF_Filter_ptr filter;
	   Filter_i* rdfltr=0;
	   matched = 0;
	   for (CORBA::ULong ix=0; ix < fltrseq->length(); ix++) {
	      filter = get_filter((*fltrseq)[ix]);
	      rdfltr = Filter_i::Filter2Filter_i(filter);
	      if ( (rdfltr && rdfltr->rdi_match(sevnt, _channel)) ||
		   (!rdfltr && filter->match_structured(*event)) ) {
		 matched = 1;
		 break;
	      }
	   }
	   delete fltrseq;
	}
	if (matched && (_myadmin->MyOperator()==CosNA_AND_OP))
	   matched = _myadmin->match_event(sevnt);
	if ( matched ) 
	   _channel->new_structured_event(sevnt);
	_oplock.unlock();
	// NOTE: we do not delete 'sevnt' -- inserted in event queue
	delete event;
  } catch (...) {
	invalid  = 1;
	_pxstate = RDI_Exception;
	_oplock.unlock();
	if ( event )
		delete event;
  }
}

void StructuredProxyPullConsumer_i::_pull_event()
{
  CORBA::Boolean hasev=0,matched=1;
  CORBA::UShort  pullp=0;
  CosN_StructuredEvent* event=0;
  RDI_StructuredEvent* sevnt=0;

  while ( 1 ) {
     _oplock.lock();
     while ( (_pxstate == RDI_NotConnected) ||
	     ((_pxstate == RDI_Connected) && ! _active) )
	_qempty->wait();
     if ( _pxstate != RDI_Connected ) {
	_thrdone = 1;
	_oplock.unlock();
	RDI_DUMP("Pull thread for proxy " << this << " exits");
	omni_thread::exit();
     }
     try {
	event = _supplier->try_pull_structured_event(hasev);
	_pulltms = RDI_TimeValue::timeofday();
	if ( hasev ) {
	   sevnt = new RDI_StructuredEvent(*event);
	   RDI_Assert(sevnt,"Memory allocation failure -- RDI_StructuredEvent");
	   _nevents += 1;
	   matched   = 1;
	   if ( _fa_helper.has_filters() ) {
	      CosNF_FilterIDSeq* fltrseq = get_all_filters();
	      CosNF_Filter_ptr filter;
	      Filter_i* rdfltr=0;
	      matched = 0;
	      for (CORBA::ULong ix=0; ix < fltrseq->length(); ix++) {
	         filter = get_filter((*fltrseq)[ix]);
		 rdfltr = Filter_i::Filter2Filter_i(filter);
		 if ( (rdfltr && rdfltr->rdi_match(sevnt, _channel)) ||
		      (!rdfltr && filter->match_structured(*event)) ) {
		    matched = 1;
		    break;
		 }
	      }
	      delete fltrseq;
	   }
	   if ( matched && 
		(_myadmin->MyOperator() == CosNA_AND_OP) )
	      matched = _myadmin->match_event(sevnt);
	   if ( matched )
	      _channel->new_structured_event(sevnt);
	   delete event; event = 0;
	}
	if ( (pullp = _channel->pull_period()) ) {
	   unsigned long nsec, usec, r_nsec, r_usec;
           r_nsec = pullp / 1000;
           r_usec = (pullp % 1000) * 1000;
           omni_thread::get_time(&nsec, &usec, r_nsec, r_usec);
	   _qempty->timedwait(nsec, usec);
	   _oplock.unlock();
	} else {
	   _oplock.unlock();
	   omni_thread::yield();
	}
     } catch (...) {
	_pxstate = RDI_Exception;
	RDI_DUMP("Pull thread for proxy " << this << " got exception");
	_oplock.unlock();
	if ( event ) 
		delete event;
     }
  }
}

ostream& operator << (ostream& out, const StructuredProxyPullConsumer_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype) << 
	 " ID " << setw(3) << prx._pserial;
   switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " Not Connected "; break;
     case RDI_Connected:    out << " Connected     "; break;
     case RDI_Disconnected: out << " Disconnected  "; break;
     case RDI_Exception:    out << " Exception     "; break;
  }
  out << (prx._active ? "Active " : "Suspended ");
  return out << " #Pull " << prx._nevents;
}

void StructuredProxyPullConsumer_i::disconnect_client_and_dispose()
{
  _oplock.lock();
  _pxstate = RDI_Disconnected;
  _active  = 0;
  remove_all_filters();
  // If we are using our own thread for dispatching, signal the
  // thread that it should terminate itself and wait for 5 secs
  if ( _worker && _qempty ) {
	_qempty->signal();
	_oplock.unlock();
	while ( ! _thrdone ) {
		omni_thread::yield();
		_qempty->signal();
	}
	_oplock.lock();
	delete _qempty;
	_qempty = 0;
  }
  if ( ! CORBA::is_nil(_supplier) ) {
        // WRAPPED_RELEASE(CosNC_StructuredPullSupplier, _supplier);
        _supplier = CosNC_StructuredPullSupplier::_nil();
  }

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}

// ------------------------------------------------------------- //
// SequenceProxyPullConsumer_i                                   //
// ------------------------------------------------------------- //

SequenceProxyPullConsumer_i::SequenceProxyPullConsumer_i(
                        		SupplierAdmin_i* admin,
                        		EventChannel_i*  chann,
					const CosNA_ProxyID&  prxid) : 
		_oplock(), _fa_helper(), _channel(chann), _myadmin(admin),
		_prxtype(CosNA_PULL_SEQUENCE),
		_pserial(prxid), _nevents(0), _pulltms(),
		_pxstate(RDI_NotConnected), _active(0), _qempty(0), 
		_worker(0), _thrdone(0), _qosprop(0), _offauto(0),
		_evtypes(RDI_EventType::hash, RDI_EventType::rank)
{
  _supplier = CosNC_SequencePullSupplier::_nil();
  // When the number of pull threads allocated at the channel level is
  // 0, each proxy uses its own thread to pull events from its supplier
  if ( _channel->pull_threads() == 0 ) {
	_qempty = new omni_condition(&_oplock);
	RDI_Assert(_qempty, "Memory allocation failed -- omni_condition");
	_worker = new SeqPullWorker(this, &SequenceProxyPullConsumer_i::_pull_event);
	RDI_Assert(_worker, "Memory allocation failed -- omni_thread");
	_worker->start();
	_thrdone = 0;
	RDI_DUMP("Pull thread for proxy " << this << " -- " << _worker->id());
  }
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_SupplierAdmin_ptr SequenceProxyPullConsumer_i::MyAdmin( WRAPPED_IMPLARG_VOID )
{ return WRAPPED_IMPL2OREF(CosNA_SupplierAdmin, _myadmin); }

CosN_EventTypeSeq*
SequenceProxyPullConsumer_i::obtain_subscription_types(CosNA_ObtainInfoMode mode
						       WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq* etypeseq=0;
  if ( mode == CosNA_ALL_NOW_UPDATES_OFF ) {
        etypeseq = _channel->obtain_subscription_types();
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_ALL_NOW_UPDATES_ON ) {
        etypeseq = _channel->obtain_subscription_types();
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_OFF ) {
        etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_ON ) {
        etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  }
  return etypeseq;
}

CosN_QoSProperties* 
SequenceProxyPullConsumer_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_C_StrPRX);
}

void SequenceProxyPullConsumer_i::set_qos(
		const CosN_QoSProperties& r_qos WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_SeqPRX,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos);
}

void SequenceProxyPullConsumer_i::validate_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_SeqPRX,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void SequenceProxyPullConsumer_i::validate_event_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void SequenceProxyPullConsumer_i::disconnect_sequence_pull_consumer( WRAPPED_IMPLARG_VOID )
{
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void SequenceProxyPullConsumer_i::offer_change(
			const CosN_EventTypeSeq& added,
			const CosN_EventTypeSeq& deled
			WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq new_added;
  CosN_EventTypeSeq old_deled;
  CORBA::ULong ix=0, sz=0, vl=0;
  if ( ! RDI_EventType::valid_sequence(added, ix) )
     throw CosNC_InvalidEventType(added[ix]);
  if ( ! RDI_EventType::valid_sequence(deled, ix) )
     throw CosNC_InvalidEventType(deled[ix]);
  new_added.length(0);
  old_deled.length(0);
  _oplock.lock();
  for (sz=0, ix=0; ix < added.length(); ix++) {
     if ( _evtypes.lookup(added[ix], vl) ) {
        vl += 1; _evtypes.replace(added[ix], vl);
     } else {
        vl  = 1; _evtypes.insert(added[ix], vl);
        new_added.length(sz + 1);
        new_added[sz].domain_name = added[ix].domain_name;
        new_added[sz++].type_name = added[ix].type_name;
     }
  }
  for (sz=0, ix=0; ix < deled.length(); ix++) {
     if ( _evtypes.lookup(deled[ix], vl) ) { 
        if ( vl == 1 ) {
           _evtypes.remove(deled[ix]);
           old_deled.length(sz + 1);
           old_deled[sz].domain_name = deled[ix].domain_name;
           old_deled[sz++].type_name = deled[ix].type_name;
        } else {
           vl -= 1; _evtypes.replace(deled[ix], vl);
        }
     } else {
        RDI_DUMP("Invalid "<<deled[ix].domain_name<<"::"<<deled[ix].type_name);
     }
  }
  _oplock.unlock();
  // If any new event types were added or existing event types were
  // deleted, notify the parent administrative object
  if ( new_added.length() || old_deled.length() ) 
     _myadmin->offer_change(new_added, old_deled);
}

void SequenceProxyPullConsumer_i::connect_sequence_pull_supplier(
	CosNC_SequencePullSupplier_ptr supplier WRAPPED_IMPLARG )
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(supplier) || (_pxstate != RDI_NotConnected) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  if ( _pxstate == RDI_Connected ) 
        throw CosEventChannelAdmin::AlreadyConnected();
  _supplier = WRAPPED_DUPLICATE(CosNC_SequencePullSupplier, supplier);
  _pxstate  = RDI_Connected;
  _active   = 1;
  if ( _worker && _qempty )	// Notify worker thread
	_qempty->signal();
}

void SequenceProxyPullConsumer_i::suspend_connection( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected )
        throw CosNA_NotConnected();
  if ( ! _active )
        throw CosNA_ConnectionAlreadyInactive();
  _active = 0;
}

void SequenceProxyPullConsumer_i::resume_connection( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected ) 
        throw CosNA_NotConnected();
  if ( _active )
        throw CosNA_ConnectionAlreadyActive();
  _active = 1;
  if ( _worker && _qempty )     // Notify worker thread
        _qempty->signal();
}

void SequenceProxyPullConsumer_i::pull_event(CORBA::Boolean& invalid)
{
  CORBA::Boolean hasev=0;
  CORBA::ULong   mxnum=5;
  CosN_EventBatch* events=0;
  RDI_StructuredEvent* sevnt=0;

  invalid = 0;
  _oplock.lock();
  if ( (_pxstate != RDI_Connected) || ! _active ) {
	_oplock.unlock();
	return;
  }
  try {
        events = _supplier->try_pull_structured_events(mxnum, hasev);
	_pulltms = RDI_TimeValue::timeofday();
	if ( ! hasev ) {
	   if ( events )
	      delete events;
	   _oplock.unlock();
	   return;
	}
	_nevents += events->length();
	if ( ! _fa_helper.has_filters() && ! _myadmin->has_filters() ) {
	   _channel->new_sequence_event(*events);
	} else {
	   CosNF_FilterIDSeq* fltrseq = get_all_filters();
	   CosNF_Filter_ptr filter;
	   Filter_i* rdfltr = 0;
	   CORBA::Boolean matched = 0;
	   for (CORBA::ULong ex=0; ex < events->length(); ex++) {
	      matched = 0; sevnt = new RDI_StructuredEvent((*events)[ex]);
	      for (CORBA::ULong ix=0; ix < fltrseq->length(); ix++) {
	         filter = get_filter((*fltrseq)[ix]);
		 rdfltr = Filter_i::Filter2Filter_i(filter);
		 if ( (rdfltr && rdfltr->rdi_match(sevnt, _channel)) ||
		      (!rdfltr && filter->match_structured((*events)[ex])) ) {
		    matched = 1;
		    break;
		 }
		 if ( matched &&
	              (_myadmin->MyOperator()==CosNA_AND_OP) )
		    matched = _myadmin->match_event(sevnt);
		 if ( matched )
		    _channel->new_structured_event(sevnt);
	      }
	      // NOTE: we do not delete 'sevnt' -- inserted in event queue
	   }
	}
	_oplock.unlock();
	delete events; events = 0;
  } catch (...) {
	invalid  = 1;
	_pxstate = RDI_Exception;
	_oplock.unlock();
	if ( events )
		delete events;
  }
}

void SequenceProxyPullConsumer_i::_pull_event()
{
  CORBA::Boolean hasev=0;
  CORBA::ULong   mxnum=5;
  CORBA::UShort  pullp=0;
  CosN_EventBatch* events=0;
  RDI_StructuredEvent* sevnt=0;

  while ( 1 ) {
     _oplock.lock();
     while ( (_pxstate == RDI_NotConnected) ||
	     ((_pxstate == RDI_Connected) && ! _active) )
	_qempty->wait();
     if ( _pxstate != RDI_Connected ) {
	_thrdone = 1;
	_oplock.unlock();
	RDI_DUMP("Pull thread for proxy " << this << " exits");
	omni_thread::exit();
     }
     try {
	events = _supplier->try_pull_structured_events(mxnum, hasev);
	_pulltms = RDI_TimeValue::timeofday();
	if ( hasev ) {
	   _nevents += events->length();
	   if ( ! _fa_helper.has_filters() && ! _myadmin->has_filters() ) {
	      _channel->new_sequence_event(*events);
	   } else {
	      CosNF_FilterIDSeq* fltrseq = get_all_filters();
	      CosNF_Filter_ptr filter;
	      Filter_i* rdfltr = 0;
	      CORBA::Boolean matched = 0;
	      for (CORBA::ULong ex=0; ex < events->length(); ex++) {
		 matched = 0; sevnt = new RDI_StructuredEvent((*events)[ex]);
		 for (CORBA::ULong ix=0; ix < fltrseq->length(); ix++) {
		    filter = get_filter((*fltrseq)[ix]);
		    rdfltr = Filter_i::Filter2Filter_i(filter);
	 	    if ((rdfltr && rdfltr->rdi_match(sevnt, _channel)) ||
			(!rdfltr && filter->match_structured((*events)[ex]))) {
		       matched = 1;
		       break;
		    }
		    if (matched &&
			(_myadmin->MyOperator()==CosNA_AND_OP))
		       matched = _myadmin->match_event(sevnt);
		    if ( matched )
		       _channel->new_structured_event(sevnt);
		 }
		 // NOTE: we do not delete 'sevnt' -- inserted in event queue
	      }
	      delete fltrseq;
	   }
	}
	if ( events ) {
	   delete events;
	    events = 0;
	}
	if ( (pullp = _channel->pull_period()) ) {
	   unsigned long nsec, usec, r_nsec, r_usec;
           r_nsec = pullp / 1000;
           r_usec = (pullp % 1000) * 1000;
           omni_thread::get_time(&nsec, &usec, r_nsec, r_usec);
	   _qempty->timedwait(nsec, usec);
	   _oplock.unlock();
	} else {
	   _oplock.unlock();
	   omni_thread::yield();
	}
     } catch (...) {
	_pxstate = RDI_Exception;
	RDI_DUMP("Pull thread for proxy " << this << " got exception");
	_oplock.unlock();
	if ( events )
		delete events;
     }
  }
}

ostream& operator << (ostream& out, const SequenceProxyPullConsumer_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype) <<
         " ID " << setw(3) << prx._pserial;
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " Not Connected "; break;
     case RDI_Connected:    out << " Connected     "; break;
     case RDI_Disconnected: out << " Disconnected  "; break;
     case RDI_Exception:    out << " Exception     "; break;
  }
  out << (prx._active ? "Active " : "Suspended ");
  return out << " #Pull " << prx._nevents;
}

void SequenceProxyPullConsumer_i::disconnect_client_and_dispose()
{
  _oplock.lock();
  _pxstate = RDI_Disconnected;
  _active  = 0;
  remove_all_filters();
  // If we are using our own thread for dispatching, signal the
  // thread that it should terminate itself and wait for 5 secs
  if ( _worker && _qempty ) {
        _qempty->signal();
        _oplock.unlock();
	while ( ! _thrdone ) {
		omni_thread::yield();
        	_qempty->signal();
	}
        _oplock.lock();
        delete _qempty;
	_qempty = 0;
  }
  if ( ! CORBA::is_nil(_supplier) ) {
        // WRAPPED_RELEASE(CosNC_SequencePullSupplier, _supplier);
        _supplier   = CosNC_SequencePullSupplier::_nil();
  }

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}
