// -*- Mode: C++; -*-
//                              File      : RDIChannelUtil.h
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    proprietary interface
//
 
/*
$Log: RDIChannelUtil.h,v $
Revision 1.12  2000/08/22 18:23:48  alcfp
added description to each file

Revision 1.11  2000/08/16 20:19:04  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#ifndef _RDI_CHANNEL_UTIL_H_
#define _RDI_CHANNEL_UTIL_H_

#include "RDIUtil.h"
#include "RDIList.h"
#include "RDITime.h"
#include "omnithread.h"

#include "RDIThreadPriority.h"
#include "CosNotify.h"
#include "CosNotification_i.h"

#if defined (__OMNIORB3__)
#include <omniORB3/CORBA.h>
#elif defined (__OMNIORB2__)
#include <omniORB2/CORBA.h>
#elif defined (__ORBIX3__)
#include "CORBA.h"
#endif

class RDI_RVM;
class EventChannel_i;
class ConsumerAdmin_i;
class SupplierAdmin_i;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
// Event Channel Statistics related structures                           //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

typedef struct RDI_ThreadStat {
  omni_mutex   _lock;
  RDI_RVM*     _rvm;
  CORBA::ULong _num_rdi_match;
  CORBA::ULong _num_rvm_eval;
  CORBA::ULong _num_announcements;
  CORBA::ULong _num_notifications;
  CORBA::ULong _qsize_acum;
  CORBA::ULong _qsize_ctr;
  CORBA::ULong _dummy[16]; 	// 64 bytes to force cache line separation?
} RDI_ThreadStat;

// max thread id must be < this number
#define RDI_TH_ARRAY_SZ 1024

#define RDI_CHECK_THREAD_ID(id) \
        RDI_Assert(id<RDI_TH_ARRAY_SZ, "Recompile with larger RDI_TH_ARRAY_SZ")

// report a delta every 1000 annoucements and/or notifications
// (the delta ctr increments every time a single thread has done 100 of these)
#define RDI_STATS_MINOR_INCREMENT 100
#define RDI_STATS_DELTA_INCREMENT 10

#define RDI_STATS_LOCK     _stats_lock.lock()
#define RDI_STATS_UNLOCK   _stats_lock.unlock()

#define RDI_THREAD_STATS_LOCK(id)     _thread_stats[id]._lock.lock()
#define RDI_THREAD_STATS_UNLOCK(id)   _thread_stats[id]._lock.unlock()

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
// Some Event Type related support functions                             //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

inline int rdi_list_member(const RDI_List<CosN_EventType>& tlist,
			   const CosN_EventType&           etype)
{
  RDI_ListCursor<CosN_EventType> curs = tlist.cursor();
  unsigned int indx;
  for (indx = 0; indx < tlist.length(); ++indx, ++curs) {
	const CosN_EventType& node = *curs;
	if ( (strcmp(node.type_name, etype.type_name) == 0) && 
	     (strcmp(node.domain_name, etype.domain_name) == 0) )
		return 1;
  }
  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
// Various Support Classes and Utility Functions                         //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

/** RDI_DummyProxy_i
  *
  * This class does not contain any members. It is only used for the
  * registration of Supplier Proxy filters with the TypeMap.  Hence,
  * all ProxySupplier objects should inherit from this class.
  */

class RDI_DummyProxy_i {
public:
  RDI_DummyProxy_i() {;}
};

/** RDIProxyPullConsumer and RDIProxyPushSupplier
  *
  * Support classes that are need by the pool of threads we create to
  * push notifications to consumers and pull events from suppliers...
  */

class RDIProxyPullConsumer {
public:
  virtual void pull_event(CORBA::Boolean& invalid) = 0;
  virtual const RDI_TimeValue& last_pull_tms() const = 0;
};

class RDIProxyPushSupplier {
public:
  virtual void push_event(CORBA::Boolean& invalid) = 0;
  virtual CORBA::Boolean has_events() const = 0;
};

/** The state of a proxy can be one of the following:
  *   - RDI_NotConnected : no supplier/consumer has been connected yet
  *   - RDI_Connected    : a supplier/consumer  is interacting with it
  *   - RDI_Disconnected : the supplier/consumer broke  the connection
  *   - RDI_Exception    : exception caught  during client interaction
  */

enum RDI_ProxyState { RDI_NotConnected, RDI_Connected, 
		      RDI_Disconnected, RDI_Exception };

/** The following is used during event dispatching to record the state
  * of filter evaluation at the ConsumerAdmin level
  */

enum RDI_FilterState_t { NoFilters, OrMatch, AndMatch, OrFailed };

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
// Event Channel Thread related structures                               //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

/** A number of threads are used by each Event Channel for event filtering.
  * The total number of created threads and the work carried out by each of
  * them are set during channel creation.  The following class  is used for
  * this purpose.
  */

typedef void (EventChannel_i::*EventChannelMethod)(void);

class EventChannelDispatch {
public:
  EventChannelDispatch(EventChannel_i*    event_channel,
                       unsigned int       num_admin_threads,
                       EventChannelMethod admin_method,
                       unsigned int       num_proxy_threads,
                       EventChannelMethod proxy_method);
  ~EventChannelDispatch();
private:
  unsigned int    _num_athreads;
  omni_thread**   _admin_thread;
  unsigned int    _num_pthreads;
  omni_thread**   _proxy_thread;
  EventChannel_i* _evnt_channel;

  EventChannelDispatch()        {;}
  EventChannelDispatch(const EventChannelDispatch&)     {;}
};

/** Admin and proxy objects may by grouped together for several reasons.
  * Here, we group them so that only one thread can evaluate filters for
  * all entries in a given group.
  */

class ChannelAdminGroup {
        friend class ChannelAdminGroup_m;
public:
  ChannelAdminGroup();
  ~ChannelAdminGroup();

  void insert(ConsumerAdmin_i* entry);
  void remove(ConsumerAdmin_i* entry);

  // Iterate over all entries in the group -- NULL when done
  ConsumerAdmin_i* next();

  friend ostream& operator << (ostream& out, ChannelAdminGroup& agr);
private:
  omni_mutex                        _oplock;
  RDI_List<ConsumerAdmin_i *>       _calist;
  RDI_ListCursor<ConsumerAdmin_i *> _cacurs;
  unsigned int                      _length;
};

class ChannelAdminGroup_m {
public:
  ChannelAdminGroup_m(unsigned int numGroups, unsigned int numThreads);
  ~ChannelAdminGroup_m();

  // Insert/Remove an entry into/from a group -- the selection is
  // based on the number of groups and the ID assigned to 'entry'

  void insert(ConsumerAdmin_i* entry);
  void remove(ConsumerAdmin_i* entry);

  // The groups are accessed by a number of threads.  This number
  // is supplied to the above constructor.  Each thread MUST call
  // 'allocate_range()'  to get an inclusive range of groups that
  // it will be responsible for accessing using the [] operator.

  CORBA::Boolean allocate_range(unsigned int& lo, unsigned int &hi);
  ChannelAdminGroup* operator [] (unsigned int grpix)
                        { return (grpix >= _length) ? 0 : & _groups[grpix]; }

  // From time to time we need to be able to debug the module
  friend ostream& operator << (ostream& out, ChannelAdminGroup_m& agrm);

private:
  omni_mutex         _oplock;
  ChannelAdminGroup* _groups;
  unsigned int       _length;
  unsigned int       _nextgr;
  unsigned int       _numthr;
  unsigned int       _rmgrps;
  unsigned int       _rmthrs;
};

/** The number of pull suppliers and push consumers connected to a channel
  * may be large and,  hence, we should use a pool of threads instead of a
  * thread per client for pulling events from suppliers or pushing matched
  * events to consumers.  The following support classes are used for this.
  */

class RDI_PullSupplier {
public:
  RDI_PullSupplier(unsigned int numThreads=1, unsigned int periodMSecs=100);
  ~RDI_PullSupplier()           { this->destroy(); }

  void destroy();

  // Register/Deregister a ProxyConsumer so that created threads can pull
  // its connected supplier for available events
  void insert_proxy(RDIProxyPullConsumer* proxy);
  void remove_proxy(RDIProxyPullConsumer* proxy);

  // Update the pull period --- We do not have to get the lock here since
  // up-to-date information about '_period' is not crucial
  void set_pull_period(unsigned long nmsecs=0)	{ _period = nmsecs; }

  // The member function that gets executed by all created threads
  void pull_event();

  // Support structures needed for the implementation of the class
  struct ProxyEntry_t {
        RDIProxyPullConsumer* _proxy;
        unsigned char         _inuse;
        unsigned char         _deled;
        ProxyEntry_t*         _next;
        ProxyEntry_t(RDIProxyPullConsumer* prx=0) :
                        _proxy(prx), _inuse(0), _deled(0), _next(0) {;}
  };
private:
  omni_mutex     _oplock;
  omni_condition _nonempty;
  unsigned long  _period;
  omni_thread**  _threads;
  unsigned int   _nactive;      // Number of threads still running
  unsigned int   _ndeleted;     // Number of deleted proxy entries
  unsigned char  _terminate;    // Set to true  during destruction
  ProxyEntry_t*  _entries;      // Registered  push  ProxySupplier
  ProxyEntry_t*  _lastone;      // Last entry accessed by a thread

  // Locate the next entry that is not being worked on by an active
  // thread -- the invocation is protected by the '_oplock' mutex
  ProxyEntry_t*  _next_available();

  // Destroy the entries that are not used  and have been marked as
  // deleted -- the invocation is protected by the '_oplock' mutex
  void _gcollect();
};

class RDI_NotifyConsumer {
public:
  RDI_NotifyConsumer(unsigned int numThreads=1);
  ~RDI_NotifyConsumer()         { this->destroy(); }

  void destroy();

  // The following is called by the channel when an event is added
  // to the queue of a ProxySupplier. We need this because we want
  // to avoid having the push threads spinning doing nothing .....

  void signal_push_threads()	{ _nonempty.broadcast(); }

  // Register/Deregister a ProxySupplier so that notifications are
  // sent to its connected consumer by one of the created theads
  void insert_proxy(RDIProxyPushSupplier* proxy);
  void remove_proxy(RDIProxyPushSupplier* proxy);

  // The member function that gets executed by all created threads
  void notify();

  // Support structures needed for the implementation of the class
  struct ProxyEntry_t {
        RDIProxyPushSupplier* _proxy;
        unsigned char         _inuse;
        unsigned char         _deled;
        ProxyEntry_t*         _next;
        ProxyEntry_t(RDIProxyPushSupplier* prx=0) :
                        _proxy(prx), _inuse(0), _deled(0), _next(0) {;}
  };
private:
  omni_mutex     _oplock;
  omni_condition _nonempty;
  omni_thread**  _threads;
  unsigned int   _nactive;      // Number of threads still running
  unsigned int   _ndeleted;     // Number of deleted proxy entries
  unsigned char  _terminate;    // Set to true  during destruction
  ProxyEntry_t*  _entries;      // Registered  push  ProxySupplier
  ProxyEntry_t*  _lastone;      // Last entry accessed by a thread

  // Locate the next entry that is not being worked on by an active
  // thread -- the invocation is protected by the '_oplock' mutex
  ProxyEntry_t*  _next_available();

  // Destroy the entries that are not used  and have been marked as
  // deleted -- the invocation is protected by the '_oplock' mutex
  void _gcollect();
};

/** Each event channel uses a number of threads for carrying out filter
  * evaluation, garbage collection etc..  The classes below provide the
  * "wrappers" for creating these threads.
  */

class EventChannelWorker : public omni_thread {
public:
  EventChannelWorker(EventChannel_i* channel, EventChannelMethod method,
                     priority_t priority = RDI_CHANNEL_WORKER_PRIORITY);
  void run(void *);
private:
  EventChannel_i*    _channel;
  EventChannelMethod _method;
  EventChannelWorker()  {;}
};

class EventChannelBoundWorker : public omni_thread {
public:
  EventChannelBoundWorker(EventChannel_i* channel, EventChannelMethod method,
			  priority_t priority = RDI_CHANNEL_WORKER_PRIORITY);
  void* run_undetached(void *);
private:
  EventChannel_i*    _channel;
  EventChannelMethod _method;
  EventChannelBoundWorker() {;}
};

#endif
