// -*- Mode: C++; -*-
//                              File      : CosNotifyChannelAdmin_i.h
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    set of C++ definitions for the CosNotifyChannelAdmin module
//
 
/*
$Log: CosNotifyChannelAdmin_i.h,v $
Revision 1.79  2000/10/04 02:40:04  alcfp
small fixes to avoid some compiler warnings

Revision 1.78  2000/10/01 13:29:09  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.77.2.1  2000/09/30 15:39:50  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.77  2000/08/22 18:23:47  alcfp
added description to each file

Revision 1.76  2000/08/16 20:18:55  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#ifndef _COS_NOTIFY_CHANNEL_ADMIN_I_H_
#define _COS_NOTIFY_CHANNEL_ADMIN_I_H_

#include "CosNotify.h"
#include "corba_wrappers.h"
#include "RDIEvent.h"
#include "RDIEventQueue.h"
#include "RDIChannelUtil.h"
#include "CosNotifyFilter_i.h"
#include "CosNotification_i.h"
#include "COS/BOA/AttNotifyChannelAdmin.hh"

class RDI_TypeMap;
class EventChannel_i;
class SupplierAdmin_i;
class ConsumerAdmin_i;
class EventChannelFactory_i;
class EventProxyPushSupplier_i;
class EventProxyPullSupplier_i;
class EventProxyPushConsumer_i;
class EventProxyPullConsumer_i;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//               Push-based Consumer Proxy implementations               //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

/** ProxyPushConsumer
  * This proxy object is connected to a push supplier and receives
  * CORBA::Any events
  */
class ProxyPushConsumer_i :
	WRAPPED_SKELETON_SUPER(CosNA::, ProxyPushConsumer) 
{
  friend class EventChannel_i;
  friend class SupplierAdmin_i;
public:
  ProxyPushConsumer_i(SupplierAdmin_i* admin, 
		      EventChannel_i*  chann, const CosNA_ProxyID& prxID);

  // Methods from CosNA_ProxyConsumer Interface
  CosNA_ProxyType         MyType( WRAPPED_DECLARG_VOID ) { return _prxtype; }
  CosNA_ProxyID           MyID( WRAPPED_DECLARG_VOID )   { return _pserial; }
  CosNA_SupplierAdmin_ptr MyAdmin( WRAPPED_DECLARG_VOID );
  CosN_EventTypeSeq*      obtain_subscription_types(CosNA_ObtainInfoMode mode
			  	                    WRAPPED_DECLARG );
  void validate_event_qos(const CosN_QoSProperties& r_qos,
         WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
         WRAPPED_DECLARG);

  // Methods from CosNA_ProxyPushConsumer Interface
  void connect_any_push_supplier(CosEventComm::PushSupplier_ptr supplier
				 WRAPPED_DECLARG);

  // Methods from CosEventComm::PushConsumer Interface
  void push(const CORBA::Any& data WRAPPED_DECLARG );
  void disconnect_push_consumer( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
 	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
       	WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void          remove_filter(CosNF_FilterID fltrID  WRAPPED_DECLARG ) 
			{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
			{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
			{ return _fa_helper.get_all_filters(); }
  void          remove_all_filters( WRAPPED_DECLARG_VOID ) 
			{ _fa_helper.remove_all_filters(); }

  CosNF_FilterID add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG); 

  // also map has_filters extra method up to this class
  CORBA::Boolean has_filters() const { return _fa_helper.has_filters(); }

  // Methods from CosNC_NotifyPublish Interface
  void offer_change(const CosN_EventTypeSeq& added, 
		    const CosN_EventTypeSeq& deled WRAPPED_DECLARG );

  friend ostream& operator << (ostream& out, const ProxyPushConsumer_i& prx);
private:
  omni_mutex       _oplock;	// Serialize operation on object
  FAdminHelper     _fa_helper;
  EventChannel_i*  _channel;
  SupplierAdmin_i* _myadmin;
  CosNA_ProxyType  _prxtype;	// This must be PUSH_ANY
  CosNA_ProxyID    _pserial;
  CORBA::Boolean   _cosevnt;	// TRUE if CosEventComm supplier
  CORBA::ULong     _nevents;
  RDI_ProxyState   _pxstate;
  RDI_NotifQoS*    _qosprop;
  CORBA::Boolean   _offauto;	// TRUE if subscription_change() is disabled
  RDI_Hash<CosN_EventType, CORBA::ULong> _evtypes;
  CosEventComm::PushSupplier_var _supplier;

  // This is invoked by the administrative object during destruction
  void disconnect_client_and_dispose();
};

/** StructuredProxyPushConsumer
  * This proxy object is connected to a push supplier and receives
  * CosN_StructuredEvent events
  */
class StructuredProxyPushConsumer_i : 
   	WRAPPED_SKELETON_SUPER(CosNA::, StructuredProxyPushConsumer) 
{
  friend class EventChannel_i;
  friend class SupplierAdmin_i;
public:
  StructuredProxyPushConsumer_i(SupplierAdmin_i*     admin,
				EventChannel_i*      chann, 
				const CosNA_ProxyID& prxID);

  // Methods from CosNA_ProxyConsumer Interface
  CosNA_ProxyType         MyType( WRAPPED_DECLARG_VOID )  { return _prxtype; }
  CosNA_ProxyID           MyID( WRAPPED_DECLARG_VOID ) 	  { return _pserial; }
  CosNA_SupplierAdmin_ptr MyAdmin( WRAPPED_DECLARG_VOID );
  CosN_EventTypeSeq*      obtain_subscription_types(CosNA_ObtainInfoMode mode
					            WRAPPED_DECLARG );
  void validate_event_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
	WRAPPED_DECLARG );

  // Methods from CosNA_StructuredProxyPushConsumer Interface
  void connect_structured_push_supplier(
		CosNC_StructuredPushSupplier_ptr supplier WRAPPED_DECLARG );

  // Methods from CosNC_StructuredPushConsumer Interface
  void push_structured_event(const CosN_StructuredEvent& event WRAPPED_DECLARG);
  void disconnect_structured_push_consumer( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void               remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void               remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }
  CosNF_FilterID     add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG )
    				{ return _fa_helper.add_filter_i(filter); }

  // also map has_filters extra method up to this class
  CORBA::Boolean has_filters() const { return _fa_helper.has_filters(); }

  // Methods from CosNC_NotifyPublish Interface
  void offer_change(const CosN_EventTypeSeq& added, 
		    const CosN_EventTypeSeq& deled WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

#ifdef __FASTPATH__
  void push_structured_event(const CosN_StructuredEvent&, cdrMemoryStream*);
  CORBA::Boolean dispatch(GIOP_S &s, const char *op, CORBA::Boolean response);
#endif

  friend ostream& operator << (ostream& out, 
			       const StructuredProxyPushConsumer_i& prx);
private:
  omni_mutex       _oplock;	// Serialize operation on object
  FAdminHelper     _fa_helper;
  EventChannel_i*  _channel;
  SupplierAdmin_i* _myadmin;
  CosNA_ProxyType  _prxtype;	// This must be PUSH_STRUCTURED
  CosNA_ProxyID    _pserial;
  CORBA::ULong     _nevents;
  RDI_ProxyState   _pxstate;
  RDI_NotifQoS*    _qosprop;
  CORBA::Boolean   _offauto;	// TRUE if subscription_change() is disabled
  RDI_Hash<CosN_EventType, CORBA::ULong> _evtypes;
  CosNC_StructuredPushSupplier_var _supplier;

  // This is invoked by the administrative object during destruction
  void disconnect_client_and_dispose();
};

/** SequenceProxyPushConsumer
  * This proxy object is connected to a push supplier and receives
  * CosN_EventBatch events
  */

class SequenceProxyPushConsumer_i : 
  	WRAPPED_SKELETON_SUPER(CosNA::, SequenceProxyPushConsumer) 
{
  friend class EventChannel_i;
  friend class SupplierAdmin_i;
 public:
  SequenceProxyPushConsumer_i(SupplierAdmin_i* admin, 
			      EventChannel_i*  chann, 
			      const CosNA_ProxyID& prxID);

  // Methods from CosNA_ProxyConsumer Interface
  CosNA_ProxyType         MyType( WRAPPED_DECLARG_VOID ) { return _prxtype; }
  CosNA_ProxyID           MyID( WRAPPED_DECLARG_VOID )   { return _pserial; }
  CosNA_SupplierAdmin_ptr MyAdmin( WRAPPED_DECLARG_VOID );
  CosN_EventTypeSeq*      obtain_subscription_types(CosNA_ObtainInfoMode mode
					            WRAPPED_DECLARG );
  void validate_event_qos(const CosN_QoSProperties& r_qos,
          WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
	  WRAPPED_DECLARG );

  // Methods from CosNA_SequenceProxyPushConsumer Interface
  void connect_sequence_push_supplier(CosNC_SequencePushSupplier_ptr supplier
				      WRAPPED_DECLARG );

  // Methods from CosNC_SequencePushConsumer Interface
  void push_structured_events(const CosN_EventBatch& events WRAPPED_DECLARG );
  void disconnect_sequence_push_consumer( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
            WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
            WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void               remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void               remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }
  CosNF_FilterID     add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG )
    				{ return _fa_helper.add_filter_i(filter); }

  // also map has_filters extra method up to this class
  CORBA::Boolean has_filters() const { return _fa_helper.has_filters(); }

  // Methods from CosNC_NotifyPublish Interface
  void offer_change(const CosN_EventTypeSeq& added,
		    const CosN_EventTypeSeq& deled WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

  friend ostream& operator << (ostream& out, 
			       const SequenceProxyPushConsumer_i& prx);
private:
  omni_mutex       _oplock;	// Serialize operation on object
  FAdminHelper     _fa_helper;
  EventChannel_i*  _channel;
  SupplierAdmin_i* _myadmin;
  CosNA_ProxyType  _prxtype;	// This must be PUSH_SEQUENCE
  CosNA_ProxyID    _pserial;
  CORBA::ULong     _nevents;
  RDI_ProxyState   _pxstate;
  RDI_NotifQoS*    _qosprop;
  CORBA::Boolean   _offauto;	// TRUE if subscription_change() is disabled
  RDI_Hash<CosN_EventType, CORBA::ULong> _evtypes;
  CosNC_SequencePushSupplier_var _supplier;

   // This is invoked by the administrative object during destruction
  void disconnect_client_and_dispose();
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//               Pull-based Supplier Proxy implementations               //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

/** ProxyPullSupplier
  * This proxy object is connected to a pull consumer and provides
  * CORBA::Any events
  */

class ProxyPullSupplier_i : 
	public RDI_DummyProxy_i, 
	public RDINotifySubscribe, 
	WRAPPED_SKELETON_SUPER(CosNA::, ProxyPullSupplier) 
{
  friend class EventChannel_i;
  friend class ConsumerAdmin_i;
public:
  ProxyPullSupplier_i(ConsumerAdmin_i* admin,
		      EventChannel_i*  chann, const CosNA_ProxyID& prxID);

  // Methods from CosNA_ProxySupplier Interface
  CosNA_ProxyType         MyType( WRAPPED_DECLARG_VOID )   { return _prxtype; }
  CosNA_ProxyID           MyID( WRAPPED_DECLARG_VOID )     { return _pserial; }
  CosNA_ConsumerAdmin_ptr MyAdmin( WRAPPED_DECLARG_VOID );
  CosNF_MappingFilter_ptr priority_filter( WRAPPED_DECLARG_VOID );
  void                    priority_filter(CosNF_MappingFilter_ptr map_filter
				          WRAPPED_DECLARG );
  CosNF_MappingFilter_ptr lifetime_filter( WRAPPED_DECLARG_VOID );
  void                    lifetime_filter(CosNF_MappingFilter_ptr life_filter
				          WRAPPED_DECLARG );
  CosN_EventTypeSeq*      obtain_offered_types(CosNA_ObtainInfoMode mode
				                WRAPPED_DECLARG );
  void validate_event_qos(const CosN_QoSProperties& r_qos,
          WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
          WRAPPED_DECLARG );

  // Methods from CosNA_ProxyPullSupplier Interface
  void connect_any_pull_consumer(CosEventComm::PullConsumer_ptr cons
			         WRAPPED_DECLARG );

  // Methods from CosEventComm::PullSupplier Interface
  CORBA::Any* pull( WRAPPED_DECLARG_VOID );
  CORBA::Any* try_pull(CORBA::Boolean& has_event WRAPPED_DECLARG );
  void        disconnect_pull_supplier( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void               remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void               remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }
  CosNF_FilterID     add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG );

  // augmented version of has_filters
  CORBA::Boolean has_filters() const
		{ return (_rqstypes.length() ? 1 : _fa_helper.has_filters()); }

  // Methods from CosNC_NotifySubscribe Interface
  void subscription_change(const CosN_EventTypeSeq& added,
			   const CosN_EventTypeSeq& deled WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

  void add_event(RDI_StructuredEvent* event);
  void subscription_change_i(const CosN_EventTypeSeq& added,
			     const CosN_EventTypeSeq& deled, Filter_i* flter);
  void filter_destroy_i(Filter_i* filter);

  friend ostream& operator << (ostream& out, const ProxyPullSupplier_i& prx);
private:
  omni_mutex         _oplock;	// Serialize operations on object
  FAdminHelper       _fa_helper;
  omni_condition     _noempty;	// Block until queue has an entry
  EventChannel_i*    _channel;
  ConsumerAdmin_i*   _myadmin;
  CosNA_ProxyType    _prxtype;	// This must be PULL_ANY
  CosNA_ProxyID      _pserial;
  CORBA::Boolean     _cosevnt;	// TRUE if CosEventComm consumer
  CORBA::ULong       _nevents;
  CORBA::ULong       _blkpull;	// Number of blocked pull() threads
  RDI_ProxyState     _pxstate;
  CosNF_MappingFilter_ptr _pfilter;
  CosNF_MappingFilter_ptr _lfilter;
  RDI_NotifQoS*      _qosprop;
  CORBA::Boolean     _offauto;	// TRUE if offer_change() is disabled
  CosN_EventTypeSeq   _rqstypes;
  CosEventComm::PullConsumer_var  _consumer;
  RDI_List<RDI_StructuredEvent *> _ntfqueue;

  // This is invoked by the administrative object during destruction
  void disconnect_client_and_dispose();
};

/** StructuredProxyPullSupplier
  * This proxy object is connected to a pull consumer and provides
  * CosN_StructuredEvent events 
  */

class StructuredProxyPullSupplier_i : 
  	public RDI_DummyProxy_i, 
  	public RDINotifySubscribe, 
  	WRAPPED_SKELETON_SUPER(CosNA::, StructuredProxyPullSupplier) 
{
  friend class EventChannel_i;
  friend class ConsumerAdmin_i;
 public:
  StructuredProxyPullSupplier_i(ConsumerAdmin_i* admin,
				EventChannel_i*  chann, 
				const CosNA_ProxyID& pxid);

  // Methods from CosNA_ProxySupplier Interface
  CosNA_ProxyType         MyType( WRAPPED_DECLARG_VOID )   { return _prxtype; }
  CosNA_ProxyID           MyID( WRAPPED_DECLARG_VOID )     { return _pserial; }
  CosNA_ConsumerAdmin_ptr MyAdmin( WRAPPED_DECLARG_VOID );
  CosNF_MappingFilter_ptr priority_filter( WRAPPED_DECLARG_VOID );
  void                    priority_filter(CosNF_MappingFilter_ptr map_filter
			                  WRAPPED_DECLARG );
  CosNF_MappingFilter_ptr lifetime_filter( WRAPPED_DECLARG_VOID );
  void                    lifetime_filter(CosNF_MappingFilter_ptr life_filter
				          WRAPPED_DECLARG );
  CosN_EventTypeSeq*      obtain_offered_types(CosNA_ObtainInfoMode mode
				               WRAPPED_DECLARG );
  void validate_event_qos(const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
	WRAPPED_DECLARG );

  // Methods from CosNA_StructuredProxyPullSupplier Interface
  void connect_structured_pull_consumer( 
		CosNC_StructuredPullConsumer_ptr consumer WRAPPED_DECLARG );

  // Methods from CosNC_StructuredPullSupplier Interface
  CosN_StructuredEvent* pull_structured_event( WRAPPED_DECLARG_VOID );
  CosN_StructuredEvent* try_pull_structured_event(CORBA::Boolean& has_event
					          WRAPPED_DECLARG );
  void disconnect_structured_pull_supplier( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void               remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void               remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }
  CosNF_FilterID     add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG );

  // augmented version of has_filters
  CORBA::Boolean has_filters() const
	{ return (_rqstypes.length() ? 1 : _fa_helper.has_filters()); }

  // Methods from CosNC_NotifySubscribe Interface
  void subscription_change(const CosN_EventTypeSeq& added,
			   const CosN_EventTypeSeq& deled WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

  void subscription_change_i(const CosN_EventTypeSeq& added,
			     const CosN_EventTypeSeq& deled, Filter_i* flter);
  void filter_destroy_i(Filter_i* filter);

  // Add a new event in the notification queue
  void add_event(RDI_StructuredEvent* event);

  friend ostream& operator << (ostream& out, 
			       const StructuredProxyPullSupplier_i& prx);
private:
  omni_mutex         _oplock;	// Serialize operations on object
  FAdminHelper       _fa_helper;
  omni_condition     _noempty;	// Block until queue has an entry
  EventChannel_i*    _channel;
  ConsumerAdmin_i*   _myadmin;
  CosNA_ProxyType    _prxtype;	// This must be PULL_STRUCTURED
  CosNA_ProxyID      _pserial;
  CORBA::ULong       _nevents;
  CORBA::ULong       _blkpull;	// Number of blocked pull() threads
  RDI_ProxyState     _pxstate;
  CosNF_MappingFilter_ptr _pfilter;
  CosNF_MappingFilter_ptr _lfilter;
  RDI_NotifQoS*      _qosprop;
  CORBA::Boolean     _offauto;	// TRUE if offer_change() is disabled
  CosN_EventTypeSeq  _rqstypes;
  CosNC_StructuredPullConsumer_var _consumer;
  RDI_List<RDI_StructuredEvent *>  _ntfqueue;

  // This is invoked by the administrative object during destruction
  void disconnect_client_and_dispose();
};

/** SequenceProxyPullSupplier
  * This proxy object is connected to a pull consumer and provides
  * CosN_EventBatch events
  */

class SequenceProxyPullSupplier_i : 
  	public RDI_DummyProxy_i, 
  	public RDINotifySubscribe, 
  	WRAPPED_SKELETON_SUPER(CosNA::, SequenceProxyPullSupplier) 
{
  friend class EventChannel_i;
  friend class ConsumerAdmin_i;
 public:
  SequenceProxyPullSupplier_i(ConsumerAdmin_i* admin, 
			      EventChannel_i*  chann, 
			      const CosNA_ProxyID& prxID);

  // Methods from CosNA_ProxySupplier Interface
  CosNA_ProxyType         MyType( WRAPPED_DECLARG_VOID )   { return _prxtype; }
  CosNA_ProxyID           MyID( WRAPPED_DECLARG_VOID )     { return _pserial; }
  CosNA_ConsumerAdmin_ptr MyAdmin( WRAPPED_DECLARG_VOID );
  CosNF_MappingFilter_ptr priority_filter( WRAPPED_DECLARG_VOID );
  void                    priority_filter(CosNF_MappingFilter_ptr map_filter
				          WRAPPED_DECLARG );
  CosNF_MappingFilter_ptr lifetime_filter( WRAPPED_DECLARG_VOID );
  void                    lifetime_filter(CosNF_MappingFilter_ptr life_filter
				          WRAPPED_DECLARG );
  CosN_EventTypeSeq*      obtain_offered_types(CosNA_ObtainInfoMode mode
				               WRAPPED_DECLARG );
  void validate_event_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // Methods from CosNA_SequenceProxyPullSupplier Interface
  void connect_sequence_pull_consumer(
		CosNC_SequencePullConsumer_ptr consumer WRAPPED_DECLARG );

  // Methods from CosNC_SequencePullSupplier Interface
  CosN_EventBatch* pull_structured_events(CORBA::Long max_number 
					  WRAPPED_DECLARG );
  CosN_EventBatch* try_pull_structured_events(CORBA::Long max_number,
                                              CORBA::Boolean& has_event
				              WRAPPED_DECLARG );
  void disconnect_sequence_pull_supplier( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void               remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void               remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }
  CosNF_FilterID     add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG );

  // augmented version of has_filters
  CORBA::Boolean has_filters() const
		{ return (_rqstypes.length() ? 1 : _fa_helper.has_filters()); }

  // Methods from CosNC_NotifySubscribe Interface
  void subscription_change(const CosN_EventTypeSeq& added,
                           const CosN_EventTypeSeq& deled WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

  void subscription_change_i(const CosN_EventTypeSeq& added,
			     const CosN_EventTypeSeq& deled, Filter_i* flter);
  void filter_destroy_i(Filter_i* filter);

  // Add a new event in the notification queue
  void add_event(RDI_StructuredEvent* event);

  friend ostream& operator << (ostream& out, 
			       const SequenceProxyPullSupplier_i& prx);
private:
  omni_mutex         _oplock;	// Serialize operations on object
  FAdminHelper       _fa_helper;
  omni_condition     _noempty;	// Block until queue has an entry
  EventChannel_i*    _channel;
  ConsumerAdmin_i*   _myadmin;
  CosNA_ProxyType    _prxtype;	// This must be PULL_SEQUENCE
  CosNA_ProxyID      _pserial;
  CORBA::ULong       _nevents;
  CORBA::ULong       _blkpull;	// Number of blocked pull() threads
  RDI_ProxyState     _pxstate;
  CosNF_MappingFilter_ptr _pfilter;
  CosNF_MappingFilter_ptr _lfilter;
  RDI_NotifQoS*      _qosprop;
  CORBA::Boolean     _offauto;	// TRUE if offer_change() is disabled
  CosN_EventTypeSeq  _rqstypes;
  CosNC_SequencePullConsumer_var _consumer;
  RDI_List<RDI_StructuredEvent *> _ntfqueue;

  // This is invoked by the administrative object during destruction
  void disconnect_client_and_dispose();
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//               Pull-based Consumer Proxy implementations               //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

/** ProxyPullConsumer
  * This proxy object is connected to a pull supplier and retrieves
  * CORBA::Any events
  */

class ProxyPullConsumer_i : 
  	public virtual RDIProxyPullConsumer, 
  	WRAPPED_SKELETON_SUPER(CosNA::, ProxyPullConsumer) 
{
  friend class EventChannel_i;
  friend class SupplierAdmin_i;
 public:
  ProxyPullConsumer_i(SupplierAdmin_i* admin, 
		      EventChannel_i*  chann, const CosNA_ProxyID& prxID);

  // Methods from CosNA_ProxyConsumer Interface
  CosNA_ProxyType         MyType( WRAPPED_DECLARG_VOID )   { return _prxtype; }
  CosNA_ProxyID           MyID( WRAPPED_DECLARG_VOID )     { return _pserial; }
  CosNA_SupplierAdmin_ptr MyAdmin( WRAPPED_DECLARG_VOID );
  CosN_EventTypeSeq*      obtain_subscription_types(CosNA_ObtainInfoMode mode
				                    WRAPPED_DECLARG );
  void validate_event_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
  	WRAPPED_DECLARG );

  // Methods from CosNA_ProxyPullConsumer Interface
  void connect_any_pull_supplier(
		CosEventComm::PullSupplier_ptr supplier WRAPPED_DECLARG );
  void suspend_connection( WRAPPED_DECLARG_VOID );
  void resume_connection( WRAPPED_DECLARG_VOID ); 

  // Methods from CosEventComm::PullConsumer Interface
  void disconnect_pull_consumer( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
    	WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void               remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void               remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }
  CosNF_FilterID     add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG )
    				{ return _fa_helper.add_filter_i(filter); }

  // also map has_filters extra method up to this class
  CORBA::Boolean has_filters() const { return _fa_helper.has_filters(); }

  // Methods from CosNC_NotifyPublish Interface
  void           offer_change(const CosN_EventTypeSeq& added, 
			      const CosN_EventTypeSeq& deled);

  // (Local only -- not available via RPC)

  void subscription_change_i(const CosN_EventTypeSeq& added,
			     const CosN_EventTypeSeq& deled, Filter_i* fltr);

  const RDI_TimeValue& last_pull_tms() const	{ return _pulltms; }
  CORBA::Boolean       is_active() const	{ return _active; }

  void pull_event(CORBA::Boolean& invalid);

  friend ostream& operator << (ostream& out, const ProxyPullConsumer_i& prx);
private:
  omni_mutex       _oplock;     // Serialize operations on object
  FAdminHelper     _fa_helper;
  EventChannel_i*  _channel;
  SupplierAdmin_i* _myadmin;
  CosNA_ProxyType  _prxtype;    // This must be PULL_ANY
  CosNA_ProxyID    _pserial;
  CORBA::Boolean   _cosevnt;	// TRUE if CosEventComm supplier
  CORBA::ULong     _nevents;
  RDI_TimeValue    _pulltms;	// Timestamp of last pull
  RDI_ProxyState   _pxstate;
  CORBA::Boolean   _active;	// TRUE if connection  is active
  omni_condition*  _qempty;
  omni_thread*     _worker;
  CORBA::Boolean   _thrdone;    // true when worker thread exits
  RDI_NotifQoS*    _qosprop;
  CORBA::Boolean   _offauto;	// TRUE if subscription_change() is disabled
  RDI_Hash<CosN_EventType, CORBA::ULong> _evtypes;
  CosEventComm::PullSupplier_var _supplier;

  void _pull_event();

  // This is invoked by the administrative object during destruction
  void disconnect_client_and_dispose();
};

/** StructuredProxyPullConsumer
  * This proxy object is connected to a push supplier and retrieves
  * CosN_StructuredEvent events
  */

class StructuredProxyPullConsumer_i : 
  	public virtual RDIProxyPullConsumer, 
  	WRAPPED_SKELETON_SUPER(CosNA::, StructuredProxyPullConsumer) 
{
  friend class EventChannel_i;
  friend class SupplierAdmin_i;
 public:
  StructuredProxyPullConsumer_i(SupplierAdmin_i* admin,
				EventChannel_i*  chann, 
				const CosNA_ProxyID& prxID);

  // Methods from CosNA_ProxyConsumer Interface
  CosNA_ProxyType         MyType( WRAPPED_DECLARG_VOID )   { return _prxtype; }
  CosNA_ProxyID           MyID( WRAPPED_DECLARG_VOID )     { return _pserial; }
  CosNA_SupplierAdmin_ptr MyAdmin( WRAPPED_DECLARG_VOID );
  CosN_EventTypeSeq*      obtain_subscription_types(CosNA_ObtainInfoMode mode
					            WRAPPED_DECLARG );
  void validate_event_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // Methods from CosNA_StructuredProxyPullConsumer Interface
  void connect_structured_pull_supplier(
		CosNC_StructuredPullSupplier_ptr supplier WRAPPED_DECLARG );
  void suspend_connection( WRAPPED_DECLARG_VOID );
  void resume_connection( WRAPPED_DECLARG_VOID );

  // Methods from CosNC_StructuredPullConsumer Interface
  void disconnect_structured_pull_consumer( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void               remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void               remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }
  CosNF_FilterID     add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG )
    				{ return _fa_helper.add_filter_i(filter); }

  // also map has_filters extra method up to this class
  CORBA::Boolean has_filters() const { return _fa_helper.has_filters(); }

  // Methods from CosNC_NotifyPublish Interface
  void           offer_change(const CosN_EventTypeSeq& added, 
			      const CosN_EventTypeSeq& deled WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

  const RDI_TimeValue& last_pull_tms() const	{ return _pulltms; }
  CORBA::Boolean       is_active() const	{ return _active;  }

  void pull_event(CORBA::Boolean& invalid);

  friend ostream& operator << (ostream& out,
			       const StructuredProxyPullConsumer_i& prx);
private:
  omni_mutex       _oplock;     // Serialize operations on object
  FAdminHelper     _fa_helper;
  EventChannel_i*  _channel;
  SupplierAdmin_i* _myadmin;
  CosNA_ProxyType  _prxtype;    // This must be PULL_STRUCTURED
  CosNA_ProxyID    _pserial;
  CORBA::ULong     _nevents;
  RDI_TimeValue    _pulltms;    // Timestamp of last pull
  RDI_ProxyState   _pxstate;
  CORBA::Boolean   _active;     // TRUE if connection  is active
  omni_condition*  _qempty;
  omni_thread*     _worker;
  CORBA::Boolean   _thrdone;    // true when worker thread exits
  RDI_NotifQoS*    _qosprop;
  CORBA::Boolean   _offauto;	// TRUE if subscription_change() is disabled
  RDI_Hash<CosN_EventType, CORBA::ULong> _evtypes;
  CosNC_StructuredPullSupplier_var _supplier;

  void _pull_event();

  // This is invoked by the administrative object during destruction
  void disconnect_client_and_dispose();
};

/** SequenceProxyPullConsumer
  * This proxy object is connected to a push supplier and retrieves
  * CosN_EventBatch events
  */

class SequenceProxyPullConsumer_i : 
  	public virtual RDIProxyPullConsumer, 
  	WRAPPED_SKELETON_SUPER(CosNA::, SequenceProxyPullConsumer) 
{
  friend class EventChannel_i;
  friend class SupplierAdmin_i;
 public:
  SequenceProxyPullConsumer_i(SupplierAdmin_i* admin, 
			      EventChannel_i*  chann, 
			      const CosNA_ProxyID& prxID);

  // Methods from CosNA_ProxyConsumer Interface
  CosNA_ProxyType         MyType( WRAPPED_DECLARG_VOID )   { return _prxtype; }
  CosNA_ProxyID           MyID( WRAPPED_DECLARG_VOID )     { return _pserial; }
  CosNA_SupplierAdmin_ptr MyAdmin( WRAPPED_DECLARG_VOID );
  CosN_EventTypeSeq*      obtain_subscription_types(CosNA_ObtainInfoMode mode
					            WRAPPED_DECLARG );
  void validate_event_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
  	WRAPPED_DECLARG );

  // Methods from CosNA_SequenceProxyPullConsumer Interface
  void connect_sequence_pull_supplier(
		CosNC_SequencePullSupplier_ptr supplier WRAPPED_DECLARG );
  void suspend_connection( WRAPPED_DECLARG_VOID );
  void resume_connection( WRAPPED_DECLARG_VOID );

  // Methods from CosNC_SequencePullConsumer Interface
  void disconnect_sequence_pull_consumer( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void               remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void                remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }
  CosNF_FilterID      add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG )
    				{ return _fa_helper.add_filter_i(filter); }

  // also map has_filters extra method up to this class
  CORBA::Boolean has_filters() const { return _fa_helper.has_filters(); }

  // Methods from CosNC_NotifyPublish Interface
  void           offer_change(const CosN_EventTypeSeq& added, 
			      const CosN_EventTypeSeq& deled WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

  const RDI_TimeValue& last_pull_tms() const	{ return _pulltms; }
  CORBA::Boolean       is_active() const	{ return _active;  }

  void pull_event(CORBA::Boolean& invalid);

  friend ostream& operator << (ostream& out, 
			       const SequenceProxyPullConsumer_i& prx);
private:
  omni_mutex       _oplock;     // Serialize operations on object
  FAdminHelper     _fa_helper;
  EventChannel_i*  _channel;
  SupplierAdmin_i* _myadmin;
  CosNA_ProxyType  _prxtype;    // This must be PULL_SEQUENCE
  CosNA_ProxyID    _pserial;
  CORBA::ULong     _nevents;
  RDI_TimeValue    _pulltms;    // Timestamp of last pull
  RDI_ProxyState   _pxstate;
  CORBA::Boolean   _active;     // TRUE if connection  is active
  omni_condition*  _qempty;
  omni_thread*     _worker;
  CORBA::Boolean   _thrdone;    // true when worker thread exits
  RDI_NotifQoS*    _qosprop;
  CORBA::Boolean   _offauto;	// TRUE if subscription_change() is disabled
  RDI_Hash<CosN_EventType, CORBA::ULong> _evtypes;
  CosNC_SequencePullSupplier_var _supplier;

  void _pull_event();

  // This is invoked by the administrative object during destruction
  void disconnect_client_and_dispose();
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//               Push-based Supplier Proxy implementations               //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

/** ProxyPushSupplier
  * This proxy object is connected to a push consumer and provides
  * CORBA::Any events
  */

class ProxyPushSupplier_i : 
	public RDI_DummyProxy_i, 
	public RDINotifySubscribe, 
	public virtual RDIProxyPushSupplier, 
	WRAPPED_SKELETON_SUPER(CosNA::, ProxyPushSupplier) {
  friend class EventChannel_i;
  friend class ConsumerAdmin_i;
public:
  ProxyPushSupplier_i(ConsumerAdmin_i* admin, 
		      EventChannel_i*  chann, const CosNA_ProxyID& prxID);

  // Methods from CosNA_ProxySupplier Interface
  CosNA_ProxyType         MyType( WRAPPED_DECLARG_VOID )   { return _prxtype; }
  CosNA_ProxyID           MyID( WRAPPED_DECLARG_VOID )     { return _pserial; }
  CosNA_ConsumerAdmin_ptr MyAdmin( WRAPPED_DECLARG_VOID );
  CosNF_MappingFilter_ptr priority_filter( WRAPPED_DECLARG_VOID );
  void                    priority_filter(CosNF_MappingFilter_ptr map_filter
				          WRAPPED_DECLARG );
  CosNF_MappingFilter_ptr lifetime_filter( WRAPPED_DECLARG_VOID );
  void                    lifetime_filter(CosNF_MappingFilter_ptr life_filter
				          WRAPPED_DECLARG );
  CosN_EventTypeSeq*      obtain_offered_types(CosNA_ObtainInfoMode mode
				               WRAPPED_DECLARG );
  void validate_event_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
  	WRAPPED_DECLARG );

  // Methods from CosNA_ProxyPushSupplier Interface
  void connect_any_push_consumer(
		CosEventComm::PushConsumer_ptr consumer WRAPPED_DECLARG );
  void suspend_connection( WRAPPED_DECLARG_VOID );
  void resume_connection( WRAPPED_DECLARG_VOID );

  // Methods from CosEventComm::PushSupplier Interface
  void disconnect_push_supplier( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
    	WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void               remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void               remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }
  CosNF_FilterID     add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG );

  // augmented version of has_filters
  CORBA::Boolean has_filters() const
		{ return (_rqstypes.length() ? 1 : _fa_helper.has_filters()); }

  // Methods from CosNC_NotifySubscribe Interface
  void subscription_change(const CosN_EventTypeSeq& added,
			   const CosN_EventTypeSeq& deled WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

  void subscription_change_i(const CosN_EventTypeSeq& added,
			     const CosN_EventTypeSeq& deled, Filter_i* flter);
  void filter_destroy_i(Filter_i* filter);

  CORBA::Boolean is_active(void) const	{ return _active; }
  CORBA::Boolean has_events() const	{ return _ntfqueue.length() ? 1 : 0; }

  void add_event(RDI_StructuredEvent* event);
  void push_event(CORBA::Boolean& invalid);

  friend ostream& operator << (ostream& out, const ProxyPushSupplier_i& prx);
private:
  omni_mutex         _oplock;	// Serialize operations on object
  FAdminHelper     _fa_helper;
  EventChannel_i*    _channel;
  ConsumerAdmin_i*   _myadmin;
  CosNA_ProxyType    _prxtype;	// This must be PUSH_ANY
  CosNA_ProxyID      _pserial;
  CORBA::Boolean     _cosevnt;	// TRUE if CosEventComm supplier
  CORBA::ULong       _nevents;
  RDI_ProxyState     _pxstate;
  CosNF_MappingFilter_ptr _pfilter;
  CosNF_MappingFilter_ptr _lfilter;
  CORBA::Boolean     _active;	// TRUE if connection is active
  omni_condition*    _qempty;
  omni_thread*       _worker;
  RDI_NotifQoS*      _qosprop;
  CORBA::Boolean     _offauto;	// TRUE if offer_change() is disabled
  CosN_EventTypeSeq  _rqstypes;
  CosEventComm::PushConsumer_var  _consumer;
  RDI_List<RDI_StructuredEvent *> _ntfqueue;

  void _push_event();

  // This is invoked by the administrative object during destruction
  void disconnect_client_and_dispose();
};

/** StructuredProxyPushSupplier
  * This proxy object is connected to a push consumer and provides
  * CosN_StructuredEvent events
  */

class StructuredProxyPushSupplier_i : 
	public RDI_DummyProxy_i, 
	public RDINotifySubscribe, 
	public virtual RDIProxyPushSupplier, 
	WRAPPED_SKELETON_SUPER(CosNA::, StructuredProxyPushSupplier) 
{
  friend class EventChannel_i;
  friend class ConsumerAdmin_i;
 public:
  StructuredProxyPushSupplier_i(ConsumerAdmin_i* admin, 
				EventChannel_i*  chann, 
				const CosNA_ProxyID& prxID);

  // Methods from CosNA_ProxySupplier Interface
  CosNA_ProxyType         MyType( WRAPPED_DECLARG_VOID )   { return _prxtype; }
  CosNA_ProxyID           MyID( WRAPPED_DECLARG_VOID )     { return _pserial; }
  CosNA_ConsumerAdmin_ptr MyAdmin( WRAPPED_DECLARG_VOID );
  CosNF_MappingFilter_ptr priority_filter( WRAPPED_DECLARG_VOID );
  void                    priority_filter(CosNF_MappingFilter_ptr map_filter
				          WRAPPED_DECLARG );
  CosNF_MappingFilter_ptr lifetime_filter( WRAPPED_DECLARG_VOID );
  void                    lifetime_filter(CosNF_MappingFilter_ptr life_filter
				          WRAPPED_DECLARG );
  CosN_EventTypeSeq*      obtain_offered_types(CosNA_ObtainInfoMode mode
				               WRAPPED_DECLARG );
  void validate_event_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
	WRAPPED_DECLARG );

  // Methods from CosNA_StructuredProxyPushSupplier Interface
  void connect_structured_push_consumer(
		CosNC_StructuredPushConsumer_ptr consumer WRAPPED_DECLARG );
  void suspend_connection( WRAPPED_DECLARG_VOID );
  void resume_connection( WRAPPED_DECLARG_VOID );

  // Methods from CosNC_StructuredPushSupplier Interface
  void disconnect_structured_push_supplier( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void               remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void               remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }
  CosNF_FilterID     add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG );

  // augmented version of has_filters
  CORBA::Boolean has_filters() const
		{ return (_rqstypes.length() ? 1 : _fa_helper.has_filters()); }

  // Methods from CosNC_NotifySubscribe Interface
  void subscription_change(const CosN_EventTypeSeq& added,
			   const CosN_EventTypeSeq& deled WRAPPED_DECLARG ); 

  // (Local only -- not available via RPC)

  void subscription_change_i(const CosN_EventTypeSeq& added,
			     const CosN_EventTypeSeq& deled, Filter_i* flter);
  void filter_destroy_i(Filter_i* filter);

  CORBA::Boolean is_active(void) const	{ return _active; }
  CORBA::Boolean has_events() const	{ return _ntfqueue.length() ? 1 : 0; }

  void add_event(RDI_StructuredEvent* event);
  void push_event(CORBA::Boolean& invalid);

  friend ostream& operator << (ostream& out, 
			       const StructuredProxyPushSupplier_i& prx);
private:
  omni_mutex         _oplock;   // Serialize operations on object
  FAdminHelper     _fa_helper;
  EventChannel_i*    _channel;
  ConsumerAdmin_i*   _myadmin;
  CosNA_ProxyType    _prxtype;  // This must be PUSH_STRUCTURED
  CosNA_ProxyID      _pserial;
  CORBA::ULong       _nevents;
  RDI_ProxyState     _pxstate;
  CosNF_MappingFilter_ptr _pfilter;
  CosNF_MappingFilter_ptr _lfilter;
  CORBA::Boolean     _active;   // TRUE if connection is active
  omni_condition*    _qempty;
  omni_thread*       _worker;
  RDI_NotifQoS*      _qosprop;
  CORBA::Boolean     _offauto;	// TRUE if offer_change() is disabled
  CosN_EventTypeSeq  _rqstypes;
  CosNC_StructuredPushConsumer_var _consumer;
  RDI_List<RDI_StructuredEvent *>  _ntfqueue;

  void _push_event();

  // This is invoked by the administrative object during destruction
  void disconnect_client_and_dispose();
};

/** SequenceProxyPushSupplier
  * This proxy object is connected to a push consumer and provides
  * CosN_EventBatch events
  */

class SequenceProxyPushSupplier_i : 
	public RDI_DummyProxy_i,  
	public RDINotifySubscribe, 
	public virtual RDIProxyPushSupplier, 
	WRAPPED_SKELETON_SUPER(CosNA::, SequenceProxyPushSupplier) 
{
  friend class EventChannel_i;
  friend class ConsumerAdmin_i;
public:
  SequenceProxyPushSupplier_i(ConsumerAdmin_i* admin, 
			      EventChannel_i*  chann, 
			      const CosNA_ProxyID& prxID);

  // Methods from CosNA_ProxySupplier Interface
  CosNA_ProxyType         MyType( WRAPPED_DECLARG_VOID )   { return _prxtype; }
  CosNA_ProxyID           MyID( WRAPPED_DECLARG_VOID )     { return _pserial; }
  CosNA_ConsumerAdmin_ptr MyAdmin( WRAPPED_DECLARG_VOID );
  CosNF_MappingFilter_ptr priority_filter( WRAPPED_DECLARG_VOID );
  void                    priority_filter(CosNF_MappingFilter_ptr map_filter
				          WRAPPED_DECLARG );
  CosNF_MappingFilter_ptr lifetime_filter( WRAPPED_DECLARG_VOID );
  void                    lifetime_filter(CosNF_MappingFilter_ptr life_filter
				          WRAPPED_DECLARG );
  CosN_EventTypeSeq* obtain_offered_types(CosNA_ObtainInfoMode mode
			                  WRAPPED_DECLARG );
  void validate_event_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // Methods from CosNA_SequenceProxyPushSupplier Interface
  void connect_sequence_push_consumer(
		CosNC_SequencePushConsumer_ptr consumer WRAPPED_DECLARG );
  void suspend_connection( WRAPPED_DECLARG_VOID );
  void resume_connection( WRAPPED_DECLARG_VOID );

  // Methods from CosNC_StructuredPushSupplier Interface
  void disconnect_sequence_push_supplier( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
	WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void               remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void               remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }
  CosNF_FilterID     add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG );

  // augmented version of has_filters
  CORBA::Boolean has_filters() const
		{ return (_rqstypes.length() ? 1 : _fa_helper.has_filters()); }

  // Methods from CosNC_NotifySubscribe Interface
  void subscription_change(const CosN_EventTypeSeq& added,
			   const CosN_EventTypeSeq& deled WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

  void subscription_change_i(const CosN_EventTypeSeq& added,
			     const CosN_EventTypeSeq& deled, Filter_i* flter);
  void filter_destroy_i(Filter_i* filter);

  CORBA::Boolean is_active(void) const { return _active; }
  CORBA::Boolean has_events() const	 { return _ntfqueue.length() ? 1 : 0; }

  void add_event(RDI_StructuredEvent* event);
  void push_event(CORBA::Boolean& invalid);

  friend ostream& operator << (ostream& out, 
			       const SequenceProxyPushSupplier_i& prx);
private:
  omni_mutex         _oplock;   // Serialize operations on object
  FAdminHelper     _fa_helper;
  EventChannel_i*    _channel;
  ConsumerAdmin_i*   _myadmin;
  CosNA_ProxyType    _prxtype;  // This must be PUSH_SEQUENCE
  CosNA_ProxyID      _pserial;
  CORBA::ULong       _nevents;
  RDI_ProxyState     _pxstate;
  CosNF_MappingFilter_ptr _pfilter;
  CosNF_MappingFilter_ptr _lfilter;
  CORBA::Boolean     _active;   // TRUE if connection is active
  omni_condition*    _qempty;
  omni_thread*       _worker;
  RDI_NotifQoS*      _qosprop;
  CORBA::Boolean     _offauto;	// TRUE if offer_change() is disabled
  CosN_EventTypeSeq  _rqstypes;
  CosNC_SequencePushConsumer_var _consumer;
  RDI_List<RDI_StructuredEvent *> _ntfqueue;

  void _push_event();

  // This is invoked by the administrative object during destruction
  void disconnect_client_and_dispose();
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//            ConsumerAdmin and SupplierAdmin implementations            //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

/** ConsumerAdmin_i objects are responsible for creating and managing
  * proxy supplier objects  within instances of  Notification Service
  * Event Channels.  ConsumerAdmin_i objects are also responsible for
  * managing a common set of QoS requirements and filter objects that
  * apply to all of their proxy objects.
  */

class ConsumerAdmin_i :	 
	public RDINotifySubscribe, 
	WRAPPED_SKELETON_SUPER(CosNA::, ConsumerAdmin) 
{
  friend class EventChannel_i;
public:
  ~ConsumerAdmin_i()	{ /* destroy() takes care of cleaning up */}

  // Methods from CosEventChannelAdmin::ConsumerAdmin Interface
  CosEventChannelAdmin::ProxyPushSupplier_ptr obtain_push_supplier();
  CosEventChannelAdmin::ProxyPullSupplier_ptr obtain_pull_supplier();

  // Methods from CosNC_ConsumerAdmin Interface
  CosNA_AdminID                  MyID( WRAPPED_DECLARG_VOID ) 
	{ return _serial ; }
  CosNA_EventChannel_ptr         MyChannel( WRAPPED_DECLARG_VOID );	
  CosNA_InterFilterGroupOperator MyOperator( WRAPPED_DECLARG_VOID ) 
	{ return _and_or_oper; }
  CosNF_MappingFilter_ptr priority_filter( WRAPPED_DECLARG_VOID );
  void  priority_filter(CosNF_MappingFilter_ptr fltr WRAPPED_DECLARG );
  CosNF_MappingFilter_ptr lifetime_filter();
  void  lifetime_filter(CosNF_MappingFilter_ptr fltr WRAPPED_DECLARG );
  CosNA_ProxyIDSeq*       pull_suppliers( WRAPPED_DECLARG_VOID );
  CosNA_ProxyIDSeq*       push_suppliers( WRAPPED_DECLARG_VOID );
  CosNA_ProxySupplier_ptr get_proxy_supplier(CosNA_ProxyID proxy_id 
					     WRAPPED_DECLARG );
  CosNA_ProxySupplier_ptr obtain_notification_pull_supplier(
				CosNA_ClientType ctype, 
			        CosNA_ProxyID&   proxy_id WRAPPED_DECLARG );
  CosNA_ProxySupplier_ptr obtain_notification_push_supplier(
				CosNA_ClientType ctype, 
			        CosNA_ProxyID&   proxy_id WRAPPED_DECLARG );
  void destroy( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void               remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr   get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void               remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }

  CosNF_FilterID add_filter(CosNF_Filter_ptr fltr WRAPPED_DECLARG )
    { return _fa_helper.add_filter_i(fltr, (RDINotifySubscribe_ptr) this); } 

  // also map has_filters extra method up to this class
  CORBA::Boolean has_filters() const { return _fa_helper.has_filters(); }

  // Methods from CosNC_NotifySubscribe Interface
  void subscription_change(const CosN_EventTypeSeq& added,
                           const CosN_EventTypeSeq& deled WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

  void subscription_change_i(const CosN_EventTypeSeq& added,
			     const CosN_EventTypeSeq& deled, Filter_i* flter);
  void filter_destroy_i(Filter_i* filter);

  // Proxy de-registration members
  void remove_proxy(ProxyPushSupplier_i* prx);
  void remove_proxy(ProxyPullSupplier_i* prx);
  void remove_proxy(EventProxyPushSupplier_i* prx);
  void remove_proxy(EventProxyPullSupplier_i* prx);
  void remove_proxy(StructuredProxyPushSupplier_i* prx);
  void remove_proxy(StructuredProxyPullSupplier_i* prx);
  void remove_proxy(SequenceProxyPushSupplier_i* prx);
  void remove_proxy(SequenceProxyPullSupplier_i* prx);

  CORBA::ULong         NumProxies() const	{ return _num_proxies; }
  const RDI_TimeValue& TimeBorn() const		{ return _time_born;   }
  const RDI_TimeValue& LastProxy() const	{ return _prxy_exit;   }
  RDI_NotifQoS*        qos_properties();

  friend ostream& operator << (ostream& out, ConsumerAdmin_i& adm);
private:
  omni_mutex                _oplock;		// Serialize operations
  FAdminHelper     	    _fa_helper;
  EventChannel_i*           _channel;
  RDI_NotifQoS*             _qosprop;
  CosNA_AdminID                  _serial;
  CosNA_InterFilterGroupOperator _and_or_oper;
  CosNF_MappingFilter_ptr        _prio_filter;	// Priority mapping filter
  CosNF_MappingFilter_ptr        _life_filter;	// Lifetime mapping filter
  CosNA_ProxyID                  _prx_serial;	// Factory for proxy IDs
  CORBA::ULong              _num_proxies;	// Number of active proxies
  RDI_TimeValue	            _time_born;		// Timestamp of creation
  RDI_TimeValue	            _prxy_exit;		// Destruction of a proxy

  // Since the Event Service proxy objects do not have a unique ID
  // associated with them, we keep these proxies in a regular list

  RDI_List<EventProxyPushSupplier_i *> _cosevent_push;
  RDI_List<EventProxyPullSupplier_i *> _cosevent_pull;

  // All Notification Service proxy objects have unique IDs. Thus, 
  // we maintain them using hash tables for fast lookups 

  RDI_Hash<CosNA_ProxyID, ProxyPushSupplier_i *>           _prx_any_push;
  RDI_Hash<CosNA_ProxyID, ProxyPullSupplier_i *>           _prx_any_pull;
  RDI_Hash<CosNA_ProxyID, StructuredProxyPushSupplier_i *> _prx_struc_push;
  RDI_Hash<CosNA_ProxyID, StructuredProxyPullSupplier_i *> _prx_struc_pull;
  RDI_Hash<CosNA_ProxyID, SequenceProxyPushSupplier_i *>   _prx_batch_push;
  RDI_Hash<CosNA_ProxyID, SequenceProxyPullSupplier_i *>   _prx_batch_pull;

  ConsumerAdmin_i(EventChannel_i* channel, CosNA_InterFilterGroupOperator op, 
		  const CosNA_AdminID& serial);

  // Dispatch an event to all consumers using the CORBA Event Service
  // interface -- no filtering is performed here

  void dispatch_event(RDI_StructuredEvent* evnt);

  // Dispatch an event to all consumers using the CORBA Notification
  // Service interface -- filtering is performed here

  void dispatch_event(RDI_StructuredEvent* evnt, 
		      RDI_FilterState_t fltstat, RDI_TypeMap* typemap);

  // Propagate changes in the event types announced by suppliers to
  // all connected consumers

  void offer_change(const CosN_EventTypeSeq& added, 
		    const CosN_EventTypeSeq& deled);

  void disconnect_clients_and_dispose();
};

/** SupplierAdmin_i objects are responsible for creating and managing
  * proxy consumer objects  within instances of  Notification Service
  * Event Channels.  ConsumerAdmin_i objects are also responsible for
  * managing a common set of QoS requirements that apply to all their
  * proxy objects.
  */

class SupplierAdmin_i : 
	WRAPPED_SKELETON_SUPER(CosNA::, SupplierAdmin) 
{
  friend class EventChannel_i;
public:
  ~SupplierAdmin_i()	{ /* destroy() takes care of cleaning up */}

  // Methods from CosEventChannelAdmin::SupplierAdmin Interface
  CosEventChannelAdmin::ProxyPushConsumer_ptr obtain_push_consumer();
  CosEventChannelAdmin::ProxyPullConsumer_ptr obtain_pull_consumer();

  // Methods from CosNA_SupplierAdmin Interface
  CosNA_AdminID                  MyID( WRAPPED_DECLARG_VOID ) 
	{ return _serial; }
  CosNA_EventChannel_ptr         MyChannel( WRAPPED_DECLARG_VOID );
  CosNA_InterFilterGroupOperator MyOperator( WRAPPED_DECLARG_VOID )
    	{ return _and_or_oper; }
  CosNA_ProxyIDSeq*       pull_consumers( WRAPPED_DECLARG_VOID );
  CosNA_ProxyIDSeq*       push_consumers( WRAPPED_DECLARG_VOID );
  CosNA_ProxyConsumer_ptr get_proxy_consumer(CosNA_ProxyID proxy_id 
					     WRAPPED_DECLARG );
  CosNA_ProxyConsumer_ptr obtain_notification_pull_consumer(
				CosNA_ClientType ctype, 
				CosNA_ProxyID&   proxy_id WRAPPED_DECLARG );
  CosNA_ProxyConsumer_ptr obtain_notification_push_consumer(
				CosNA_ClientType ctype, 
			        CosNA_ProxyID&   proxy_id WRAPPED_DECLARG );
  void destroy( WRAPPED_DECLARG_VOID );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // Methods from CosNF_FilterAdmin Interface
  void              remove_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ _fa_helper.remove_filter(fltrID); }
  CosNF_Filter_ptr  get_filter(CosNF_FilterID fltrID WRAPPED_DECLARG ) 
				{ return _fa_helper.get_filter(fltrID); }
  CosNF_FilterIDSeq* get_all_filters( WRAPPED_DECLARG_VOID ) 
				{ return _fa_helper.get_all_filters(); }
  void               remove_all_filters( WRAPPED_DECLARG_VOID ) 
				{ _fa_helper.remove_all_filters(); }
  CosNF_FilterID     add_filter(CosNF_Filter_ptr filter WRAPPED_DECLARG )
    				{ return _fa_helper.add_filter_i(filter); } 

  // also map has_filters extra method up to this class
  CORBA::Boolean has_filters() const { return _fa_helper.has_filters(); }

  // Methods from CosNC_NotifyPublish Interface
  void offer_change(const CosN_EventTypeSeq& added, 
		    const CosN_EventTypeSeq& removed WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

  // Proxy de-registration members
  void remove_proxy(ProxyPushConsumer_i* prx);
  void remove_proxy(ProxyPullConsumer_i* prx);
  void remove_proxy(EventProxyPushConsumer_i* prx);
  void remove_proxy(EventProxyPullConsumer_i* prx);
  void remove_proxy(StructuredProxyPushConsumer_i* prx);
  void remove_proxy(StructuredProxyPullConsumer_i* prx);
  void remove_proxy(SequenceProxyPushConsumer_i* prx);
  void remove_proxy(SequenceProxyPullConsumer_i* prx);

  CORBA::ULong         NumProxies() const	{ return _num_proxies; }
  const RDI_TimeValue& TimeBorn() const		{ return _time_born;   }
  const RDI_TimeValue& LastProxy() const	{ return _prxy_exit;   }
  RDI_NotifQoS*        qos_properties();

  // While this module does not support CosNC_NotifySubscribe,
  // we added the following operation to propagate changes in the event
  // types referenced in consumer filters to the actual suppliers .....

  void subscription_change(const CosN_EventTypeSeq& added,
			   const CosN_EventTypeSeq& deled);

  // The following methods are invoked by connected proxies to check if
  // a given supplied event can be inserted into the channel

  CORBA::Boolean match_event(const CORBA::Any& event);
  CORBA::Boolean match_event(RDI_StructuredEvent* event);

  friend ostream& operator << (ostream& out, SupplierAdmin_i& adm);

private:
  omni_mutex                _oplock;            // Serialize operations
  FAdminHelper              _fa_helper;
  EventChannel_i*           _channel;
  RDI_NotifQoS*             _qosprop;
  CosNA_AdminID                  _serial;
  CosNA_InterFilterGroupOperator _and_or_oper;
  CosNA_ProxyID                  _prx_serial;        // Factory for proxy IDs
  CORBA::ULong              _num_proxies;
  RDI_TimeValue	            _time_born;		// Timestamp of creation
  RDI_TimeValue	            _prxy_exit;		// Destruction of a proxy

  // To support offer_change() we need a hash table of event types

  RDI_Hash<CosN_EventType, CORBA::ULong> _evtypes;

  // Since the Event Service proxy objects do not have a unique ID
  // associated with them, we keep these proxies in a regular list

  RDI_List<EventProxyPushConsumer_i *> _cosevent_push;
  RDI_List<EventProxyPullConsumer_i *> _cosevent_pull;

  // All Notification Service proxy objects have unique IDs. Thus,
  // we maintain them using hash tables for fast lookups

  RDI_Hash<CosNA_ProxyID, ProxyPushConsumer_i *>           _prx_any_push;
  RDI_Hash<CosNA_ProxyID, ProxyPullConsumer_i *>           _prx_any_pull;
  RDI_Hash<CosNA_ProxyID, StructuredProxyPushConsumer_i *> _prx_struc_push;
  RDI_Hash<CosNA_ProxyID, StructuredProxyPullConsumer_i *> _prx_struc_pull;
  RDI_Hash<CosNA_ProxyID, SequenceProxyPushConsumer_i *>   _prx_batch_push;
  RDI_Hash<CosNA_ProxyID, SequenceProxyPullConsumer_i *>   _prx_batch_pull;

  SupplierAdmin_i(EventChannel_i* channel, CosNA_InterFilterGroupOperator op, 
		  const CosNA_AdminID& serial);

  void disconnect_clients_and_dispose();
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//         EventChannel and EventChannelFactory implementations          //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

/** Each event channel has a default filter factory that is used 
  * for creating filter objects. 
  */

class EventChannel_i : 
	WRAPPED_SKELETON_SUPER(AttNotifyChannelAdmin::, EventChannel) 
{
public:

  EventChannel_i(EventChannelFactory_i*  cfactory,
		 const CosN_QoSProperties&   init_qos,
	         const CosN_AdminProperties& init_adm,
		 const CosNA_ChannelID&       myserial);
  ~EventChannel_i() 	{/* destroy() takes care of cleaning up */}

  // Methods from CosEventChannelAdmin::EventChannel Interface
  CosEventChannelAdmin::ConsumerAdmin_ptr for_consumers( WRAPPED_DECLARG_VOID );
  CosEventChannelAdmin::SupplierAdmin_ptr for_suppliers( WRAPPED_DECLARG_VOID );
  void destroy( WRAPPED_DECLARG_VOID );

  // Methods from CosNA_EventChannel Interface
  CosNA_EventChannelFactory_ptr MyFactory( WRAPPED_DECLARG_VOID );
  CosNA_ConsumerAdmin_ptr default_consumer_admin( WRAPPED_DECLARG_VOID );
  CosNA_SupplierAdmin_ptr default_supplier_admin( WRAPPED_DECLARG_VOID );
  CosNF_FilterFactory_ptr default_filter_factory( WRAPPED_DECLARG_VOID );
  CosNA_ConsumerAdmin_ptr new_for_consumers(
					CosNA_InterFilterGroupOperator op,
					CosNA_AdminID& id WRAPPED_DECLARG );
  CosNA_SupplierAdmin_ptr new_for_suppliers(
					CosNA_InterFilterGroupOperator op,
                                        CosNA_AdminID& id WRAPPED_DECLARG );
  CosNA_ConsumerAdmin_ptr get_consumeradmin(CosNA_AdminID id  WRAPPED_DECLARG );
  CosNA_SupplierAdmin_ptr get_supplieradmin(CosNA_AdminID id  WRAPPED_DECLARG );
  CosNA_AdminIDSeq *      get_all_consumeradmins( WRAPPED_DECLARG_VOID );
  CosNA_AdminIDSeq *      get_all_supplieradmins( WRAPPED_DECLARG_VOID );


  // Methods from CosN_AdminPropertiesAdmin Interface
  CosN_AdminProperties* get_admin( WRAPPED_DECLARG_VOID );
  void set_admin(const CosN_AdminProperties& admin WRAPPED_DECLARG );

  // Methods from CosN_QoSAdmin Interface
  CosN_QoSProperties* get_qos( WRAPPED_DECLARG_VOID );
  void set_qos(const CosN_QoSProperties& qos WRAPPED_DECLARG );
  void validate_qos(const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) a_qos
        WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

  RDI_NotifQoS*   qos_properties() 	{ return _qosprop; }
  CosNA_ChannelID MyID()		{ return _serial;  }

  // An event is sent to the channel by some connected supplier

  int new_any_event(const CORBA::Any& event);
  int new_structured_event(const CosN_StructuredEvent& event);
  int new_structured_event(RDI_StructuredEvent* event);
  int new_sequence_event(const CosN_EventBatch& event);

  // Increment/decrement the number of consumers and suppliers
 
  bool incr_consumers();
  void decr_consumers();
  bool incr_suppliers();
  void decr_suppliers();

  // Retrieve administrative properties of the channel. We do not
  // acquire a lock since we assume that they do not change often
  // and, in addition, a snapshop is sufficient

  CORBA::Long   max_consumers() const	{ return _admin_qos.maxConsumers;    }
  CORBA::Long   max_suppliers() const	{ return _admin_qos.maxSuppliers;    }
  CORBA::UShort push_threads() const 	{ return _admin_qos.numPushThreads;  }
  CORBA::UShort pull_threads() const 	{ return _admin_qos.numPullThreads;  }
  CORBA::UShort pull_period() const 	{ return _admin_qos.pullEventPeriod; }

  // Unregister Admin objects

  void unregister(SupplierAdmin_i* suplAdmin, CORBA::Boolean holdslock=0);
  void unregister(ConsumerAdmin_i* consAdmin, CORBA::Boolean holdslock=0);

  // Update the event type mapping for a ConsumerAdmin/ProxySupplier

  bool update_mapping(const CosN_EventTypeSeq& added,
			const CosN_EventTypeSeq& deled,
			ConsumerAdmin_i* admin, Filter_i* filter);

  // XXX USED TO BE: CosNA_ProxySupplier_ptr proxy
  bool update_mapping(const CosN_EventTypeSeq& added,
		      const CosN_EventTypeSeq& deled,
		      RDI_DummyProxy_i* proxy, Filter_i* filter);

  struct ProxyDispatch_t {
	RDI_StructuredEvent* _event;
	ConsumerAdmin_i*     _admin;
	RDI_FilterState_t    _state;

	ProxyDispatch_t(RDI_StructuredEvent* e=0, 
		        ConsumerAdmin_i* a=0, RDI_FilterState_t s=NoFilters) : 
			_event(e), _admin(a), _state(s) {;}
	ProxyDispatch_t(const ProxyDispatch_t& p) : 
			_event(p._event), _admin(p._admin), _state(p._state) {;}
	ProxyDispatch_t& operator = (const ProxyDispatch_t& p) {
			_event=p._event; _admin=p._admin; 
			_state=p._state; return *this;         }
  };

  // Update the hash table of offered event types.  When a new entry
  // is created or an existing entry is deleted, we notify consumers

  void offer_change(const CosN_EventTypeSeq& added,
		    const CosN_EventTypeSeq& deled);

  // Notify suppliers about changes in event types referenced in the
  // filters registered by consumers

  void subscription_change(const CosN_EventTypeSeq& added,
			   const CosN_EventTypeSeq& deled);

  // Retrieve the sequence of event types supplied by suppliers and
  // the sequence of event types referenced in all consumer filters 

  CosN_EventTypeSeq* obtain_offered_types();
  CosN_EventTypeSeq* obtain_subscription_types();

  AttNotifyChannelAdmin::PerformanceStats obtain_performance_stats();

  void display_stats()		{ dump_stats(1, 1); }
  void display_state()		{ cout << *this << endl; }

  // Return a reference to 'RDI_RVM' that is thread-safe; the current
  // implementation allocates an RDI_RVM per thread

  RDI_RVM* get_rvm() { int id = omni_thread::self()->id(); 
		       RDI_CHECK_THREAD_ID(id); return _thread_stats[id]._rvm; }

  // Some performance-related functionality
  inline void incr_num_rdi_match();
  inline void incr_num_rvm_eval();
  inline void incr_num_announcements();
  inline void incr_num_notifications(unsigned int qsize);
  void        dump_stats(bool getlock = true, bool onstdout = false);

  // From time to time we may need to debug the module
  void display_event_queue()	{ _events->display_stats(1); }

  friend ostream& operator << (ostream& out, EventChannel_i& evc);

  // The thread pools used for pulling event from Pull suppliers
  // and pushing events to push consumers -- they may be NULL

  RDI_PullSupplier*      _pull_supplier;
  RDI_NotifyConsumer*    _push_consumer;

private:
  EventChannelFactory_i* _my_channel_factory;
  ConsumerAdmin_i*       _def_consumer_admin;
  SupplierAdmin_i*       _def_supplier_admin;
  FilterFactory_i*       _def_filter_factory;
  RDI_NotifQoS*          _qosprop;
  RDI_TypeMap*           _type_map;
  RDI_AdminQoS           _admin_qos;
  CORBA::ULong           _serial;
  CORBA::ULong           _admin_serial;
  CORBA::ULong           _num_consadmin;
  CORBA::ULong           _num_suppadmin;
  CORBA::ULong           _num_consumers;
  CORBA::ULong           _num_suppliers;

  omni_mutex             _opera_lock;	// Channel  operation mutex
  omni_mutex             _proxy_lock;	// Proxy event queue mutex
  omni_condition         _proxy_emty;	// Proxy event queue condition

  // Event filtering and dispatching thread pool
  EventChannelDispatch*  _rdi_dispatch;

  // Global thread responsible for channel maintainance and cleanup.
  // The conditional variable is used to signal channel destruction,
  // as well as implement perioding garbage collection.  
  // When 'destroy' is called on the channel,  '_shutmedown' becomes
  // non zero. 
  // When the garbage collection thread exits, '_gcisactive' becomes
  // zero.

  omni_thread*   _gcollector;
  omni_condition _going_down;
  unsigned char  _gcisactive;
  unsigned char  _shutmedown;

  // Lists of announced events and entries for proxy dispatching
  RDI_EventQueue*        _events;

  omni_mutex             _stats_lock;
  RDI_Watch              _performance_timer;
  RDI_ThreadStat*        _thread_stats;

  CORBA::ULong           _gq_acm;
  CORBA::ULong           _gq_ctr;
  CORBA::ULong           _pq_acm;
  CORBA::ULong           _pq_ctr;

  CORBA::ULong           _prev_num_rdi_match;
  CORBA::ULong           _prev_num_rvm_eval;
  CORBA::ULong           _prev_num_announcements;
  CORBA::ULong           _prev_num_notifications;

  CORBA::ULong           _stat_update_counter;
  CORBA::ULong           _stat_delta_target;

  unsigned long long     _cum_msecs;
  bool                   _second_delta;
  // sleep this many nanosecs on each global event queue insert
  // (starts zero, adjusted according to average notif queue size) 
  unsigned long          _gq_sleep_nanosecs;
  double                 _prev_avg_nq_sz;

  RDI_List<ProxyDispatch_t> _proxy_events;

  // The following is the group manager for ConsumerAdmin_i objects
  // and it is used by the threads that perfrom filtering on behalf
  // of ConsumerAdmin_i objects

  ChannelAdminGroup_m*   _admin_group;

  // Hash tables for the various admin objects

  RDI_Hash<CosNA_AdminID, SupplierAdmin_i *> _supl_admin;
  RDI_Hash<CosNA_AdminID, ConsumerAdmin_i *> _cons_admin;

  // To support offer_change() we need a hash table of event types

  RDI_Hash<CosN_EventType, CORBA::ULong> _evtypes;

  // Carry out event filtering and dispatching for ConsumerAdmin 
  // and ProxySupplier objects

  void  admin_dispatch();
  void  proxy_dispatch();

  // Does any of the filters, if any, of a given ConsumerAdmin_i
  // object match an announced event having the provided type?

  CORBA::Boolean match_event(ConsumerAdmin_i* admin,
			     RDI_StructuredEvent* event,
			     RDI_FilterState_t& fstate);

  void  gcollect();
};

inline void EventChannel_i::incr_num_rdi_match()
{ 
  int id = omni_thread::self()->id();
  RDI_CHECK_THREAD_ID(id);
  RDI_THREAD_STATS_LOCK(id);
  ++_thread_stats[id]._num_rdi_match;
  RDI_THREAD_STATS_UNLOCK(id);
}

inline void EventChannel_i::incr_num_rvm_eval()
{ 
  int id = omni_thread::self()->id();
  RDI_CHECK_THREAD_ID(id);
  RDI_THREAD_STATS_LOCK(id);
  ++_thread_stats[id]._num_rvm_eval;
  RDI_THREAD_STATS_UNLOCK(id);
}

inline void EventChannel_i::incr_num_announcements()
{
  int id = omni_thread::self()->id();
  RDI_CHECK_THREAD_ID(id);
  RDI_THREAD_STATS_LOCK(id);
  if (++_thread_stats[id]._num_announcements % RDI_STATS_MINOR_INCREMENT == 0) {
    RDI_THREAD_STATS_UNLOCK(id);
    RDI_STATS_LOCK;
    _gq_ctr++; _gq_acm += _events->length();
    _pq_ctr++; _pq_acm += _proxy_events.length();
    if (++_stat_update_counter == _stat_delta_target) {
      _stat_delta_target += RDI_STATS_DELTA_INCREMENT;
      dump_stats(false);
    }
    RDI_STATS_UNLOCK;
  } else {
    RDI_THREAD_STATS_UNLOCK(id);
  }
}

inline void EventChannel_i::incr_num_notifications(unsigned int qsize)
{
  int id = omni_thread::self()->id();
  RDI_CHECK_THREAD_ID(id);
  RDI_THREAD_STATS_LOCK(id);
  if (++_thread_stats[id]._num_notifications % RDI_STATS_MINOR_INCREMENT == 0) {
    _thread_stats[id]._qsize_ctr++;
    _thread_stats[id]._qsize_acum += qsize;
    RDI_THREAD_STATS_UNLOCK(id);
    RDI_STATS_LOCK;
    _gq_ctr++; _gq_acm += _events->length();
    _pq_ctr++; _pq_acm += _proxy_events.length();
    if (++_stat_update_counter == _stat_delta_target) {
      _stat_delta_target += RDI_STATS_DELTA_INCREMENT;
      dump_stats(false);
    }
    RDI_STATS_UNLOCK;
  } else {
    RDI_THREAD_STATS_UNLOCK(id);
  }
}

/** The event channel factory is responsible for creating event
  * channels.  This object supports concurrency and maintains a
  * list of all the channels it has created so far. 
  */

class EventChannelFactory_i : 
	WRAPPED_SKELETON_SUPER(CosNA::, EventChannelFactory) 
{
public:
  EventChannelFactory_i();
  EventChannelFactory_i(const RDI_NotifQoS& defqos, const RDI_AdminQoS& defadm);
  EventChannelFactory_i(const WRAPPED_OBJKEY_TYPE& key, 
		        const RDI_NotifQoS& defqos, const RDI_AdminQoS& defadm);
  virtual ~EventChannelFactory_i();

  CosNA_EventChannel_ptr 
	create_channel(const CosN_QoSProperties&   init_qos,
		       const CosN_AdminProperties& init_adm,
		       CosNA_ChannelID& myserial WRAPPED_DECLARG );

  CosNA_ChannelIDSeq* get_all_channels( WRAPPED_DECLARG_VOID );

  CosNA_EventChannel_ptr get_event_channel(CosNA_ChannelID id WRAPPED_DECLARG );

  // (Local only -- not available via RPC)

  EventChannel_i* create_channel(CosNA_ChannelID& myserial); 
  void remove_channel(CosNA_ChannelID id);

  // Print on standard output information about a specific channel
  // or all channels that are currently present in the hash table.

  void display_info(CosNA_ChannelID id=0, CORBA::Boolean event_queue=0);

private:
  omni_mutex   _oplock;
  CORBA::ULong _serial;
  RDI_NotifQoS _defqos;
  RDI_AdminQoS _defadm;
  RDI_Hash<CosNA_ChannelID, EventChannel_i*>  _channel;
};

#endif
