// -*- Mode: C++; -*-
//                              File      : RDIDebug.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of RDI_ErrorLog and RDI_Debug 
//
 
/*
$Log: RDIDebug.cc,v $
Revision 1.13  2000/08/22 18:23:55  alcfp
added description to each file

Revision 1.12  2000/08/16 20:19:49  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <errno.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stddef.h>
#include <syslog.h>
#include "RDIDebug.h"

extern "C" void ast_error_str(int e, char* fmt, char* str);

static char       __buffer;
static ostrstream shared_stream(&__buffer, 1);
static char       log_buff[16384];
static char       tmp_buff[16384];

// ------------------------------------------------------------- //

static inline int Level2Priority(RDI_ErrorLog::level_t level)
{
  if ( level == RDI_ErrorLog::Fatal )
	return LOG_EMERG;
  else if ( level == RDI_ErrorLog::Critical)
	return LOG_CRIT;
  else if ( level == RDI_ErrorLog::Error)
	return LOG_ERR;
  else if ( level == RDI_ErrorLog::Warning)
	return LOG_WARNING;
  else if ( level == RDI_ErrorLog::Notice)
	return LOG_NOTICE;
  else if ( level == RDI_ErrorLog::Info)
	return LOG_INFO;
  else if ( level == RDI_ErrorLog::Debug)
	return LOG_DEBUG;
  // this is never reached
  return LOG_ERR; 
}

static inline const char* Level2String(RDI_ErrorLog::level_t level)
{
  if ( level == RDI_ErrorLog::Fatal )
        return "FAT";
  else if ( level == RDI_ErrorLog::Critical)
        return "CRT";
  else if ( level == RDI_ErrorLog::Error)
        return "ERR";
  else if ( level == RDI_ErrorLog::Warning)
        return "WRN";
  else if ( level == RDI_ErrorLog::Notice)
        return "NOT";
  else if ( level == RDI_ErrorLog::Info)
        return "INF";
  else if ( level == RDI_ErrorLog::Debug)
        return "DBG";
  // this is never reached
  return "ERR";
}

// ------------------------------------------------------------- //

RDI_ErrorLog::RDI_ErrorLog(const char* indentity, 
			   RDI_ErrorLog::level_t level, const char* medium) :
		_active(false), _medium(Ether), _llevel(level), 
		_elfile(0), _elname(0), _stream(log_buff, 1024)
{
  _stream.tie(&shared_stream);
  _elname = CORBA_STRING_DUP(indentity ? indentity : "READY_error_log"); 
  
  if ( strcmp(medium, "ether") == 0 ) {
	_medium = Ether;
  } else if ( strcmp(medium, "atsvclog") == 0 ) {
	_medium = AtSvcLog;
  } else if ( strcmp(medium, "stderr") == 0 ) {
	_medium = Stderr;
	_elfile = stderr;
  } else if ( strcmp(medium, "syslog") == 0 ) {
	_medium = Syslog;
	errno   = 0;
	openlog(_elname, LOG_PID | LOG_CONS | LOG_NOWAIT, LOG_USER);
	if ( errno != 0 ) {
		cerr << "RDI_ErrorLog: openlog() error " << errno << endl;
		cerr << "              logging disabled" << endl;
		_medium = Ether;
		return;
	}
  } else {
	_medium = File;
	if ( ! (_elfile = fopen(medium, "a+")) ) {
		cerr << "RDI_ErrorLog: fopen() failed " << medium << endl;
		cerr << "              logging disabled" << endl;
		_medium = Ether;
		return;
	}
  }
  _active = true;
}

void RDI_ErrorLog::close(void)
{
  if ( _medium == Syslog )
	closelog();
  else if ( (_medium == File) && _elfile )
	fclose(_elfile);
  _active = false;
  _elfile = 0;
}

void RDI_ErrorLog::log(RDI_ErrorLog::level_t level, const char *format, ...)
{
  int     prio = Level2Priority(level);
  va_list args;
  va_start(args, format);

  if ( level > _llevel )
	return;

  this->flush();

  if ( _active == true ) {
	if ( (_medium == Stderr) || (_medium == File) ) {
		fprintf(_elfile, "%s: ", Level2String(level));
		vfprintf(_elfile, format, args);
		fprintf(_elfile, "\n");
		fflush(_elfile);
	} else if ( _medium == Syslog ) {
		vsprintf(log_buff, format, args);
		syslog(prio, log_buff);
	} else if ( _medium == AtSvcLog ) {
	        sprintf(tmp_buff, "%s: ", Level2String(level));
		vsprintf(tmp_buff + strlen(tmp_buff), format, args);
		ast_error_str(0, "%s", tmp_buff);
	}
  }
  va_end(args);
  _stream.seekp(ios::beg);
  if ( level == Fatal )
	abort();
}

void RDI_ErrorLog::flush()
{
  int prio = Level2Priority(_llevel);

  if ( _active == false )
	return;

  _stream << ends;
  if ( strlen(_stream.str()) > 0 ) {
    if ( (_medium == Stderr) || (_medium == File) ) {
      fprintf(_elfile, "%s: %s\n", Level2String(_llevel), _stream.str());
      fflush(_elfile);
    } else if ( _medium == Syslog ) {
      syslog(prio, _stream.str());
    } else if ( _medium == AtSvcLog ) {
      sprintf(tmp_buff, "%s: %s", Level2String(_llevel), _stream.str());
      ast_error_str(0, "%s", tmp_buff);
    }
  }
  _stream.seekp(ios::beg);
}

// ----------------------------------------------------------- //

static const int RDI_MAX_DBG_FILES = 128;

RDI_Debug* RDI_Debug::_def_dbg = 0;

void RDI_Debug::init_default_debug(const char* files, const char* medium)
{
  delete_default_debug();
  _def_dbg = new RDI_Debug("DefaultDebug", files, medium);
}

void RDI_Debug::delete_default_debug()
{
  if ( _def_dbg != (RDI_Debug *) 0 )
	delete _def_dbg;
  _def_dbg = 0;
}

RDI_Debug::RDI_Debug(const char* indentity,
		     const char* list_of_files, const char* medium) :
		RDI_ErrorLog(indentity, RDI_ErrorLog::Debug, medium)
{
  _file_name = new char * [RDI_MAX_DBG_FILES];
  for (int i=0; i < RDI_MAX_DBG_FILES; i++)
	_file_name[i] = 0;
  _num_files = 0;
  _all_files = false;
  this->enable(list_of_files);
}

RDI_Debug::~RDI_Debug()
{
  for (unsigned int i=0; i<_num_files; i++)
	delete [] _file_name[i];
  delete [] _file_name;
  _file_name = 0;
  _num_files = 0;
  _all_files = false; 
}

void RDI_Debug::enable(const char* list_of_files)
{
  unsigned char  over=0;
  unsigned int   indx=0;
  char* strt=0;
  char* cptr=(char *)list_of_files;

  if ( ! list_of_files || (strlen(list_of_files) == 0) )
	return;
  while ( ! over ) {
	// Eat any white spaces that may exist 
	while ( *cptr == ' ' ||  *cptr == '\t' ) 
		cptr++;
 	strt = cptr;
	// Find end of file name or end of string
  	while (*cptr != ' ' && *cptr !=  '\t' && *cptr != ',' && *cptr != '\0')
		cptr++;
	if ( strt == cptr )
		break;
	if ( *cptr == '\0' )
		over = 1;
	else 
		*cptr++ = '\0';
	// If file name exists, do nothing
  	for (indx=0; indx < _num_files; indx++) {
		if ( strcmp(_file_name[indx], strt) == 0 )
			break;
  	}
	if ( indx == _num_files ) {	// Name did not exist
		_file_name[indx] = CORBA_STRING_DUP(strt);
		_num_files += 1;
	}
	strt = cptr;
  }
}

void RDI_Debug::disable(const char* list_of_files)
{
  unsigned char  over=0;
  unsigned char  shft=0;
  char* strt=0;
  char* cptr=(char *)list_of_files;

  if ( ! list_of_files || (strlen(list_of_files) == 0) )
	return;

  while ( ! over ) {
	// Eat any white spaces that may exist 
	while ( *cptr == ' ' ||  *cptr == '\t' ) 
		cptr++;
 	strt = cptr;
	// Find end of file name or end of string
  	while (*cptr != ' ' && *cptr !=  '\t' && *cptr != ',' && *cptr != '\0')
		cptr++;
	if ( strt == cptr )
		break;
	if ( *cptr == '\0' ) 
		over = true;
	else 
		*cptr++ = '\0';
	shft = 0;
  	for (unsigned int i=0; i<_num_files; i++) {
		if ( shft ) 
			_file_name[i-1] = _file_name[i];
		else if ( strcmp(_file_name[i], strt) == 0 ) {
			delete [] _file_name[i];
			shft = 1;
		}
  	}
  	if ( shft )
		_num_files -= 1;
	strt = cptr;
  }
}

bool RDI_Debug::is_enabled(const char* fname) const
{
  if ( get_active() == false )
	return false;
  if ( _all_files == true )
	return true;
  for (unsigned int i=0; i<_num_files; i++) {
	if ( strcmp(_file_name[i], fname) == 0 )
		return true;
	if ((strlen(_file_name[i]) > 2) &&
	    (_file_name[i][0] == '*') && (_file_name[i][1] == '/') &&
	    (strlen(fname) >= (strlen(_file_name[i]) - 2))) {
	  char* wildcard_suffix = _file_name[i] + 2;
	  if ( strlen(wildcard_suffix) == strlen(fname) ) {
	    // matching */foo against ??? (foo and ??? have same length)
	    if ( strcmp(wildcard_suffix, fname) == 0 )
	      return true;
	  } else {
	    // matching */foo against ?????? (latter is longer than foo, 
	    // just length of foo) will only match if has the form ??/XXX
	    //  where XXX has length of foo
	    char* tail = (char*)fname + strlen(fname) - strlen(wildcard_suffix);
	    char* tail_back1 = tail - 1;
	    if ((tail_back1[0] == '/') &&
		(strcmp(wildcard_suffix, tail) == 0 ))
	      return true;
	  }
	}
  }
  return false;
}

void RDI_Debug::current_files(char** file_list, unsigned int max_entries) const
{
  unsigned int indx;
  for (indx = 0; indx < max_entries; indx++ )
	file_list[indx] = 0;
  if ( _num_files == 0 )
	return;
  max_entries = (_num_files > max_entries) ? _num_files : max_entries;
  for (indx = 0; indx < max_entries; indx++ )
	file_list[indx] = _file_name[indx];
}
