// -*- Mode: C++; -*-
//                              File      : PushConsumer_i.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of PushConsumer proxies
//
 
/*
$Log: PushConsumer_i.cc,v $
Revision 1.30  2000/08/22 18:23:54  alcfp
added description to each file

Revision 1.29  2000/08/16 20:19:37  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <iomanip.h>

#include "corba_wrappers.h"

#include "CosNotifyChannelAdmin_i.h"

extern const char* RDI_PRX_TYPE(const CosNA_ProxyType& type);

// ------------------------------------------------------------- //
// ProxyPushConsumer_i                                           //
// ------------------------------------------------------------- //

ProxyPushConsumer_i::ProxyPushConsumer_i(SupplierAdmin_i* admin,
					 EventChannel_i*  evchn,
					 const CosNA_ProxyID&  prxid) : 
		_oplock(), _fa_helper(), _channel(evchn), _myadmin(admin),
		_prxtype(CosNA_PUSH_ANY), 
		_pserial(prxid), _cosevnt(0), _nevents(0), 
		_pxstate(RDI_NotConnected), _qosprop(0), _offauto(0),
		_evtypes(RDI_EventType::hash, RDI_EventType::rank)
{
  _supplier = CosEventComm::PushSupplier::_nil();
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_SupplierAdmin_ptr ProxyPushConsumer_i::MyAdmin( WRAPPED_IMPLARG_VOID )
{ return WRAPPED_IMPL2OREF(CosNA_SupplierAdmin, _myadmin); }

CosN_EventTypeSeq*
ProxyPushConsumer_i::obtain_subscription_types(CosNA_ObtainInfoMode mode
					       WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq* etypeseq=0;
  if ( mode == CosNA_ALL_NOW_UPDATES_OFF ) {
        etypeseq = _channel->obtain_subscription_types();
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_ALL_NOW_UPDATES_ON ) {
        etypeseq = _channel->obtain_subscription_types();
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_OFF ) {
        etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_ON ) {
        etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  }
  return etypeseq;
}

CosN_QoSProperties* 
ProxyPushConsumer_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_C_AnyPRX);
}

void ProxyPushConsumer_i::set_qos(const CosN_QoSProperties& r_qos
				  WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_AnyPRX,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos);
}

void ProxyPushConsumer_i::validate_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_AnyPRX,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void ProxyPushConsumer_i::validate_event_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void ProxyPushConsumer_i::push(const CORBA::Any& event WRAPPED_IMPLARG )
{
  CORBA::Boolean matched = 1;
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected ) {
	throw CosEventComm::Disconnected();
  }
  _nevents += 1;
  if ( has_filters() ) {
        CosNF_FilterIDSeq* fseq = get_all_filters();
        CosNF_Filter_ptr   fltr = CosNF_Filter::_nil();
        matched = 0;
        for (CORBA::ULong ix=0; ix < fseq->length(); ix++) {
                fltr = get_filter((*fseq)[ix]);
                if ( fltr->match(event) ) {
                        matched = 1;
                        break;
                }
        }
        delete fseq;
  }
  if ( matched && (_myadmin->MyOperator() == CosNA_AND_OP) )
	matched = _myadmin->match_event(event);
  if ( matched && _channel->new_any_event(event) )
	throw CORBA::IMP_LIMIT(0, CORBA::COMPLETED_NO);
}

CosNF_FilterID 
ProxyPushConsumer_i::add_filter(CosNF_Filter_ptr fltr 
			        WRAPPED_DECLARG)
{
  return _fa_helper.add_filter_i(fltr);
}

void ProxyPushConsumer_i::disconnect_push_consumer( WRAPPED_IMPLARG_VOID )
{
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void ProxyPushConsumer_i::offer_change(
			const CosN_EventTypeSeq& added,
			const CosN_EventTypeSeq& deled
			WRAPPED_IMPLARG )
{ 
  CosN_EventTypeSeq new_added;
  CosN_EventTypeSeq old_deled;
  CORBA::ULong ix=0, sz=0, vl=0;
  if ( ! RDI_EventType::valid_sequence(added, ix) )
     throw CosNC_InvalidEventType(added[ix]);
  if ( ! RDI_EventType::valid_sequence(deled, ix) )
     throw CosNC_InvalidEventType(deled[ix]);
  new_added.length(0);
  old_deled.length(0);
  _oplock.lock();
  for (sz=0, ix=0; ix < added.length(); ix++) {
     if ( _evtypes.lookup(added[ix], vl) ) {
        vl += 1; _evtypes.replace(added[ix], vl);
     } else {
        vl  = 1; _evtypes.insert(added[ix], vl);
        new_added.length(sz + 1);
        new_added[sz].domain_name = added[ix].domain_name;
        new_added[sz++].type_name = added[ix].type_name;
     }
  }
  for (sz=0, ix=0; ix < deled.length(); ix++) {
     if ( _evtypes.lookup(deled[ix], vl) ) {
        if ( vl == 1 ) {
           _evtypes.remove(deled[ix]);
           old_deled.length(sz + 1);
           old_deled[sz].domain_name = deled[ix].domain_name;
           old_deled[sz++].type_name = deled[ix].type_name;
        } else {
           vl -= 1; _evtypes.replace(deled[ix], vl);
        }
     } else {
        RDI_DUMP("Invalid "<<deled[ix].domain_name<<"::"<<deled[ix].type_name);
     }
  }
  _oplock.unlock();
  // If any new event types were added or existing event types were
  // deleted, notify the parent administrative object
  if ( new_added.length() || old_deled.length() )
     _myadmin->offer_change(new_added, old_deled);

}

void ProxyPushConsumer_i::connect_any_push_supplier(
		CosEventComm::PushSupplier_ptr supplier WRAPPED_IMPLARG)
{
  CosEventComm::PushSupplier_var evntsupl;
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(supplier) )
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  if ( (_pxstate != RDI_NotConnected) || ! CORBA::is_nil(_supplier) )
	throw CosEventChannelAdmin::AlreadyConnected();
  _pxstate  = RDI_Connected;
  // Need to find if the given reference corresponds to a CosEventComm
  // supplier -- this supplier does not support subscription_change()
  evntsupl  = CosEventComm::PushSupplier::_narrow(supplier);
  _cosevnt  = CORBA::is_nil(evntsupl) ? 0 : 1;
  _supplier = WRAPPED_DUPLICATE(CosEventComm::PushSupplier, supplier);
}

ostream& operator << (ostream& out, const ProxyPushConsumer_i& prx)
{
  out << & prx << " -- " << RDI_PRX_TYPE(prx._prxtype);
  if ( prx._cosevnt ) out << " CosEventComm Supplier";
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " Not Connected"; break;
     case RDI_Connected:    out << " Connected";     break;
     case RDI_Disconnected: out << " Disconnected";  break;
     case RDI_Exception:    out << " Exception";     break;
  }
  return out << " #Events " << prx._nevents;
}

void ProxyPushConsumer_i::disconnect_client_and_dispose()
{
  _oplock.lock();
  _pxstate = RDI_Disconnected;
  remove_all_filters();
  if ( ! CORBA::is_nil(_supplier) ) {
        // WRAPPED_RELEASE(CosEventComm::PushSupplier, _supplier);
        _supplier = CosEventComm::PushSupplier::_nil();
  }
  if ( _qosprop ) 
	delete _qosprop;

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}

// ------------------------------------------------------------- //
// StructuredProxyPushConsumer_i                                 //
// ------------------------------------------------------------- //

StructuredProxyPushConsumer_i::StructuredProxyPushConsumer_i(
					SupplierAdmin_i* admin, 	
					EventChannel_i*  evchn,
					const CosNA_ProxyID&  prxid) :
 		_oplock(), _fa_helper(), _channel(evchn), _myadmin(admin),
                _prxtype(CosNA_PUSH_STRUCTURED),
		_pserial(prxid), _nevents(0), 
		_pxstate(RDI_NotConnected), _qosprop(0), _offauto(0),
		_evtypes(RDI_EventType::hash, RDI_EventType::rank)
{
  _supplier = CosNC_StructuredPushSupplier::_nil();
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_SupplierAdmin_ptr StructuredProxyPushConsumer_i::MyAdmin(WRAPPED_IMPLARG_VOID)
{ return WRAPPED_IMPL2OREF(CosNA_SupplierAdmin, _myadmin); }

CosN_EventTypeSeq*
StructuredProxyPushConsumer_i::obtain_subscription_types(CosNA_ObtainInfoMode mode
						         WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq* etypeseq=0;
  if ( mode == CosNA_ALL_NOW_UPDATES_OFF ) {
        etypeseq = _channel->obtain_subscription_types();
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_ALL_NOW_UPDATES_ON ) {
        etypeseq = _channel->obtain_subscription_types();
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_OFF ) {
        etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_ON ) {
        etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  }
  return etypeseq;
}

CosN_QoSProperties* 
StructuredProxyPushConsumer_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_C_StrPRX);
}

void StructuredProxyPushConsumer_i::set_qos(
		const CosN_QoSProperties& r_qos WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_StrPRX,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos);
}

void StructuredProxyPushConsumer_i::validate_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
	WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_StrPRX,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void StructuredProxyPushConsumer_i::validate_event_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void StructuredProxyPushConsumer_i::push_structured_event(
			const CosN_StructuredEvent& event   WRAPPED_IMPLARG )
{
  CORBA::Boolean matched = 1;
  RDI_StructuredEvent* sevnt = new RDI_StructuredEvent(event);
  RDI_Assert(sevnt, "Memory allocation failure -- RDI_StructuredEvent");
  omni_mutex_lock lock(_oplock);

  if ( _pxstate != RDI_Connected ) {
	RDI_DUMP("Supplier is not connected to proxy ... ");
        throw CosEventComm::Disconnected();
  }
  _nevents += 1;
  if ( has_filters() ) {
	CosNF_FilterIDSeq* fltrseq = get_all_filters();
	CosNF_Filter_ptr filter = CosNF_Filter::_nil();
	Filter_i* rdfltr=0;
	matched = 0;
	for (CORBA::ULong ix=0; ix < fltrseq->length(); ix++) {
		filter = get_filter((*fltrseq)[ix]);
                rdfltr = Filter_i::Filter2Filter_i(filter);
		if ( rdfltr ) {         // This is a READY filter
                        if ( rdfltr->rdi_match(sevnt, _channel) ) {
                                matched = 1;
                                break;  
                        }
                } else if ( filter->match_structured(event) ) {
                        matched = 1;
                        break;
                }
	}
	delete fltrseq;
  }
  if ( matched && (_myadmin->MyOperator() == CosNA_AND_OP) )
	matched = _myadmin->match_event(sevnt);
  if ( matched && _channel->new_structured_event(sevnt) ) {
	RDI_DUMP("Failed to insert structured event into channel queue");
	throw CORBA::IMP_LIMIT(0, CORBA::COMPLETED_NO);
  }
  // NOTE: we do not delete 'sevnt' -- inserted in event queue
}

#ifdef __FASTPATH__
void StructuredProxyPushConsumer_i::push_structured_event(
			const CosN_StructuredEvent& event, cdrMemoryStream* cdrm_s)
{
  omni_mutex_lock lock(_oplock);
  RDI_StructuredEvent* sevnt = new RDI_StructuredEvent(event);
  if ( (sevnt == (RDI_StructuredEvent *) 0) || CORBA::is_nil(_supplier) )
        throw CosEventComm::Disconnected();
  sevnt->_rawd = cdrm_s;
  if ( _channel->new_structured_event(sevnt) != 0 )
	sevnt->decr_ref_counter();	// force destruction of event plus
					// deletion of the cdrMemortStream
}
#endif

void StructuredProxyPushConsumer_i::disconnect_structured_push_consumer( WRAPPED_IMPLARG_VOID )
{
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void StructuredProxyPushConsumer_i::offer_change(
			const CosN_EventTypeSeq& added,
			const CosN_EventTypeSeq& deled
		        WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq new_added;
  CosN_EventTypeSeq old_deled;
  CORBA::ULong ix=0, sz=0, vl=0;
  if ( ! RDI_EventType::valid_sequence(added, ix) )
     throw CosNC_InvalidEventType(added[ix]);
  if ( ! RDI_EventType::valid_sequence(deled, ix) )
     throw CosNC_InvalidEventType(deled[ix]);
  new_added.length(0);
  old_deled.length(0);
  _oplock.lock();
  for (sz=0, ix=0; ix < added.length(); ix++) {
     if ( _evtypes.lookup(added[ix], vl) ) {
        vl += 1; _evtypes.replace(added[ix], vl);
     } else {
        vl  = 1; _evtypes.insert(added[ix], vl);
        new_added.length(sz + 1);
        new_added[sz].domain_name = added[ix].domain_name;
        new_added[sz++].type_name = added[ix].type_name;
     }
  }
  for (sz=0, ix=0; ix < deled.length(); ix++) {
     if ( _evtypes.lookup(deled[ix], vl) ) {
        if ( vl == 1 ) {
           _evtypes.remove(deled[ix]);
           old_deled.length(sz + 1);
           old_deled[sz].domain_name = deled[ix].domain_name;
           old_deled[sz++].type_name = deled[ix].type_name;
        } else {
           vl -= 1; _evtypes.replace(deled[ix], vl);
        }
     } else {
        RDI_DUMP("Invalid "<<deled[ix].domain_name<<"::"<<deled[ix].type_name);
     }
  }
  _oplock.unlock();
  // If any new event types were added or existing event types were
  // deleted, notify the parent administrative object
  if ( new_added.length() || old_deled.length() )
     _myadmin->offer_change(new_added, old_deled);
}

void StructuredProxyPushConsumer_i::connect_structured_push_supplier(
	CosNC_StructuredPushSupplier_ptr supplier WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(supplier) ) {
	RDI_DUMP("Supplier is already connected .... ");
        throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  }
  if ( _pxstate != RDI_NotConnected ) {
	RDI_DUMP("Invalid proxy state ... ");
        throw CosEventChannelAdmin::AlreadyConnected();
  }
  _pxstate = RDI_Connected;
  _supplier= WRAPPED_DUPLICATE(CosNC_StructuredPushSupplier, supplier);
}

ostream& operator << (ostream& out, const StructuredProxyPushConsumer_i& prx)
{
  out << &prx  << " -- "  << RDI_PRX_TYPE(prx._prxtype) << 
	 " ID " << setw(3) << prx._pserial;
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " Not Connected"; break;
     case RDI_Connected:    out << " Connected";     break;
     case RDI_Disconnected: out << " Disconnected";  break;
     case RDI_Exception:    out << " Exception";     break;
  }
  return out << " #Events " << prx._nevents;
}

#ifdef __FASTPATH__
CORBA::Boolean StructuredProxyPushConsumer_i::dispatch(GIOP_S &_giop_s, 
				                       const char *operation, 
					    	       CORBA::Boolean response)
{
  cdrStream&       _0RL_s = (cdrStream&)_giop_s;
  cdrMemoryStream* cdrm_s; 

  if ( strcmp(operation, "push_structured_event") == 0  ) {
	CosN_StructuredEvent notif;
	notif <<= _0RL_s;
	// We need to copy the unmarshalled event into the memory buffer
	// before we mark the incoming giop buffer as available ........
	cdrm_s = 0;
	_giop_s.RequestReceived();
	try {
		push_structured_event(notif, cdrm_s); 
	} catch ( CosEventComm::Disconnected & _0RL_ex ) {
		_giop_s.InitialiseReply(GIOP::USER_EXCEPTION);
		CORBA::ULong _len = 42;
		_len >>= _0RL_s;
		_0RL_s.put_char_array((CORBA::Char *)CosEventComm_Disconnected_IntfRepoID, 42);
		_0RL_ex >>= _0RL_s;
		_giop_s.ReplyCompleted();
		return 1;
	}
    	_giop_s.InitialiseReply(GIOP::NO_EXCEPTION);
	_giop_s.ReplyCompleted();
    	return 1;
  } else if ( CosNA::_sk_StructuredProxyPushConsumer::dispatch(
						_giop_s, operation, response) ) {
	return 1;
  } else {
	return 0;
  }
}
#endif

void StructuredProxyPushConsumer_i::disconnect_client_and_dispose()
{
  _oplock.lock();
  _pxstate = RDI_Disconnected;
  remove_all_filters();
  if ( ! CORBA::is_nil(_supplier) ) {
        // WRAPPED_RELEASE(CosNC_StructuredPushSupplier, _supplier);
        _supplier = CosNC_StructuredPushSupplier::_nil();
  }
  if ( _qosprop )
	delete _qosprop;

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}

// ------------------------------------------------------------- //
// SequenceProxyPushConsumer_i                                   //
// ------------------------------------------------------------- //

SequenceProxyPushConsumer_i::SequenceProxyPushConsumer_i(
                        		SupplierAdmin_i* admin,
                        		EventChannel_i*  evchn,
					const CosNA_ProxyID&  prxid) :
		_oplock(), _fa_helper(), _channel(evchn), _myadmin(admin),
		_prxtype(CosNA_PUSH_SEQUENCE),
		_pserial(prxid), _nevents(0), 
		_pxstate(RDI_NotConnected), _qosprop(0), _offauto(0),
		_evtypes(RDI_EventType::hash, RDI_EventType::rank)
{
  _supplier = CosNC_SequencePushSupplier::_nil();
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosNA_SupplierAdmin_ptr SequenceProxyPushConsumer_i::MyAdmin( WRAPPED_IMPLARG_VOID )
{ return WRAPPED_IMPL2OREF(CosNA_SupplierAdmin, _myadmin); }

CosN_EventTypeSeq*
SequenceProxyPushConsumer_i::obtain_subscription_types(CosNA_ObtainInfoMode mode
						       WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq* etypeseq=0;
  if ( mode == CosNA_ALL_NOW_UPDATES_OFF ) {
        etypeseq = _channel->obtain_subscription_types();
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_ALL_NOW_UPDATES_ON ) {
        etypeseq = _channel->obtain_subscription_types();
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_OFF ) {
        etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 1; _oplock.unlock();
  } else if ( mode == CosNA_NONE_NOW_UPDATES_ON ) {
        etypeseq = new CosN_EventTypeSeq; etypeseq->length(0);
        _oplock.lock(); _offauto = 0; _oplock.unlock();
  }
  return etypeseq;
}

CosN_QoSProperties* 
SequenceProxyPushConsumer_i::get_qos( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_oplock);
  RDI_NotifQoS*   qosp = (_qosprop ? _qosprop : _myadmin->qos_properties());
  RDI_Assert(qosp, "Event Channel should have RDI_NotifQoS set");
  return qosp->get_qos(RDI_C_SeqPRX);
}

void SequenceProxyPushConsumer_i::set_qos(
		const CosN_QoSProperties& r_qos WRAPPED_IMPLARG)
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  if (! RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_SeqPRX,eseq,rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
  if ( ! _qosprop ) {
        _qosprop = new RDI_NotifQoS();
        RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");
  }
  _qosprop->set_qos(r_qos);
}

void SequenceProxyPushConsumer_i::validate_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_C_SeqPRX,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void SequenceProxyPushConsumer_i::validate_event_qos(
	const CosN_QoSProperties& r_qos,
        WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
        WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_oplock);
  CORBA::Boolean  child = 0;
  RDI_NotifQoS*   p_qos = _myadmin->qos_properties();
  RDI_NotifQoS*   a_qos = (_qosprop ? _qosprop : p_qos);

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (!RDI_NotifQoS::validate(r_qos,*a_qos,RDI_EMESSAGE,eseq,*rseq,child,p_qos))
        throw CosN_UnsupportedQoS(eseq);
}

void 
SequenceProxyPushConsumer_i::push_structured_events(const CosN_EventBatch& events
						    WRAPPED_IMPLARG )
{
  RDI_StructuredEvent* sevnt=0;
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected )
	throw CosEventComm::Disconnected();
  _nevents += events.length();
  if ( ! has_filters() && ! _myadmin->has_filters() ) {
	if ( _channel->new_sequence_event(events) )
		throw CORBA::IMP_LIMIT(0, CORBA::COMPLETED_NO);
  } else {
	CosNF_FilterIDSeq* fltrseq = get_all_filters();
	CosNF_Filter_ptr filter = CosNF_Filter::_nil();
	Filter_i* rdfltr = 0;
	for (CORBA::ULong ex=0; ex < events.length(); ex++) {
		CORBA::Boolean matched = 0;
		sevnt = new RDI_StructuredEvent(events[ex]);
		for (CORBA::ULong ix=0; ix < fltrseq->length(); ix++) {
			filter = get_filter((*fltrseq)[ix]);
                	rdfltr = Filter_i::Filter2Filter_i(filter);
			if ( rdfltr ) {		// This is a READY filter
				if ( rdfltr->rdi_match(sevnt, _channel) ) {
					matched = 1;
					break;
				}
			} else if ( filter->match_structured(events[ex]) ) {
				matched = 1;
				break;
			}
		}
		if ( matched && 
		     (_myadmin->MyOperator() == CosNA_AND_OP) )
			 matched = _myadmin->match_event(sevnt);
		if ( matched && _channel->new_structured_event(sevnt) )
			throw CORBA::IMP_LIMIT(0, CORBA::COMPLETED_NO);
  		// NOTE: we do not delete 'sevnt' -- inserted in event queue
	}
        delete fltrseq;
  }
}

void SequenceProxyPushConsumer_i::disconnect_sequence_push_consumer( WRAPPED_IMPLARG_VOID )
{
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void SequenceProxyPushConsumer_i::offer_change(
			const CosN_EventTypeSeq& added,
			const CosN_EventTypeSeq& deled
		        WRAPPED_IMPLARG )
{
  CosN_EventTypeSeq new_added;
  CosN_EventTypeSeq old_deled;
  CORBA::ULong ix=0, sz=0, vl=0;
  if ( ! RDI_EventType::valid_sequence(added, ix) )
     throw CosNC_InvalidEventType(added[ix]);
  if ( ! RDI_EventType::valid_sequence(deled, ix) )
     throw CosNC_InvalidEventType(deled[ix]);
  new_added.length(0);
  old_deled.length(0);
  _oplock.lock();
  for (sz=0, ix=0; ix < added.length(); ix++) {
     if ( _evtypes.lookup(added[ix], vl) ) {
        vl += 1; _evtypes.replace(added[ix], vl);
     } else {
        vl  = 1; _evtypes.insert(added[ix], vl);
        new_added.length(sz + 1);
        new_added[sz].domain_name = added[ix].domain_name;
        new_added[sz++].type_name = added[ix].type_name;
     }
  }
  for (sz=0, ix=0; ix < deled.length(); ix++) {
     if ( _evtypes.lookup(deled[ix], vl) ) {
        if ( vl == 1 ) {
           _evtypes.remove(deled[ix]);
           old_deled.length(sz + 1);
           old_deled[sz].domain_name = deled[ix].domain_name;
           old_deled[sz++].type_name = deled[ix].type_name;
        } else {
           vl -= 1; _evtypes.replace(deled[ix], vl);
        }
     } else {
        RDI_DUMP("Invalid "<<deled[ix].domain_name<<"::"<<deled[ix].type_name);
     }
  }
  _oplock.unlock();
  // If any new event types were added or existing event types were
  // deleted, notify the parent administrative object
  if ( new_added.length() || old_deled.length() )
     _myadmin->offer_change(new_added, old_deled);
}

void SequenceProxyPushConsumer_i::connect_sequence_push_supplier(
	CosNC_SequencePushSupplier_ptr supplier WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(supplier) )
        throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  if ( _pxstate != RDI_NotConnected )
        throw CosEventChannelAdmin::AlreadyConnected();
  _pxstate  = RDI_Connected;
  _supplier = WRAPPED_DUPLICATE(CosNC_SequencePushSupplier, supplier);
}

ostream& operator << (ostream& out, const SequenceProxyPushConsumer_i& prx)
{
  out << & prx  << " -- "  << RDI_PRX_TYPE(prx._prxtype) << 
	 " ID " << setw(3) << prx._pserial;
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " Not Connected"; break;
     case RDI_Connected:    out << " Connected";     break;
     case RDI_Disconnected: out << " Disconnected";  break;
     case RDI_Exception:    out << " Exception";     break;
  }
  return out << " #Events " << prx._nevents;
}

void SequenceProxyPushConsumer_i::disconnect_client_and_dispose()
{
  _oplock.lock();
  _pxstate = RDI_Disconnected;
  remove_all_filters();
  if ( ! CORBA::is_nil(_supplier) ) {
        // WRAPPED_RELEASE(CosNC_SequencePushSupplier, _supplier);
        _supplier = CosNC_SequencePushSupplier::_nil();
  }
  if ( _qosprop )
	delete _qosprop;

  // NOTE: we have to release the mutex in order to avoid EBUSY errors 
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}
