// -*- Mode: C++; -*-
//                              File      : CosEventProxy.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of CORBA Event Service Proxies
//
 
/*
$Log: CosEventProxy.cc,v $
Revision 1.15  2000/10/01 13:39:52  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.14.2.1  2000/09/30 15:39:51  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.14  2000/08/22 18:23:53  alcfp
added description to each file

Revision 1.13  2000/08/16 20:19:25  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include "RDI.h"
#include "CosEventChannelAdmin_i.h"
#include "CosNotifyChannelAdmin_i.h"

// ------------------------------------------------------------ //
// EventProxyPushConsumer_i                                     //
// ------------------------------------------------------------ //

EventProxyPushConsumer_i::EventProxyPushConsumer_i(SupplierAdmin_i* admin, 
					 	   EventChannel_i*  evchn) :
		_oplock(), _channel(evchn), _myadmin(admin), 
		_nevents(0), _pxstate(RDI_NotConnected)
{
  _supplier = CosEventComm::PushSupplier::_nil();
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

void EventProxyPushConsumer_i::connect_push_supplier(
		CosEventComm::PushSupplier_ptr supplier WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(supplier) )
        throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  if ( _pxstate != RDI_NotConnected )
	throw CosEventChannelAdmin::AlreadyConnected();
  _pxstate  = RDI_Connected;
  _supplier = WRAPPED_DUPLICATE(CosEventComm::PushSupplier, supplier);
}

void EventProxyPushConsumer_i::push(const CORBA::Any& data WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( _pxstate != RDI_Connected ) {
	throw CosEventComm::Disconnected();
  }
  if ( _channel->new_any_event(data) )
	throw CORBA::IMP_LIMIT(0, CORBA::COMPLETED_NO);
  _nevents += 1;
}

void EventProxyPushConsumer_i::disconnect_push_consumer(WRAPPED_IMPLARG_VOID)
{
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void EventProxyPushConsumer_i::disconnect_client_and_dispose()
{
  omni_mutex_lock lock(_oplock);
  _pxstate = RDI_Disconnected;
  if ( ! CORBA::is_nil(_supplier) ) {
	// WRAPPED_RELEASE(CosEventComm::PushSupplier, _supplier);
	_supplier = CosEventComm::PushSupplier::_nil();
  }
  WRAPPED_DISPOSE_IMPL(this);
}

ostream& operator << (ostream& out, const EventProxyPushConsumer_i& prx)
{
  out << & prx << " CosEvent ProxyPushConsumer";
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " NotConnected "; break;
     case RDI_Connected:    out << " Connected    "; break;
     case RDI_Disconnected: out << " Disconnected "; break;
     case RDI_Exception:    out << " Exception    "; break;
  }
  return out << " #Push "<< prx._nevents;
}

// ------------------------------------------------------------ //
// EventProxyPullConsumer_i                                     //
// ------------------------------------------------------------ //

/** We use unbound threads here for we assume that in the majority
  * of configuations a thread pool will be used for pulling events
  * from suppliers and pushing events to consumers.
  */

class EventPullWorker : public omni_thread {
public:
  typedef void (EventProxyPullConsumer_i::*Method)(void);
  EventPullWorker(EventProxyPullConsumer_i* p, Method m) :
                omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  EventProxyPullConsumer_i* _proxy;
  Method                    _method;
  EventPullWorker()  {;}
};

// ------------------------------------------------------------ //

EventProxyPullConsumer_i::EventProxyPullConsumer_i(SupplierAdmin_i* admin, 
                                                   EventChannel_i*  evchn) :
                _oplock(), _channel(evchn), _myadmin(admin), 
		_pworker(0), _iamdone(0), _thrdone(0), _nevents(0), 
		_pxstate(RDI_NotConnected), _pulltms()
{
  _supplier = CosEventComm::PullSupplier::_nil();
  // When the number of pull threads allocated at the channel level is
  // 0, each proxy uses its own thread to pull events from its supplier
  if ( _channel->pull_threads() == 0 ) {
	_iamdone = new omni_condition(&_oplock);
	RDI_Assert(_iamdone, "Memory allocation failed -- omni_condition");
	_pworker = new EventPullWorker(this, &EventProxyPullConsumer_i::_pull);
	RDI_Assert(_pworker, "Memory allocation failed -- omni_thread");
	_pworker->start();
  	_thrdone = 0;
        RDI_DUMP("Pull thread for proxy " << this << " -- " << _pworker->id());
  }
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

void EventProxyPullConsumer_i::connect_pull_supplier(
		CosEventComm::PullSupplier_ptr supplier WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(supplier) )
        throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  if ( _pxstate != RDI_NotConnected ) 
        throw CosEventChannelAdmin::AlreadyConnected();
  _pxstate  = RDI_Connected;
  _supplier = WRAPPED_DUPLICATE(CosEventComm::PullSupplier, supplier);
  if ( _pworker && _iamdone )
	_iamdone->signal();
}

void EventProxyPullConsumer_i::disconnect_pull_consumer(WRAPPED_IMPLARG_VOID)
{
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void EventProxyPullConsumer_i::disconnect_client_and_dispose()
{
  _oplock.lock();
  _pxstate = RDI_Disconnected;
  // If we are using our own thread for event pulling, signal the
  // thread that it should terminate itself
  if ( _pworker && _iamdone ) {
	_iamdone->signal();
	_oplock.unlock();
	while ( ! _thrdone ) {
		omni_thread::yield();
		_iamdone->signal();
	}
	_oplock.lock();
	delete _iamdone;
  }
  if ( ! CORBA::is_nil(_supplier) ) {
        // WRAPPED_RELEASE(CosEventComm::PullSupplier, _supplier);
        _supplier = CosEventComm::PullSupplier::_nil();
  }
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}

void EventProxyPullConsumer_i::pull_event(CORBA::Boolean& invalid)
{
  CORBA::Any*     event=0;
  CORBA::Boolean  hasev=0;
  omni_mutex_lock glock(_oplock);

  invalid = 0;
  if ( _pxstate != RDI_Connected ) {
	return;
  }
  try {
	event = _supplier->try_pull(hasev);
	if ( hasev ) {
		if ( ! _channel->new_any_event(*event) )
			_nevents += 1;
		delete event;
	}
	_pulltms = RDI_TimeValue::timeofday();
  } catch (...) {
	invalid  = 1;
	_pxstate = RDI_Exception;
  }
}

void EventProxyPullConsumer_i::_pull()
{
  CORBA::Any*    event=0;
  CORBA::UShort  pullp=0;
  CORBA::Boolean hasev=0;

  _oplock.lock();
  while ( _pxstate == RDI_NotConnected )
	_iamdone->wait();

  while ( 1 ) {
	if ( _pxstate != RDI_Connected ) {
		_thrdone = 1;
		_oplock.unlock();
		RDI_DUMP("Pull thread for proxy " << this << " exits");
		omni_thread::exit();
	}
  	try {
		event = _supplier->try_pull(hasev);
		pullp = _channel->pull_period();
		if ( hasev ) {
			if ( ! _channel->new_any_event(*event) )
				_nevents += 1;
			delete event;
		}
		_pulltms = RDI_TimeValue::timeofday();
		if ( pullp ) {
			// We use 'timedwait' below so that we get notified
			// when the connection is terminated before we time
			// out.  We hold the lock when 'timedwait' returns.
			unsigned long nsec, usec, r_nsec, r_usec;
			r_nsec = pullp / 1000;
			r_usec = (pullp % 1000) * 1000;
			omni_thread::get_time(&nsec, &usec, r_nsec, r_usec);
			_iamdone->timedwait(nsec, usec);
		} else {
			_oplock.unlock();
			omni_thread::yield();
			_oplock.lock();
		}
  	} catch (...) {
		_pxstate = RDI_Exception;
  	}
  }
}

ostream& operator << (ostream& out, const EventProxyPullConsumer_i& prx)
{
  out << & prx << " CosEvent ProxyPullConsumer";
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " NotConnected "; break;
     case RDI_Connected:    out << " Connected    "; break;
     case RDI_Disconnected: out << " Disconnected "; break;
     case RDI_Exception:    out << " Exception    "; break;
  }
  return out << "#Push "<< prx._nevents;
}

// ------------------------------------------------------------ //
// EventProxyPushSupplier_i                                     //
// ------------------------------------------------------------ //

/** We use unbound threads here for we assume that in the majority
  * of configuations a thread pool will be used for pulling events
  * from suppliers and pushing events to consumers.
  */

class EventPushWorker : public omni_thread {
public:
  typedef void (EventProxyPushSupplier_i::*Method)(void);
  EventPushWorker(EventProxyPushSupplier_i* p, Method m) :
                omni_thread(NULL,PRIORITY_NORMAL), _proxy(p), _method(m) {;}
  void run(void *)      { (_proxy->*_method)(); }
private:
  EventProxyPushSupplier_i* _proxy;
  Method                    _method;
  EventPushWorker()  {;}
};

// ------------------------------------------------------------ //

EventProxyPushSupplier_i::EventProxyPushSupplier_i(ConsumerAdmin_i* admin, 
						   EventChannel_i*  evchn) :
		_oplock(), _channel(evchn), _myadmin(admin), 
		_pworker(0),  _iamdone(0),  _thrdone(0), _nevents(0), 
		_pxstate(RDI_NotConnected), _ntfqueue()
{
  _consumer = CosEventComm::PushConsumer::_nil();
  // When the number of push threads allocated at the channel level is
  // 0, each proxy uses its own thread to pull events from its supplier
  if ( _channel->push_threads() == 0 ) {
	_iamdone = new omni_condition(&_oplock);
	RDI_Assert(_iamdone, "Memory allocation failed -- omni_condition");
	_pworker = new EventPushWorker(this, &EventProxyPushSupplier_i::_push);
	RDI_Assert(_pworker, "Memory allocation failed -- omni_thread");
	_pworker->start();
	_thrdone = 0;
	RDI_DUMP("Pull thread for proxy " << this << " -- " << _pworker->id());
  }
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

void EventProxyPushSupplier_i::connect_push_consumer(
		CosEventComm::PushConsumer_ptr consumer WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  if ( CORBA::is_nil(consumer) ) 
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  if ( _pxstate != RDI_NotConnected )
	throw CosEventChannelAdmin::AlreadyConnected();
  _pxstate  = RDI_Connected;
  _consumer = WRAPPED_DUPLICATE(CosEventComm::PushConsumer, consumer);
  if ( _pworker && _iamdone )
	_iamdone->signal();
}

void EventProxyPushSupplier_i::disconnect_push_supplier(WRAPPED_IMPLARG_VOID)
{
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void EventProxyPushSupplier_i::add_event(RDI_StructuredEvent* event)
{
  _oplock.lock();
  if ( (_pxstate == RDI_Connected) && event ) {
	event->incr_ref_counter( RDI::lockEvent() );
	_ntfqueue.insert_tail(event);
	if ( _pworker && _iamdone )
		_iamdone->signal();
  }
  _oplock.unlock();
}

void EventProxyPushSupplier_i::push_event(CORBA::Boolean& invalid)
{
  RDI_StructuredEvent* event=0;
  unsigned int         qsize=0;

  invalid = 0;
  _oplock.lock();
  if ( (_pxstate != RDI_Connected) || (_ntfqueue.length() == 0) ) {
	_oplock.unlock();
	return;
  }
  event = _ntfqueue.get_head();
  qsize = _ntfqueue.length() - 1;
  _ntfqueue.remove_head();
  _nevents += 1;
  // We are not holding the lock while pushing events to a consumer
  // to avoid blocking the thread adding new events to to our queue
  _oplock.unlock();

  if (  RDI::lockEvent()  ) event->mylock().lock();
  try {
	const CosN_StructuredEvent& cosev = event->get_cos_event();
	if ( strcmp(event->get_type_name(), "%ANY") == 0 ) {
		_consumer->push(cosev.remainder_of_body);
	} else {
		CORBA::Any anyev;
		anyev <<= cosev;
		_consumer->push(anyev);
	}
	_channel->incr_num_notifications(qsize);
  } catch (...) {
	invalid  = 1;
	_pxstate = RDI_Exception;
  }
  if (  RDI::lockEvent()  ) event->mylock().unlock();
  event->decr_ref_counter( RDI::lockEvent() );
}

void EventProxyPushSupplier_i::_push()
{
  register RDI_StructuredEvent* event=0;
  register unsigned int         qsize=0;

  while ( 1 ) {
	_oplock.lock();
	while ( ((_pxstate == RDI_Connected) && (_ntfqueue.length() == 0)) ||
		(_pxstate == RDI_NotConnected) )
		_iamdone->wait();
	if ( _pxstate != RDI_Connected ) {
		_thrdone = 1;
		_oplock.unlock();
		RDI_DUMP("Push thread for proxy " << this << " exits");
		omni_thread::exit();
	}
	event = _ntfqueue.get_head();
	qsize = _ntfqueue.length() - 1;
	_ntfqueue.remove_head();
	_nevents += 1;
	// We are not holding the lock while pushing events to a consumer
	// to avoid blocking the thread adding new events to to our queue
	_oplock.unlock();

	if (  RDI::lockEvent()  ) event->mylock().lock();
	try {
		const CosN_StructuredEvent& cosev = 
							event->get_cos_event();
		if ( strcmp(event->get_type_name(), "%ANY") == 0 ) {
			_consumer->push(cosev.remainder_of_body);
		} else {
			CORBA::Any anyev;
			anyev <<= cosev;
			_consumer->push(anyev);
		}
		_channel->incr_num_notifications(qsize);
	} catch (...) {
		_pxstate = RDI_Exception;
	}
	if (  RDI::lockEvent()  ) event->mylock().unlock();
	event->decr_ref_counter( RDI::lockEvent() );
  }
}

void EventProxyPushSupplier_i::disconnect_client_and_dispose()
{
  _oplock.lock();
  _pxstate = RDI_Disconnected;
  // If we are using our own thread for dispatching, signal the
  // thread that it should terminate itself and wait for 5 secs
  if ( _pworker && _iamdone ) {
	_iamdone->signal();
        _oplock.unlock();
	while ( ! _thrdone ) {
		omni_thread::yield();
		_iamdone->signal();
	}
        _oplock.lock();
        delete _iamdone;
	_iamdone = 0;
  }
  if ( ! CORBA::is_nil(_consumer) ) {
        // WRAPPED_RELEASE(CosEventComm::PushConsumer, _consumer);
        _consumer = CosEventComm::PushConsumer::_nil();
  }
  while ( _ntfqueue.length() )          // Remove all events
	_ntfqueue.remove_head();

  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}

ostream& operator << (ostream& out, const EventProxyPushSupplier_i& prx)
{
  out << & prx << " CosEvent ProxyPushSupplier";
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " NotConnected "; break;
     case RDI_Connected:    out << " Connected    "; break;
     case RDI_Disconnected: out << " Disconnected "; break;
     case RDI_Exception:    out << " Exception    "; break;
  }
  return out << "QSize "<< prx._ntfqueue.length() << " #Push "<< prx._nevents;
}

// ------------------------------------------------------------ //
// EventProxyPullSupplier_i                                     //
// ------------------------------------------------------------ //

EventProxyPullSupplier_i::EventProxyPullSupplier_i(ConsumerAdmin_i* admin, 
						   EventChannel_i*  evchn) :
			_oplock(), _isempty(&_oplock), _channel(evchn), 
			_myadmin(admin), _blkpull(0), _nevents(0), 
			_pxstate(RDI_NotConnected), _ntfqueue()
{
  _consumer = CosEventComm::PullConsumer::_nil();
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

void EventProxyPullSupplier_i::connect_pull_consumer(
		CosEventComm::PullConsumer_ptr consumer WRAPPED_IMPLARG)
{
  omni_mutex_lock lock(_oplock);
  RDI_DUMP("CosEvent consumer to connect to " << this);
  if ( CORBA::is_nil(consumer) ) 
	throw CORBA::BAD_PARAM(0, CORBA::COMPLETED_NO);
  if ( _pxstate == RDI_Connected )
	throw CosEventChannelAdmin::AlreadyConnected();
  _pxstate  = RDI_Connected;
  _consumer = WRAPPED_DUPLICATE(CosEventComm::PullConsumer, consumer);
  RDI_DUMP("CosEvent consumer connected to " << this);
}

void EventProxyPullSupplier_i::disconnect_pull_supplier(WRAPPED_IMPLARG_VOID)
{
  _myadmin->remove_proxy(this);
  disconnect_client_and_dispose();
}

void EventProxyPullSupplier_i::add_event(RDI_StructuredEvent* event)
{
  _oplock.lock();
  if ( (_pxstate == RDI_Connected) && event ) {
	event->incr_ref_counter( RDI::lockEvent() );
	_ntfqueue.insert_tail(event);
	_isempty.signal();
  }
  _oplock.unlock();
}

CORBA::Any* EventProxyPullSupplier_i::pull(WRAPPED_IMPLARG_VOID)
{
  _oplock.lock();
  if ( _pxstate == RDI_NotConnected ) {
	_oplock.unlock();
	throw CosEventComm::Disconnected();
  }
  _blkpull += 1;
  while ( (_pxstate == RDI_Connected) && (_ntfqueue.length() == 0) )
	_isempty.wait();
  if ( _pxstate != RDI_Connected ) {
  	_blkpull -= 1;
	_oplock.unlock();
	throw CosEventComm::Disconnected();
  } else {
	RDI_StructuredEvent* event = _ntfqueue.get_head();
	const CosN_StructuredEvent& cosev = event->get_cos_event();
	_ntfqueue.remove_head();
	_nevents += 1;
  	_blkpull -= 1;
	_oplock.unlock();
  	RDI_DUMP("pull event " << event << " from queue of " << this);
	if ( strcmp(event->get_type_name(), "%ANY") == 0 ) {
		return new CORBA::Any(cosev.remainder_of_body);
	} else {
		CORBA::Any* anyev = new CORBA::Any;
		(*anyev) <<= cosev;
		return anyev;
	}
  }
}

CORBA::Any* EventProxyPullSupplier_i::try_pull(
			CORBA::Boolean& has_event WRAPPED_IMPLARG)
{
  _oplock.lock();
  if ( _pxstate != RDI_Connected ) {
	_oplock.unlock();
	throw CosEventComm::Disconnected();
  } else if (_ntfqueue.length() == 0 ) {
	_oplock.unlock();
	has_event = 0;
	return new CORBA::Any();
  } else {
	RDI_StructuredEvent* event = _ntfqueue.get_head();
	const CosN_StructuredEvent& cosev = event->get_cos_event();
	_ntfqueue.remove_head();
	_nevents += 1;
	_oplock.unlock();
	has_event = 1;
  	RDI_DUMP("try_pull event " << event << " from queue of " << this);
	if ( strcmp(event->get_type_name(), "%ANY") == 0 ) {
                return new CORBA::Any(cosev.remainder_of_body);
        } else {
                CORBA::Any* anyev = new CORBA::Any;
                (*anyev) <<= cosev;
                return anyev;
        }
  }
}

void EventProxyPullSupplier_i::disconnect_client_and_dispose()
{
  RDI_StructuredEvent* event=0;
  RDI_DUMP("Consumer to disconnect from " << this);
  _oplock.lock();
  _pxstate = RDI_Disconnected;
  // The consumer(s) may be blocked on a pull() at this point; signal
  // the conditional variable and give the consumer some time 
  _isempty.broadcast();
  _oplock.unlock();
  while ( _blkpull )  {
	omni_thread::yield();
	_isempty.signal();
  }
  _oplock.lock();

  if ( ! CORBA::is_nil(_consumer) ) {
        // WRAPPED_RELEASE(CosEventComm::PullConsumer, _consumer);
        _consumer = CosEventComm::PullConsumer::_nil();
  }
  while ( (event = _ntfqueue.get_head()) ) {    // Remove all events
	_ntfqueue.remove_head();
	event->decr_ref_counter( RDI::lockEvent() );
  }
  _oplock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}

ostream& operator << (ostream& out, const EventProxyPullSupplier_i& prx)
{
  out << & prx << " CosEvent ProxyPullSupplier";
  switch ( prx._pxstate ) {
     case RDI_NotConnected: out << " NotConnected "; break;
     case RDI_Connected:    out << " Connected    "; break;
     case RDI_Disconnected: out << " Disconnected "; break;
     case RDI_Exception:    out << " Exception    "; break;
  }
  return out << "QSize "<< prx._ntfqueue.length() << " #Push "<< prx._nevents;
}
