// This application implements a push supplier which announces to
// an event channel CosNotification::StructuredEvent events.  The
// announced events are of two kinds.  The event type used by the
// first kind is Stock::Ticker, and it contains 2 variable_header
// fields and 3 filterable_data fields.  The second kind uses the
// Stock::Tocker event type, and it does not contain any fields.
 
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iostream.h>
#include "corba_wrappers.h"
#include "CosNotify.h"
#include "CosNotifyComm_i.h"

void print_etypes(CosN_EventTypeSeq& etypes, int mode)
{
  cout << "UPDATES ARE NOW " << (mode ? "ON " : "OFF") << endl;
  cout << "Length of type sequence = " << (long) etypes.length() << endl;
  for (CORBA::ULong k = 0; k < etypes.length(); k++) {
     cout << "   event type "<< k << ": domain name = "<<
      	   	(const char *)etypes[k].domain_name << " type_name = " << 
		(const char *)etypes[k].type_name << endl;
  }
}

static void usage(const char* pname)
{
   cout << "Usage: " << pname << " [-d numb] [-n name]" << endl;
   cout << "       -d numb : disconnect after numb pushes" << endl;
   cout << "       -n name : channel name [\"EventChannel\"]" << endl;
}

int main(int argc, char** argv)
{
  CosNA_EventChannel_var channel;
  CosN_StructuredEvent se1, se2;
  CosNaming::NamingContext_var name_context;
  CosNaming::Name name;
  int c, num_pushes=0;
  char *channelName = (char *) "EventChannel";
  char *channelKind = (char *) "EventChannel";

  CORBA::ORB_ptr orb = WRAPPED_ORB_INIT(argc, argv);
  CORBA::BOA_ptr boa = WRAPPED_BOA_INIT(orb, argc, argv);

  StructuredPushSupplier_i* supplier = new StructuredPushSupplier_i();

  WRAPPED_ORB_REGISTER_SERVANT_NAME(orb, "filter_struct_push__supplier");
  WRAPPED_BOA_OBJ_IS_READY(boa, supplier);

  while ( (c = getopt(argc, argv,"d:n:")) != EOF ) {
  	switch (c) {
        case 'd': num_pushes = atoi(optarg);
                  cout << "# events to be pushed = " << num_pushes << endl;
                  break;
        case 'n': channelName = optarg;
		  channelKind = optarg;
                  break;
        default : usage(argv[0]);
                  exit(-1);
     }
  }

  // Locate, using the naming service, the event channel, after
  // obtaining a reference to the nameing service itself.......

  try {
	CORBA::Object_var name_service;
	cout << "** Resolving initial reference to NameService" << endl;
	name_service = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService"); 
	cout << "** Narrowing the retrieved reference" << endl;
	name_context = CosNaming::NamingContext::_narrow(name_service);
	if ( CORBA::is_nil(name_context) ) {
		cerr << "Failed to obtain context for NameService" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the naming service" << endl;
        exit(1);
  }

  name.length(1);
  name[0].id   = CORBA::string_dup(channelName);
  name[0].kind = CORBA::string_dup(channelKind);
  try {
	cout << "** Resolving the reference to the event channel" << endl;
	CORBA::Object_var channel_ref = name_context->resolve(name);
	cout << "** Narrowing the resolved reference" << endl;
	channel = CosNA_EventChannel::_narrow(channel_ref);
	if ( CORBA::is_nil(channel) ) {
		cerr << "Failed to narrow Event Channel !" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the event channel" << endl;
        exit(1);
  }
 
  // Let the BOA know that we are ready -- no blocking since we
  // need to supply events

  WRAPPED_IMPL_IS_READY_FORK(boa, "filter_struct_push_supplier");
   
  // Get supplier admin object from the channel

  CosNA_InterFilterGroupOperator ifoper = CosNA_AND_OP;
  CosNA_SupplierAdmin_var supplier_admin;
  CosNA_AdminID suppID;

  try {
	cout << "** Obtaining a supplier admin" << endl;
	supplier_admin = channel->new_for_suppliers(ifoper, suppID);
	if ( CORBA::is_nil(supplier_admin) ) {
		cerr << "Failed to find Supplier Admin !" << endl;
		exit(1);
	}
  } catch (...) {
	cerr << "Failed to obtain Supplier Admin !" << endl;
	exit(1);
  }

  cout << "Obtained SupplierAdmin from the channel" << endl;

  // Get a push proxy using the above admin object

  CosNA_ProxyID prxID;
  CosNA_ClientType ctype = CosNA_STRUCTURED_EVENT;
  CosNA_ProxyConsumer_var cons;
  CosNA_StructuredProxyPushConsumer_var pcons;
  try {
	cons = supplier_admin->obtain_notification_push_consumer(ctype,prxID);
  	pcons=CosNA_StructuredProxyPushConsumer::_narrow(cons);
	if ( CORBA::is_nil(pcons) ) {
		cerr << "Failed to find Proxy Consumer !" << endl;
		exit(1);
	}
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
	cerr << "Failed to obtain Proxy Consumer !" << endl;
	exit(1);
  }

  cout << "Obtained ProxyPushConsumer from SupplierAdmin" << endl;

  // Connect supplier to the proxy
  try {
	pcons->connect_structured_push_supplier(supplier->_this());
  }
  catch (CORBA::BAD_PARAM& ex) {
	cerr << "BAD_PARAM Exception while connecting Push Supplier !" << endl;
        exit (1);
  }
  catch (CosEventChannelAdmin::AlreadyConnected& ex) {
	cerr << "Pull Supplier already connected !" << endl;
  } catch (...) {
	cerr << "Failed to connect Push Supplier !" << endl;
	exit(1);
  }

  cout << "Connected push supplier to ProxyPushConsumer" << endl;

  se1.header.fixed_header.event_type.domain_name = CORBA::string_dup("Stock");
  se1.header.fixed_header.event_type.type_name   = CORBA::string_dup("Ticker");
  se1.header.variable_header.length(2);
  se1.header.variable_header[0].name = (const char*)"FooDouble";
  se1.header.variable_header[0].value <<= ((CORBA::Double)2.5);
  se1.header.variable_header[1].name = (const char*)"Priority";
  se1.header.variable_header[1].value <<= (CORBA::ULong)5;
  se1.filterable_data.length(3);
  se1.filterable_data[0].name = (const char*)"MsgId";
  se1.filterable_data[0].value <<= 
			(const char*)"message12793date123456789postOffice22";
  se1.filterable_data[1].name = (const char*)"From";
  se1.filterable_data[1].value <<= (const char*)"issy@research.att.com";
  se1.filterable_data[2].name = (const char*)"Subject";
  se1.filterable_data[2].value <<= (const char*)"test";

  se2.header.fixed_header.event_type.domain_name = CORBA::string_dup("Stock");
  se2.header.fixed_header.event_type.type_name   = CORBA::string_dup("Tocker");
  se2.header.variable_header.length( 0 );
  se2.filterable_data.length( 0 );

  CosN_EventTypeSeq* etypes = 
    		pcons->obtain_subscription_types(CosNA_ALL_NOW_UPDATES_ON);

  if(etypes)
    print_etypes(*etypes,1);
  else
    cerr << "No types were found" << endl;

  int flip = 0;

  for (int i=0; (num_pushes == 0) || (i < num_pushes); i=i+2) {
	se1.remainder_of_body <<= (CORBA::ULong) i;
	se2.remainder_of_body <<= (CORBA::ULong) i+1;
        se1.header.variable_header[1].value <<= (CORBA::ULong) i;
	try {
		pcons->push_structured_event( se1 );
		cout << "Pushed" << i << " -- " << 
	  (const char*)se1.header.fixed_header.event_type.domain_name << "::" <<
	  (const char*)se1.header.fixed_header.event_type.type_name << endl;
	} catch (...) {
		cerr << "push() se1 failed" << endl;
	}
	sleep(1);
	try {
		pcons->push_structured_event( se2 );
		cout << "Pushed" << i+1 << " -- " << 
	  (const char*)se2.header.fixed_header.event_type.domain_name << "::" <<
	  (const char*)se2.header.fixed_header.event_type.type_name << endl;
	} catch (...) {
		cerr << "push() se2 failed" << endl;
	}
	sleep(1);

	if ((i % 20) == 0 ) {
	  try {
	    if (flip) {
	      etypes = 
		pcons->obtain_subscription_types(CosNA_ALL_NOW_UPDATES_ON);
	      flip = 0;
	    }
	    else {
	      etypes = 
		pcons->obtain_subscription_types(CosNA_ALL_NOW_UPDATES_OFF);
	      flip = 1;
	    }
	  }
	  catch (...) {
	    cerr << "Failed to obtain subscription types!" << endl;
	    exit(1);
	  }

	  if(etypes)
	    print_etypes(*etypes, flip == 1 ? 0 : 1);
	  else
	    cerr << "No types were found" << endl;
	}
  }

  // Disconnect supplier from the service

  pcons->disconnect_structured_push_consumer();
  supplier_admin->destroy();
  WRAPPED_DISPOSE_IMPL(supplier);

  return 0;
}

