// This application implements a push consumer.  It uses filters
// and consumes CosNotification::StructuredEvent events.
// The filter used by this application selects all Stock::Ticker
// events having a Priority field with value greater than 10.
 
#include <stdlib.h>
#include <unistd.h>
#include <iostream.h>
#include "corba_wrappers.h"
#include "CosNotify.h"
#include "CosNotifyComm_i.h"

omni_mutex mutex;
omni_condition finish(&mutex);

/*****************************/
class MyPushConsumer_i : 
	WRAPPED_SKELETON_SUPER(CosNC::, StructuredPushConsumer) {
public:
  MyPushConsumer_i(int max_numb=0) : _num_push(0), _max_numb(max_numb)  {;}

  inline void push_structured_event(const CosN_StructuredEvent& event);
  inline void disconnect_structured_push_consumer();
  inline void offer_change(const CosN_EventTypeSeq& added,
                           const CosN_EventTypeSeq& removed);
protected:
  CORBA::ULong _num_push;
  CORBA::ULong _max_numb;
};

inline void MyPushConsumer_i::push_structured_event(
				const CosN_StructuredEvent& event)
{
  CORBA::ULong l;
  cout << event.header.fixed_header.event_type.domain_name << "::" <<
	  event.header.fixed_header.event_type.type_name << " -- ";
  event.remainder_of_body >>= l;
  cout << l << endl;
  if ( _max_numb && (++_num_push >= _max_numb) )
	finish.signal();
}

inline void MyPushConsumer_i::disconnect_structured_push_consumer()
{ cout << "PushConsumer: disconnected" << endl; }

inline void MyPushConsumer_i::offer_change(const CosN_EventTypeSeq&,
                                           const CosN_EventTypeSeq&)
{;}
/*****************************/

static void usage(const char* pname)
{
   cout << "Usage: " << pname << " [-d numb] [-n name]" << endl;
   cout << "       -d numb : disconnect after numb pushes" << endl;
   cout << "       -n name : channel name [\"EventChannel\"]" << endl;
}

int main(int argc, char** argv)
{
  CosNA_EventChannel_var channel;
  CosNaming::NamingContext_var name_context;
  CosNaming::Name name;
  int c, num_pushes=0;
  char pname[128];
  CosNF_FilterFactory_ptr ffp;
  CosNF_Filter_ptr        filter;

  strcpy(pname, argv[0]);

  CORBA::ORB_ptr orb = WRAPPED_ORB_INIT(argc, argv);
  CORBA::BOA_ptr boa = WRAPPED_BOA_INIT(orb, argc, argv);

  // Process command line arguments

  char *channelName = (char *) "EventChannel";
  char *channelKind = (char *) "EventChannel";

  while ( (c = getopt(argc, argv,"d:n:")) != EOF ) {
  	switch (c) {
        case 'd': num_pushes = atoi(optarg);
                  cout << "# events to be consumed = " << num_pushes << endl;
                  break;
        case 'n': channelName = optarg;
		  channelKind = optarg;
                  break;
        default : usage(pname);
                  exit(-1);
     }
  }

  MyPushConsumer_i* consumer = new MyPushConsumer_i(num_pushes);

  WRAPPED_ORB_REGISTER_SERVANT_NAME(orb, "filter_struct_push_consumer");
  WRAPPED_BOA_OBJ_IS_READY(boa, consumer);

  // Let the BOA know that we are ready -- no blocking since we
  // need to consume events

  WRAPPED_IMPL_IS_READY_FORK(boa, "filter_struct_push_consumer");

  // Locate, using the naming service, the event channel, after
  // obtaining a reference to the nameing service itself.......

  try {
	CORBA::Object_var name_service;
	name_service = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService"); 
	name_context = CosNaming::NamingContext::_narrow(name_service);
	if ( CORBA::is_nil(name_context) ) {
		cerr << "Failed to obtain context for NameService" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the naming service" << endl;
        exit(1);
  }

  name.length(1);
  name[0].id   = CORBA::string_dup(channelName);
  name[0].kind = CORBA::string_dup(channelKind);
  try {
	CORBA::Object_var channel_ref = name_context->resolve(name);
	channel = CosNA_EventChannel::_narrow(channel_ref);
	if ( CORBA::is_nil(channel) ) {
		cerr << "Failed to narrow Event Channel !" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the event channel" << endl;
        exit(1);
  }
 
  // Get consumer admin object from the channel

  CosNA_InterFilterGroupOperator ifoper = 
			CosNA_AND_OP;
  CosNA_ConsumerAdmin_var consumer_admin;
  CosNA_AdminID suppID;

  try {
	consumer_admin = channel->new_for_consumers(ifoper, suppID);
	if ( CORBA::is_nil(consumer_admin) ) {
		cerr << "Failed to find Consumer Admin !" << endl;
		exit(1);
	}
  } catch (...) {
	cerr << "Failed to obtain Consumer Admin !" << endl;
	exit(1);
  }

  cout << "Obtained ConsumerAdmin from the channel" << endl;

  // Get a push proxy using the above admin object

  CosNA_ProxyID prxID;
  CosNA_ClientType ctype = 
				CosNA_STRUCTURED_EVENT;
  CosNA_ProxySupplier_var supp;
  CosNA_StructuredProxyPushSupplier_var psupp;
  try {
	supp = consumer_admin->obtain_notification_push_supplier(ctype,prxID);
  	psupp=CosNA_StructuredProxyPushSupplier::_narrow(supp);
	if ( CORBA::is_nil(psupp) ) {
		cerr << "Failed to find Proxy Supplier !" << endl;
		exit(1);
	}
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
	cerr << "Failed to obtain Proxy Supplier !" << endl;
	exit(1);
  }

  cout << "Obtained ProxyPushSupplier from ConsumerAdmin" << endl;

  // Connect supplier to the proxy and register event types to 
  // receive using subscription_change() on the admin object..

 CosN_EventTypeSeq added, deleted;

  added.length(1);
  added[0].domain_name = CORBA::string_dup("Stock");
  added[0].type_name   = CORBA::string_dup("Ticker");
  deleted.length(0); 


  // obtain a reference to the default filter factory,
  // create a filter object 
  // and add simple type constraints to it 


  try {
    ffp    = channel->default_filter_factory();  
    filter = ffp->create_filter("EXTENDED_TCL");
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
	cerr << "Failed to obtain filter !" << endl;
	exit(1);
  }

  cout << "Obtained filter from channel" << endl;

  // construct a simple constraint expression,
  // and add it to the proxy

  CosNF_ConstraintExpSeq   exp;
  CosNF_ConstraintInfoSeq* cis1;

  exp.length(1);
  exp[0].event_types.length(1);
  exp[0].event_types[0].domain_name = CORBA::string_dup("Stock");
  exp[0].event_types[0].type_name = CORBA::string_dup("Ticker");
  exp[0].constraint_expr = CORBA::string_dup("$Priority > 10");

  try {
    cis1 = filter->add_constraints(exp);
    for (unsigned int i=0; i < cis1->length(); i++) {
	cout << "Constraint " << (*cis1)[i].constraint_id << " -- "
	     <<	(const char *) (*cis1)[i].constraint_expression.constraint_expr
	     << endl;
    }
  }
  catch(CosNF_InvalidConstraint& _exobj1) {
    cerr << "Exception thrown : Invalid Constraint given "
	 << (const char *)exp[0].constraint_expr << endl;
    cerr << "constraint not added " << endl << endl;
  }
  catch (...) {
	cerr << "Exception thrown : " << (const char *)exp[0].constraint_expr 
	     << endl << "constraint not added " << endl << endl;
  }

  try {
        CosNF_FilterID id1 = consumer_admin->add_filter(filter);
	cout << "Added filter to Admin " << id1 << endl;
  } catch (...) {
	cerr << "Failed to attach filter to admin object" << endl;
  }

  try {
	psupp->connect_structured_push_consumer(consumer->_this());
  }
  catch (CORBA::BAD_PARAM& ex) {
	cerr << "BAD_PARAM Exception while connecting Push Consumer !" << endl;
        exit (1);
  }
  catch (CosEventChannelAdmin::AlreadyConnected& ex) {
	cerr << "Push Consumer already connected !" << endl;
  } catch (...) {
	cerr << "Failed to connect Push Consumer!" << endl;
	exit(1);
  }

  cout << "Connected push consumer to ProxyPushSupplier" << endl;

  mutex.lock();
  finish.wait();
  mutex.unlock();

  // Disconnect supplier from the service

  psupp->disconnect_structured_push_supplier();
  WRAPPED_DISPOSE_IMPL(consumer);

  return 0;
}

