// This application implements a push consumer that connects to
// a PULL_SEQUENCE proxy.
// It consumes CosNotification::StructuredEvent event sequences.
// Each event in the sequence is assumed to have a CORBA::ULong
// value stored in the remainder_of_body attribute.

#include <stdlib.h>
#include <unistd.h>
#include <iostream.h>

#include "corba_wrappers.h"

#include "CosNotify.h"
#include "CosNotifyComm_i.h"

CORBA::Boolean verbose = 0;
CORBA::ULong   num_push = 0;
omni_mutex     mutex;
omni_condition finish(&mutex);

/*****************************/
class MyPushConsumer_i : 
	WRAPPED_SKELETON_SUPER(CosNC::, SequencePushConsumer) {
public:
  MyPushConsumer_i(int max_push=20) : _max_push(max_push) {;}
  inline void push_structured_events(const CosN_EventBatch& batch);
  inline void disconnect_sequence_push_consumer();
  inline void offer_change(const CosN_EventTypeSeq& added,
                           const CosN_EventTypeSeq& removed);
protected:
  CORBA::ULong _max_push;
};

inline void MyPushConsumer_i::push_structured_events(const CosN_EventBatch& batch)
{
  if ( verbose ) {
     CORBA::ULong l;
     for (unsigned int i=0; i < batch.length(); i++) {
  	cout << batch[i].header.fixed_header.event_type.domain_name << "::" <<
	        batch[i].header.fixed_header.event_type.type_name << " -- ";
  	batch[i].remainder_of_body >>= l;
  	cout << l << endl;
     }
  }
  num_push += batch.length();
  if ( _max_push && (num_push >= _max_push) )
	finish.signal();
}

inline void MyPushConsumer_i::disconnect_sequence_push_consumer()
{ if ( verbose ) cout << "PushConsumer: disconnected" << endl; }

inline void MyPushConsumer_i::offer_change(const CosN_EventTypeSeq&,
                                           const CosN_EventTypeSeq&)
{;}
/*****************************/

static void usage(const char* pname)
{
   cout << "Usage: " << pname << " [-d numb] [-n name] [-v]" << endl;
   cout << "       -d numb : disconnect after numb events"   << endl;
   cout << "       -n name : channel name [\"EventChannel\"]" << endl;
   cout << "       -v      : verbose output mode" << endl;
}

int main(int argc, char** argv)
{
  CosNA_EventChannel_var channel;
  CosNaming::NamingContext_var name_context;
  CosNaming::Name name;
  int c, num_pushes=0;
  char pname[128];

  strcpy(pname, argv[0]);

  CORBA::ORB_ptr orb = WRAPPED_ORB_INIT(argc, argv);
  CORBA::BOA_ptr boa = WRAPPED_BOA_INIT(orb, argc, argv);

  // Process command line arguments

  char *channelName = (char *) "EventChannel";
  char *channelKind = (char *) "EventChannel";

  while ( (c = getopt(argc, argv,"d:n:v")) != EOF ) {
  	switch (c) {
        case 'd': num_pushes = atoi(optarg);
                  break;
        case 'n': channelName = optarg;
		  channelKind = optarg;
                  break;
	case 'v': verbose = 1;
		  break;
        default : usage(pname);
                  exit(-1);
     }
  }

  MyPushConsumer_i* consumer = new MyPushConsumer_i(num_pushes);

  WRAPPED_ORB_REGISTER_SERVANT_NAME(orb, "batch_push_consumer");
  WRAPPED_BOA_OBJ_IS_READY(boa, consumer);

  // Let the BOA know that we are ready -- no blocking since we
  // need to consume events

  WRAPPED_IMPL_IS_READY_FORK(boa, "batch_push_consumer");

  // Locate, using the naming service, the event channel, after
  // obtaining a reference to the nameing service itself.......

  try {
	CORBA::Object_var name_service;
	name_service = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService"); 
	name_context = CosNaming::NamingContext::_narrow(name_service);
	if ( CORBA::is_nil(name_context) ) {
		cerr << "Failed to obtain context for NameService" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the naming service" << endl;
        exit(1);
  }

  name.length(1);
  name[0].id   = CORBA::string_dup(channelName);
  name[0].kind = CORBA::string_dup(channelKind);
  try {
	CORBA::Object_var channel_ref = name_context->resolve(name);
	channel = CosNA_EventChannel::_narrow(channel_ref);
	if ( CORBA::is_nil(channel) ) {
		cerr << "Failed to narrow Event Channel !" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the event channel" << endl;
        exit(1);
  }
 
  // Get consumer admin object from the channel

  CosNA_InterFilterGroupOperator ifoper = 
			CosNA_AND_OP;
  CosNA_ConsumerAdmin_var consumer_admin;
  CosNA_AdminID suppID;

  try {
	consumer_admin = channel->new_for_consumers(ifoper, suppID);
	if ( CORBA::is_nil(consumer_admin) ) {
		cerr << "Failed to find Consumer Admin !" << endl;
		exit(1);
	}
  } catch (...) {
	cerr << "Failed to obtain Consumer Admin !" << endl;
	exit(1);
  }

  if ( verbose ) 
	cout << "Obtained ConsumerAdmin from the channel" << endl;

  // Get a push proxy using the above admin object

  CosNA_ProxyID prxID;
  CosNA_ClientType ctype = 
				CosNA_SEQUENCE_EVENT;
  CosNA_ProxySupplier_var supp;
  CosNA_SequenceProxyPushSupplier_var psupp;
  try {
	supp = consumer_admin->obtain_notification_push_supplier(ctype,prxID);
  	psupp=CosNA_SequenceProxyPushSupplier::_narrow(supp);
	if ( CORBA::is_nil(psupp) ) {
		cerr << "Failed to find Proxy Supplier !" << endl;
		exit(1);
	}
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
	cerr << "Failed to obtain Proxy Supplier !" << endl;
	exit(1);
  }

  if ( verbose )
	cout << "Obtained ProxyPushSupplier from ConsumerAdmin" << endl;

  // Connect supplier to the proxy

  try {
	psupp->connect_sequence_push_consumer(consumer->_this());
  }
  catch (CORBA::BAD_PARAM& ex) {
	cerr << "BAD_PARAM Exception while connecting Push Consumer !" << endl;
        exit (1);
  }
  catch (CosEventChannelAdmin::AlreadyConnected& ex) {
	cerr << "Push Consumer already connected !" << endl;
  } catch (...) {
	cerr << "Failed to connect Push Consumer !" << endl;
	exit(1);
  }

  if ( verbose )
  	cout << "Connected push consumer to ProxyPushSupplier" << endl;

  mutex.lock();
  finish.wait();

  // Disconnect supplier from the service

  try {
  	psupp->disconnect_sequence_push_supplier();
	consumer_admin->destroy();
	WRAPPED_DISPOSE_IMPL(consumer);
  } catch (...) {
	cerr << "Exception during the cleanup process ... " << endl;
	exit(1);
  }

  cout << ::getpid() << " got " << num_push << " events " << endl;
  return 0;
}
