// This application implements a pull supplier that connects to 
// a PULL_SEQUENCE proxy.
// It generates CosNotification::StructuredEvent event sequences.
// Each event in the sequence has Sensor::Value as its event type
// and the remainder_of_body stores a CORBA::ULong value.
 
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <iostream.h>
#include "corba_wrappers.h"
#include "CosNotify.h"
#include "CosNotifyComm_i.h"

omni_mutex mutex;
omni_condition finish(&mutex);

/*****************************/

class MyPullSupplier_i : 
	WRAPPED_SKELETON_SUPER(CosNC::, SequencePullSupplier) {
public:
  MyPullSupplier_i(int max_pull=2, int batch_sz=5) : 
		_num_pull(0), _max_pull(max_pull), _batch_sz(batch_sz) {;}
  inline CosN_EventBatch* pull_structured_events(CORBA::Long max_number);
  inline CosN_EventBatch* try_pull_structured_events(CORBA::Long max_number,
						 CORBA::Boolean& has_event);
  inline void         disconnect_sequence_pull_supplier();
  inline void         subscription_change(const CosN_EventTypeSeq& added,
                                          const CosN_EventTypeSeq& removed);
protected:
  CORBA::ULong _num_pull;
  CORBA::ULong _max_pull;
  CORBA::ULong _batch_sz;
};

inline CosN_EventBatch* 
MyPullSupplier_i::pull_structured_events(CORBA::Long max_number)
{
  CosN_StructuredEvent se;
  CosN_EventBatch* bc = new CosN_EventBatch(max_number);

  se.header.fixed_header.event_type.domain_name = CORBA::string_dup("Sensor");
  se.header.fixed_header.event_type.type_name   = CORBA::string_dup("Value");
  se.header.variable_header.length(0);
  se.filterable_data.length(0);

  for (int i=0; i < max_number; i++) {
	se.remainder_of_body <<= _num_pull++;
	(*bc)[i] = se;
  }
  cout << "Pulled: " << max_number << endl;
  if ( _max_pull && (_num_pull >= _max_pull * _batch_sz) )
	finish.signal();
  return bc;
}

inline CosN_EventBatch*
MyPullSupplier_i::try_pull_structured_events(CORBA::Long max_number,
					     CORBA::Boolean& has_event)
{
  CosN_StructuredEvent se;
  CosN_EventBatch* bc = new CosN_EventBatch;

  cout << "try_pull_structured_events() invoked by the channel" << endl;
  bc->length(_batch_sz);

  se.header.fixed_header.event_type.domain_name = CORBA::string_dup("Sensor");
  se.header.fixed_header.event_type.type_name   = CORBA::string_dup("Value");
  se.header.variable_header.length(0);
  se.filterable_data.length(0);

  for (unsigned int i=0; i < _batch_sz; i++) {
        se.remainder_of_body <<= _num_pull++;
        (*bc)[i] = se;
  }
  has_event = 1;
  cout << "Pulled: " << _batch_sz << endl;

  if ( _max_pull && (_num_pull >= _max_pull * _batch_sz) )
        finish.signal();
  return bc;
}

inline void MyPullSupplier_i::disconnect_sequence_pull_supplier()
{ cout << "SequencePullSupplier: disconnected" << endl; }

inline void MyPullSupplier_i::subscription_change(
                                const CosN_EventTypeSeq& added,
                                const CosN_EventTypeSeq& removed)
{
  CORBA::ULong indx;
  CosN_EventType   type;

  cout << "MyPullSupplier_i: subscription_change contains: " << endl;
  for (indx = 0; indx < added.length(); indx++) {
        type = added[indx];
        cout << "\t+ " << (const char *)type.domain_name << 
		"::" << (const char *)type.type_name << endl;
  }
  for (indx = 0; indx < removed.length(); indx++) {
        type = removed[indx];
        cout << "\t- " << (const char *)type.domain_name << 
		"::" << (const char *)type.type_name << endl;
  }
}

/*****************************/

static void usage(const char* pname)
{
   cout << "Usage: " << pname << " [-d numb] [-b size] [-n name]" << endl;
   cout << "       -d numb : disconnect after numb batches"   << endl;
   cout << "       -b size : max number of events per batch"  << endl;
   cout << "       -n name : channel name [\"EventChannel\"]" << endl;
}

int main(int argc, char** argv)
{
  CosNA_EventChannel_var channel;
  CosN_StructuredEvent se;
  CosNaming::NamingContext_var name_context;
  CosNaming::Name name;
  int c, num_pulls=4, batch_sz=5;
  char *channelName = (char *) "EventChannel";

  CORBA::ORB_ptr orb = WRAPPED_ORB_INIT(argc, argv);
  CORBA::BOA_ptr boa = WRAPPED_BOA_INIT(orb, argc, argv);

  // Process command line arguments

  while ( (c = getopt(argc, argv,"d:b:n:")) != EOF ) {
  	switch (c) {
        case 'd': num_pulls = atoi(optarg); break;
	case 'b': batch_sz  = atoi(optarg); break;
        case 'n': channelName = optarg;     break;
        default : usage(argv[0]); exit(-1);
     }
  }

  cout << "# event batches to be pulled = " << num_pulls << endl;
  cout << "max # events per event batch = " << batch_sz  << endl;

  MyPullSupplier_i* supplier = new MyPullSupplier_i(num_pulls, batch_sz);

  WRAPPED_ORB_REGISTER_SERVANT_NAME(orb, "batch_pull_supplier");
  WRAPPED_BOA_OBJ_IS_READY(boa, supplier);

  // Locate, using the naming service, the event channel, after
  // obtaining a reference to the nameing service itself.......

  try {
	CORBA::Object_var name_service;
	cout << "** Resolving initial reference to NameService" << endl;
	name_service = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService"); 
	cout << "** Narrowing the retrieved reference" << endl;
	name_context = CosNaming::NamingContext::_narrow(name_service);
	if ( CORBA::is_nil(name_context) ) {
		cerr << "Failed to obtain context for NameService" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the naming service" << endl;
        exit(1);
  }

  name.length(1);
  name[0].id   = CORBA::string_dup(channelName);
  name[0].kind = CORBA::string_dup(channelName);
  try {
	cout << "** Resolving the reference to the event channel" << endl;
	CORBA::Object_var channel_ref = name_context->resolve(name);
	cout << "** Narrowing the resolved reference" << endl;
	channel = CosNA_EventChannel::_narrow(channel_ref);
	if ( CORBA::is_nil(channel) ) {
		cerr << "Failed to narrow Event Channel !" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the event channel" << endl;
        exit(1);
  }
 
  // Let the BOA know that we are ready -- no blocking since we
  // need to supply events

  WRAPPED_IMPL_IS_READY_FORK(boa, "batch_pull_supplier");
   
  // Get supplier admin object from the channel

  CosNA_InterFilterGroupOperator ifoper = CosNA_AND_OP;
  CosNA_SupplierAdmin_var supplier_admin;
  CosNA_AdminID suppID;

  try {
	cout << "** Obtaining a supplier admin" << endl;
	supplier_admin = channel->new_for_suppliers(ifoper, suppID);
	if ( CORBA::is_nil(supplier_admin) ) {
		cerr << "Failed to find Supplier Admin !" << endl;
		exit(1);
	}
  } catch (...) {
	cerr << "Failed to obtain Supplier Admin !" << endl;
	exit(1);
  }

  cout << "Obtained SupplierAdmin from the channel" << endl;

  // Get a push proxy using the above admin object

  CosNA_ProxyID prxID;
  CosNA_ClientType ctype=CosNA_SEQUENCE_EVENT;
  CosNA_ProxyConsumer_var cons;
  CosNA_SequenceProxyPullConsumer_var pcons;
  try {
	cons = supplier_admin->obtain_notification_pull_consumer(ctype,prxID);
  	pcons=CosNA_SequenceProxyPullConsumer::_narrow(cons);
	if ( CORBA::is_nil(pcons) ) {
		cerr << "Failed to find Proxy Consumer !" << endl;
		exit(1);
	}
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
	cerr << "Failed to obtain Proxy Consumer !" << endl;
	exit(1);
  }

  cout << "Obtained ProxyPullConsumer from SupplierAdmin" << endl;

  // Connect supplier to the proxy

  try {
	pcons->connect_sequence_pull_supplier(supplier->_this());
  }
  catch (CORBA::BAD_PARAM& ex) {
	cerr << "BAD_PARAM Exception while connecting Pull Supplier !" << endl;
        exit (1);
  }
  catch (CosEventChannelAdmin::AlreadyConnected& ex) {
	cerr << "Pull Supplier already connected !" << endl;
  } catch (...) {
	cerr << "Failed to connect Push Supplier !" << endl;
	exit(1);
  }

  cout << "Connected pull supplier to ProxyPullConsumer" << endl;

  // Wait till the desired number of events has been supplied

  mutex.lock();
  finish.wait();
  mutex.unlock();

  // Disconnect supplier from the service

  pcons->disconnect_sequence_pull_consumer();
  WRAPPED_DISPOSE_IMPL(supplier);

  return 0;
}
