// -*- Mode: C++; -*-
//                              File      : FilterAdmin_i.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of FilterFactory_i and FAdminHelper
//
 
/*
$Log: FilterAdmin_i.cc,v $
Revision 1.32  2000/08/22 18:23:53  alcfp
added description to each file

Revision 1.31  2000/08/16 20:19:29  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <iostream.h>
#include "RDIList.h"
#include "RDIDebug.h"
#include "CosNotifyFilter_i.h"
#include "CosNotifyChannelAdmin_i.h"

////////////////////////////////////////////////////////////////////

FilterFactory_i::FilterFactory_i(const char* grammar, EventChannel_i* chan) :
	_oplock(), _channel(chan), _nlangs(0)
{
  for (unsigned int i=0; i < MAXGR; i++)
	_clangs[i] = 0; 
  if ( ! (_clangs[0] = CORBA_STRING_DUP(grammar)) )
	throw CORBA::NO_MEMORY(0, CORBA::COMPLETED_NO);
  _nlangs += 1;
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

FilterFactory_i::FilterFactory_i(const WRAPPED_OBJKEY_TYPE& key,
				 const char* grammar, EventChannel_i* chan) :
	WRAPPED_SUPER_KEY(CosNotifyFilter::, FilterFactory, key),
	_oplock(), _channel(chan), _nlangs(0)
{
  for (unsigned int i=0; i < MAXGR; i++)
        _clangs[i] = 0;
  if ( ! (_clangs[0] = CORBA_STRING_DUP(grammar)) )
        throw CORBA::NO_MEMORY(0, CORBA::COMPLETED_NO);
  _nlangs += 1;
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

FilterFactory_i::~FilterFactory_i()
{
  omni_mutex_lock lock(_oplock);
  for (unsigned int i=0; i < MAXGR; i++) {
	CORBA_STRING_FREE(_clangs[i]);
	_clangs[i] = 0;
  }
  _nlangs  = 0;
  _channel = 0;
}

CosNF_Filter_ptr FilterFactory_i::create_filter(const char* grammar)
{
  Filter_i* fltr = 0;
  RDI_DUMP("Create a new filter using: " << grammar);
  if ( ! this->is_supported(grammar) ) 
     throw CosNF_InvalidGrammar();
  fltr = new Filter_i(grammar);
  RDI_Assert(fltr, "Memory allocation failure -- Filter_i");
  return WRAPPED_IMPL2OREF(CosNF_Filter, fltr);
}

CosNF_MappingFilter_ptr FilterFactory_i::create_mapping_filter(
				const char* grammar, const CORBA::Any& value)
{
  MappingFilter_i* fltr = 0;
  if ( ! this->is_supported(grammar) ) 
     throw CosNF_InvalidGrammar();
  if ( ! (fltr = new MappingFilter_i(grammar, value)) )
     throw CORBA::NO_MEMORY(0, CORBA::COMPLETED_NO);
  RDI_DUMP("Created new MappingFilter using: " << grammar);
  return WRAPPED_IMPL2OREF(CosNF_MappingFilter, fltr); 
}

int FilterFactory_i::add_grammar(const char* grammar)
{
  if ( this->is_supported(grammar) )
     return 0;
  _oplock.lock();
  if ( _nlangs == MAXGR ) {
     _oplock.unlock();
     RDI_DUMP("reached max number of supported grammars");
     return -1;
  }
  for (unsigned int i=0; i < MAXGR; i++) {
     if ( _clangs[i] == (char *) 0 ) {
	if ( ! (_clangs[i] = CORBA_STRING_DUP(grammar)) ) {
     	   _oplock.unlock();
	   RDI_DUMP("failed to allocate string for " << grammar);
	   return -1;
	} else {
	   RDI_DUMP("Adding support for: " << grammar);
	   _nlangs += 1;
	   _oplock.unlock();
	   return 0;
	}
     }
  }
  _oplock.unlock();
  RDI_FDUMP("Internal error -- inconsistent data structures.....");
  return -1;
}

void FilterFactory_i::del_grammar(const char* grammar)
{
  _oplock.lock();
  for (unsigned int i=0; i < MAXGR; i++) {
     if ( _clangs[i] && (strcmp(_clangs[i], grammar) == 0) ) {
	RDI_DUMP("Deleting support for: " << grammar);
	CORBA_STRING_FREE(_clangs[i]);
	_nlangs -= 1;
	break;
     }
  }
  _oplock.unlock();
}

CORBA::Boolean FilterFactory_i::is_supported(const char* grammar)
{
  _oplock.lock();
  for (unsigned int i=0; i < MAXGR; i++) {
     if ( _clangs[i] && (strcmp(_clangs[i], grammar) == 0) ) {
	_oplock.unlock();
	return 1;
     }
  }
  _oplock.unlock();
  return 0;
}

////////////////////////////////////////////////////////////////////

FAdminHelper::FAdminHelper() : _oplock(), _serial(1),
			       _rdcbids(RDI_ULongHash, RDI_ULongRank),
			       _filters(RDI_ULongHash, RDI_ULongRank) {;}

FAdminHelper::~FAdminHelper()
{ remove_all_filters(); }

CosNF_FilterID FAdminHelper::add_filter_i(CosNF_Filter_ptr new_filter,
				     RDINotifySubscribe_ptr filter_holder)
{
  CosNF_FilterID   fltrID;
  CosNF_CallbackID rdcbID;
  Filter_i* fltr = Filter_i::Filter2Filter_i(new_filter);
  RDI_Assert(fltr, "Filter was not created by READY"); 
  omni_mutex_lock lock(_oplock);
  fltrID = _serial++;
  if ( _filters.insert(fltrID, fltr) ) {
	RDI_DUMP("Failed to register new filter in hash table");
	return 0;
  }
  WRAPPED_DUPLICATE(CosNF_Filter, new_filter);
  rdcbID = fltr->attach_callback_i(filter_holder);
  if ( _rdcbids.insert(fltrID, rdcbID) ) {
	RDI_DUMP("Failed to register new callback ID in hash table");
	_filters.remove(fltrID);
	return 0;
  }
  RDI_DUMP("\tFilter "<<new_filter<<" ["<<fltrID<<"] for "<<filter_holder);
  return fltrID;
}

// ******************************************************************** //
// The following is invoked by SupplierAdmin and ProxyConsumer objects. //
// Filters attached to such objects are treated differently than normal //
// consumer filters.  In particular,  changes in event types referenced //
// in these filters are NOT propagated to suppliers.                    //
// ******************************************************************** //

CosNF_FilterID FAdminHelper::add_filter_i(CosNF_Filter_ptr new_filter)
{
  CosNF_FilterID fltrID;
  Filter_i* fltr = Filter_i::Filter2Filter_i(new_filter);
  RDI_Assert(fltr, "Filter was not created by READY");
  omni_mutex_lock lock(_oplock);
  fltrID = _serial++;
  if ( _filters.insert(fltrID, fltr) ) {
	RDI_DUMP("Failed to register new filter in hash table");
        return 0;
  }
  WRAPPED_DUPLICATE(CosNF_Filter, new_filter);
  return fltrID;
}

// ---------------------------------------------------------------- //
// A filter is being destroyed and it is registered with the object //
// that uses this particular FAdminHelper object. To avoid run-time //
// errors, we should remove the filter from the internal hash table //
// ---------------------------------------------------------------- //

void FAdminHelper::rem_filter_i(Filter_i* fltr)
{
  RDI_HashCursor<CosNF_FilterID, Filter_i*> curs;
  omni_mutex_lock lock(_oplock);
  for (curs = _filters.cursor(); curs.is_valid(); ++curs) {
     Filter_i*& fval = curs.val();
     if ( fval->getID() == fltr->getID() ) {
	RDI_DUMP("removing filter [" << curs.key() << "] from hash table");
	_filters.remove( curs.key() );
	_rdcbids.remove( curs.key() );
	break;
     }
  }
}

CosNF_Filter_ptr FAdminHelper::get_filter(CosNF_FilterID fltrID 
				     WRAPPED_IMPLARG )
{
  Filter_i*       fltr=0;
  omni_mutex_lock lock(_oplock);
  if ( _filters.lookup(fltrID, fltr) ) {
	RDI_DUMP("Get Filter " << fltr << " [" << fltrID << "]");
	return WRAPPED_IMPL2OREF(CosNF_Filter, fltr);
  }
  throw CosNF_FilterNotFound();
}

CosNF_FilterIDSeq* FAdminHelper::get_all_filters( WRAPPED_IMPLARG_VOID )
{
  RDI_HashCursor<CosNF_FilterID, Filter_i*> fcur;
  CORBA::ULong  indx=0;
  CosNF_FilterIDSeq* fseq=new CosNF_FilterIDSeq();
  RDI_Assert(fseq, "Memory allocation failure -- FilterIDSeq");
  omni_mutex_lock lock(_oplock);
  fseq->length(_filters.length());
  for ( fcur = _filters.cursor(); fcur.is_valid(); fcur++, indx++ ) {
	(*fseq)[indx] = fcur.key();
  }
  return fseq;
}

void FAdminHelper::remove_filter(CosNF_FilterID fltrID
				 WRAPPED_IMPLARG )
{
  Filter_i* fltr=0;
  omni_mutex_lock lock(_oplock);
  if ( ! _filters.lookup(fltrID, fltr) )
	throw CosNF_FilterNotFound();
  // cleanup_filter(fltrID);
  _filters.remove(fltrID);
  _rdcbids.remove(fltrID);
  WRAPPED_RELEASE_IMPL(CosNF_Filter, fltr);
}

void FAdminHelper::remove_all_filters( WRAPPED_IMPLARG_VOID )
{
  RDI_HashCursor<CosNF_FilterID, Filter_i*> fcur;
  omni_mutex_lock lock(_oplock);
  for ( fcur = _filters.cursor(); fcur.is_valid(); fcur++ ) {
	Filter_i*& fltr = fcur.val();
	cleanup_filter(fcur.key());
  }
  _rdcbids.clear();
  _filters.clear();
}

void FAdminHelper::cleanup_filter(CosNF_FilterID fltrID)
{
  CosN_EventTypeSeq addseq;
  CosN_EventTypeSeq delseq;
  CosNF_CallbackID  rdcbID;
  Filter_i* fltr=0;

  _filters.lookup(fltrID, fltr);
  _rdcbids.lookup(fltrID, rdcbID);
  addseq.length(0);
  delseq.length(0);

  // We have to notify all interested parties about the deletion of
  // this filter. In particular, we need to collect all event types
  // referenced in the constraints of the filter and propagate them
  // to the subscribers of 'subscription_change'

  fltr->create_ev_types_from_dom_list(delseq);
  fltr->notify_subscribers_i(addseq, delseq);
  fltr->detach_callback_i(rdcbID);

  WRAPPED_RELEASE_IMPL(CosNF_Filter, fltr);
}
