// -*- Mode: C++; -*-
//                              File      : EventChannel_i.cc
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    Implementation of EventChannel_i
//
 
/*
$Log: EventChannel_i.cc,v $
Revision 1.85  2000/10/01 13:39:53  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.84.2.1  2000/09/30 15:39:51  alcfp
Removed sleep() calls used for synchronization with unbound threads. Counters and flags are used instead

Revision 1.84  2000/08/22 18:22:47  alcfp
omnimutex release at channel destroy

Revision 1.83  2000/08/16 20:19:27  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#include <new.h> 
#include <unistd.h> 
#include <iomanip.h>
#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include "RDI.h"
#include "RDIDebug.h"
#include "RDITypeMap.h"
#include "CosNotifyChannelAdmin_i.h"

extern unsigned long QueueGCPeriod;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//                             EventChannel_i                            //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

EventChannel_i::EventChannel_i(EventChannelFactory_i*  cfactory,
			       const CosN_QoSProperties&   init_qos,
			       const CosN_AdminProperties& init_adm,
			       const CosNA_ChannelID&       myserial) :
	_pull_supplier(0), _push_consumer(0), _my_channel_factory(cfactory), 
	_def_consumer_admin(0), _def_supplier_admin(0), _def_filter_factory(0),
	_type_map(0), _admin_qos(), _serial(myserial), _admin_serial(1), 
	_num_consadmin(0), _num_suppadmin(0), _num_consumers(0), 
	_num_suppliers(0), _opera_lock(), _proxy_lock(), 
	_proxy_emty(&_proxy_lock), _rdi_dispatch(0), _gcollector(0), 
	_going_down(&_opera_lock), _gcisactive(0), _shutmedown(0), 
	_events(0), _stats_lock(), _performance_timer(), 
	_proxy_events(8192,1024), _admin_group(0),
	_supl_admin(RDI_ULongHash, RDI_ULongRank), 
	_cons_admin(RDI_ULongHash, RDI_ULongRank),
	_evtypes(RDI_EventType::hash, RDI_EventType::rank)
{
  CosNA_AdminID def_admin_id = 0;
  CosNA_InterFilterGroupOperator ifop = CosNA_OR_OP;

  // Make sure that the _admin_qos preperties are valid

  RDI_Assert(_admin_qos.numAdminGroups, "Zero value for numAdminGroups");
  RDI_Assert(_admin_qos.numAdminThreads, "Zero value for numAdminThreads");

  _qosprop = new RDI_NotifQoS();
  RDI_Assert(_qosprop, "Memory allocation failure - RDI_NotifQoS");

  _qosprop->set_qos(init_qos);		// The factory validated them
  _admin_qos.from_admin(init_adm); 	// The factory validated them

  // Create the default filter factory
  _def_filter_factory = new FilterFactory_i("EXTENDED_TCL", this);
  RDI_Assert(_def_filter_factory,"Memory allocation failure - FilterFactory_i");

  // Create the type mapping object to be used for matching
  _type_map = new RDI_TypeMap(this, 256);
  RDI_Assert(_type_map, "Memory allocation failure - RDI_TypeMap");

  // Create the manager for the queue of announced events
  _events = new RDI_EventQueue(_admin_qos.maxQueueLength, 
			       _admin_qos.numAdminThreads);
  RDI_Assert(_events, "Memory allocation failure - RDI_EventQueue");
  _events->set_gc_period(QueueGCPeriod);

  // Create the thread pools for pull consumers and push suppliers
  _pull_supplier = (_admin_qos.numPullThreads == 0) ? 0 :
  			(new RDI_PullSupplier(_admin_qos.numPullThreads, 
				              _admin_qos.pullEventPeriod));
  _push_consumer = (_admin_qos.numPushThreads == 0) ? 0 :
			(new RDI_NotifyConsumer(_admin_qos.numPushThreads));

  // Initialize all statistics-related structures

  _thread_stats = new RDI_ThreadStat[RDI_TH_ARRAY_SZ];
  RDI_Assert(_thread_stats, "Memory allocation failed - RDI_ThreadStat object");
  for (unsigned int id = 0; id < RDI_TH_ARRAY_SZ; id++) {
	_thread_stats[id]._rvm               = new RDI_RVM();
    	_thread_stats[id]._num_rdi_match     = 0;
    	_thread_stats[id]._num_rvm_eval      = 0;
    	_thread_stats[id]._num_announcements = 0;
    	_thread_stats[id]._num_notifications = 0;
    	_thread_stats[id]._qsize_acum        = 0;
    	_thread_stats[id]._qsize_ctr         = 0;
  }

  _prev_num_rdi_match = 0;       _prev_num_rvm_eval = 0;
  _prev_num_announcements = 0;   _prev_num_notifications = 0;
  _gq_acm    = 0; _gq_ctr = 0;   _pq_acm = 0; _pq_ctr = 0;
  _cum_msecs = 0; _second_delta = false;
  _gq_sleep_nanosecs = 0;
  _prev_avg_nq_sz = 0;
  _performance_timer.start();
  // the first time interval will be longer than others due to setup,
  // connection, etc.,  so it should be ignored  when computing stats
  _stat_update_counter = 0;
  _stat_delta_target   = RDI_STATS_DELTA_INCREMENT;

  // Create the ConsumerAdmin_i group manager
  _admin_group = new ChannelAdminGroup_m(_admin_qos.numAdminGroups, 
			                 _admin_qos.numAdminThreads);

  // Create the default consumer and supplier admin objects and 
  // insert them into the corresponding hash tables
  _def_consumer_admin = new ConsumerAdmin_i(this, ifop, def_admin_id);
  _cons_admin.insert(def_admin_id, _def_consumer_admin);
  _admin_group->insert(_def_consumer_admin);
  _def_supplier_admin = new SupplierAdmin_i(this, ifop, def_admin_id);
  _supl_admin.insert(def_admin_id, _def_supplier_admin);

  // Create the threads responsible for dispatching incoming events,
  // garbage collecting  the queue of announced events,  and pulling 
  // events from pull suppliers

  _rdi_dispatch = new EventChannelDispatch(this, 
		_admin_qos.numAdminThreads, &EventChannel_i::admin_dispatch, 
	        _admin_qos.numProxyThreads, &EventChannel_i::proxy_dispatch);

  _gcollector = new EventChannelWorker(this, &EventChannel_i::gcollect);
  RDI_Assert(_gcollector, "Memory allocation failed -- omni_thread");
  RDI_DUMP("garbage collection thread created -- ID " << _gcollector->id());
  _gcollector->start();
  _gcisactive = 1;

  // Register with the ORB to start receiving requests
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

CosN_AdminProperties* 
EventChannel_i::get_admin( WRAPPED_IMPLARG_VOID )
{
  omni_mutex_lock lock(_opera_lock);
  return _admin_qos.to_admin();
}

void EventChannel_i::set_admin(const CosN_AdminProperties& qos
			       WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq error;
  omni_mutex_lock lock(_opera_lock);
  if ( ! _admin_qos.validate(qos, error, 0) )
	throw CosN_UnsupportedAdmin(error);
  _admin_qos.from_admin(qos);
  // The maximum queue length may have been changed. The same may be
  // true for the pullEventPeriod -- propagate changes accordingly..
  _events->set_max_size(_admin_qos.maxQueueLength);
  if ( _pull_supplier )
	_pull_supplier->set_pull_period(_admin_qos.pullEventPeriod);
}

CosEventChannelAdmin::ConsumerAdmin_ptr 
EventChannel_i::for_consumers(WRAPPED_IMPLARG_VOID)
{ 
  omni_mutex_lock lock(_opera_lock);
  return WRAPPED_IMPL2OREF(CosEventChannelAdmin::ConsumerAdmin, _def_consumer_admin); 
}

CosEventChannelAdmin::SupplierAdmin_ptr 
EventChannel_i::for_suppliers(WRAPPED_IMPLARG_VOID)
{ 
  omni_mutex_lock lock(_opera_lock);
  return WRAPPED_IMPL2OREF(CosEventChannelAdmin::SupplierAdmin, _def_supplier_admin); 
}

CosNA_EventChannelFactory_ptr 
EventChannel_i::MyFactory(WRAPPED_IMPLARG_VOID)
{ 
  omni_mutex_lock lock(_opera_lock);
  return WRAPPED_IMPL2OREF(CosNA_EventChannelFactory, _my_channel_factory); 
}

CosNA_ConsumerAdmin_ptr 
EventChannel_i::default_consumer_admin(WRAPPED_IMPLARG_VOID)
{ 
  omni_mutex_lock lock(_opera_lock);
  return WRAPPED_IMPL2OREF(CosNA_ConsumerAdmin, _def_consumer_admin); 
}

CosNA_SupplierAdmin_ptr 
EventChannel_i::default_supplier_admin(WRAPPED_IMPLARG_VOID)
{ 
  omni_mutex_lock lock(_opera_lock);
  return WRAPPED_IMPL2OREF(CosNA_SupplierAdmin, _def_supplier_admin); 
}

CosNF_FilterFactory_ptr 
EventChannel_i::default_filter_factory(WRAPPED_IMPLARG_VOID)
{ 
  omni_mutex_lock lock(_opera_lock);
  return WRAPPED_IMPL2OREF(CosNF_FilterFactory, _def_filter_factory); 
}

CosNA_ConsumerAdmin_ptr 
EventChannel_i::new_for_consumers(CosNA_InterFilterGroupOperator op, 
				  CosNA_AdminID& id WRAPPED_IMPLARG)
{ 
  ConsumerAdmin_i* adm=0;
  _opera_lock.lock();
  id = _admin_serial;
  RDI_DUMP("Creating ConsumerAdmin -- Channel " << _serial);
  if ( ! (adm = new ConsumerAdmin_i(this, op, id)) ) {
	RDI_DUMP("\tfailed to create ConsumerAdmin object");
	_opera_lock.unlock();
	return CosNA_ConsumerAdmin::_nil();
  }
  if ( _cons_admin.insert(id, adm) ) {
	RDI_DUMP("\tfailed to register created ConsumerAdmin object");
	_opera_lock.unlock();
	delete adm;
	return CosNA_ConsumerAdmin::_nil();
  }
  _admin_group->insert(adm);
  _admin_serial  += 1;
  _num_consadmin += 1;
  _opera_lock.unlock();
  RDI_DUMP("\tcreated ConsumerAdmin [" << id << "] -- Channel " << _serial);
  return WRAPPED_IMPL2OREF(CosNA_ConsumerAdmin, adm);
}

CosNA_SupplierAdmin_ptr 
EventChannel_i::new_for_suppliers(CosNA_InterFilterGroupOperator op, 
				  CosNA_AdminID& id WRAPPED_IMPLARG)
{ 
  SupplierAdmin_i* adm=0;
  _opera_lock.lock();
  id = _admin_serial;
  if ( !(adm=new SupplierAdmin_i(this,op,id)) || _supl_admin.insert(id,adm) ) {
	_opera_lock.unlock();
        if ( adm ) delete adm;
	return CosNA_SupplierAdmin::_nil();
  }
  _admin_serial  += 1;
  _num_suppadmin += 1;
  _opera_lock.unlock();
  RDI_DUMP("Created SupplierAdmin [" << id << "] -- Channel " << _serial);
  return WRAPPED_IMPL2OREF(CosNA_SupplierAdmin, adm);
}

CosNA_ConsumerAdmin_ptr 
EventChannel_i::get_consumeradmin(CosNA_AdminID id WRAPPED_IMPLARG)
{ 
  ConsumerAdmin_i* admin = 0;
  if ( id == (CORBA::Long)-999 ) {  // XXX HACK
	RDI_DUMP("\nXXX DUMP_STATS_HACK\n");
    	dump_stats();
    	return CosNA_ConsumerAdmin::_nil();
  }
  _opera_lock.lock();
  if ( _cons_admin.lookup(id, admin) ) {
	_opera_lock.unlock();
	return WRAPPED_IMPL2OREF(CosNA_ConsumerAdmin, admin);
  }
  _opera_lock.unlock();
  throw CosNA_AdminNotFound();
}

CosNA_SupplierAdmin_ptr 
EventChannel_i::get_supplieradmin(CosNA_AdminID id WRAPPED_IMPLARG)
{ 
  SupplierAdmin_i* admin = 0;
  _opera_lock.lock();
  if ( _supl_admin.lookup(id, admin) ) {
	_opera_lock.unlock();
	return WRAPPED_IMPL2OREF(CosNA_SupplierAdmin, admin);
  }
  _opera_lock.unlock();
  throw CosNA_AdminNotFound();
}

CosNA_AdminIDSeq * EventChannel_i::get_all_consumeradmins(WRAPPED_IMPLARG_VOID)
{ 
  RDI_HashCursor<CosNA_AdminID, ConsumerAdmin_i *> c;
  unsigned int idx = 0;
  CosNA_AdminIDSeq* aseq = new CosNA_AdminIDSeq;
  if ( aseq == (CosNA_AdminIDSeq *) 0 )
	return 0;
  _opera_lock.lock();
  aseq->length(_cons_admin.length());
  for (c = _cons_admin.cursor(); c.is_valid(); ++c)
	(*aseq)[idx++] = c.key();
  _opera_lock.unlock();
  return aseq;
}

CosNA_AdminIDSeq * EventChannel_i::get_all_supplieradmins(WRAPPED_IMPLARG_VOID)
{ 
  RDI_HashCursor<CosNA_AdminID, SupplierAdmin_i *> c;
  unsigned int idx = 0;
  CosNA_AdminIDSeq* aseq = new CosNA_AdminIDSeq;
  if ( aseq == (CosNA_AdminIDSeq *) 0 )
        return 0;
  _opera_lock.lock();
  aseq->length(_supl_admin.length());
  for (c = _supl_admin.cursor(); c.is_valid(); ++c)
        (*aseq)[idx++] = c.key();
  _opera_lock.unlock();
  return aseq;
}

CosN_QoSProperties* EventChannel_i::get_qos(WRAPPED_IMPLARG_VOID)
{
  omni_mutex_lock lock(_opera_lock);
  return _qosprop->get_qos(RDI_ECHANNEL);
}

void EventChannel_i::set_qos(const CosN_QoSProperties& r_qos   
			     WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  CosN_NamedPropertyRangeSeq rseq;
  omni_mutex_lock lock(_opera_lock);
  CORBA::Boolean  child = 1;	// Channel always has default admins

  if (! RDI_NotifQoS::validate(r_qos,*_qosprop,RDI_ECHANNEL,eseq,rseq,child))
        throw CosN_UnsupportedQoS(eseq);
  _qosprop->set_qos(r_qos);
}

void EventChannel_i::validate_qos(
        const CosN_QoSProperties& r_qos,
	WRAPPED_OUTARG_TYPE(CosNotification::NamedPropertyRangeSeq) rseq
	WRAPPED_IMPLARG )
{
  CosN_PropertyErrorSeq eseq;
  omni_mutex_lock lock(_opera_lock);
  CORBA::Boolean  child = 1;    // Channel always has default admins

  rseq = new CosNotification::NamedPropertyRangeSeq();
  RDI_Assert(rseq, "Memory allocation failed - NamedPropertyRangeSeq");
  if (! RDI_NotifQoS::validate(r_qos,*_qosprop,RDI_ECHANNEL,eseq,*rseq,child))
        throw CosN_UnsupportedQoS(eseq);
}

void EventChannel_i::destroy(WRAPPED_IMPLARG_VOID)
{
  RDI_HashCursor<CosNA_AdminID, SupplierAdmin_i *> scur;
  RDI_HashCursor<CosNA_AdminID, ConsumerAdmin_i *> ccur;

  RDI_DUMP("Event Channel " << _serial << " is being destroyed");
  RDI_DUMP("** signal garbage collection thread for " << _serial);
  _shutmedown = 1;
  _going_down.signal();
  while ( _gcisactive ) {
	omni_thread::yield();
  	_going_down.signal();
  }
  _opera_lock.lock();
  _my_channel_factory->remove_channel(_serial);

  // First, destroy all consumer and supplier objects that have been
  // created so far. Their destruction will disconnect any connected
  // clients and destroy all created proxies

  RDI_DUMP("** destroy all SupplierAdmin objects for " << _serial);
  for (scur = _supl_admin.cursor(); scur.is_valid(); scur++)
        scur.val()->disconnect_clients_and_dispose();
  _supl_admin.clear();
  _def_supplier_admin=0;
  RDI_DUMP("** destroy all ConsumerAdmin objects for " << _serial);
  for (ccur = _cons_admin.cursor(); ccur.is_valid(); ccur++)
        ccur.val()->disconnect_clients_and_dispose();
  _cons_admin.clear();
  _def_consumer_admin=0;

  // Destroy all support objects that have been created -- Note that
  // we MUST destroy the event queue first so that any admin threads
  // that are blocked waiting for events are notified about it .....

  RDI_DUMP("** destroy global Event Queue for " << _serial);
  if ( _events ) delete _events; _events = 0;
  RDI_DUMP("** destroy filter evaluation objects for " << _serial);
  // We have to notify any blocked threads waiting for proxy-level
  // filter evalaution about the shut down
  _proxy_emty.broadcast();
  if ( _rdi_dispatch ) delete _rdi_dispatch; _rdi_dispatch = 0;
  RDI_DUMP("** destroy event push thread pool for " << _serial);
  if ( _push_consumer ) delete _push_consumer; _push_consumer = 0;
  RDI_DUMP("** destroy event pull thread pool for " << _serial);
  if ( _pull_supplier ) delete _pull_supplier; _pull_supplier = 0;
  RDI_DUMP("** destroy all administrative groups for " << _serial);
  if ( _admin_group ) delete _admin_group; _admin_group = 0;
  RDI_DUMP("** destroy event type map object for " << _serial);
  if ( _type_map ) delete _type_map; _type_map = 0;
  if ( _qosprop )  delete _qosprop;  _qosprop  = 0;

  for (unsigned int id = 0; id < RDI_TH_ARRAY_SZ; id++) {
	if ( _thread_stats && _thread_stats[id]._rvm )
		delete _thread_stats[id]._rvm;
  }
  if ( _thread_stats )  delete [] _thread_stats; _thread_stats = 0;

  if ( _def_filter_factory )
    	WRAPPED_DISPOSE_IMPL(_def_filter_factory);
  _opera_lock.unlock();
  WRAPPED_DISPOSE_IMPL(this);
}

void EventChannel_i::unregister(SupplierAdmin_i* suplAdmin, 
			        CORBA::Boolean   holdslock)
{
  if ( ! holdslock )
  	_opera_lock.lock();
  if ( _supl_admin.exists( suplAdmin->MyID() ) ) {
	_supl_admin.remove( suplAdmin->MyID() );
	_num_suppadmin -= 1;
  } else {
	RDI_DUMP("Invalid SupplierAdmin_i ["<<suplAdmin->MyID()<<"] given");
  }
  if ( ! holdslock ) 
  	_opera_lock.unlock();
}

void EventChannel_i::unregister(ConsumerAdmin_i* consAdmin,
				CORBA::Boolean   holdslock)
{
  // We *must* remove 'consAdmin' from the consumer group manager first to
  // avoid a deadlock in the current implementation of ChannelAdminGroup.
  // This can happen because '_admin_group->remove()' may be blocked until
  // the dispatch thread for  'consAdmin' finishes an iteration.  However,
  // this may not happen since the lock on some other ConsumerAdmin object
  // in the same group may be help due to the destruction of a proxy. This
  // proxy destruction needs to acquire  the channel lock to decrement the
  // number of consumers -- SOLUTION: need to modify the ChannelAdminGroup 
  // implementation --

  _admin_group->remove(consAdmin);

  if ( ! holdslock )
  	_opera_lock.lock();
  if ( _cons_admin.exists( consAdmin->MyID() ) ) {
	_cons_admin.remove( consAdmin->MyID() );
	_num_consadmin -= 1;
  } else {
	RDI_DUMP("Invalid ConsumerAdmin_i ["<<consAdmin->MyID()<<"] given");
  }
  if ( ! holdslock ) 
  	_opera_lock.unlock();
}

// ************************************************************* //
// Increment/Decrement the number of consumer/suppliers that are //
// connected to the channel at any particular point in time      //
// ************************************************************* //

bool EventChannel_i::incr_consumers()
{
  _opera_lock.lock();
  if ( _admin_qos.maxConsumers && 
       ((CORBA::Long)_num_consumers >= _admin_qos.maxConsumers)) {
	_opera_lock.unlock();
	return 0;
  }
  _num_consumers += 1;
  _opera_lock.unlock();
  return 1;
}

void EventChannel_i::decr_consumers()
{ _opera_lock.lock(); _num_consumers -= 1; _opera_lock.unlock(); }

bool EventChannel_i::incr_suppliers()
{
  _opera_lock.lock();
  if (_admin_qos.maxSuppliers && ((CORBA::Long)_num_suppliers >= _admin_qos.maxSuppliers)) {
	_opera_lock.unlock();
	return 0;
  }
  _num_suppliers += 1;
  _opera_lock.unlock();
  return 1;
}

void EventChannel_i::decr_suppliers()
{ _opera_lock.lock(); _num_suppliers -= 1; _opera_lock.unlock(); }

AttNotifyChannelAdmin::PerformanceStats 
EventChannel_i::obtain_performance_stats()
{
  AttNotifyChannelAdmin::PerformanceStats stats;
  stats.num_announcements = 0;
  stats.num_notifications = 0;
  RDI_STATS_LOCK;
  for (unsigned int id = 0; id < RDI_TH_ARRAY_SZ; id++) {
	RDI_THREAD_STATS_LOCK(id);
	stats.num_announcements += _thread_stats[id]._num_announcements;
	stats.num_notifications += _thread_stats[id]._num_notifications;
	RDI_THREAD_STATS_UNLOCK(id);
  }
  RDI_STATS_UNLOCK;
  return stats;
}

// ************************************************************* //
//                   dump_stats                                  //
// ************************************************************* //

void EventChannel_i::dump_stats(bool getlock, bool onstdout) 
{
  unsigned int msecs = 0;
  int thread_e = 0, thread_n = 0, thread_a = 0, thread_m = 0;
  CORBA::ULong num_e = 0, num_n = 0, num_a = 0, num_m = 0;
  CORBA::ULong nq_acm = 0, nq_ctr = 0;
  CORBA::ULong delta_e, delta_n, delta_a, delta_m;
  double avg_nq_sz = -1, avg_gq_sz = -1, avg_pq_sz = -1;

  if (getlock) RDI_STATS_LOCK;

  for (unsigned int id = 0; id < RDI_TH_ARRAY_SZ; id++) {
    RDI_THREAD_STATS_LOCK(id);
    num_n += _thread_stats[id]._num_notifications;
    num_a += _thread_stats[id]._num_announcements;
    num_m += _thread_stats[id]._num_rdi_match;
    num_e += _thread_stats[id]._num_rvm_eval;
    nq_acm += _thread_stats[id]._qsize_acum;
    nq_ctr += _thread_stats[id]._qsize_ctr;
    _thread_stats[id]._qsize_acum = 0;
    _thread_stats[id]._qsize_ctr = 0;
    if (_thread_stats[id]._num_notifications > 0) thread_n++;
    if (_thread_stats[id]._num_announcements > 0) thread_a++;
    if (_thread_stats[id]._num_rdi_match > 0)     thread_m++;
    if (_thread_stats[id]._num_rvm_eval > 0)      thread_e++;
    RDI_THREAD_STATS_UNLOCK(id);
  }

  _performance_timer.stop();

// unit is nanosecs
#define RDI_1_US_AS_NS        1000
#define RDI_pt01_MS_AS_NS    10000
#define RDI_pt1_MS_AS_NS    100000
#define RDI_pt25_MS_AS_NS   250000
#define RDI_pt5_MS_AS_NS    500000
#define RDI_1_MS_AS_NS     1000000
#define RDI_2_MS_AS_NS     2000000
#define RDI_4_MS_AS_NS     4000000
#define RDI_100_MS_AS_NS 100000000

  if ( nq_ctr ) {
    avg_nq_sz = (double)nq_acm/(double)nq_ctr;

    if ( (avg_nq_sz > 40) && (avg_nq_sz > _prev_avg_nq_sz) && 
         (_gq_sleep_nanosecs < RDI_100_MS_AS_NS)) {
      _gq_sleep_nanosecs += 
	(unsigned long) (RDI_pt01_MS_AS_NS * (avg_nq_sz - _prev_avg_nq_sz));
      if (_gq_sleep_nanosecs > RDI_100_MS_AS_NS)
	_gq_sleep_nanosecs = RDI_100_MS_AS_NS;
	// cout<<"GQ_SLEEP_NANOSECS= "<<_gq_sleep_nanosecs<<" nanosecs"<<endl;
    } else if ( (avg_nq_sz < 10) && (avg_nq_sz <= _prev_avg_nq_sz) && 
		(_gq_sleep_nanosecs > 0) ) {
      unsigned long deltns = 
	(unsigned long) (RDI_pt01_MS_AS_NS * (_prev_avg_nq_sz - avg_nq_sz));
      if ( !deltns || deltns >= _gq_sleep_nanosecs) {
	_gq_sleep_nanosecs = 0;
      } else {
	_gq_sleep_nanosecs -= deltns;
      }
      // cout<<"GQ_SLEEP_NANOSECS= "<<_gq_sleep_nanosecs<<" nanosecs"<<endl;
    }
    _prev_avg_nq_sz = avg_nq_sz;
  }
  if (_gq_ctr) {
    avg_gq_sz = (double)_gq_acm/(double)_gq_ctr;
  }
  if (_pq_ctr) {
    avg_pq_sz = (double)_pq_acm/(double)_pq_ctr;
  }
  _gq_acm = 0;  _gq_ctr = 0;
  _pq_acm = 0;  _pq_ctr = 0;

  delta_n = num_n - _prev_num_notifications;
  delta_a = num_a - _prev_num_announcements;
  delta_e = num_e - _prev_num_rvm_eval;
  delta_m = num_m - _prev_num_rdi_match;

  _prev_num_notifications = num_n;
  _prev_num_announcements = num_a;
  _prev_num_rvm_eval      = num_e;
  _prev_num_rdi_match     = num_m;

  msecs = _performance_timer.millisecs();
  if (_cum_msecs == 0) {
    _cum_msecs = msecs;
    _second_delta = true;
  } else if (_second_delta) {
    _cum_msecs = msecs * 2;
    _second_delta = false;
  } else {
    _cum_msecs += msecs;
  }

  if ( onstdout ) {
     cout << endl;
     cout << "Channel: " << _serial << " queue sizes: global " << avg_gq_sz <<
	     " proxy " << avg_pq_sz << " notif " << avg_nq_sz << endl;
     cout << "\tDELTA: #announ " << delta_a << " #notifs " << delta_n << 
	     " #rdi_match " << delta_m << " #rvm_evals " << delta_e << endl <<
	     "\t       time " << msecs << " msecs : " << 
	     (double)delta_a/((double) msecs/1000) << " a/sec " << 
	     (double)delta_n/((double) msecs/1000) << " n/sec " << 
	     (double)(delta_a+delta_n)/((double) msecs/1000) << " a+n/sec" <<
	     endl;
     cout << "\tCUM:   #announ " << num_a << " " << "#notifs " << num_n <<
              " #rdi_match " << num_m << " " << "#rvm_evals " << num_e << 
	      endl << "\t       time " << (unsigned long)_cum_msecs <<
	      " msecs : " <<
              (double)num_a/((double) _cum_msecs/1000) << " a/sec " << 
	      (double)num_n/((double) _cum_msecs/1000) << " n/sec  " << 
	      (double)(num_a+num_n)/((double) _cum_msecs/1000) << " a+n/sec" <<
	      endl;
  } else {
     RDI_DUMP("\nXXX-NUM_UNIQUE_THREAD_IDS " << "for_announ " << thread_a << " " << "for_notif " << thread_n << " " << "rdi_match " << thread_m << " " << "rvm_evals " << thread_e); 
     RDI_DUMP("XXX-SAMPLED_QUEUE_SIZES" << " glob_arriv_Q " << avg_gq_sz << " proxy_work_Q " << avg_pq_sz << " avg_notif_Q  " << avg_nq_sz);
     RDI_DUMP("XXX-DELTA " << "#announ " << delta_a << " " << "#notifs " << delta_n << " " << "#announ+notifs " << (delta_a + delta_n) << " " << "#rdi_match " << delta_m << " " << "#rvm_evals " << delta_e << " " << "time " << msecs << " msecs : " << (double)delta_a/((double) msecs/1000) << " a/sec " << (double)delta_n/((double) msecs/1000) << " n/sec " << (double)(delta_a+delta_n)/((double) msecs/1000) << " a+n/sec");
     RDI_DUMP("XXX-CUM   " << "#announ " << num_a << " " << "#notifs " << num_n << " " << "#announ+notif " << num_a+num_n << " " << "#rdi_match " << num_m << " " << "#rvm_evals " << num_e << " " << "time " << (unsigned long)_cum_msecs << " msecs : " << (double)num_a/((double) _cum_msecs/1000) << " a/sec " << (double)num_n/((double) _cum_msecs/1000) << " n/sec  " << (double)(num_a+num_n)/((double) _cum_msecs/1000) << " a+n/sec");
  }
  _performance_timer.start();
  if (getlock) RDI_STATS_UNLOCK;
}

// ************************************************************* //
// A new structured event is announced to the channel via one of //
// the proxies. Insert the event in the queue of events and then //
// signal the dispatcher thread, given that the queue was empty. //
// ************************************************************* //

#define RDI_NQ_SIZE_CHECK \
if (_gq_sleep_nanosecs) omni_thread::sleep(0, _gq_sleep_nanosecs)

int EventChannel_i::new_any_event(const CORBA::Any& event)
{
  RDI_StructuredEvent* sevnt = new RDI_StructuredEvent(event);
  RDI_Assert(sevnt, "Memory allocation failure -- RDI_StructuredEvent");
  if ( _events->insert(sevnt) ) 
	return -1;
  incr_num_announcements();
  RDI_NQ_SIZE_CHECK;
  return 0;
}

int EventChannel_i::new_structured_event(const CosN_StructuredEvent& evnt)
{
  RDI_StructuredEvent* sevnt = new RDI_StructuredEvent(evnt);
  RDI_Assert(sevnt, "Memory allocation failure -- RDI_StructuredEvent");
  if ( _events->insert(sevnt) )
	return -1;
  incr_num_announcements();
  RDI_NQ_SIZE_CHECK;
  return 0;
}

int EventChannel_i::new_structured_event(RDI_StructuredEvent* sevnt)
{
  if ( _events->insert(sevnt) )
	return -1;
  incr_num_announcements();
  RDI_NQ_SIZE_CHECK;
  return 0;
}

int EventChannel_i::new_sequence_event(const CosN_EventBatch& events)
{
  for (unsigned int n=0; n < events.length(); n++) {
	RDI_StructuredEvent* sevnt = new RDI_StructuredEvent(events[n]);
	RDI_Assert(sevnt, "Memory allocation failure -- RDI_StructuredEvent");
	if ( _events->insert(sevnt) )
		return -1;
	incr_num_announcements();
	RDI_NQ_SIZE_CHECK;
  }
  return 0;
}

// ************************************************************* //
// Update the current event type registrations for ConsumerAdmin //
// and ProxySupplier objects                                     //
// ************************************************************* //

bool EventChannel_i::update_mapping(const CosN_EventTypeSeq& added,
				    const CosN_EventTypeSeq& deled,
				    ConsumerAdmin_i* admin, Filter_i* filter)
{ return _type_map->update(added, deled, admin, filter); }


// XXX USED TO BE: CosNA_ProxySupplier_ptr proxy
bool EventChannel_i::update_mapping(const CosN_EventTypeSeq& added,
				    const CosN_EventTypeSeq& deled,
				    RDI_DummyProxy_i* proxy, Filter_i* filter)
{ return _type_map->update(added, deled, proxy, filter); }

// ************************************************************* //
// Update the hash table that keeps information about the event  //
// types supplied by suppliers.  When a new entry is created or  //
// an existing entry is deleted, notify the connected consumers  //
// ************************************************************* //

void EventChannel_i::offer_change(const CosN_EventTypeSeq& added,
				    const CosN_EventTypeSeq& deled)
{
  CORBA::ULong ix=0, sz=0, vl=0;
  CosN_EventTypeSeq new_added;
  CosN_EventTypeSeq old_deled;
  omni_mutex_lock lock(_opera_lock);

  new_added.length(0);
  old_deled.length(0);

  for (sz=0, ix = 0; ix < added.length(); ix++) {
     RDI_DUMP("+ " << added[ix].domain_name << "::" << added[ix].type_name);
     if ( _evtypes.lookup(added[ix], vl) ) {
        vl += 1; _evtypes.replace(added[ix], vl);
     } else {
        vl  = 1; _evtypes.insert(added[ix], vl);
        new_added.length(sz + 1);
        new_added[sz].domain_name = added[ix].domain_name;
        new_added[sz++].type_name = added[ix].type_name;
     }
  }
  for (sz=0, ix = 0; ix < deled.length(); ix++) {
     RDI_DUMP("- " << deled[ix].domain_name << "::" << deled[ix].type_name);
     if ( _evtypes.lookup(deled[ix], vl) ) {
        if ( vl == 1 ) {
           _evtypes.remove(deled[ix]);
           old_deled.length(sz + 1);
	   old_deled[sz].domain_name = deled[ix].domain_name;
	   old_deled[sz++].type_name = deled[ix].type_name;
        } else {
           vl -= 1; _evtypes.replace(deled[ix], vl);
        }
     } else {
        RDI_DUMP("Invalid "<<deled[ix].domain_name<<"::"<<deled[ix].type_name);
     }
  }
  // If any new event types were added or existing event types were
  // deleted, have all ConsumerAdmin objects notify their consumers
  if ( new_added.length() || old_deled.length() ) {
     RDI_HashCursor<CosNA_AdminID, ConsumerAdmin_i *> curs;
     for ( curs = _cons_admin.cursor(); curs.is_valid(); ++curs ) {
	if ( curs.val()->NumProxies() ) 
	   curs.val()->offer_change(new_added, old_deled);
     }
  }
}

CosN_EventTypeSeq* EventChannel_i::obtain_offered_types( )
{
  RDI_HashCursor<CosN_EventType, CORBA::ULong> curs;
  CosN_EventTypeSeq* etypeseq = new CosN_EventTypeSeq;
  RDI_Assert(etypeseq, "Memory allocation failed -- EventTypeSeq");
  CORBA::ULong    indx=0;
  omni_mutex_lock lock(_opera_lock);
  etypeseq->length(_evtypes.length());
  for ( indx=0, curs = _evtypes.cursor(); curs.is_valid(); ++indx, ++curs )
	(*etypeseq)[indx] = curs.key();
  return etypeseq;
}

// ************************************************************* //
// There have been some changes in the event types referenced in //
// consumers filters. Notify all connected suppliers about this. //
// [This method is invoked by the RDI_TypeMap module]            //
// ************************************************************* //

void EventChannel_i::subscription_change(
				const CosN_EventTypeSeq& added,
				const CosN_EventTypeSeq& deled)
{
  RDI_HashCursor<CosNA_AdminID, SupplierAdmin_i *> scur;
  // If we are being shut down, we skip this operation to avoid 
  // deadlocks since we are already holding the channel lock ..
  if ( _shutmedown )
	return;
  _opera_lock.lock();
  for (scur = _supl_admin.cursor(); scur.is_valid(); ++scur)
     scur.val()->subscription_change(added, deled);
  _opera_lock.unlock();
}

CosN_EventTypeSeq* EventChannel_i::obtain_subscription_types()
{ 
  return _type_map->obtain_subscription_types(); 
}

// ************************************************************* //
// The following is executed by threads that carry out filtering //
// on behalf of ConsumerAdmin objects.                           //
// ************************************************************* //

void EventChannel_i::admin_dispatch()
{
  CORBA::Boolean        adone = 0;
  RDI_StructuredEvent*  pevnt = 0;
  RDI_StructuredEvent*  sevnt = 0;
  ConsumerAdmin_i*      admin = 0;
  ChannelAdminGroup*    group = 0;
  RDI_FilterState_t     fstat;
  ProxyDispatch_t       pxdis;
  unsigned int ix, mylo=0, myhi=0;

  adone = _admin_group->allocate_range(mylo, myhi);
  RDI_Assert(adone, "Failed to allocated group range to work on");
  RDI_DUMP("Thread "<<omni_thread::self()->id()<<" L: "<<mylo<<" H: "<<myhi);

  while ( 1 ) {
     // Wait until an event becomes available. Since we are blocked 
     // waiting for an event, if we get a NULL event back we should
     // be in the process of terminating execution.  
     if ( (sevnt=_events->next_event(pevnt, 1)) == (RDI_StructuredEvent *) 0 ) {
	if ( _shutmedown ) {	// Channel is being destroyed
	   int tid = omni_thread::self()->id();
	   RDI_DUMP("   - ADispatch thread "<<tid<<" for "<<_serial<<" exits");
	   omni_thread::exit();
	} else {
	   RDI_Fatal("Internal error -- NULL event");
	}
     }
     sevnt->set_state(RDI_StructuredEvent::DISPATCHED);
     pevnt = sevnt;

     for ( ix = mylo; ix <= myhi; ix++ ) {
	group = (*_admin_group)[ix];
       	while ( (admin = group->next()) ) {
   	   admin->_oplock.lock();
	   if ( admin->NumProxies() == 0 ) {
	      admin->_oplock.unlock();
	      omni_thread::yield();
	      continue;
	   }
	   // Dispatch the event to all proxies based on the CORBA Event
 	   // Service -- no filtering is performed in this case
	   admin->dispatch_event(sevnt);

	   if ( this->match_event(admin, sevnt, fstat) ) {
	      if ( _admin_qos.numProxyThreads == 0 ) {
		 // We are responsible for dispatching this event to all
	 	 // proxies of this ConsumerAdmin object; we do not need
		 // to increment the reference counter of the event here
		 // since it is <> 1 due to the 'next_event()' semantics 
	         admin->dispatch_event(sevnt, fstat, _type_map);
		 admin->_oplock.unlock();
	      } else {
		 admin->_oplock.unlock();
		 pxdis._event = sevnt;
		 pxdis._admin = admin;
		 pxdis._state = fstat;
	         sevnt->incr_ref_counter( RDI::lockEvent() );
		 _proxy_lock.lock();
		 _proxy_events.insert_tail(pxdis);
		 _proxy_emty.broadcast();
		 _proxy_lock.unlock();
	      }
	   } else {
	      admin->_oplock.unlock();
	   }
	   omni_thread::yield();
	}
     }
  }
}

// ************************************************************* //
// The following is executed by threads that carry out filtering //
// on behalf of ProxySupplier objects.                           //
// ************************************************************* //

void EventChannel_i::proxy_dispatch()
{
  ProxyDispatch_t pxdis;
  unsigned int    numev=0;

  while ( 1 ) {
     _proxy_lock.lock();
     while ( ! _shutmedown && ! _proxy_events.length() ) {
	numev = 0; _proxy_emty.wait();
     }
     if ( _shutmedown ) {	// Channel is being destroyed
	int tid = omni_thread::self()->id();
	RDI_DUMP("   - PDispatch thread "<<tid<<" for "<<_serial<<" exits");
	_proxy_lock.unlock();
	omni_thread::exit();
     }

     pxdis = _proxy_events.get_head();
     _proxy_events.remove_head();
     // Before releasing the lock on the proxy queue,  we need to 
     // grab the lock on the admin to prevent another thread from
     // dispatching an event to the same admin out of order......
     pxdis._admin->_oplock.lock();
     _proxy_lock.unlock();
     pxdis._admin->dispatch_event(pxdis._event, pxdis._state, _type_map);
     pxdis._admin->_oplock.unlock();
     // Decrement reference counter -- was incremented when event
     // was inserted into the proxy queue
     pxdis._event->decr_ref_counter( RDI::lockEvent() );
     pxdis._event = 0; pxdis._admin = 0;
     if ( ++numev >= 10 ) {
	numev = 0;
     	omni_thread::yield();
     }
  }
}

// ************************************************************* //
// Does any of the filters, if any,  of a ConsumerAdmin_i object //
// match a structured event having the provided type?            //
//								 //
// NOTE: all appropriate locks must be acquired by the caller of //
//       this method                                             //
// ************************************************************* //

CORBA::Boolean EventChannel_i::match_event(ConsumerAdmin_i*     admin,
					   RDI_StructuredEvent* event,
					   RDI_FilterState_t&   fstat)
{
  RDI_TypeMap::FList_t flist;
  unsigned char match_found = 0;
  RDI_TypeMap::FNode_t* fnode = 0;

  if ( ! admin->has_filters() ) {	// No filters -- success
	fstat = NoFilters;
	return  1;
  }

  const char* dname = event->get_domain_name();
  const char* tname = event->get_type_name();

  _type_map->lookup_begin(dname, tname, admin, flist);

  if ( ! flist._star_star && ! flist._domn_star && 
       ! flist._star_type && ! flist._domn_type ) {
	_type_map->lookup_end();
	if ( admin->MyOperator() == CosNA_OR_OP ) {
		fstat = OrFailed;
		return 1;
	}
	return 0;
  }

  for ( fnode=flist._star_star; fnode; fnode=fnode->_next ) {
	if ( fnode->_fltr->rdi_match(event, this) ) 
	{ match_found = 1; break; }
  }
  if ( ! match_found ) {
	for ( fnode=flist._domn_star; fnode; fnode=fnode->_next ) {
		if ( fnode->_fltr->rdi_match(event, this) ) 
		{ match_found = 1; break; }
	}
  } 
  if ( ! match_found ) {
	for ( fnode=flist._star_type; fnode; fnode=fnode->_next ) {
		if ( fnode->_fltr->rdi_match(event, this) ) 
		{ match_found = 1; break; }
        }
  } 
  if ( ! match_found ) {
	for ( fnode=flist._domn_type; fnode; fnode=fnode->_next ) {
		if ( fnode->_fltr->rdi_match(event, this) ) 
		{ match_found = 1; break; }
        }
  } 

  _type_map->lookup_end();

  if ( match_found ) {
	fstat = (admin->MyOperator() == CosNA_OR_OP) ?
		 		                     OrMatch : AndMatch;
	return 1; 
  } else if ( admin->MyOperator() == CosNA_OR_OP ) {
	fstat = OrFailed;
	return 1;
  }

  return 0;
}

// ************************************************************* //

void EventChannel_i::gcollect()
{
  RDI_HashCursor<CosNA_AdminID, ConsumerAdmin_i *> ccur;
  RDI_HashCursor<CosNA_AdminID, SupplierAdmin_i *> scur;
  RDI_List<ConsumerAdmin_i *> clst;
  RDI_ListCursor<ConsumerAdmin_i *> clst_c;
  RDI_List<SupplierAdmin_i *> slst;
  RDI_ListCursor<SupplierAdmin_i *> slst_c;
  ConsumerAdmin_i* cadm = 0;
  SupplierAdmin_i* sadm = 0;
  unsigned int  cntr;
  int thrid = omni_thread::self()->id();

  while ( 1 ) {
     RDI_TimeValue delt(300);
     RDI_TimeValue tnow;
     unsigned long nsec, usec;

     _opera_lock.lock();
     if ( _shutmedown ) {
	RDI_DUMP("   - GC thread " << thrid << " for " << _serial << " exits");
	_gcisactive = 0;
        _opera_lock.unlock();
	omni_thread::exit();
     }
     omni_thread::get_time(&nsec, &usec, 180, 0);
     _going_down.timedwait(nsec, usec);

     if ( _shutmedown ) {
	RDI_DUMP("   - GC thread " << thrid << " for " << _serial << " exits");
	_gcisactive = 0;
	_opera_lock.unlock();
	omni_thread::exit();
     }

     // Go through ConsumerAdmin_i and SupplierAdmin_i objects and
     // garbage collect those that do not have any proxies and the
     // last proxy disconnect is at least 5 minutes old -- Default
     // administrative objects are NOT garbage collected

     for ( ccur = _cons_admin.cursor(); ccur.is_valid(); ++ccur ) {
        cadm = ccur.val();
        tnow = RDI_TimeValue::timeofday();
        if ( (cadm->MyID() != 0) && (cadm->NumProxies() == 0) &&
             (RDI_TimeValue::delta(tnow, cadm->LastProxy()) > delt) )
                clst.insert_tail(cadm);
     }
     for ( scur = _supl_admin.cursor(); scur.is_valid(); ++scur ) {
        sadm = scur.val();
        tnow = RDI_TimeValue::timeofday();
        if ( (sadm->MyID() != 0) && (sadm->NumProxies() == 0) &&
             (RDI_TimeValue::delta(tnow, sadm->LastProxy()) > delt) )
                slst.insert_tail(sadm);
     }

     // First, unregister all objects from the event channel, release
     // '_opera_lock',  and invoke 'disconnect_clients_and_dispose()'
     // on them.  We must release the mutex because filters which are
     // associated with these admins are removed and this may trigger
     // 'subscription_change()' to be called, which acqires the mutex
     // on the channel

     clst_c = clst.cursor();
     for (cntr=0; cntr < clst.length(); cntr++, clst_c++) {
        cadm = *clst_c;
        unregister(cadm, 1);
     }
     slst_c = slst.cursor();
     for (cntr=0; cntr < slst.length(); cntr++, slst_c++) {
        sadm = *slst_c;
        unregister(sadm, 1);
     }

     _opera_lock.unlock();

     while ( clst.length() != 0 ) {
        cadm = clst.get_head();
        clst.remove_head();
        cadm->disconnect_clients_and_dispose();
     }
     clst.drain();
     while ( slst.length() != 0 ) {
        sadm = slst.get_head();
        slst.remove_head();
        sadm->disconnect_clients_and_dispose();
     }
     slst.drain();
  }
}

ostream& operator << (ostream& out, EventChannel_i& evc)
{
  RDI_HashCursor<CosNA_AdminID, SupplierAdmin_i *> suplcur;
  RDI_HashCursor<CosNA_AdminID, ConsumerAdmin_i *> conscur;
  evc._opera_lock.lock();
  out << endl << "Event Channel: "   << setw(2) << evc._serial << 
	 endl << "=-=-=-=-=-=-=-=-=" << endl << endl;
  out << *(evc._qosprop) << endl << endl;
  out << evc._admin_qos << endl;
  evc._events->display_stats();
  out << endl << *(evc._admin_group);
  out << endl << "SupplierAdmin_i" << endl << "---------------" << endl;
  for ( suplcur = evc._supl_admin.cursor(); suplcur.is_valid(); ++suplcur)
	out << *suplcur.val() << endl;
  out << "ConsumerAdmin_i" << endl << "---------------" << endl;
  for ( conscur = evc._cons_admin.cursor(); conscur.is_valid(); ++conscur)
        out << *conscur.val() << endl;
  evc._type_map->debug(out);
  evc._opera_lock.unlock();
  return out;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //
//                         EventChannelFactory_i                         //
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= //

EventChannelFactory_i::EventChannelFactory_i() :
		_oplock(), _serial(0), _defqos(),
		_defadm(), _channel(RDI_ULongHash, RDI_ULongRank)
{
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

EventChannelFactory_i::EventChannelFactory_i(const RDI_NotifQoS& defqos,
					     const RDI_AdminQoS& defadm) :
		WRAPPED_SUPER(CosNA::, EventChannelFactory), 
		_oplock(), _serial(0), _defqos(defqos), 
		_defadm(defadm), _channel(RDI_ULongHash, RDI_ULongRank)
{
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

EventChannelFactory_i::EventChannelFactory_i(const WRAPPED_OBJKEY_TYPE& key,
					     const RDI_NotifQoS& defqos,
					     const RDI_AdminQoS& defadm) :
		WRAPPED_SUPER_KEY(CosNA::, EventChannelFactory, key),
		_oplock(), _serial(0), _defqos(defqos), 
		_defadm(defadm), _channel(RDI_ULongHash, RDI_ULongRank)
{
  WRAPPED_BOA_OBJ_IS_READY(CORBA::BOA::getBOA(), this);
}

EventChannelFactory_i::~EventChannelFactory_i()
{
  omni_mutex_lock lock(_oplock);
  _channel.clear();
}

CosNA_EventChannel_ptr
EventChannelFactory_i::create_channel(const CosN_QoSProperties&   qosP,
                                      const CosN_AdminProperties& admP,
                                      CosNA_ChannelID& myID  WRAPPED_IMPLARG)
{
  CosN_NamedPropertyRangeSeq pran;
  CosN_PropertyErrorSeq      perr;
  CORBA::Boolean  child=1;    // Event Channel always has default Admins
  EventChannel_i* chann=0;
  omni_mutex_lock lock(_oplock);

  // Validate the requested QoS based on the '_defqos' settings
  if ( ! RDI_NotifQoS::validate(qosP,_defqos,RDI_ECHANNEL,perr,pran,child) ) {
        for (CORBA::ULong ix=0; ix < perr.length(); ix++)
                RDI_DUMP("Invalid QoS: " << (const char*)(perr[ix].name));
        throw CosN_UnsupportedQoS(perr);
  }
  // Validate the requested Admin QoS based on the '_defadm' settings
  if ( ! _defadm.validate(admP, perr, 1) ) {
        for (CORBA::ULong ix=0; ix < perr.length(); ix++)
                RDI_DUMP("Invalid AdminQoS: " << (const char*)(perr[ix].name));
        throw CosN_UnsupportedAdmin(perr);
  }
  myID = _serial++;
  if ( ! (chann = new EventChannel_i(this, qosP, admP, myID)) ) {
        return CosNA_EventChannel::_nil();
  } else if ( _channel.insert(myID, chann) != 0 ) {
        delete chann;
        return CosNA_EventChannel::_nil();
  }
  return WRAPPED_IMPL2OREF(CosNA_EventChannel, chann);
}

EventChannel_i* 
EventChannelFactory_i::create_channel(CosNA_ChannelID& myID WRAPPED_IMPLARG)
{
  CosN_QoSProperties*   qosP=0; 
  CosN_AdminProperties* admP=0;
  EventChannel_i* chan = 0;
  omni_mutex_lock lock(_oplock);

  qosP = _defqos.get_qos(RDI_ECHANNEL);
  admP = _defadm.to_admin();
  myID = _serial++;
  chan = new EventChannel_i(this, *qosP, *admP, myID);
  delete qosP;
  delete admP;
  if ( ! chan || _channel.insert(myID, chan) ) {
	if ( chan ) delete chan;
	return 0;
  }
  return chan;
}

CosNA_ChannelIDSeq* 
EventChannelFactory_i::get_all_channels(WRAPPED_IMPLARG_VOID)
{
  unsigned int index = 0;
  CosNA_ChannelIDSeq* idseq = new CosNA_ChannelIDSeq;
  if ( idseq == (CosNA_ChannelIDSeq *) 0 ) 
	return 0;
  RDI_HashCursor<CosNA_ChannelID, EventChannel_i*> cursor;
  _oplock.lock();
  idseq->length( _channel.length() );
  for ( cursor = _channel.cursor(); cursor.is_valid(); index++, ++cursor )
	(*idseq)[index] = cursor.key();
  _oplock.unlock();
  return idseq;
}

CosNA_EventChannel_ptr 
EventChannelFactory_i::get_event_channel(CosNA_ChannelID id WRAPPED_IMPLARG)
{
  EventChannel_i* channel = 0;
  _oplock.lock();
  if ( _channel.lookup(id, channel) ) {
	_oplock.unlock();
	return WRAPPED_IMPL2OREF(CosNA_EventChannel, channel);
  } else {
	_oplock.unlock();
	throw CosNA_ChannelNotFound();
  }
}

void EventChannelFactory_i::remove_channel(CosNA_ChannelID id)
{ _oplock.lock(); _channel.remove(id); _oplock.unlock(); }

void EventChannelFactory_i::display_info(CosNA_ChannelID id, 
					 CORBA::Boolean  evqueue)
{
  _oplock.lock();
  if ( id != 0 ) {
	EventChannel_i* channel=0;
  	if ( _channel.lookup(id, channel) ) {
		if ( evqueue )
			channel->display_event_queue();
		else
			cout << *channel << endl;
	} else {
		cout << "Failed to locate Channel with ID " << id << endl;
	}
  } else {
	RDI_HashCursor<CosNA_ChannelID, EventChannel_i*> curs;
  	for ( curs = _channel.cursor(); curs.is_valid(); ++curs ) {
		if ( evqueue ) {
			curs.val()->display_event_queue();
		} else {
			cout << *(curs.val()) << endl;
		}
	}
  }
  _oplock.unlock();
}
