// -*- Mode: C++; -*-
//                              File      : RDIEventQueue.h
//                              Package   : omniNotify-Library
//                              Created on: 1-Jan-1998
//                              Authors   : gruber&panagos
//
//    Copyright (C) 1998-2000 AT&T Laboratories -- Research
//
//    This file is part of the omniNotify library
//    and is distributed with the omniNotify release.
//
//    The omniNotify library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Library General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Library General Public License for more details.
//
//    You should have received a copy of the GNU Library General Public
//    License along with this library; if not, write to the Free
//    Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
//    02111-1307, USA
//
//
// Description:
//    proprietary interface
//
 
/*
$Log: RDIEventQueue.h,v $
Revision 1.22  2000/08/22 18:23:50  alcfp
added description to each file

Revision 1.21  2000/08/16 20:19:16  alcfp
Added licensing notice to each .h and .cc file where library files get GLPL notice and daemon file gets GPL notice -- examples do not claim any license but point out that the library and daemon code does have a license notice

*/
 
#ifndef __RDI_EVENT_QUEUE_H__
#define __RDI_EVENT_QUEUE_H__

#include <iostream.h>
#include <omnithread.h>
#include "CosNotify.h"
#include "RDIEvent.h"

/** Each EventChannel_i maintains a global queue that contains the
  * announced events. The queue can be scanned by multiple threads
  * consurrently and, hence, we need to know the number of them at
  * creation time.  This number is used for reference counting the
  * events inserted in the queue.  Finally,  the queue manages the
  * space associated with events once they are inserted into it.
  */

class RDI_EventQueue {
public:
  RDI_EventQueue(unsigned int maxLength=0, unsigned int concurrency=4,
		 CORBA::Short disPolicy = CosN_AnyOrder);
  ~RDI_EventQueue();

  // Insert a new event into the queue -- reference counter of the
  // event must be 1 at this point. On success, 0 is returned.  If
  // we are shutting down or reached the capacity of the queue and
  // we cannot drop existing events, -1 is returned

  int insert(RDI_StructuredEvent* event);

  // Retrieve the next event based on a previously accessed event.
  // The very first time this is called,  the value of 'prevEvent'
  // MUST be NULL. If no next event exists yet and 'block' is set,
  // the caller is blocked until an event becomes available.  NULL
  // is returned if 'block' is not set and no event is avaialble.
  
  RDI_StructuredEvent* next_event(RDI_StructuredEvent* prevEvent, 
			          unsigned char        block = 1);

  // The following can be used to view a snapshot of the status of
  // the queue. Since no lock is held, the numbers may not reflect
  // the most up-to-date information. 

  unsigned int  length() const		{ return _length; }
  unsigned int  max_size() const	{ return _maxnum; }
  unsigned long announced() const	{ return _announ; }
  unsigned long dropped() const		{ return _nmdrop; }

  // Change the maximun size of the queue - it does not affect any
  // of the events already present in the queue

  void set_max_size(unsigned int new_max_size);

  // Change the garbage collection period in seconds - it replaces
  // the old period immediately

  void set_gc_period(unsigned long nsecs);

  // Utility function for printing on standard output the queue --
  // the lock on the queue is held.  When 'showEvents' is set, some
  // information about each entry in queue is also displayed.

  void display_stats(unsigned char showEvents=0);

private:
  omni_mutex           _oplock;
  omni_condition       _qempty;
  omni_mutex           _gclock;
  omni_condition       _qclean;
  unsigned long        _period; // Period (sec) of garbage collector
  unsigned char        _gcdone; // Set on inactive garbage collector
  unsigned char        _finish; // Set when queue is beeing disposed
  omni_thread*         _worker;	// Garbage collection  worker thread
  RDI_StructuredEvent* _evhead;	// Head of the linked list of events
  RDI_StructuredEvent* _evtail;	// Tail of the linked list of events
  CORBA::Short         _disqos; // Discard policy for  queued events
  unsigned int         _length; // Current length  of the event list
  unsigned int         _maxnum;	// Max number of events  in the list
  unsigned long        _announ;	// Number of announced events so far
  unsigned long        _nmdrop;	// Number of dropped  events  so far
  unsigned int         _numreg;	// Number of threads accessing queue
  unsigned int         _numblk;	// Number of  blocked access threads

  // When the queue is full, we should apply the discard policy to
  // make room in the queue, if possible
  
  int  discard_events();

  // Garbage collect the queue of events. Garbage collection takes
  // place for events having reference counter == 1 and state that
  // is not equal to RDI_StructuredEvent::NEWBORN

  void garbage_collect();

  // The garbage collection thread is activated when the queue size
  // exceeds a threshold

  inline int run_garbage_collect() const 
		{ return ((!_maxnum && (_length < 4096)) || 
			  (_maxnum && (_length < 3*_maxnum/4))) ? 0 : 1; }

};

#endif
