// This application implements a pull consumer.  It uses filters
// and consumes CosNotification::StructuredEvent events.
// The filter used by this application depends on several command
// line arguments.

#include <stdlib.h>
#include <unistd.h>
#include <iostream.h>
#include "corba_wrappers.h"
#include "CosNotify.h"
#include "CosNotifyComm_i.h"

void print_constraints(CosNF_ConstraintInfoSeq *cons);
void parse_cons_str(char *cons_str, CosN_EventTypeSeq *evs);

static void usage(const char* pname)
{
   cout << "Usage: " << pname << " [-d numb] [-n name]" << endl;
   cout << "       -d numb : disconnect after numb pulls" << endl;
   cout << "       -n name : channel name [\"EventChannel\"]" << endl;
   cout << "       -c constraint list string [\"*::*\"]" << endl;
   cout << "       -a constraint list (call add_constraints again) " << endl;
   cout << "       -m constraint list (call modify_constraints) " << endl;
   cout << "       -p attach filter to proxy (default is admin) " << endl;
   cout << "       -r cid : remove constraint with id cid " << endl;
   cout << "       -e constraint expression - for the first constraint" << endl;
}

int main(int argc, char** argv)
{
  CosNA_EventChannel_var channel;
  CosNaming::NamingContext_var name_context;
  CosNaming::Name name;
  CosNF_FilterFactory_ptr  ffp;
  CosNF_Filter_ptr         filter;

  int c, num_pulls=10;
  char pname[128];

  strcpy(pname, argv[0]);

  CORBA::ORB_ptr orb = WRAPPED_ORB_INIT(argc, argv);
  CORBA::BOA_ptr boa = WRAPPED_BOA_INIT(orb, argc, argv);

  // Process command line arguments

  char *channelName = (char *) "EventChannel";
  char *channelKind = (char *) "EventChannel";
  char *cons_str = NULL;
  char *add_cons_str = NULL;
  char *add2_cons_str = NULL;
  char *mod_cons_str = NULL;
  char *cons_exp = NULL;
  CORBA::Boolean mod_flag  = 0;
  CORBA::Boolean add_flag  = 0;
  CORBA::Boolean add2_flag  = 0;
  CORBA::Boolean exp_flag   = 0;
  CORBA::Boolean proxy_flag  = 0;
  CORBA::Boolean del_flag  = 0;
  int  cid = 0;

  while ( (c = getopt(argc, argv,"d:n:c:m:a:r:e:p")) != EOF ) {
  	switch (c) {
        case 'd': num_pulls = atoi(optarg);
                  cout << "# events to be consumed = " << num_pulls << endl;
                  break;
        case 'n': channelName = optarg;
		  channelKind = optarg;
                  break;
	case 'c': cons_str = optarg;
                  break;
	case 'a': if (add_flag){
	            add2_flag = 1;
		    add2_cons_str = optarg;
                  }
                  else {
		    add_flag = 1;
		    add_cons_str = optarg;
		  }
                  break;
	case 'm': mod_flag = 1;
	          mod_cons_str = optarg;
                  break;
	case 'p': proxy_flag = 1;
                  break;
	case 'r': del_flag = 1;
	          cid = atoi(optarg);
                  break;
	case 'e': exp_flag = 1;
	          cons_exp = optarg; 
                  break;
        default : usage(pname);
                  exit(-1);
     }
  }

  StructuredPullConsumer_i* consumer = new StructuredPullConsumer_i();

  WRAPPED_ORB_REGISTER_SERVANT_NAME(orb, "filter_struct_pull_consumer");
  WRAPPED_BOA_OBJ_IS_READY(boa, consumer);

  // Let the BOA know that we are ready -- no blocking since we
  // need to consume events

  WRAPPED_IMPL_IS_READY_FORK(boa, "filter_struct_pull_consumer");

  // Locate, using the naming service, the event channel, after
  // obtaining a reference to the nameing service itself.......

  try {
	CORBA::Object_var name_service;
	name_service = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService"); 
	name_context = CosNaming::NamingContext::_narrow(name_service);
	if ( CORBA::is_nil(name_context) ) {
		cerr << "Failed to obtain context for NameService" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the naming service" << endl;
        exit(1);
  }

  name.length(1);
  name[0].id   = CORBA::string_dup(channelName);
  name[0].kind = CORBA::string_dup(channelKind);
  try {
	CORBA::Object_var channel_ref = name_context->resolve(name);
	channel = CosNA_EventChannel::_narrow(channel_ref);
	if ( CORBA::is_nil(channel) ) {
		cerr << "Failed to narrow Event Channel !" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the event channel" << endl;
        exit(1);
  }
 
  // Get consumer admin object from the channel

  CosNA_InterFilterGroupOperator ifoper = CosNA_AND_OP;
  CosNA_ConsumerAdmin_var consumer_admin;
  CosNA_AdminID suppID;

  try {
	consumer_admin = channel->new_for_consumers(ifoper, suppID);
	if ( CORBA::is_nil(consumer_admin) ) {
		cerr << "Failed to find Consumer Admin !" << endl;
		exit(1);
	}
  } catch (...) {
	cerr << "Failed to obtain Consumer Admin !" << endl;
	exit(1);
  }

  cout << "Obtained ConsumerAdmin from the channel" << endl;

  // Get a pull proxy using the above admin object

  CosNA_ProxyID prxID;
  CosNA_ClientType ctype = 
				CosNA_STRUCTURED_EVENT;
  CosNA_ProxySupplier_var supp;
  CosNA_StructuredProxyPullSupplier_var psupp;
  try {
	supp = consumer_admin->obtain_notification_pull_supplier(ctype,prxID);
  	psupp=CosNA_StructuredProxyPullSupplier::_narrow(supp);
	if ( CORBA::is_nil(psupp) ) {
		cerr << "Failed to find Proxy Supplier !" << endl;
		exit(1);
	}
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
	cerr << "Failed to obtain Proxy Supplier !" << endl;
	exit(1);
  }

  cout << "Obtained ProxyPullSupplier from ConsumerAdmin" << endl;


  // obtain a reference to the default filter factory,
  // create a filter object 
  // and add simple type constraints to it 


  try {
    ffp     = channel->default_filter_factory();  
    filter     = ffp->create_filter("EXTENDED_TCL");
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
	cerr << "Failed to obtain filter !" << endl;
	exit(1);
  }

  cout << "Obtained filter from channel" << endl;

  //
  // construct a single constraint expression,
  // and add it to the proxy
  _CORBA_ULong len1 = 1;

  CosNF_ConstraintExpSeq exp = CosNF_ConstraintExpSeq(len1);
  exp.length(len1);

  if (exp_flag)
    exp[0].constraint_expr = CORBA::string_dup(cons_exp);
  else
    exp[0].constraint_expr = CORBA::string_dup("true");

  // exp[0].constraint_expr = CORBA::string_dup("$.header.variable_header(priority) > 5");
  // exp[0].constraint_expr = CORBA::string_dup("($.header.variable_header[1].name == `priority' and $.header.variable_header[1].value > 5)");

  parse_cons_str(cons_str, &exp[0].event_types);
  
  CosNF_ConstraintInfoSeq *cis1 = NULL;
  CosNF_ConstraintInfoSeq *cis2 = NULL;


  try {
    cis1 = filter->add_constraints(exp);
    cis2 = filter->get_all_constraints();
    print_constraints(cis2);
  }
  catch(CosNF_InvalidConstraint& _exobj1) {
    cerr << "Exception thrown : Invalid Constraint given "
 	 << (const char *) exp[0].constraint_expr << endl;
    cerr << "constraint not added " << endl << endl;
  }


  // add filter to proxy or admin

  CosNF_FilterID fltid;

  if (proxy_flag) 
    fltid = psupp->add_filter(filter);
  else 
    fltid = consumer_admin->add_filter(filter);

  //
  // Connect supplier to the proxy
  //

  try {
    psupp->connect_structured_pull_consumer(consumer->_this());
  }
  catch (CORBA::BAD_PARAM& ex) {
	cerr << "BAD_PARAM Exception while connecting Pull Consumer !" << endl;
        exit (1);
  }
  catch (CosEventChannelAdmin::AlreadyConnected& ex) {
	cerr << "Pull Consumer already connected !" << endl;
  } catch (...) {
	cerr << "Failed to connect Pull Consumer !" << endl;
	exit(1);
  }

  cout << "Connected pull consumer to ProxyPullSupplier" << endl;

  for (int i=0; i < num_pulls; i++) {
	CORBA::ULong l;
	CosN_StructuredEvent* event= psupp->pull_structured_event();
	cout << 
	(const char*)event->header.fixed_header.event_type.domain_name<<"::"<<
        (const char*)event->header.fixed_header.event_type.type_name << " -- ";
  	event->remainder_of_body >>= l;
  	cout << l << endl;
	delete event;
  }


  if (add_flag) {

    // add another constraint here 
    CosNF_ConstraintExpSeq add_exp = CosNF_ConstraintExpSeq(1);
    add_exp.length(1);
    // add_exp[0].constraint_expr = CORBA::string_dup("true");
    add_exp[0].constraint_expr = CORBA::string_dup("$.header.variable_header(priority) < 2");

    parse_cons_str(add_cons_str, &add_exp[0].event_types);
    CosNF_ConstraintInfoSeq *cis1, *cis2;

    filter->add_constraints(add_exp);
    cis2 = filter->get_all_constraints();
    print_constraints(cis2);

    // check results 

    for (int i=0; i < num_pulls; i++) {
      CORBA::ULong l;
      CosN_StructuredEvent* event= psupp->pull_structured_event();
      cout << 
	(const char*)event->header.fixed_header.event_type.domain_name<<"::"<<
	(const char*)event->header.fixed_header.event_type.type_name << " -- ";
      event->remainder_of_body >>= l;
      cout << l << endl;
      delete event;
    }
  }


  //
  // now check modify constraints 
  //
  
  if (mod_flag) {

    // modify first expression for now 

    CosNF_ConstraintIDSeq* del_cons = new CosNF_ConstraintIDSeq(0);

    CosNF_ConstraintInfoSeq* mod_cons = new CosNF_ConstraintInfoSeq(1);
    mod_cons->length(1);
    (*mod_cons)[0].constraint_id = (*cis2)[0].constraint_id;

    CosNF_ConstraintExp& mod_exp = (*mod_cons)[0].constraint_expression;
    mod_exp.constraint_expr = CORBA::string_dup("true");
    parse_cons_str(mod_cons_str, &mod_exp.event_types);

    try {

      filter->modify_constraints(*del_cons, *mod_cons);
      cis2 = filter->get_all_constraints();
      print_constraints(cis2);
    }
    catch(CosNF_InvalidConstraint& _exobj1) {
      cerr << "Exception thrown : Invalid Constraint given "
	   << (const char *) exp[0].constraint_expr << endl;
      cerr << "constraint not added " << endl << endl;
    }
    
    // pull again, this time with a modified  filter
    for (int i=0; i < num_pulls; i++) {
      CORBA::ULong l;
      CosN_StructuredEvent* event= psupp->pull_structured_event();
      cout << 
	(const char*)event->header.fixed_header.event_type.domain_name<<"::"<<
	(const char*)event->header.fixed_header.event_type.type_name << " -- ";
      event->remainder_of_body >>= l;
      cout << l << endl;
      delete event;
    }
  }


  if (del_flag) {

    CosNF_ConstraintIDSeq* del_cons = new CosNF_ConstraintIDSeq(1);
    del_cons->length(1);
    (*del_cons)[0] = (CosNF_ConstraintID) cid;
    
    CosNF_ConstraintInfoSeq* mod_cons = new CosNF_ConstraintInfoSeq(0);
    try {

      filter->modify_constraints(*del_cons, *mod_cons);
      cis2 = filter->get_all_constraints();
      print_constraints(cis2);
    }
    catch(CosNF_InvalidConstraint& _exobj1) {
      cerr << "Exception thrown : Invalid Constraint given "
	   << (const char *) exp[0].constraint_expr << endl;
      cerr << "constraint not added " << endl << endl;
    }
    
    // pull again, this time with a modified  filter
    for (int i=0; i < num_pulls; i++) {
      CORBA::ULong l;
      CosN_StructuredEvent* event= psupp->pull_structured_event();
      cout << 
	(const char*)event->header.fixed_header.event_type.domain_name<<"::"<<
	(const char*)event->header.fixed_header.event_type.type_name << " -- ";
      event->remainder_of_body >>= l;
      cout << l << endl;
      delete event;
    }
  }


  if (add2_flag) {

    // add another constraint here 
    CosNF_ConstraintExpSeq add_exp = CosNF_ConstraintExpSeq(1);
    add_exp.length(1);
    add_exp[0].constraint_expr = CORBA::string_dup("true");

    parse_cons_str(add2_cons_str, &add_exp[0].event_types);
    CosNF_ConstraintInfoSeq *cis1, *cis2;

    filter->add_constraints(add_exp);
    cis2 = filter->get_all_constraints();
    print_constraints(cis2);

    // check results 

    for (int i=0; i < num_pulls; i++) {
      CORBA::ULong l;
      CosN_StructuredEvent* event= psupp->pull_structured_event();
      cout << 
	(const char*)event->header.fixed_header.event_type.domain_name<<"::"<<
	(const char*)event->header.fixed_header.event_type.type_name << " -- ";
      event->remainder_of_body >>= l;
      cout << l << endl;
      delete event;
    }
  }

  // Disconnect supplier from the service

  psupp->disconnect_structured_pull_supplier();
  consumer_admin->destroy();
  WRAPPED_DISPOSE_IMPL(consumer);

  return 0;
}

// ------------------------------------------------------------------------- //

void print_constraints(CosNF_ConstraintInfoSeq *cons) 
{
  CORBA::ULong conslen = cons->length();
  cout << "*** Printing Constraint Sequence *** " << endl;
  cout << "Size of constraint sequence = " << conslen << endl;

  for (CORBA::ULong j = 0; j < conslen ; j++) {
    CosN_EventTypeSeq& etypes = (*cons)[j].constraint_expression.event_types;
    cout << "Constraint " << j << " ID = " << (*cons)[j].constraint_id << endl;
    cout << "    constraint expression = " <<    
      (const char *) (*cons)[j].constraint_expression.constraint_expr << endl;
    cout << "    length of type sequence = " << etypes.length() << endl;
    for (CORBA::ULong k = 0; k < etypes.length(); k++) {
        cout << "    event type "<< k << ": domain name = "<< 
		(const char*) etypes[k].domain_name << " type_name = " << 
		(const char*) etypes[k].type_name << endl;
    }
  }
}

void parse_cons_str(char *cons_str, CosN_EventTypeSeq *evs)
{
  char *domain_name;
  char *type_name;
  char *str_beg;
  char *str_end;
  char *tok; 

  if ( ! cons_str )
     cons_str = CORBA::string_dup("*::*");	// Default is *::*
  tok  = strtok(cons_str, " ");
  *evs = CosN_EventTypeSeq(1);

  for (CORBA::ULong i=0; tok != NULL; i++) {
    domain_name = CORBA::string_dup(tok);
    str_end = strchr(domain_name, ':');
    str_beg = str_end+2;
    *str_end = '\0';
    type_name = CORBA::string_dup(str_beg);

    (*evs).length(i+1);
    (*evs)[i].domain_name = domain_name;
    (*evs)[i].type_name = type_name;
    tok = strtok(NULL, " ");
  }
  return;
}

