// This application implements an Event Service push supplier.
// It consumes CORBA::Any events that contain a CORBA::ULong. 
 
#include <stdlib.h>
#include <unistd.h>
#include <iomanip.h>
#include <iostream.h>
#include "corba_wrappers.h"
#include "CosNotify.h"

CORBA::Boolean verbose = 0;
CORBA::ULong   numpush = 0;	// Events pushed so far
omni_mutex     is_over;	

/////////////////////////////////////////////////////////////////////
// *************************************************************** //
/////////////////////////////////////////////////////////////////////

class MyPushConsumer_i : WRAPPED_SKELETON_SUPER(CosNC::, PushConsumer) {
public:
  MyPushConsumer_i(int max_numb=0) : _max_numb(max_numb)  {;}

  void push(const CORBA::Any& data  WRAPPED_DECLARG );
  void disconnect_push_consumer( WRAPPED_DECLARG_VOID )	{;}
  void offer_change(const CosN_EventTypeSeq& added,
		    const CosN_EventTypeSeq& removed WRAPPED_DECLARG ) {;}
protected:
  CORBA::ULong _max_numb;
};

void MyPushConsumer_i::push(const CORBA::Any& data  WRAPPED_IMPLARG )
{
  CORBA::ULong l;
  data >>= l;
  if ( verbose ) {
	cout << setw(4) << numpush << " : Got " << l << endl;
  }
  if ( _max_numb && (++numpush >= _max_numb) )
	is_over.unlock();
}

/////////////////////////////////////////////////////////////////////
// *************************************************************** //
/////////////////////////////////////////////////////////////////////

static void usage(const char* pname)
{
   cout << "Usage: " << pname << " [-d numb] [-n name] [-v]" << endl;
   cout << "       -d numb : disconnect after numb pushes" << endl;
   cout << "       -n name : channel name [\"EventChannel\"]" << endl;
   cout << "       -v      : verbose output mode" << endl;
}

int main(int argc, char** argv)
{
  char *channelName = (char *) "EventChannel";
  char *channelKind = (char *) "EventChannel";
  int evnum = 0, c;

  // Parse command line arguments and initialize global variables

  while ( (c = getopt(argc, argv,"d:n:v")) != EOF ) {
  	switch (c) {
        case 'd': evnum = atoi(optarg);
                  break;
        case 'n': channelName = optarg;
		  channelKind = optarg;
                  break;
 	case 'v': verbose = 1;
		  break;
        default : usage(argv[0]);
                  exit(-1);
     }
  }

  CORBA::ORB_ptr orb = WRAPPED_ORB_INIT(argc, argv);
  CORBA::BOA_ptr boa = WRAPPED_BOA_INIT(orb, argc, argv);

  MyPushConsumer_i* cons = new MyPushConsumer_i(evnum);

  WRAPPED_ORB_REGISTER_SERVANT_NAME(orb, "event_push_consumer");
  WRAPPED_BOA_OBJ_IS_READY(boa, cons);

  WRAPPED_IMPL_IS_READY_FORK(boa, "event_push_consumer");

  try {
  	CosNaming::Name                             name;
	CORBA::Object_var                           echannel_ref;
	CORBA::Object_var                           name_service;
  	CosNaming::NamingContext_var                name_context;
  	CosEventChannelAdmin::EventChannel_var      channel;
  	CosEventChannelAdmin::ConsumerAdmin_var     cadmin;
	CosEventChannelAdmin::ProxyPushSupplier_var psupp;

	if ( verbose ) 
		cout << "Resolving reference to: 'NameService'" << endl;
	name_service = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService"); 
	name_context = CosNaming::NamingContext::_narrow(name_service);
	if ( CORBA::is_nil(name_context) ) {
		cerr << "Failed to obtain context for NameService" << endl;
		exit(1);
	}
	name.length(1);
	name[0].id   = CORBA::string_dup(channelName);
	name[0].kind = CORBA::string_dup(channelKind);
	if ( verbose ) 
		cout << "Resolving reference to: '"<<channelName<<"'" << endl;
	echannel_ref = name_context->resolve(name);
        channel = CosNA_EventChannel::_narrow(echannel_ref);
        if ( CORBA::is_nil(channel) ) {
                cerr << "Failed to narrow Event Channel !" << endl;
		exit(1);
        }

	if ( verbose ) 
		cout << "Creating new ConsumerAdmin object" << endl;
	cadmin = channel->for_consumers();
	if ( CORBA::is_nil(cadmin) ) {
		cerr << "Failed to find Consumer Admin !" << endl;
		exit(1);
	}

	psupp = cadmin->obtain_push_supplier();
	if ( CORBA::is_nil(psupp) ) {
		cerr << "Failed to create ProxyPushSupplier Supplier !" << endl;
		exit(-1);
	}
	psupp->connect_push_consumer(cons->_this());

	is_over.lock();
	// The following will succeed when we are done with the events
	// we are expecting to receive
	is_over.lock();
	is_over.unlock();

	psupp->disconnect_push_supplier();
	WRAPPED_DISPOSE(cons->_this());
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (CosEventChannelAdmin::AlreadyConnected& ex) {
	cerr << "Push Consumer already connected !" << endl;
	exit(1);
  } catch (CORBA::BAD_PARAM& ex) {
	cerr << "BAD_PARAM Exception while connecting Push Consumer !" << endl;
	exit (1);
  } catch (...) {
        cerr << "Caught exception while resolving the naming service" << endl;
        exit(1);
  }
  return 0;
}
