// This application implements a pull consumer that connects to 
// a PULL_SEQUENCE proxy.
// It consumes CosNotification::StructuredEvent event sequences.
// Each event in the sequence is assumed to have a CORBA::ULong
// value stored in the remainder_of_body attribute.
 
#include <stdlib.h>
#include <unistd.h>
#include <iostream.h>
#include "corba_wrappers.h"
#include "CosNotify.h"
#include "CosNotifyComm_i.h"

static void usage(const char* pname)
{
   cout << "Usage: " << pname << " [-d numb] [-b size] [-n name]" << endl;
   cout << "       -d numb : disconnect after numb pulls"  << endl;
   cout << "       -b size : max size of each event batch" << endl;
   cout << "       -n name : channel name [\"EventChannel\"]" << endl;
}

int main(int argc, char** argv)
{
  CosNA_EventChannel_var channel;
  CosNaming::NamingContext_var name_context;
  CosNaming::Name name;
  CORBA::Boolean  has_events;
  int c, num_pulls=20, batch_sz=5;
  char *channelName = (char *) "EventChannel";

  CORBA::ORB_ptr orb = WRAPPED_ORB_INIT(argc, argv);
  CORBA::BOA_ptr boa = WRAPPED_BOA_INIT(orb, argc, argv);

  while ( (c = getopt(argc, argv,"d:b:n:")) != EOF ) {
  	switch (c) {
        case 'd': num_pulls = atoi(optarg); break;
	case 'b': batch_sz = atoi(optarg);  break;
        case 'n': channelName = optarg;     break;
        default : usage(argv[0]); exit(-1);
     }
  }
  cout << "# batches to be pulled = " << num_pulls << endl;
  cout << "max # events per batch = " << batch_sz  << endl;

  SequencePullConsumer_i* consumer = new SequencePullConsumer_i();

  WRAPPED_ORB_REGISTER_SERVANT_NAME(orb, "batch_pull_consumer");
  WRAPPED_BOA_OBJ_IS_READY(boa, consumer);

  // Let the BOA know that we are ready -- no blocking since we
  // need to consume events

  WRAPPED_IMPL_IS_READY_FORK(boa, "batch_pull_consumer");

  // Locate, using the naming service, the event channel, after
  // obtaining a reference to the naming service itself........
  try {
	CORBA::Object_var name_service;
	name_service = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService"); 
	name_context = CosNaming::NamingContext::_narrow(name_service);
	if ( CORBA::is_nil(name_context) ) {
		cerr << "Failed to obtain context for NameService" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the naming service" << endl;
        exit(1);
  }

  name.length(1);
  name[0].id   = CORBA::string_dup(channelName);
  name[0].kind = CORBA::string_dup(channelName);
  try {
	CORBA::Object_var channel_ref = name_context->resolve(name);
	channel = CosNA_EventChannel::_narrow(channel_ref);
	if ( CORBA::is_nil(channel) ) {
		cerr << "Failed to narrow Event Channel !" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the event channel" << endl;
        exit(1);
  }
 
  // Get consumer admin object from the channel

  CosNA_InterFilterGroupOperator ifoper = CosNA_AND_OP;
  CosNA_ConsumerAdmin_var consumer_admin;
  CosNA_AdminID suppID;

  try {
	consumer_admin = channel->new_for_consumers(ifoper, suppID);
	if ( CORBA::is_nil(consumer_admin) ) {
		cerr << "Failed to find Consumer Admin !" << endl;
		exit(1);
	}
  } catch (...) {
	cerr << "Failed to obtain Consumer Admin !" << endl;
	exit(1);
  }

  cout << "Obtained ConsumerAdmin from the channel" << endl;

  // Get a pull proxy using the above admin object

  CosNA_SequenceProxyPullSupplier_var psupp;
  CosNA_ProxySupplier_var             supp;
  CosNA_ClientType                    ctype = CosNA_SEQUENCE_EVENT;
  CosNA_ProxyID                       prxID;

  try {
	supp = consumer_admin->obtain_notification_pull_supplier(ctype,prxID);
  	psupp=CosNA_SequenceProxyPullSupplier::_narrow(supp);
	if ( CORBA::is_nil(psupp) ) {
		cerr << "Failed to find Proxy Supplier !" << endl;
		exit(1);
	}
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
	cerr << "Failed to obtain Proxy Supplier !" << endl;
	exit(1);
  }
  cout << "Obtained ProxyPullSupplier from ConsumerAdmin" << endl;

  // Connect supplier to the proxy

  try {
	psupp->connect_sequence_pull_consumer(consumer->_this());
  }
  catch (CORBA::BAD_PARAM& ex) {
	cerr << "BAD_PARAM Exception while connecting Pull Consumer !" << endl;
        exit (1);
  }
  catch (CosEventChannelAdmin::AlreadyConnected& ex) {
	cerr << "Pull Consumer already connected !" << endl;
  } catch (...) {
	cerr << "Failed to connect Pull Consumer !" << endl;
	exit(1);
  }
  cout << "Connected pull consumer to ProxyPullSupplier" << endl;

  while ( num_pulls > 0 ) {
	CORBA::ULong l;
	CosN_EventBatch* batch = 
			psupp->try_pull_structured_events(batch_sz, has_events);
	if ( has_events ) {
		for (unsigned int j=0; j < batch->length(); j++) {
		   CosN_StructuredEvent ev = (*batch)[j];
		   cout << 
	(const char *)ev.header.fixed_header.event_type.domain_name << "::" <<
	(const char *)ev.header.fixed_header.event_type.type_name << " - ";
		   ev.remainder_of_body >>= l;
		   cout << l << endl;
		}
		num_pulls -= 1;
	}
	delete batch;
  }

  // Disconnect supplier from the service

  try {
  	psupp->disconnect_sequence_pull_supplier();
  	consumer_admin->destroy();
	WRAPPED_DISPOSE_IMPL(consumer);
  } catch (...) {
	cerr << "Exception during the cleanup process ...." << endl;
	exit(1);
  }

  return 0;
}
