// This application implements a push supplier which uses the new
// interface (provided by the Notification Service), for creating
// administrative objects and proxies of ANY_EVENT kind.
// It generates CORBA::Any events that contain a CORBA::Ulong.
 
#include <stdlib.h>
#include <unistd.h>
#include <iostream.h>
#include "corba_wrappers.h"
#include "CosNotify.h"

/* -------------------------------------------------------------- */

class MyPushSupplier_i : WRAPPED_SKELETON_SUPER(CosNC::, PushSupplier) {
public:
  MyPushSupplier_i() {;}

  inline void disconnect_push_supplier();
  inline void subscription_change(const CosN_EventTypeSeq& added,
                                  const CosN_EventTypeSeq& deled);
};

inline void MyPushSupplier_i::disconnect_push_supplier()
{ cout << "PushSupplier: disconnected" << endl; }

inline void MyPushSupplier_i::subscription_change(
				const CosN_EventTypeSeq& added,
                                const CosN_EventTypeSeq& deled)
{
  CORBA::ULong indx;
  CosN_EventType type;

  cout << "PushSupplier: subscription_change contains: " << endl;
  for (indx = 0; indx < added.length(); indx++) {
        type = added[indx];
        cout << "\t+ " << (const char*)type.domain_name << 
		"::" << (const char*)type.type_name << endl;
  }
  for (indx = 0; indx < deled.length(); indx++) {
        type = deled[indx];
        cout << "\t- " << (const char*)type.domain_name << 
	        "::" << (const char*)type.type_name << endl;
  }
}

/* -------------------------------------------------------------- */

static void usage(const char* pname)
{
   cout << "Usage: " << pname << " [-d numb] [-n name]" << endl;
   cout << "       -d numb : disconnect after numb pushes" << endl;
   cout << "       -n name : channel name [\"EventChannel\"]" << endl;
}

int main(int argc, char** argv)
{
  MyPushSupplier_i* supplier = 0;
  const char* channelName = (const char *) "EventChannel";
  int c, num_pushes = 100;


  CORBA::ORB_var orb = WRAPPED_ORB_INIT(argc, argv);
  CORBA::BOA_var boa = WRAPPED_BOA_INIT(orb, argc, argv);

  if ( ! (supplier = new MyPushSupplier_i()) ) {
	cerr << "Failed to create push supplier object" << endl;
	return -1;
  }
  WRAPPED_ORB_REGISTER_SERVANT_NAME(orb, "any_push_supplier");
  WRAPPED_BOA_OBJ_IS_READY(boa, supplier);

  while ( (c = getopt(argc, argv,"d:n:")) != EOF ) {
  	switch (c) {
        case 'd': num_pushes = atoi(optarg);
                  break;
        case 'n': channelName = optarg;
                  break;
        default : usage(argv[0]);
                  return -1;
     }
  }

  try {
	CosNaming::Name              name;
	CORBA::Object_var            name_service;
  	CORBA::Object_var            echannel_ref;
	CosNaming::NamingContext_var name_context;
  	CosNA_EventChannel_var       channel;

	name_service = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService"); 
	name_context = CosNaming::NamingContext::_narrow(name_service);
	if ( CORBA::is_nil(name_context) ) {
		cerr << "Failed to obtain context for NameService" << endl;
		return -1;
	}

  	name.length(1);
  	name[0].id   = CORBA::string_dup(channelName);
  	name[0].kind = CORBA::string_dup(channelName);

	echannel_ref = name_context->resolve(name);
	channel = CosNA_EventChannel::_narrow(echannel_ref);
	if ( CORBA::is_nil(channel) ) {
		cerr << "Failed to retrieve reference to Event Channel" << endl;
		return -1;
	}

  	// Let the BOA know that we are ready -- no blocking since we
  	// need to supply events
	cout << "** Call to WRAPPED_IMPL_IS_READY()" << endl;
	WRAPPED_IMPL_IS_READY_FORK(boa, "any_push_supplier");
   
  	CosNA_InterFilterGroupOperator ifoper = CosNA_AND_OP;
  	CosNA_SupplierAdmin_var supplier_admin;
  	CosNA_AdminID suppID;

	supplier_admin = channel->new_for_suppliers(ifoper, suppID);
	if ( CORBA::is_nil(supplier_admin) ) {
		cerr << "Failed to find Supplier Admin !" << endl;
		return -1;
	}
  	// Get a push proxy using the above admin object

  	CosNA_ProxyID prxID;
  	CosNA_ProxyConsumer_var consumer;
  	CosNA_ProxyPushConsumer_var proxy_consumer;

	consumer = supplier_admin->obtain_notification_push_consumer(
				CosNA_ANY_EVENT, prxID);
  	proxy_consumer = CosNA_ProxyPushConsumer::_narrow(consumer);
	if ( CORBA::is_nil(proxy_consumer) ) {
		cerr << "Failed to find Proxy Consumer !" << endl;
		return -1;
	}
	proxy_consumer->connect_any_push_supplier(supplier->_this());

  	for (int i=0; (num_pushes == 0) || (i < num_pushes); i++) {
		CORBA::Any any;
		any <<= (CORBA::ULong) i;
		proxy_consumer->push(any);
	}

	proxy_consumer->disconnect_push_consumer();
	WRAPPED_DISPOSE_IMPL(supplier);
  } catch (CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        return -1;
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        return -1;
#endif
  } catch (CosEventChannelAdmin::AlreadyConnected& ex) {
        cerr << "Push Supplier already connected!" << endl;
        return -1;
  } catch (CORBA::BAD_PARAM& ex) {
        cerr << "BAD_PARAM Exception while connecting Push Supplier!" << endl;
        return -1;
  } catch (...) {
        cerr << "Caught exception while resolving the naming service" << endl;
        return -1;
  }

  return 0;
}
