// This application implements a push consumer which uses the new
// interface (provided by the Notification Service), for creating
// administrative objects and proxies of ANY_EVENT kind.
// It expects CORBA::Any events that contain a CORBA::Ulong value.
 
#include <unistd.h>
#include <stdlib.h>
#include <iostream.h>
#include "corba_wrappers.h"
#include "CosNotify.h"

/* -------------------------------------------------------------- */

class MyPushConsumer_i : WRAPPED_SKELETON_SUPER(CosNC::, PushConsumer) {
public:
  MyPushConsumer_i() : _num_push(0) {;}

  inline void  push(const CORBA::Any & data);
  inline void  disconnect_push_consumer();
  inline void  offer_change(const CosN_EventTypeSeq& added, 
			    const CosN_EventTypeSeq& deled);
  CORBA::ULong num_push() const		{ return _num_push; }
protected:
  CORBA::ULong _num_push;
};

inline void MyPushConsumer_i::push(const CORBA::Any & data  WRAPPED_IMPLARG )
{ 
  CORBA::ULong l;
  data >>= l;
  cout << "Push Consumer: got " << l << endl;
  _num_push += 1;
}

inline void MyPushConsumer_i::disconnect_push_consumer( WRAPPED_IMPLARG_VOID )
{ cout << "PushConsumer: disconnected" << endl; }

inline void MyPushConsumer_i::offer_change(
		const CosN_EventTypeSeq& added,
                const CosN_EventTypeSeq& deled WRAPPED_IMPLARG )
{
  CORBA::ULong indx;
  CosN_EventType type;

  cout << "PushConsumer: offer_change contains: " << endl;
  for (indx = 0; indx < added.length(); indx++) {
        type = added[indx];
        cout << "\t+ " << (const char*)type.domain_name << 
                "::" << (const char*)type.type_name << endl;
  }
  for (indx = 0; indx < deled.length(); indx++) {
        type = deled[indx];
        cout << "\t- " << (const char*)type.domain_name << 
                "::" << (const char*)type.type_name << endl;
  }
}

/* -------------------------------------------------------------- */

static void usage(const char* pname)
{
   cout << "Usage: " << pname << " [-d numb] [-n name]" << endl;
   cout << "       -d numb : disconnect after numb pushes" << endl;
   cout << "       -n name : channel name [\"EventChannel\"]" << endl;
}

int main(int argc, char** argv)
{
  CosNA_EventChannel_ptr channel;
  CosNaming::NamingContext_var name_context;
  CosNaming::Name name;
  const char *channelName = (const char *) "EventChannel";
  int c, num_pushes=0;
  char pname[128];

  strcpy(pname, argv[0]);

  CORBA::ORB_ptr orb = WRAPPED_ORB_INIT(argc, argv);
  CORBA::BOA_ptr boa = WRAPPED_BOA_INIT(orb, argc, argv);

  // Process command line arguments

  while ( (c = getopt(argc, argv,"d:n:")) != EOF ) {
  	switch (c) {
        case 'd': num_pushes = atoi(optarg);
                  cout << "# events to be consumed = " << num_pushes << endl;
                  break;
        case 'n': channelName = optarg;
                  break;
        default : usage(pname);
                  exit(-1);
     }
  }

  MyPushConsumer_i* consumer = new MyPushConsumer_i();
  WRAPPED_ORB_REGISTER_SERVANT_NAME(orb, "any_push_consumer");
  WRAPPED_BOA_OBJ_IS_READY(boa, consumer);

  // Let the BOA know that we are ready -- no blocking since we
  // need to supply events
  WRAPPED_IMPL_IS_READY_FORK(boa, "any_push_consumer");

  try {
	CORBA::Object_var name_service;
	name_service = WRAPPED_RESOLVE_INITIAL_REFERENCES(orb, "NameService"); 
	name_context = CosNaming::NamingContext::_narrow(name_service);
	if ( CORBA::is_nil(name_context) ) {
		cerr << "Failed to obtain context for NameService" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the naming service" << endl;
        exit(1);
  }

  name.length(1);
  name[0].id   = CORBA::string_dup(channelName);
  name[0].kind = CORBA::string_dup(channelName);
  try {
	CORBA::Object_var channel_ref = name_context->resolve(name);
	channel = CosNA_EventChannel::_narrow(channel_ref);
	if ( CORBA::is_nil(channel) ) {
		cerr << "Failed to narrow Event Channel !" << endl;
		exit(1);
	}
  } catch(CORBA::ORB::InvalidName& ex) {
        cerr << "Service required is invalid [does not exist]" << endl;
        exit(1);
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
        cerr << "Caught exception while resolving the event channel" << endl;
        exit(1);
  }
 
  // Get consumer admin object from the channel

  CosNA_InterFilterGroupOperator ifoper = CosNA_AND_OP;
  CosNA_ConsumerAdmin_var consumer_admin;
  CosNA_AdminID suppID;

  try {
	consumer_admin = channel->new_for_consumers(ifoper, suppID);
	if ( CORBA::is_nil(consumer_admin) ) {
		cerr << "Failed to find Consumer Admin !" << endl;
		exit(1);
	}
  } catch (...) {
	cerr << "Failed to obtain Consumer Admin !" << endl;
	exit(1);
  }

  cout << "Obtained ConsumerAdmin from the channel" << endl;

  // Get a push proxy using the above admin object

  CosNA_ProxyID prxID;
  CosNA_ClientType ctype = CosNA_ANY_EVENT;
  CosNA_ProxySupplier_var supplier;
  CosNA_ProxyPushSupplier_var proxy_supplier;
  try {
	supplier =
		consumer_admin->obtain_notification_push_supplier(ctype,prxID);
  	proxy_supplier = 
		CosNA_ProxyPushSupplier::_narrow(supplier);
	if ( CORBA::is_nil(proxy_supplier) ) {
		cerr << "Failed to find Proxy Supplier !" << endl;
		exit(1);
	}
#if defined (__OMNIORB2__) || defined (__OMNIORB3__)
  } catch (CORBA::COMM_FAILURE& ex) {
        cerr << "Caught system exception COMM_FAILURE" << endl;
        exit(1);
#endif
  } catch (...) {
	cerr << "Failed to obtain Proxy Supplier !" << endl;
	exit(1);
  }

  cout << "Obtained ProxyPushSupplier from ConsumerAdmin" << endl;

  // Connect supplier to the proxy

  try {
	proxy_supplier->connect_any_push_consumer(consumer->_this());
  } catch (CORBA::BAD_PARAM& ex) {
	cerr << "BAD_PARAM Exception while connecting Push Consumer !" << endl;
        exit (1);
  } catch (CosEventChannelAdmin::AlreadyConnected& ex) {
	cerr << "Push Consumer already connected !" << endl;
  } catch (...) {
	cerr << "Failed to connect Push Consumer !" << endl;
	exit(1);
  }

  cout << "Connected push consumer to ProxyPushSupplier" << endl;

  while ( (int)consumer->num_push() < num_pushes ) 
	omni_thread::yield();

  // Disconnect supplier from the service

  try {
    	proxy_supplier->disconnect_push_supplier();
	WRAPPED_DISPOSE_IMPL(consumer);
  } catch (...) {
	cerr << "Exception while cleaning up ....." << endl;
	exit(1);
  }

  return 0;
}
