#include "RPii.h"

/*Prototypes*/
//Recording Functions
int Rec_Data(struct Log_Control *Control);
int Rec_Acc_Data(struct Log_Control *Control);
int Rec_DataTime(struct Log_Control *Control);
int Rec_dataFileHeader(struct Log_Control *Control);
//Print Functions
int Print_Ctrl_Settings(struct Log_Control Control);
void Print_Countdown(int wait, int values);
void Print_App_Ctrls( void );
void	Print_EditSet_Ctrls ( void );
void Print_wiimote_data(struct RPii *RPii, bool all);
void RPii_Prnt_Ctrl_Settings(struct Log_Control Control, bool all);
//LED Functions
int LEDs_Mod4(struct RPii *RPii, int samplecount);
int LEDs_All_Flash(struct RPii *RPii);

//
/********Rec Functions*********/
//
//
//A Function to print out the Acc_Data and Time data
//
int Rec_Data(struct Log_Control *Control){
	Rec_Acc_Data(Control);
	Rec_DataTime(Control);
	return 1;
}
//
//A function to print out data depending on which data is set to print.
//
int Rec_Acc_Data(struct Log_Control *Control){
	
	//Check that structures have been declared
	if(Control == NULL){
		fprintf(stderr, "Error, Pointer to Control struct is NULL\n");
		exit(-1);
	}
	
	//Check that the PRNT settings are valid
	int fields = Control->log_flags.PRNTACC.X+Control->log_flags.PRNTACC.Y+Control->log_flags.PRNTACC.Z;
	if (fields == 0){
		fprintf(stderr, "no accelerometer axis have been set to print\n");
		exit(-1);
	}
	if ((fields < 0)||(fields > 3 )){
		fprintf(stderr, "Error, Invalid Print Settings\n");
		exit(-1);
	}
	
	// Print depending of which axes are set to print
	if (Control->log_flags.PRNTACC.X == 1){
		fprintf(Control->files.dataFile, " %f,", Process_Calibrate_Value(Control, CALX));
	}
	if (Control->log_flags.PRNTACC.Y == 1){
		fprintf(Control->files.dataFile, " %f,", Process_Calibrate_Value(Control, CALY));
	}
	if (Control->log_flags.PRNTACC.Z == 1){
		fprintf(Control->files.dataFile, " %f,", Process_Calibrate_Value(Control, CALZ));
	}
	Control->samples.count ++;
	return 1;
}
//
// A Function to print a Time to file
//
int Rec_DataTime(struct Log_Control *Control){
	gettimeofday(&Control->sampleTime, NULL);
	fprintf(Control->files.dataFile, " %f\n", Process_Time_diff(Control->startTime, Control->sampleTime)/1000000);
	return 0;
}
//
//A Function to Record a Header to an opened DataFile
//
int Rec_dataFileHeader(struct Log_Control *Control){
	//Check that files exist
	if(Control->files.dataFile == NULL){
		fprintf(stderr, "Error, no dataFile to print to\n");
		fprintf(stderr, "No Header created in file\n");
		exit(-1);
	}
	
	int fields = Control->log_flags.PRNTACC.X+Control->log_flags.PRNTACC.Y+Control->log_flags.PRNTACC.Z;
	//Save Header to dataFile and output to console which axis will be recorded
	if (fields == 0){
		fprintf(stderr, "no accelerometer axes have been set to print\n");
		exit(-1);
	}
	if ((fields < 0)||(fields > 3 )){
		fprintf(stderr, "Error, Invalid Print Settings");
		exit(-1);
	}
	if (Control->log_flags.PRNTACC.X == 1){
		fprintf(Control->files.dataFile, " %s,", "X");
	}
	if (Control->log_flags.PRNTACC.Y == 1){
		fprintf(Control->files.dataFile, " %s,", "Y");
	}
	if (Control->log_flags.PRNTACC.Z == 1){
		fprintf(Control->files.dataFile, " %s,", "Z");
	}
	fprintf(Control->files.dataFile, " %s\n", "Time (s)");
	return 1;
}
	
//
//
/*****Print Functions*******/
//
//
//A function to Print the Control settings to the terminal and csv header file
//
int Print_Ctrl_Settings(struct Log_Control Control){
	
	int samples_max = Control.samples.max;
	float samples_wait = (float)Control.samples.wait/1000;
	float samples_LogDuration = (float)Control.samples.LogDuration/1000000;
	float Est_LogDuration = (float)samples_max*samples_wait/1000;
	float Est_samples = 1000*(float)(samples_LogDuration)/(float)(samples_wait);
	
	printf("LOGGING VARIABLES\n");
	
	// Print settings to terminal
	switch (Control.log_flags.Exit_type){
		
		case EXIT_TIME:
			printf("Exit_type: TIME\n");
			printf("Control Settings:\n");
			printf("			Est.samples: %f\n", Est_samples);
			printf("			samples.wait(ms): %f\n", samples_wait);
			printf("			samples.LogDuration(secs):%f\n", samples_LogDuration);
		break;
		
		case EXIT_SAMPLES:
			printf("Exit_type: SAMPLES\n");
			printf("Control Settings:\n");
			printf("			samples.max: %d\n", samples_max);
			printf("			samples.wait(ms): %f\n", samples_wait);
			printf("			Est.LogDuration(secs):%f\n", Est_LogDuration);
		break;
		
		case EXIT_BUTTONB:
			printf("Exit_type: BUTTONB\n");
			printf("Control Settings:\n");
			printf("			Est.samples: NA\n");
			printf("			samples.wait(ms): %f\n", samples_wait);
			printf("			Est.LogDuration(secs): NA\n");
		break;
		
		default:
			fprintf(stderr, "invalid Exit_type\n");
			exit(-1);
	};
	
	//Print whether data is being collected from mesgStream or State
	char *OffOrOn;
	if (Control.log_flags.mesgStream_on){
		OffOrOn = "On";
	}
	else {
		OffOrOn = "Off";
	}
	printf("ACCmesgStream: %s\n", OffOrOn);
	
	int fields = Control.log_flags.PRNTACC.X+Control.log_flags.PRNTACC.Y+Control.log_flags.PRNTACC.Z;
	//Save Header to dataFile and output to console which axis will be recorded
	if (fields == 0){
		printf("fields in Print_ctrl_settings: %d\n", fields);
		fprintf(stderr, "no accelerometer axes have been set to print\n");
		return -1;
	}
	if ((fields < 0)||(fields > 3 )){
		fprintf(stderr, "Error, Invalid Print Settings");
		exit(-1);
	}
	printf("Recording axes: ");
	if (Control.log_flags.PRNTACC.X == 1){
		printf("X ");
	}
	if (Control.log_flags.PRNTACC.Y == 1){
		printf("Y ");
	}
	if (Control.log_flags.PRNTACC.Z == 1){
		printf("Z ");
	}
	
	printf("\n\n");
	printf("Calibration Constants:\n mx:	%f\n cx:	%f\n my:	%f\n cy:	%f\n mz:	%f\n cz:	%f\n", 
		Control.Cal_Consts.mx, Control.Cal_Consts.cx, Control.Cal_Consts.my, Control.Cal_Consts.cy, Control.Cal_Consts.mz, Control.Cal_Consts.cz);
	
	return 1;
}
//
// A Function to countdown on the console before continuing
//
void Print_Countdown(int wait, int values){
	int i = 0;
	for (i = values; i >0; i--){
		printf("%d\n", i);
		usleep(wait);
	}
			
}
//
// A Function to print out the Controls for the application
//
void Print_App_Ctrls( void ) {
	printf("CONTROLS:			WIIMOTE BUTTON(S)\n\n");
	printf("Edit Settings:			+&- \n");
	printf("Calibrate Wiimote Acc Sensors:	1&2\n");
	printf("Start Logging:			A\n");
	printf("Stop Logging(if set):		B\n");
	printf("Exit RPiiMotionLogger:		Home\n\n");
}
//
//A function to print out the Edit Settings Controls
//
void Print_EditSet_Ctrls( void ) {

	printf("EDIT SETTINGS:\n");
	printf("Toggle Exit_Type:		UP\n");
	printf("Toggle sample.max:		RIGHT\n");
	printf("Toggle sample.wait:		DOWN\n");
	printf("Toggle sample.LogDuration:	LEFT\n");
	printf("Toggle ACC axes:		1 or 2\n");
	printf("Save and Exit:			+&-\n");
}
//
// A function to call whenever the contents of wiimote.data need to be printed out
//
void Print_wiimote_data(struct RPii *RPii, bool all){
	
	
	printf("This is a printout of the values of the variables in wiimote.data\n\n");
	RPii->Control = *(Pointto_Control_Data(RPii->wiimote));
	
	if (all){

		printf("RPii->Control.samples.count: %d\n",RPii->Control.samples.count);
		printf("RPii->Control.samples.max: %d\n", RPii->Control.samples.max);
		printf("RPii->Control.samples.wait: %d\n", RPii->Control.samples.wait);
		printf("RPii->Control.samples.LogDuration: %d\n\n", RPii->Control.samples.LogDuration);
	
		printf("RPii->Control.log_flags.Enable_Log: %s\n", (RPii->Control.log_flags.Enable_Log)?"true":"false");
		printf("RPii->Control.log_flags.End_Loop: %s\n", (RPii->Control.log_flags.End_Loop)?"true":"false");
		printf("RPii->Control.log_flags.dataFile_Open: %s\n", (RPii->Control.log_flags.dataFile_Open)?"true":"false");
		printf("RPii->Control.log_flags.PRNTACC.X: %d\n",RPii->Control.log_flags.PRNTACC.X);
		printf("RPii->Control.log_flags.PRNTACC.Y: %d\n",RPii->Control.log_flags.PRNTACC.Y);
		printf("RPii->Control.log_flags.PRNTACC.Z: %d\n\n",RPii->Control.log_flags.PRNTACC.Z);
	
		printf("RPii->Control.Cal_Consts.mx: %f\n",RPii->Control.Cal_Consts.mx);
		printf("RPii->Control.Cal_Consts.cx: %f\n", RPii->Control.Cal_Consts.cx );
		printf("RPii->Control.Cal_Consts.my: %f\n", RPii->Control.Cal_Consts.my);
		printf("RPii->Control.Cal_Consts.cy: %f\n", RPii->Control.Cal_Consts.cy);
		printf("RPii->Control.Cal_Consts.mz: %f\n", RPii->Control.Cal_Consts.mz);
		printf("RPii->Control.Cal_Consts.cz: %f\n\n", RPii->Control.Cal_Consts.cz);
	}
	
	printf("RPii->Control.Acc_Data.X: %d\n", RPii->Control.Acc_Data.X);
	printf("RPii->Control.Acc_Data.Y: %d\n", RPii->Control.Acc_Data.Y);
	printf("RPii->Control.Acc_Data.Z: %d\n\n", RPii->Control.Acc_Data.Z);
	
	printf("RPii->Control.btn_flags.A_flag: %s\n", (RPii->Control.btn_flags.A_flag)?"true":"false");
	printf("RPii->Control.btn_flags.B_flag: %s\n", (RPii->Control.btn_flags.B_flag)?"true":"false");
	printf("RPii->Control.btn_flags.HOME_flag: %s\n", (RPii->Control.btn_flags.HOME_flag)?"true":"false");
	printf("RPii->Control.btn_flags.PLUS_flag: %s\n", (RPii->Control.btn_flags.PLUS_flag)?"true":"false");
	printf("RPii->Control.btn_flags.MINUS_flag: %s\n",(RPii->Control.btn_flags.MINUS_flag )?"true":"false");
	printf("RPii->Control.btn_flags.RIGHT_flag: %s\n", (RPii->Control.btn_flags.RIGHT_flag)?"true":"false");
	printf("RPii->Control.btn_flags.LEFT_flag: %s\n",(RPii->Control.btn_flags.LEFT_flag)?"true":"false");
	printf("RPii->Control.btn_flags.UP_flag: %s\n",(RPii->Control.btn_flags.UP_flag)?"true":"false");
	printf("RPii->Control.btn_flags.DOWN_flag: %s\n", (RPii->Control.btn_flags.DOWN_flag)?"true":"false");
}
//
// A function to print out the contents of a control struct
//
void RPii_Prnt_Ctrl_Settings(struct Log_Control Control, bool all){
	
	
	printf("This is a printout of the values of the variables in the Log_Control structure passed\n\n");
	
	if (all){

		printf("Control.samples.count: %d\n",Control.samples.count);
		printf("Control.samples.max: %d\n", Control.samples.max);
		printf("Control.samples.wait: %d\n",  Control.samples.wait);
		printf("Control.samples.LogDuration: %d\n\n",  Control.samples.LogDuration);
	
		printf("Control.log_flags.Enable_Log: %s\n", ( Control.log_flags.Enable_Log)?"true":"false");
		printf("Control.log_flags.End_Loop: %s\n", ( Control.log_flags.End_Loop)?"true":"false");
		printf("Control.log_flags.dataFile_Open: %s\n", ( Control.log_flags.dataFile_Open)?"true":"false");
		printf("Control.log_flags.PRNTACC.X: %d\n", Control.log_flags.PRNTACC.X);
		printf("Control.log_flags.PRNTACC.Y: %d\n", Control.log_flags.PRNTACC.Y);
		printf("Control.log_flags.PRNTACC.Z: %d\n\n", Control.log_flags.PRNTACC.Z);
	
		printf("Control.Cal_Consts.mx: %f\n", Control.Cal_Consts.mx);
		printf("Control.Cal_Consts.cx: %f\n",  Control.Cal_Consts.cx );
		printf("Control.Cal_Consts.my: %f\n",  Control.Cal_Consts.my);
		printf("Control.Cal_Consts.cy: %f\n",  Control.Cal_Consts.cy);
		printf("Control.Cal_Consts.mz: %f\n",  Control.Cal_Consts.mz);
		printf("Control.Cal_Consts.cz: %f\n\n",  Control.Cal_Consts.cz);
	}
	
	printf("Control.Acc_Data.X: %d\n",  Control.Acc_Data.X);
	printf("Control.Acc_Data.Y: %d\n",  Control.Acc_Data.Y);
	printf("Control.Acc_Data.Z: %d\n\n",  Control.Acc_Data.Z);
	
	printf("Control.btn_flags.A_flag: %s\n", ( Control.btn_flags.A_flag)?"true":"false");
	printf("Control.btn_flags.B_flag: %s\n", ( Control.btn_flags.B_flag)?"true":"false");
	printf("Control.btn_flags.HOME_flag: %s\n", ( Control.btn_flags.HOME_flag)?"true":"false");
	printf("Control.btn_flags.PLUS_flag: %s\n", ( Control.btn_flags.PLUS_flag)?"true":"false");
	printf("Control.btn_flags.MINUS_flag: %s\n",( Control.btn_flags.MINUS_flag )?"true":"false");
	printf("Control.btn_flags.RIGHT_flag: %s\n", ( Control.btn_flags.RIGHT_flag)?"true":"false");
	printf("Control.btn_flags.LEFT_flag: %s\n",( Control.btn_flags.LEFT_flag)?"true":"false");
	printf("Control.btn_flags.UP_flag: %s\n",( Control.btn_flags.UP_flag)?"true":"false");
	printf("Control.btn_flags.DOWN_flag: %s\n", ( Control.btn_flags.DOWN_flag)?"true":"false");
}
//
/******LED Functions *******/
//
//
//Displays LEDs according to samplecount mod 4.
//
int LEDs_Mod4(struct RPii *RPii, int samplecount){
	
	unsigned char led_state = 0;
	
	led_state = 1 << (samplecount % 4);
	if(cwiid_set_led(RPii->wiimote, led_state)){
		return -1;
	}
	return 0;
}
//
//A function to provide a flash of LED activity to highlight the end of a process
//
int LEDs_All_Flash(struct RPii *RPii){
	int i;
	for(i = 0; i<4; i++){
		if( cwiid_set_led(RPii->wiimote, 0x0F) ){
			fprintf(stderr, "Report send error (led)\n");
			return -1;
		}
		usleep(150000);
		if( cwiid_set_led(RPii->wiimote, 0x00) ){
			fprintf(stderr, "Report send error (led)\n");
			return -1;
		}
		usleep(150000);
	}
	if( cwiid_set_led(RPii->wiimote, 0x01) ){
			fprintf(stderr, "Report send error (led)\n");
			return -1;
	}
	return 0;
}
//
