#! /usr/bin/env python
###############################################################################
#                                                                             #
#   Copyright 2005 University of Cambridge Computer Laboratory.               #
#                                                                             #
#   This file is part of Nprobe.                                              #
#                                                                             #
#   Nprobe is free software; you can redistribute it and/or modify            #
#   it under the terms of the GNU General Public License as published by      #
#   the Free Software Foundation; either version 2 of the License, or         #
#   (at your option) any later version.                                       #
#                                                                             #
#   Nprobe is distributed in the hope that it will be useful,                 #
#   but WITHOUT ANY WARRANTY; without even the implied warranty of            #
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             #
#   GNU General Public License for more details.                              #
#                                                                             #
#   You should have received a copy of the GNU General Public License         #
#   along with Nprobe; if not, write to the Free Software                     #
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA #
#                                                                             #
###############################################################################


##############################################################################
##############################################################################

#
# Emulate unsigned 32bit integer arithmetic using Python longs
#
from  np_longutil import Longstring

##############################################################################

SEQ_MAX = 0xffffffffL 

##############################################################################

#
# seq r = seq a + int b
#

def seq_add(a, b):

    r = a+b
    if r > 0xffffffffL:
	r = r - 0xffffffffL
    elif r < 0:
	# a < |b| b  < 0
	r = r + 0xffffffffL

    return r

##############################################################################

#
# seq r = seq a - int b
#

def seq_sub(a, b):

    r = a-b
    if r < 0:
	r = r + 0xffffffffL

    return r

##############################################################################

#
# int r = seq a - seq b
#

def seq_diff(a, b):

    if a == b:
	return 0

    r = a-b
##     print 'in seq_diff %s-%s = %s' % (Longstring(a, width=10), 
## 				   Longstring(b, width=10), 
## 				   Longstring(r, width=10)) 

    if r > 0L:
	if abs(r) <= 0x7fffffffL:
	    return int(r)
	else:
	    return int(r-0xffffffffL)

    else:
	if abs(r) <= 0x7fffffffL:
	    return int(r)
	else:
	    return int(r+0xffffffffL)
	

##############################################################################

def seq_gt(a, b):
    if a == b:
	return 0
    d = a - b
    if d > 0xfffffff:
	return 0
    elif d < 0 and d > -0xfffffff:
	return 0
    else:
	return 1

##############################################################################

def seq_lt(a, b):
    return seq_gt(b, a)

##############################################################################

def seq_gte(a, b):
    if a == b:
	return 1
    else:
	d = a - b
	if d > 0xfffffff:
	    return 0
	elif d < 0 and d > -0xfffffff:
	    return 0
	else:
	    return 1

##############################################################################

def seq_lte(a, b):
    return seq_gte(b, a)

##############################################################################

def seq_min(a, b):
    if seq_lt(a, b):
	return b
    else:
	return a

##############################################################################

def seq_max(a, b):
    if seq_gt(a, b):
	return a
    else:
	return b
    

##############################################################################
##############################################################################

# Call main when run as script to check correctness
	
gte_vals = [
    (0xffffffffL, 0xfffffffeL),
    (0xfffffffeL, 0xffffffffL),
    (0xfffffffeL, 0xfffL),
    (0xfffL, 0xffffffffL),
    (0xffffL, 0xfffeL),
    (0xfffeL, 0xffffL),
    (0xffffffffL, 0xffffffeL),
    (0xffffffffL, 0xfffffffL),
    (0xffffffffL, 0xfffffff1L),
]

add_vals = [
    (0xfffffffeL, 0),
    (0xfffffffeL, 1),
    (0xfffffffeL, 2),
    (0xffffffffL, 1),
    (0x7fffffffL, 0x7ffffffe),
    (0x7fffffffL, 0x7fffffff),
    (0x80000000L, 0x7fffffff),
    (0x80000001L, 0x7fffffff),
    (0x1L,         0x7ffffffe),
    (0x1L,         0x7fffffff),
    (0x1L,        1),
    (0x1L,        -1),
    (0x1L,        -2),
]

def main():

    for v in gte_vals: 
	l1 = v[0]
	l2 = v[1]
	if seq_gte(l1, l2):
	    s = 'T'
	else:
	    s = 'F'

	print '%s gte %s %s' % (Longstring(l1, width=10), Longstring(l2,width=10), s)

    print

    for v in add_vals: 
	l1 = v[0]
	l2 = v[1]
	s = seq_add(l1, l2)
	if s > 0xffffffffL or s < 0:
	    print 'WHOOPS - too large'
	print 'add seq %s + %6d = %s' % (Longstring(l1, width=10), l2, Longstring(s, width = 10))
	r = seq_diff(s, l1)
	if r != l2:
	    print 'WHOOPS'
	print 'seq diff %s - %s = %d' % (Longstring(s, width=10), Longstring(l1,width=10), r)

if __name__ == '__main__':
        main()
