/*
 * An implementation of the hold-release operations in which threads
 * encountering contention attempt to remove the obstructing thread by
 * suspending it and updating its PC.  
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/lwp.h>
#include <procfs.h>
#include <errno.h>
#include <synch.h>
#include <signal.h>
#include <schedctl.h>
#include "holdrel.h"

#define WREST_SPIN_LIMIT 1000

#define TAG_OWNED   1
#define TAG_UNOWNED 0

typedef union {
  struct {
    unsigned int val : 31;
    unsigned int tag : 1;
  } fields;
  word_t composite;
} hold_rec_t;

per_thread_t *per_thread[MAX_THREADS];

/*......................................................................*/

/* Signal fatal and unexpected error conditions */

void fail(char *msg) {
  fprintf(stderr, "%s\n", msg);
  abort();
}

/*......................................................................*/

void hr_init (per_thread_t *me, int idx)
{
  char     name[128];
  lwpid_t  lwp;
  thread_t thr;
  int      fd;
  int      i;

  per_thread[idx] = me;

  // We actually just need LWP since we assume bound threads, but keep
  // track of thr as well to prevent obscure bugs if the assumption fails
  lwp = _lwp_self();
  thr = thr_self();

  // Open control interface to LWP
  sprintf (name, "/proc/self/lwp/%d/lwpctl", lwp);
  fd = open (name, O_WRONLY);

  me -> fd_ctl = fd;
  me -> lwp = lwp;
  me -> thr = thr;
  me -> sc = schedctl_init ();
  me -> idx = idx;
  me -> holds_started = 0;
  me -> holds_completed = 0;
}

/*......................................................................*/

/*
 * The lwp-level operations displace threads to "displacement_target"
 * if ownership of a held location is wrested from them.  This
 * function is responsible for finding the appropriate per_thread_t
 * structure and resuming execution from the saved jmp_buf within it.
 * This approach simplifies the displacement code because we just need
 * to change the PC which can be achieved more easily through the
 * Solaris interface than updating the other registers.  
 */

void displacement_target(void)
{
  thread_t me;
  int      i;

  me = thr_self ();
  for (i = 1; per_thread[i] != NULL; i ++) 
    {
      if (per_thread[i] -> thr == me)
	{
	  if (per_thread[i] -> lwp != _lwp_self()) {
	    fail ("LWP changed");
	  }
	  release (per_thread[i]);
	  longjmp (per_thread[i] -> jb, 1);
	}
    }

  fail ("Could not find displacement target");
}

/*......................................................................*/

/*
 * Cause calling thread "me" to take ownership of address "addr".
 *
 * This implementation is blocking with portions executing under mutual
 * exclusion managed by lwp_suspend_lock.  This simplifies the design because
 * we do not need to consider concurrent executions of that portion of fetch.
 * 
 * We assume that actually displacing the current owner of "addr" is
 * likely to be extremely costly; we therefore spin up to
 * WREST_SPIN_LIMIT before attempting displacement.
 */

static lwp_mutex_t lwp_suspend_lock = {0};

hold_rec_t fetch (per_thread_t *me, addr_t addr)
{
  hold_rec_t seen;
  int        i;

  /*
   * First phase: wait for a while in the hope that the location if released
   * by the current thread (if any) that is holding it.
   */ 

  i = 0;
  seen.composite = *addr; 
  while (i < WREST_SPIN_LIMIT && seen.fields.tag == TAG_OWNED) {
    RMB();
    seen.composite = *addr;
    i++;
  }

  /*
   * Second phase: if the location is still held then attempt to
   * displace the current holder.
   */

  while (seen.fields.tag == TAG_OWNED) 
    {
      long command[4];
      int  tidx = seen.fields.val;
      int  r;
      int  cl;

      schedctl_start (me -> sc); // Hint: do not pre-empt

      /* Acquire suspension lock and attempt to suspend target thread.
       * The _lwp_suspend call may legitimately fail if the target
       * exits before we attemp to suspend it.
       */

      r = _lwp_mutex_lock (&lwp_suspend_lock);
      if (r != 0) fail ("_lwp_mutex_lock");
      r = _lwp_suspend (per_thread[tidx] -> lwp);

      if (r != ESRCH)
	{
	  /* We go through a series of operations to get access to the
           * PC of the suspended thread.  We need to get it stopped on
           * an event of interest which means stopping it with
           * PCDSTOP, then releasing its suspension and then waiting
           * for it with PCWSTOP.
	   */

	  if (r != 0) fail ("_lwp_suspend");
	  command[0] = PCDSTOP;
	  r = write(per_thread[tidx] -> fd_ctl, &command[0], sizeof (long));
	  if (r != sizeof(long)) { fail("PCDSTOP"); }

	  r = _lwp_continue (per_thread[tidx] -> lwp);
	  if (r != 0) fail ("_lwp_continue");

	  command[0] = PCWSTOP;
	  r = write(per_thread[tidx] -> fd_ctl, &command[0], sizeof (long));
	  if (r != sizeof(long)) fail ("PCWSTOP");
	  
	  /* Now we've got the target to a state where we can set its
	   * PC.  Check that it really is holding the location we are
	   * interested in (it may have ran if we waited to suspend
	   * it).  If so then move it to "displacement_target" and
	   * update its per_thread_t to indicate the revocation. 
	   */

	  cl = 0;
	  if (*addr == seen.composite) 
	    {
	      hold_rec_t desired;
	      RMB();
	      command[cl++] = PCSVADDR;
	      command[cl++] = (long) displacement_target;
	      desired.fields.tag = TAG_UNOWNED;
	      desired.fields.val = per_thread[tidx] -> displaced;
	      if (CASIO (addr,
			  seen.composite,
			  desired.composite) != seen.composite) {
		fail ("CAS unexpectedly failed");
	      }
	      per_thread[tidx] -> addr = NULL;

	      /* Target may have been mid-way through hold() */
	      per_thread[tidx] -> holds_completed = per_thread[tidx] -> holds_started;
	    }
	  
	  /* Resume target */
	  command[cl++] = PCRUN;
	  command[cl++] = 0;
	  r = write(per_thread[tidx] -> fd_ctl, &command[0], cl * sizeof (long));
	  if (r != cl * sizeof(long)) { fail ("PCRUN"); }
	}
      else
	{
	  r =_lwp_continue (per_thread[tidx] -> lwp);
	}
      
      r = _lwp_mutex_unlock (&lwp_suspend_lock);
      if (r != 0) fail ("_lwp_mutex_unlock");
      schedctl_stop (me -> sc); 
      seen.composite = *addr; 
    }

  return seen;
}

/*......................................................................*/

void hold (per_thread_t *me, addr_t addr) 
{
  hold_rec_t expected;
  hold_rec_t desired;

  me -> holds_started ++;

  do
    {
      // Revoke ownership if necessary
      expected = fetch (me, addr);

      // Displace current value and attempt to install ownership
      me -> addr = addr;
      me -> displaced = expected.fields.val;
      desired.fields.tag = TAG_OWNED;
      desired.fields.val = me -> idx;
      MB();
    }
  while (CASIO(addr, expected.composite, desired.composite) != expected.composite);

  me -> holds_completed ++;
}

/*......................................................................*/

void release (per_thread_t *me)
{
  hold_rec_t expected;
  hold_rec_t desired;
  addr_t     addr;

  addr = me -> addr;
  if (addr != NULL)
    {
      // Release ownership restoring the displaced value
      expected.fields.tag = TAG_OWNED;
      expected.fields.val = me -> idx;
      desired.fields.tag = TAG_UNOWNED;
      desired.fields.val = me -> displaced;
      if (CASIO (addr, expected.composite, desired.composite) != expected.composite) {
	fail ("CAS failed during release");
      }
      MB();
      me -> addr = NULL;
    }
}

/*......................................................................*/

word_t hr_read0 (per_thread_t *me, addr_t addr)
{
  word_t result;
  hold_rec_t seen;
    
  retry_read:    
    seen.composite = *addr;
    if (seen.fields.tag == TAG_OWNED)
      {
	int owner;
	unsigned int ver;
	addr_t owner_addr;
	owner = seen.fields.val;
	ver = per_thread[owner] -> holds_completed;
	RMB();
	owner_addr = per_thread[owner] -> addr;
	result = per_thread[owner] -> displaced;
	RMB();
	if ((owner_addr != addr) || 
	    (per_thread[owner] -> holds_started != ver))
	  {
	    goto retry_read;
	  }
      }
    else 
      {
	result = seen.fields.val;
      }

  return result;
}

/*......................................................................*/

void hr_write0 (per_thread_t *me, addr_t addr, word_t val)
{
    hold_rec_t expected, desired;
    int idx = me->idx;
    
    do
      {
	// Revoke ownership if necessary
	expected = fetch (me, addr);

	// Update location without taking ownership
	desired.fields.tag = TAG_UNOWNED;
	desired.fields.val = val;
      }
    while (CASIO(addr, 
		  expected.composite, 
		  desired.composite) != expected.composite);
}

/*......................................................................*/
