// Simple Key Search Protocol Client
// (c) Andy Brown 1995


#include "stdafx.h"
#include "sksp.h"
#include "SKSPclnt.h"
#include "resource.h"
#include "hourglas.h"
#include "misc.h"


#ifdef _DEBUG
FILE *dbg_out;
#endif


/***************/
/* Constructor */
/***************/

CSKSP::CSKSP()
{
// initialise the winsock DLL

  WSADATA wsaData;
	if(WSAStartup(0x0101,&wsaData)!=0)
	{
		AfxMessageBox(ERR_WINSOCKINIT);
		AfxAbort();
	}

// set internal variables

  m_Socket=INVALID_SOCKET;
  m_State=Ready;
  m_ErrorString.Empty();

#ifdef _DEBUG
  dbg_out=fopen("sksp.log","wb");
#endif
}


/**************/
/* Destructor */
/**************/

CSKSP::~CSKSP()
{
// kill off any active socket

  if(m_Socket!=INVALID_SOCKET)
    closesocket(m_Socket);

// clean up the sockets DLL

	WSACleanup();

#ifdef _DEBUG
  fclose(dbg_out);
#endif
}


/*************************/
/* Connect to the server */
/*************************/

void CSKSP::Start(CWnd *wnd,const CString& server,unsigned short port,
                  const CString& email,const CString& name,const CString& command)
{
  if(m_State!=Ready)
    return;

  CHourglass hg;

// set the internal variables

  m_Email=email;
  m_Name=name;
  m_Command=command;

	m_ReadBuffer.Empty();
	m_WriteBuffer.Empty();
  m_Advice.Empty();
  m_ReturnData.RemoveAll();

  m_Socket=INVALID_SOCKET;
  m_State=Ready;
  m_ErrorString.Empty();

// set up the connection parameters

	LPHOSTENT hostent;
  struct sockaddr_in si;

	if(IsIPAddress(server))												// numbers and dots == IP address
	{
		unsigned long ipaddr=inet_addr((const char *)server);

	// validate the IP address

		if(ipaddr==INADDR_NONE)
		{
			AfxMessageBox(ERR_INADDR);
			return;
		}

	// convert to a hostent

		if((hostent=gethostbyaddr((const char *)&ipaddr,4,PF_INET))==NULL)
		{
			AfxMessageBox(ERR_GETHOSTBYADDR);
			return;
		}

	}
	else																					// must be a hostname
	{
		if((hostent=gethostbyname((const char *)server))==NULL)
		{
			AfxMessageBox(ERR_GETHOSTBYNAME);
			return;
		}
	}

// copy the hostent entry and port into the sockaddr structure

	memcpy((LPSTR)&si.sin_addr,hostent->h_addr,hostent->h_length);
	si.sin_family=AF_INET;		
	si.sin_port=htons(port);

// create a socket for this session

	if((m_Socket=socket(PF_INET,SOCK_STREAM,0))==INVALID_SOCKET)
	{
    AfxMessageBox(ERR_SOCKET);
		return;
	}

// set asynchronous mode for all activity on this socket

	if(WSAAsyncSelect(m_Socket,wnd->GetSafeHwnd(),WM_SOCKET,FD_CLOSE|FD_READ|FD_WRITE|FD_CONNECT)==SOCKET_ERROR)
	{
		AfxMessageBox(ConvertSocketError(WSAGetLastError()));
		return;
	}

// set off the connection request to the SKSP server

	if((connect(m_Socket,(const struct sockaddr *)&si,sizeof(struct sockaddr)))==SOCKET_ERROR)
	{
		if(WSAGetLastError()!=WSAEWOULDBLOCK)
		{
      AfxMessageBox(ConvertSocketError(WSAGetLastError()));
			return;
		}
	}

// indicate that we are trying to connect

	m_State=Connecting;
  MYAPP->m_ViewWnd->EnableButtons(FALSE);
  SendUpdate(WM_SKSPUPDATE);
}


/*************************************/
/* Does this look like an IP address */
/*************************************/

BOOL CSKSP::IsIPAddress(const CString& host)
{
	for(int i=0;i<host.GetLength();i++)
		if((host[i]<'0' || host[i]>'9') && host[i]!='.')
			return FALSE;

	return TRUE;
}


/*********************/
/* display the error */
/*********************/

void CSKSP::DisplayError(void) const
{
  if(m_ErrorString.GetLength())
    AfxMessageBox(m_ErrorString);
}


/***********************************************/
/* Convert Winsock error to one of our strings */
/***********************************************/

int CSKSP::ConvertSocketError(const int err)
{
	const int codemap[21][2]=
	{
		WSANOTINITIALISED,IDS_WSANOTINITIALISED,
		WSAENETDOWN,IDS_WSAENETDOWN,
		WSAEAFNOSUPPORT,IDS_WSAEAFNOSUPPORT,
		WSAEINPROGRESS,IDS_WSAEINPROGRESS,
		WSAEMFILE,IDS_WSAEMFILE,
		WSAENOBUFS,IDS_WSAENOBUFS,
		WSAEPROTONOSUPPORT,IDS_WSAEPROTONOSUPPORT,
		WSAEPROTOTYPE,IDS_WSAEPROTOTYPE,
		WSAESOCKTNOSUPPORT,IDS_WSAESOCKTNOSUPPORT,
		WSAEADDRINUSE,IDS_WSAEADDRINUSE,
		WSAEFAULT,IDS_WSAEFAULT,
		WSAEINVAL,IDS_WSAEINVAL,
		WSAENOTSOCK,IDS_WSAENOTSOCK,
		WSAEISCONN,IDS_WSAEISCONN,
		WSAEOPNOTSUPP,IDS_WSAEOPNOTSUPP,
    WSAEADDRNOTAVAIL,IDS_WSAEADDRNOTAVAIL,
    WSAECONNREFUSED,IDS_WSAECONNREFUSED,
    WSAEDESTADDRREQ,IDS_WSAEDESTADDRREQ,
    WSAENETUNREACH,IDS_WSAENETUNREACH,
    WSAENOTCONN,IDS_WSAENOTCONN,
    WSAETIMEDOUT,IDS_WSAETIMEDOUT
	};

// locate the error code

	for(int i=0;i<21;i++)
		if(err==codemap[i][0])
			break;

	return(i==15 ? IDS_NETERROR : codemap[i][1]);
}


/*********************************/
/* Forcibly abort the connection */
/*********************************/

void CSKSP::Abort(void)
{
  if(m_Socket!=INVALID_SOCKET)
    closesocket(m_Socket);

  m_Socket=INVALID_SOCKET;
  m_State=Ready;
  m_ErrorString.Empty();
}


/**********************************/
/* Get a descriptive state string */
/**********************************/

CString CSKSP::GetStateString(void) const
{
  switch(m_State)
  {
    case Ready:
      return CString("Ready");
    case Connecting:
      return CString("Connecting to server");
    case ReadingGreeting:
      return CString("Reading greeting");
    case WritingHelo:
      return CString("Saying hello to the server");
    case ReadingHeloReply:
      return CString("Reading reply to hello");
    case WritingComm:
      return CString("Writing version to the server");
    case ReadingCommReply:
      return CString("Reading version reply from the server");
    case WritingCommand:
      return CString("Writing command");
    case ReadingReplyCode:
      return CString("Reading reply to command");
    case ReadingDataReply:
      return CString("Reading data from server");
    case WritingQuit:
      return CString("Writing quit command");
    case ReadingQuitReply:
      return CString("Reading reply to quit command");
    default:
      return CString("Unknown state - not good at all");
  }
}


/******************/
/* Read some data */
/******************/

void CSKSP::Read(void)
{
	if(m_State==Ready)
		return;

// Try reading 512 characters.  Responses from an SKSP server are generally
// a lot shorter than this so we should get any response in one try

	char buffer[513];

	int size=recv(m_Socket,buffer,512,0);
	if(size==SOCKET_ERROR && WSAGetLastError()!=WSAEWOULDBLOCK)
	{
		m_ErrorString.LoadString(ConvertSocketError(WSAGetLastError()));
		m_State=Ready;
    SendUpdate(WM_SKSPERROR);
		return;
	}

  if(size>0)
  {  
#ifdef _DEBUG
  fwrite(buffer,size,1,dbg_out);
#endif

    buffer[size]='\0';
	  m_ReadBuffer+=buffer;
    ProcessReadbuffer();
  }
}


/****************************************/
/* Process the read buffer if necessary */
/****************************************/

void CSKSP::ProcessReadbuffer(void)
{
// if the read buffer contains at least one \n then process the line

  int index;
  if((index=m_ReadBuffer.Find('\n'))!=-1)
  {
  // if we're reading multi-line data then get as many lines out of
  // m_ReadBuffer as we can

    if(m_State==ReadingDataReply)
    {
      do
      {
        CString str=m_ReadBuffer.Find('\r')==-1 ?
                    m_ReadBuffer.Left(index) :
                    m_ReadBuffer.Left(index-1);

        if(str==".")                            // end of data, move on
         AdvanceToNextState();
        else
        {
          if(str.GetLength() && str[0]=='.')
            str=str.Right(str.GetLength()-1);   // strip leading '.'

          m_ReturnData.Add(str);
        }
        if(m_ReadBuffer.GetLength())
          m_ReadBuffer=m_ReadBuffer.Right(m_ReadBuffer.GetLength()-index-1);
      } while((index=m_ReadBuffer.Find('\n'))!=-1);
    }
    else                                        // must be a status code, only one line
    {
      CString str=m_ReadBuffer.Find('\r')==-1 ?
                  m_ReadBuffer.Left(index) :
                  m_ReadBuffer.Left(index-1);

      if(str[0]=='5')                           // error, abort now
      {
        m_ErrorString.LoadString(ConvertSKSPError(str.Left(3)));
        SendUpdate(WM_SKSPERROR);
      }
      else if(str[0]=='6')
      {
        m_ErrorString=str.Right(str.GetLength()-4);
        SendUpdate(WM_SKSPERROR);
      }
      else
      {
        if(str.Left(3)=="210")                  // keys allocation
          m_ReturnData.Add(str.Right(str.GetLength()-4));
         AdvanceToNextState();                  // OK to continue
      }
    }
  } 
}


/*******************/
/* Write some data */
/*******************/

void CSKSP::Write(void)
{
	if(m_State==Ready)
		return;

// try to write out all of m_WriteBuffer in one go

  int len=m_WriteBuffer.GetLength();

  if(len>0)
  {
    int size=send(m_Socket,m_WriteBuffer,len,0);
    if(size==SOCKET_ERROR && WSAGetLastError()!=WSAEWOULDBLOCK)
    {
  		m_ErrorString.LoadString(ConvertSocketError(WSAGetLastError()));
	  	m_State=Ready;
      SendUpdate(WM_SKSPERROR);
		  return;
	  }
    else                                  // no error, shorten m_WriteBuffer as required
    {
#ifdef _DEBUG
      fwrite((const char *)m_WriteBuffer,size,1,dbg_out);
#endif
      if(size==len)
        m_WriteBuffer.Empty();
      else
        m_WriteBuffer=m_WriteBuffer.Right(len-size);
    }

		if(m_WriteBuffer.GetLength()==0)			// all gone, advance to next state
			AdvanceToNextState();
  }
}


/*************************/
/* Successful connection */
/*************************/

void CSKSP::Connect(void)
{
	ASSERT(m_State!=Ready);

// connection was successful

	m_State=ReadingGreeting;
}


/*********************/
/* connection closed */
/*********************/

void CSKSP::Close(void)
{
  if(m_Socket!=INVALID_SOCKET)
    closesocket(m_Socket);

  m_Socket=INVALID_SOCKET;
  m_State=Ready;
  m_ErrorString.Empty();
}


/*****************************/
/* Advance to the next state */
/*****************************/

void CSKSP::AdvanceToNextState(void)
{
  switch(m_State)
  {
    case ReadingGreeting:
      m_WriteBuffer=CString("HELO 1 ")+m_Email+CString(" ")+m_Name+CString("\r\n");
			m_State=WritingHelo;
      Write();
      SendUpdate(WM_SKSPUPDATE);
      break;

    case WritingHelo:
      m_State=ReadingHeloReply;
			m_ReadBuffer.Empty();
      SendUpdate(WM_SKSPUPDATE);
      break;

    case ReadingHeloReply:
      m_WriteBuffer="COMM W32C1.0\r\n";
      m_State=WritingComm;
      Write();
      SendUpdate(WM_SKSPUPDATE);
      break;

    case WritingComm:
      m_State=ReadingCommReply;
      m_ReadBuffer.Empty();
      SendUpdate(WM_SKSPUPDATE);
      break;

    case ReadingCommReply:
      if(m_ReadBuffer.Left(3)=="222")
        m_Advice=m_ReadBuffer.Right(m_ReadBuffer.GetLength()-4);

      m_WriteBuffer=m_Command+CString("\r\n");
			m_State=WritingCommand;
      Write();
      SendUpdate(WM_SKSPUPDATE);
      break;

    case WritingCommand:
      m_State=ReadingReplyCode;
			m_ReadBuffer.Empty();
      SendUpdate(WM_SKSPUPDATE);
      break;

    case ReadingReplyCode:
      if(m_ReadBuffer[0]=='3')                  // multi-line reply
      {
        m_State=ReadingDataReply;
        int index=m_ReadBuffer.Find('\n');      // strip off the command
        m_ReadBuffer=m_ReadBuffer.Right(m_ReadBuffer.GetLength()-index-1);
        if(m_ReadBuffer.GetLength())
          ProcessReadbuffer();
      }
      else
      {
        m_WriteBuffer="QUIT\r\n";
			  m_State=WritingQuit;
        Write();
      }
      SendUpdate(WM_SKSPUPDATE);
      break;

    case ReadingDataReply:
      m_WriteBuffer="QUIT\r\n";
  	  m_State=WritingQuit;
      Write();
      SendUpdate(WM_SKSPUPDATE);
      break;

    case WritingQuit:
      m_State=ReadingQuitReply;
      m_ReadBuffer.Empty();
      SendUpdate(WM_SKSPUPDATE);
      break;

    case ReadingQuitReply:
      m_State=Ready;
      SendUpdate(WM_SKSPCOMPLETED);
      break;
  }
}


/*************************************/
/* Send a message to the main window */
/*************************************/

void CSKSP::SendUpdate(UINT msg) const
{
  MYAPP->m_ViewWnd->ProcessMessage(msg);
}


/*************************************/
/* Convert SKSP error to resource ID */
/*************************************/

int CSKSP::ConvertSKSPError(const CString& errstr)
{
  int code;
  sscanf((const char *)errstr,"%d",&code);

  switch(code)
  {
    case 500:
      return SERR_SYNTAX;
    case 501:
      return SERR_ARGS;
    case 502:
      return SERR_VERSION;
    case 503:
      return SERR_COMMAND;
    case 510:
      return SERR_NOSUCHPROJECT;
    case 511:
      return SERR_PROJECTFINISHED;
    case 512:
      return SERR_CANTALLOCATE;
    case 513:
      return SERR_NOPROJECTS;
    case 514:
      return SERR_OUTOFRANGE;
    case 515:
      return SERR_CHECKSUM;
    case 550:
      return SERR_INTERNAL;
    case 551:
      return SERR_NOHELO;
    case 552:
      return SERR_HELO2;
    default:
      return SERR_UNKNOWN;
  }
}
