// Simple Key Search Protocol Client
// (c) Andy Brown 1995


#include "stdafx.h"
#include "search.h"
#include "startdlg.h"
#include "hourglas.h"
#include "resource.h"
#include "utils.h"


extern int time_trial(void);
extern int search(const char *fname, const char *check_s,const char *start_seg_s, 
                  const char *n_seg_s,char *result,int *abortnow);

UINT DoSearch(LPVOID pParam);


/***************/
/* Constructor */
/***************/

CSearch::CSearch()
{
}


/**************/
/* Destructor */
/**************/

CSearch::~CSearch()
{
}


/***************************/
/* Is project a known type */
/***************************/

BOOL CSearch::IsKnownType(const CString& type) const
{
  CString known,c(":");
  known.LoadString(IDS_TYPES);

  return known.Find(c+type+c)!=-1;
}


/******************/
/* Start a search */
/******************/

void CSearch::Start(const CProject *project,const CString& keystr)
{
// display the starting dialog

  CStartSearchDlg dlg;

  dlg.m_Type=project->GetType();
  dlg.m_Parameters=keystr;
  dlg.m_Time=GetTime(keystr,project->GetType(),GetKeysPerSecond(project->GetType()));
  dlg.m_Priority=2;

  if(dlg.DoModal()==IDOK)
  {
  // initialise a SEARCHDATA structure that will be passed as a parameter to the
  // worker thread that does the searching

    SEARCHDATA *sd=new SEARCHDATA;
    sd->keystr=new CString(keystr);
    sd->project=new CProject(*project);
    sd->result=(char *)GlobalAlloc(GPTR,256);
    sd->abortnow=FALSE;

  // create the worker thread in suspended mode

    CWinThread *thread=AfxBeginThread(DoSearch,
                                      sd,
                                      dlg.m_Priority==0 ? THREAD_PRIORITY_HIGHEST :
                                        dlg.m_Priority==1 ? THREAD_PRIORITY_ABOVE_NORMAL :
                                        dlg.m_Priority==2 ? THREAD_PRIORITY_NORMAL :
                                        dlg.m_Priority==3 ? THREAD_PRIORITY_BELOW_NORMAL :
                                        THREAD_PRIORITY_LOWEST,
                                      CREATE_SUSPENDED,
                                      NULL);

  // make the thread object persist after it's finished

    thread->m_bAutoDelete=FALSE;

  // add its address to the array that we are keeping
  
    m_Threads.Add(thread);
    m_ThreadData.Add(sd);
    m_SuspendedThreads.Add(0);

  // now it's OK to let the thread start processing

    thread->ResumeThread();
  }
}


/**********************************/
/* Get the estimated running time */
/**********************************/

CString CSearch::GetTime(const CString& keystr,const CString& type,int keyspersec)
{
// keystr is always in the format "start keys", advance to 2nd argument

  for(int i=0;!isspace(keystr[i]);i++);
  while(isspace(keystr[i])) i++;

// extract the decimal number of keys

  if(keyspersec==-1)
    return CString("unknown");

  int nKeys;
  sscanf(((const char *)keystr)+i,"%d",&nKeys);

// return the number of hours and minutes required

  unsigned long nMins;
  
  if(type=="ssl")
    nMins=MulDiv(nKeys,16777216,keyspersec);
    
  nMins/=60;

  char buffer[100];
  wsprintf(buffer,"%d hrs, %d mins",nMins/60,nMins%60);
  return CString(buffer);
}


/********************************************/
/* Get the keys/sec of this type of project */
/********************************************

return -1 if unknown */

int CSearch::GetKeysPerSecond(const CString& type)
{
  for(int i=0;i<m_KnownTypes.GetSize();i++)
    if(m_KnownTypes[i]==type)
      return m_KeysPerSecond[i];

  return -1;
}


/********************************/
/* Recalibrate the search speed */
/********************************

returns # keys per sec */

int CSearch::Recalibrate(const CString& type)
{
  int keyspersec;

// this may take some time...

  CHourglass hg;

// do the time trial if the type is known

  if(type=="ssl")
    keyspersec=(int)((unsigned long)16777216L/(unsigned long)time_trial());
  else
    return 0;

// locate and update the known keys/sec

  for(int i=0;i<m_KnownTypes.GetSize();i++)
  {
    if(m_KnownTypes[i]==type)
    {
      m_KeysPerSecond[i]=keyspersec;
      return keyspersec;
    }
  }

// not in array, add a new entry

  m_KnownTypes.Add(type);
  m_KeysPerSecond.Add(keyspersec);

  return keyspersec;
}


/**************************************/
/* Worker thread controlling function */
/**************************************

returns zero if the search is unsuccessful or is aborted, or one if
we hit the jackpot.  The result member of the SEARCHDATA structure is
filled out with the type-dependent result */

UINT DoSearch(LPVOID pParam)
{
  SEARCHDATA *sd=(SEARCHDATA *)pParam;

// things are different depending on the project type

  if(sd->project->GetType()=="ssl")
  {
  // create a temporary file with the project information in it

    char *tempfile=tmpnam(NULL);
    FILE *fp;
    if((fp=fopen(tempfile,"wb"))==NULL)
      return 0;

    CString work=sd->project->GetWork();
    if(fwrite((const char *)work,work.GetLength(),1,fp)!=1)
    {
      fclose(fp);
      remove(tempfile);
      return 0;
    }
    fclose(fp);

  // get the start and # of segments from the keystring

    char start[100],keys[100],*ptr;
    ptr=(char *)(const char *)(*(sd->keystr));

    for(int i=0;!isspace(ptr[i]);i++)
      start[i]=ptr[i];

    start[i]='\0';
    while(isspace(ptr[i]))
      i++;

    for(int j=0;!isspace(ptr[i]);i++,j++)
      keys[j]=ptr[i];
    keys[j]='\0';

  // get the checksum for the project

    unsigned short checksum=sd->project->GetChecksum();
    char cktext[20];
    sprintf(cktext,"%04x",(unsigned int)checksum);

  // kick off the search

    return(search(tempfile,cktext,start,keys,sd->result,&sd->abortnow)==0);
  }
  else
    return 0;
}


/*********************************/
/* Write the known types to disk */
/*********************************/

void CSearch::Write(CFile& file)
{
  ut_WriteStringArray(file,m_KnownTypes);

  ut_WriteLong(file,m_KeysPerSecond.GetSize());
  for(int i=0;i<m_KeysPerSecond.GetSize();i++)
    ut_WriteLong(file,m_KeysPerSecond[i]);
}


/**********************************/
/* Read the known types from disk */
/**********************************/

void CSearch::Read(CFile& file)
{
  ut_ReadStringArray(file,m_KnownTypes);

  int size=(int)ut_ReadLong(file);
  for(int i=0;i<size;i++)
    m_KeysPerSecond.Add(ut_ReadLong(file));
}
