// Simple Key Search Protocol Client
// (c) Andy Brown 1995


#include "stdafx.h"
#include "SKSPclnt.h"
#include "clview.h"
#include "projdlg.h"
#include "keysdlg.h"
#include "foundkey.h"
#include "misc.h"


#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif


IMPLEMENT_DYNCREATE(CClientView, CFormView)


/***************/
/* Message map */
/***************/

BEGIN_MESSAGE_MAP(CClientView, CFormView)
	//{{AFX_MSG_MAP(CClientView)
	ON_CBN_SELCHANGE(IDC_PROJECT, OnSelchangeProject)
	ON_LBN_SELCHANGE(IDC_ALLOCATED, OnSelchangeAllocated)
	ON_BN_CLICKED(IDC_RETRIEVE, OnRetrieve)
	ON_BN_CLICKED(IDC_UPDATE, OnUpdate)
	ON_BN_CLICKED(IDC_PROPERTIES, OnProperties)
	ON_BN_CLICKED(IDC_GETKEYS, OnGetkeys)
	ON_BN_CLICKED(IDC_REMOVE, OnRemove)
	ON_BN_CLICKED(IDC_REJECT, OnReject)
	ON_BN_CLICKED(IDC_ELIMINATE, OnEliminate)
	ON_BN_CLICKED(IDC_FOUNDKEY, OnFoundkey)
	ON_EN_CHANGE(IDC_PORT, OnChangePort)
	ON_CBN_EDITCHANGE(IDC_SERVERS, OnEditchangeServers)
	ON_CBN_SELCHANGE(IDC_SERVERS, OnSelchangeServers)
	ON_BN_CLICKED(IDC_CANCEL, OnCancel)
	ON_BN_CLICKED(IDC_SEARCH, OnSearch)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


/***************/
/* Constructor */
/***************/

CClientView::CClientView()
 : CFormView(CClientView::IDD)
{
	//{{AFX_DATA_INIT(CClientView)
	//}}AFX_DATA_INIT

  MYAPP->m_ViewWnd=this;
}


/**************/
/* Destructor */
/**************/

CClientView::~CClientView()
{
// nothing to do
}


/*************************/
/* Dynamic data exchange */
/*************************/

void CClientView::DoDataExchange(CDataExchange* pDX)
{
	CFormView::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CClientView)
	DDX_Control(pDX, IDC_SERVERS, m_Servers);
	DDX_Control(pDX, IDC_PORT, m_Port);
	DDX_Control(pDX, IDC_STATUS, m_Status);
	DDX_Control(pDX, IDC_ALLOCATED, m_Allocated);
	DDX_Control(pDX, IDC_SEARCH, m_Search);
	DDX_Control(pDX, IDC_ELIMINATE, m_Eliminate);
	DDX_Control(pDX, IDC_REJECT, m_Reject);
	DDX_Control(pDX, IDC_FOUNDKEY, m_FoundKey);
	DDX_Control(pDX, IDC_PROJECT, m_Project);
	DDX_Control(pDX, IDC_UPDATE, m_Update);
	DDX_Control(pDX, IDC_REMOVE, m_Remove);
	DDX_Control(pDX, IDC_PROPERTIES, m_Properties);
	DDX_Control(pDX, IDC_GETKEYS, m_GetKeys);
	DDX_Control(pDX, IDC_RETRIEVE, m_Retrieve);
	DDX_Control(pDX, IDC_CANCEL, m_Cancel);
	//}}AFX_DATA_MAP
}


#ifdef _DEBUG

void CClientView::AssertValid() const
{
	CFormView::AssertValid();
}

void CClientView::Dump(CDumpContext& dc) const
{
	CFormView::Dump(dc);
}

CClientDoc *CClientView::GetDocument()
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CClientDoc)));
	return (CClientDoc *)m_pDocument;
}

#endif


/******************/
/* Initial update */
/******************/

void CClientView::OnInitialUpdate() 
{
  ResizeParentToFit(FALSE);
  CFormView::OnInitialUpdate();
}


/**********************************/
/* Enable/disable retrieve button */
/**********************************/

void CClientView::EnableRetrieve(void)
{
  CClientDoc *pDoc=GetDocument();

  m_Retrieve.EnableWindow(
    GetDlgItem(IDC_SERVERS)->GetWindowTextLength()>0 &&
    GetDlgItem(IDC_PORT)->GetWindowTextLength()>0 &&
    pDoc->m_Email.GetLength()>0 &&
    pDoc->m_Name.GetLength()>0);
}


/*************************/
/* new project selection */
/*************************/

void CClientView::OnSelchangeProject() 
{
  RefillKeysList();
  SetControls();
}


/*********************************/
/* Set the control button states */
/*********************************/

void CClientView::SetControls(void)
{
  int nPrj=m_Project.GetCurSel();
  BOOL bEnable=nPrj!=CB_ERR;
  
  m_GetKeys.EnableWindow(bEnable);	
  m_Properties.EnableWindow(bEnable);
  m_Remove.EnableWindow(bEnable);
  m_Update.EnableWindow(bEnable);

	bEnable=m_Allocated.GetCurSel()!=LB_ERR;

  CProject *project=nPrj==CB_ERR ? NULL : GetDocument()->m_Projects[nPrj];
	m_Eliminate.EnableWindow(bEnable && project!=NULL && project->GetWork().GetLength());
  m_Reject.EnableWindow(bEnable);
  m_FoundKey.EnableWindow(bEnable);
  m_Search.EnableWindow(bEnable && project!=NULL && MYAPP->m_Search.IsKnownType(project->GetType()));
}


/***********************************/
/* Allocated key selection changed */
/***********************************/

void CClientView::OnSelchangeAllocated() 
{
  SetControls();
}


/***********************/
/* Port number changed */
/***********************/

void CClientView::OnChangePort() 
{
  EnableRetrieve();
}


/******************/
/* Server changed */
/******************/

void CClientView::OnEditchangeServers() 
{
	EnableRetrieve();
}


/***************************/
/* Server picked from list */
/***************************/

void CClientView::OnSelchangeServers() 
{
  CString server,newserver;
  unsigned short port;
  CClientDoc *pDoc=GetDocument();

// get selected server

  int index=m_Servers.GetCurSel();
  if(index==CB_ERR)
    return;
  m_Servers.GetLBText(index,newserver);

// search in known server array

  for(int i=0;i<pDoc->m_Servers.GetSize();i++)
  {
    pDoc->m_Servers.GetAt(i,server,port);
    if(newserver==server)
    {
      char buffer[20];
      wsprintf(buffer,"%hu",port);
      m_Port.SetWindowText(buffer);
      m_Retrieve.EnableWindow(TRUE);
      return;
    }
  }
}


/***************************/
/* Refill the project list */
/***************************/

void CClientView::RefillProjectList(void)
{
  m_Project.SetRedraw(FALSE);
  m_Project.ResetContent();

  CProjectArray *array=&GetDocument()->m_Projects;
  for(int i=0;i<array->GetSize();i++)
    m_Project.AddString((*array)[i]->GetDescription());

  m_Project.SetRedraw(TRUE);

  if(array->GetSize())
    m_Project.SetCurSel(0);

  SetControls();
}


/****************************/
/* Refill the keys list box */
/****************************/

void CClientView::RefillKeysList(void)
{
  m_Allocated.SetRedraw(FALSE);
  m_Allocated.ResetContent();

  int index=m_Project.GetCurSel();
  if(index!=CB_ERR)
  {
    CStringArray *keys=GetDocument()->m_Projects[index]->GetAllocated();
    for(int i=0;i<keys->GetSize();i++)
      m_Allocated.AddString((*keys)[i]);
  }

  m_Allocated.SetRedraw(TRUE);
  SetControls();
}


/***************************/
/* Refill the servers list */
/***************************/

void CClientView::RefillServerList(void)
{
  m_Servers.SetRedraw(FALSE);
  m_Servers.ResetContent();

  CClientDoc *pDoc=GetDocument();
  unsigned short port;
  CString server;

  for(int i=0;i<pDoc->m_Servers.GetSize();i++)
  {
    pDoc->m_Servers.GetAt(i,server,port);
    m_Servers.AddString(server);
  }

  m_Servers.SetRedraw(TRUE);
}


/*********************************/
/* Intercept our socket messages */
/*********************************/

BOOL CClientView::PreTranslateMessage(MSG* pMsg) 
{
// check to see if message concerns our socket

	if(pMsg->message==WM_SOCKET)
	{
		int err=(int)WSAGETSELECTERROR(pMsg->lParam);
    if(err)
    {
      MYAPP->m_SKSP.Abort();
      EnableButtons(TRUE);

      CString errstr;
      errstr.LoadString(CSKSP::ConvertSocketError(err));
      AfxMessageBox(errstr);
      return 1;
    }

		SOCKET s=(SOCKET)pMsg->wParam;

		switch(WSAGETSELECTEVENT(pMsg->lParam))
		{
			case FD_READ:															// data is ready to read
				MYAPP->m_SKSP.Read();
				return 1;
			case FD_WRITE:														// ready to write
				MYAPP->m_SKSP.Write();
				return 1;
			case FD_CLOSE:														// socket has closed
        if(MYAPP->m_SKSP.IsRunning())
          EnableButtons(TRUE);
				MYAPP->m_SKSP.Close();
				return 1;
			case FD_CONNECT:													// a connection has been made
				MYAPP->m_SKSP.Connect();
				return 1;
		}
	}

// let the framework get on with processing the message

	return CFormView::PreTranslateMessage(pMsg);
}


/***********************************/
/* Process an SKSP command message */
/***********************************/

void CClientView::ProcessMessage(UINT msg)
{
  switch(msg)
  {
    case WM_SKSPERROR:                          // communications error
      MYAPP->m_SKSP.DisplayError();
      MYAPP->m_SKSP.Abort();
      EnableButtons(TRUE);
      break;

    case WM_SKSPUPDATE:                         // update status display
      m_Status.SetWindowText(MYAPP->m_SKSP.GetStateString());
      m_Status.UpdateWindow();
      break;

    case WM_SKSPCOMPLETED:
      EnableButtons(TRUE);

      if(MYAPP->m_SKSP.GetCommand().Left(4)=="INFO")
        ProcessInfo();
      else if(MYAPP->m_SKSP.GetCommand().Left(4)=="WORK")
        ProcessWork();
      else if(MYAPP->m_SKSP.GetCommand().Left(4)=="KEYS")
        ProcessKeys();
      else if(MYAPP->m_SKSP.GetCommand().Left(3)=="ACK")
        ProcessAck();

      if(MYAPP->m_SKSP.GetAdvice().GetLength())
      {
        CString advice=CString("Advice returned from server: ")+MYAPP->m_SKSP.GetAdvice();
        AfxMessageBox(advice);
      }
      break;
  }
}


/*****************************/
/* Cancel button was pressed */
/*****************************/

void CClientView::OnCancel() 
{
  MYAPP->m_SKSP.Abort();
  EnableButtons(TRUE);
}


/*******************************/
/* Disable/restore all buttons */
/*******************************/

void CClientView::EnableButtons(const BOOL bEnable)
{
  static BOOL states[9];

  if(bEnable)
  {
    m_Retrieve.EnableWindow(states[0]);
    m_GetKeys.EnableWindow(states[1]);
    m_Properties.EnableWindow(states[2]);
    m_Remove.EnableWindow(states[3]);
    m_Update.EnableWindow(states[4]);
    m_Eliminate.EnableWindow(states[5]);
    m_Reject.EnableWindow(states[6]);
    m_FoundKey.EnableWindow(states[7]);
    m_Search.EnableWindow(states[8]);
  }
  else
  {
    states[0]=m_Retrieve.EnableWindow(FALSE)==0;
    states[1]=m_GetKeys.EnableWindow(FALSE)==0;
    states[2]=m_Properties.EnableWindow(FALSE)==0;
    states[3]=m_Remove.EnableWindow(FALSE)==0;
    states[4]=m_Update.EnableWindow(FALSE)==0;
    states[5]=m_Eliminate.EnableWindow(FALSE)==0;
    states[6]=m_Reject.EnableWindow(FALSE)==0;
    states[7]=m_FoundKey.EnableWindow(FALSE)==0;
    states[8]=m_Search.EnableWindow(FALSE)==0;
  }

  m_Cancel.ShowWindow(bEnable ? SW_HIDE : SW_SHOW);
  if(bEnable)
    m_Status.SetWindowText("Ready."); 
}


/***************************/
/* Process an INFO command */
/***************************/

void CClientView::ProcessInfo(void)
{
// pass through each line of data returned by the server

  CStringArray *array=MYAPP->m_SKSP.GetReturnData();
  CProjectArray *projects=&GetDocument()->m_Projects;

  for(int i=0;i<array->GetSize();i++)
  {
  // break down the : separated string

    char *buffer=(char *)GlobalAlloc(GPTR,(*array)[i].GetLength()+1);
    ASSERT(buffer!=NULL);

    lstrcpy(buffer,(const char *)(*array)[i]);
    for(int j=0;buffer[j];j++)
      if(buffer[j]==':')
        buffer[j]='\0';

    char *id,*type,*description,*status,*email,*server;

    id=buffer;
    type=id+lstrlen(id)+1;
    description=type+lstrlen(type)+1;
    status=description+lstrlen(description)+1;
    email=status+lstrlen(status)+1;
    server=email+lstrlen(email)+1;

  // check ID against known IDs

    for(j=0;j<projects->GetSize();j++)
      if((*projects)[j]->GetID()==id)
        break;

    if(j==projects->GetSize())                  // create a new project if ID is unknown
    {
      CProject *prj=new CProject(id,type,description,status,email,server,GetServer(),GetPort());
      projects->Add(prj);
    }
    else                                        // update the current project
      (*projects)[j]->Set(id,type,description,status,email,server,GetServer(),GetPort());

    GlobalFree((HGLOBAL)buffer);
  }

// INFO was successful, so we must have a good server/port - add it to the list

  GetDocument()->m_Servers.Add(GetServer(),GetPort());

// refill the project list

  RefillProjectList();
  RefillKeysList();
  RefillServerList();
}


/**************************/
/* Process a WORK command */
/**************************/

void CClientView::ProcessWork(void)
{
  CStringArray *array=MYAPP->m_SKSP.GetReturnData();
  CProjectArray *projects=&GetDocument()->m_Projects;

// get the project identifier that we were getting work for

  CString pid=MYAPP->m_SKSP.GetCommand();
  int index=pid.Find(' ');

  if(index==-1)
    return;

  pid=pid.Right(pid.GetLength()-index-1);

// locate the correct project

  if((index=GetProject(pid))!=-1)
   (*projects)[index]->UpdateWork(array);
}


/**************************/
/* Process a KEYS command */
/**************************/

void CClientView::ProcessKeys(void)
{
// get the project identifier that we were getting work for

  CString pid=MYAPP->m_SKSP.GetCommand();
  int index=pid.Find(' ');

  if(index==-1)
    return;

  pid=pid.Right(pid.GetLength()-index-1);
  index=pid.Find(' ');

  if(index==-1)
    return;

  pid=pid.Left(index);

// get the project from the ID
  
  if((index=GetProject(pid))==-1)
    return;

// add the keys to the project

  CProjectArray *projects=&GetDocument()->m_Projects;
  CStringArray *array=MYAPP->m_SKSP.GetReturnData();

// it is possible for the server to not return any information

  if(array->GetSize())
    (*projects)[index]->AddKeys((*array)[0]);
  else
    AfxMessageBox(ERR_NODATA);

// refill the keys list box

  RefillKeysList();
}


/**************************/
/* Process an ACK command */
/**************************/

void CClientView::ProcessAck(void)
{
// if it's a FOUND ACK then don't do anything

  CString cmd=MYAPP->m_SKSP.GetCommand();
  int index=cmd.Find("FOUND");
  if(index!=-1)
    return;

  char *addr=(char *)(const char *)cmd;
  for(int i=0;i<4;i++)
  {
    while(!isspace(*addr++));
    addr--;
    while(isspace(*addr++));
    addr--;
  }
  CString arg(addr);

// get the project identifier that we were getting work for

  index=cmd.Find(' ');
  if(index==-1)
    return;

  CString pid=cmd.Right(cmd.GetLength()-index-1);
  index=pid.Find(' ');

  if(index==-1)
    return;

  pid=pid.Left(index);

// get the project from the ID
  
  if((index=GetProject(pid))==-1)
    return;

// locate the key string from the project list and delete it

  CProjectArray *projects=&GetDocument()->m_Projects;
  CStringArray *keys=(*projects)[index]->GetAllocated();

  for(i=0;i<keys->GetSize();i++)
  {
    if((*keys)[i]==arg)
    {
      keys->RemoveAt(i);
      RefillProjectList();
      RefillKeysList();
      return;
    }
  }
}


/********************/
/* Get project info */
/********************/

void CClientView::OnRetrieve() 
{
  UpdateData(TRUE);

  CClientDoc *pDoc=GetDocument();
  MYAPP->m_SKSP.Start(this,
                      GetServer(),
                      GetPort(),
                      pDoc->m_Email,
                      pDoc->m_Name,
                      "INFO");
}


/**********************************/
/* Update the project information */
/**********************************/

void CClientView::OnUpdate() 
{
// get the selected project

  int index=m_Project.GetCurSel();
  if(index==CB_ERR)
    return;

  CClientDoc *pDoc=GetDocument();
  CProjectArray *array=&pDoc->m_Projects;

  if(index>=array->GetSize())
    return;

// construct an SKSP command

  CString cmd=CString("WORK ")+(*array)[index]->GetID();
  MYAPP->m_SKSP.Start(this,
                      (*array)[index]->GetWorkServer(),
                      (*array)[index]->GetWorkPort(),
                      pDoc->m_Email,
                      pDoc->m_Name,
                      cmd);
}


/*******************************/
/* Show the project properties */
/*******************************/

void CClientView::OnProperties() 
{
// get the selected project

  int index=m_Project.GetCurSel();
  if(index==CB_ERR)
    return;

// show the dialog box

  CProjectArray *array=&GetDocument()->m_Projects;
	CProjectDlg dlg;

  dlg.m_ID=(*array)[index]->GetID();
  dlg.m_Type=(*array)[index]->GetType();
  dlg.m_Description=(*array)[index]->GetDescription();
  dlg.m_Status=(*array)[index]->GetStatus();
  dlg.m_Manager=(*array)[index]->GetEmail();
  dlg.m_Server=(*array)[index]->GetKeyServer();

  char buffer[20];
  wsprintf(buffer,"%d",(int)(*array)[index]->GetKeyPort());
  dlg.m_Port=buffer;

  CString work=(*array)[index]->GetWork();
  if(work.GetLength())
  {
    CTime t((*array)[index]->GetDate());
    dlg.m_Update=t.Format("%#c");
    dlg.m_Work=work;
  }
  else
  {
    dlg.m_Update="Not yet updated";
    dlg.m_Work="(project requires updating)";
  }

  dlg.DoModal();
}


/******************************/
/* Get a project from its PID */
/******************************

return -1 if not found */

int CClientView::GetProject(const CString& pid)
{
  CProjectArray *projects=&GetDocument()->m_Projects;

  for(int i=0;i<projects->GetSize();i++)
    if((*projects)[i]->GetID()==pid)
      return i;

  return -1;
}


/*********************************/
/* Get some keys from the server */
/*********************************/

void CClientView::OnGetkeys() 
{	
// get the selected project

  int index=m_Project.GetCurSel();
  if(index==CB_ERR)
    return;

  CClientDoc *pDoc=GetDocument();
  CProjectArray *array=&pDoc->m_Projects;

  if(index>=array->GetSize())
    return;

// ask the user how many keys to get

  CGetKeysDlg dlg;
  if(dlg.DoModal()==IDOK)
  {
  // make and start a KEYS command

    if(!dlg.m_Keys)
      AfxMessageBox(ERR_ZEROKEYS);
    else
    {
      char buffer[50];
      wsprintf(buffer,"%lu",dlg.m_Keys);
      CString cmd=CString("KEYS ")+(*array)[index]->GetID()+CString(" ")+buffer;

      MYAPP->m_SKSP.Start(this,
                          (*array)[index]->GetKeyServer(),
                          (*array)[index]->GetKeyPort(),
                          pDoc->m_Email,
                          pDoc->m_Name,
                          cmd);
    }
  }
}


/**********************************/
/* Remove a project from the list */
/**********************************/

void CClientView::OnRemove() 
{
// get the selected project

  int index=m_Project.GetCurSel();
  if(index==CB_ERR)
    return;

// give the user a chance to back off

  if(AfxMessageBox(IDS_REALLYREMOVE,MB_YESNO)==IDNO)
    return;

// remove the project from the array

  CProjectArray *array=&GetDocument()->m_Projects;
  array->RemoveAt(index);

// redraw projects and keys

  RefillProjectList();
  RefillKeysList();
}


/**************************/
/* Reject a range of keys */
/**************************/

void CClientView::OnReject() 
{
// get the selected project

  int index=m_Project.GetCurSel();
  if(index==CB_ERR)
    return;

  CClientDoc *pDoc=GetDocument();
  CProjectArray *array=&pDoc->m_Projects;

  if(index>=array->GetSize())
    return;

// get the selected keys

  int keyidx=m_Allocated.GetCurSel();
  if(index==LB_ERR)
    return;

// give the user a chance to back off

  if(AfxMessageBox(IDS_REALLYREJECT,MB_YESNO)==IDNO)
    return;

// construct the ACK command

  CString cmd=CString("ACK ")+
              (*array)[index]->GetID()+
              CString(" REJECTED 0 ")+
              (*(*array)[index]->GetAllocated())[keyidx];

// start the communication

  MYAPP->m_SKSP.Start(this,
                      (*array)[index]->GetKeyServer(),
                      (*array)[index]->GetKeyPort(),
                      pDoc->m_Email,
                      pDoc->m_Name,
                      cmd);
}


/*****************************/
/* Eliminate a range of keys */
/*****************************/

void CClientView::OnEliminate() 
{
// get the selected project

  int index=m_Project.GetCurSel();
  if(index==CB_ERR)
    return;

  CClientDoc *pDoc=GetDocument();
  CProjectArray *array=&pDoc->m_Projects;

  if(index>=array->GetSize())
    return;

// get the selected keys

  int keyidx=m_Allocated.GetCurSel();
  if(index==LB_ERR)
    return;

// give the user a chance to back off

  if(AfxMessageBox(IDS_REALLYELIMINATE,MB_YESNO)==IDNO)
    return;

// checksum the project file (WORK command) and add in the parameters

  char buffer[200];
  unsigned short checksum=(*array)[index]->GetChecksum();
  unsigned long start,count;
  CString allocated=(*(*array)[index]->GetAllocated())[keyidx];

  sscanf((const char *)allocated,"%lx%lu",&start,&count);
  checksum+=(unsigned short)start+(unsigned short)count;
  wsprintf(buffer,"%x ",(unsigned int)checksum);

// construct the ACK command

  CString cmd=CString("ACK ")+
              (*array)[index]->GetID()+
              CString(" SEARCHED ")+
              CString(buffer)+
              allocated;

// start the communication

  MYAPP->m_SKSP.Start(this,
                      (*array)[index]->GetKeyServer(),
                      (*array)[index]->GetKeyPort(),
                      pDoc->m_Email,
                      pDoc->m_Name,
                      cmd);
}


/****************************/
/* Report successful search */
/****************************/

void CClientView::OnFoundkey() 
{
// get the selected project

  int index=m_Project.GetCurSel();
  if(index==CB_ERR)
    return;

  CClientDoc *pDoc=GetDocument();
  CProjectArray *array=&pDoc->m_Projects;

  if(index>=array->GetSize())
    return;

// ask the user which is the successful key

  CFoundKeyDlg dlg;
  if(dlg.DoModal()==IDOK)
  {
  // give the user a chance to back off

    if(AfxMessageBox(IDS_REALLYFOUND,MB_YESNO)==IDNO)
      return;

  // construct the ACK command

    CString cmd=CString("ACK ")+
                (*array)[index]->GetID()+
                CString(" FOUND 0 ")+
                dlg.m_Key+
                CString(" 0");

  // start the SKSP session

    MYAPP->m_SKSP.Start(this,
                        (*array)[index]->GetKeyServer(),
                        (*array)[index]->GetKeyPort(),
                        pDoc->m_Email,
                        pDoc->m_Name,
                        cmd);
  }
}


/*************************/
/* Get the server string */
/*************************/

CString CClientView::GetServer(void)
{
  CString str;
  m_Servers.GetWindowText(str);
  return str;
}


/****************/
/* Get the port */
/****************/

unsigned short CClientView::GetPort(void)
{
  CString str;
  m_Port.GetWindowText(str);
  unsigned short port;
  sscanf((const char *)str,"%hu",&port);

  return port;
}


/**************************************/
/* Start searching a particular range */
/**************************************/

void CClientView::OnSearch() 
{
// get the selected project

  int nPrj=m_Project.GetCurSel();
  if(nPrj==CB_ERR)
    return;

  CClientDoc *pDoc=GetDocument();
  CProjectArray *array=&pDoc->m_Projects;

  if(nPrj>=array->GetSize())
    return;

  int nAlloc=m_Allocated.GetCurSel();
  if(nAlloc==LB_ERR)
    return;

  CString keystr;
  m_Allocated.GetText(nAlloc,keystr);

// start the search

  MYAPP->m_Search.Start((*array)[nPrj],keystr);
}
