/*----------------------------------------------------------------------------

                                brutessl 0.5

                        Copyright (C) 1995 Andrew Roos
                             All Rights Reserved

This program is licensed for academic and educational use only. It may not be
used  for  any commercial purpose. You may modify the program if you wish to,
provided that the original copyright  notice  and  license  restrictions  are
retained,  and  that  you include a notice stating that you have modified the
program and giving details of the changes that you have made. This program is
distributed  without  any warranty including, but not limited to, the implied 
warranty of merchantability or fitness for a particular purpose.  

synopsis:  A program for brute-force searching SSL data which has been 
           encrypted using 40-bit (export) RC4 keys.

usage:	   brutessl [-q] -t [hh:[mm]]
		   brutessl [-q] <file> <checksum> <start segment> <no of segments>
		   brutessl [-q] -r <start segment> [<no of segments>]

Andrew Roos <andrewr@vironix.co.za>

Hacked for inclusion in the Win32 search client by Andy Brown
<a.brown@nexor.co.uk>.

----------------------------------------------------------------------------*/


#include "stdafx.h"
#include <stdarg.h>
#include "brutessl.h"


/*-------------------------------- Macros ----------------------------------*/

#define TIMER_WAIT 				200000U		/* iterations waiting for timer */
#define SSL_MT_CLIENT_FINISHED	3			/* client-finished message id */
#define SSL_MT_SERVER_VERIFY	5 			/* server-verify message id */

/*-------------------------------- Constants -------------------------------*/

const char *version = "0.5";

/* Test data. The test used requires 2^19 iterations to find the key. */

const byte test_ssl_block[52] = {
0x51,0xb7,0x63,0xf8,0xe0,0xe0,0xa6,0x33,0x5a,0x2d,0xcc,0xd5,0x7b,0x00,0x00,0x06,
0x30,0x1d,0x56,0x7b,0xce,0x00,0x9d,0x1d,0xc5,0xbd,0xb8,0x11,0x56,0x07,0x21,0x3e,
0xd9,0xa5,0x16,0x38,0x9f,0x72,0x8f,0x67,0x85,0xa9,0x73,0x66,0xc9,0x5f,0x0b,0xa9,
0x1c,0x80,0x00,0x00
};

const byte test_stream[8] = { 0xd2,0x19,0x75,0x76,0xb9,0x1c,0xf6,0xc1 };

const byte test_master[16] = {
0x51,0xb7,0x63,0xf8,0xe0,0xe0,0xa6,0x33,0x5a,0x2d,0xcc,0xd5,0x7b,0xff,0xff,0x07
};

/*---------------------------- Global Variables ----------------------------*/

int big_endian;						/* true if MSB of word stored first */
static int quiet=1;	  		/* terse output, set by the -q flag */
static char* myname = "brutessl";	/* program name taken from argv[0] */

/*--------------------------- Function Prototypes --------------------------*/

/* parameter file parsing */
static unsigned parse_file(const char *fname, ssl_block *ssl, byte *stream);
static void check_checksum(unsigned checksum, const char *check_s);

/* key search */
static int  search_segment(ssl_block *ssl, word seg, const byte *stream, int *abortnow);

/* I/O functions */
static int  get_hex_bytes(const char *s, byte *dest, int max_bytes);
static int  hex_val(char hex_digit);
static void eureka(ssl_block *ssl);
static void print_bytes(const byte *data, int n);
static void print_time(word secs, int print_secs);

/* Diagnostics */
static int  check_hardware(void);
int  time_trial(void);
static word time_in_ms(void);

/* Miscellaneous */
static void set_low_priority(void);
static void usage(void);
static void fatal(const char* format,...);

/*-------------------------------- Top Level -------------------------------*/


int search(const char *fname, const char *check_s, const char *start_seg_s, 
	const char *n_seg_s,char *result,int *abortnow)
/* Perform a key search, reading parameter file "fname", or standard input */
/* if fname is "-". start_seg_s and n_seg_s are string representations of  */
/* starting segment (in hex) and number of segments (decimal). Returns 0   */
/* if successful, 1 otherwise.                                             */
{
	word start_seg=0, n_seg, current_seg,percent,lastpercent;
	int i,x;
	unsigned checksum, completed_sum, check_arg=0;
	byte stream[8];
	ssl_block ssl;

  strcpy(result,"Initialising");
  *abortnow=0;

	/* get starting segment */
	for (i=0;i<4;i++)
	{
		if ((x=hex_val(*start_seg_s++))<0) 
    {
			fatal("starting segment must be four hex digits");
      return -1;
    }
		start_seg = start_seg<<4|x;
	}

	/* get number of segments */
	n_seg = (word)atol(n_seg_s);
	if (n_seg<1 || n_seg>65536) 
  {
		fatal("number of segments must be a decimal number from 1 to 65536");
    return -1;
  }

	/* read other parameters from parameter file */
	checksum = parse_file(fname,&ssl,stream);

/* HACK! fname is assumed to be a temporary file and is removed after use */

  remove(fname);

	/* check the checksum */
	for (i=0;i<4;i++)
	{
		if ((x=hex_val(*check_s++))<0) 
    {
			fatal("checksum must be four hex digits");
      return -1;
    }
		check_arg = check_arg<<4|x;
	}
	if (checksum != check_arg)
  {
		fatal("checksum doesn't match - should be %x",checksum);
    return -1;
  }
	completed_sum = (checksum+start_seg+n_seg) & 0xFFFF;

	/* outermost search loop */	

  lastpercent=101;
	for (current_seg=start_seg; current_seg<start_seg+n_seg; current_seg++)
	{

    percent=MulDiv(current_seg-start_seg,100,n_seg);
    if(percent!=lastpercent)
    {
      sprintf(result,"%u%%",percent);
      lastpercent=percent;
    }

		if (search_segment(&ssl,current_seg,stream,abortnow))
		{
      int n;

      for(n=0;n<5;n++)
        sprintf(result+(n*2),"%02x",(unsigned int)ssl.master[11+n]);
		  return 0;
		}
    if(*abortnow)
      break;
	}

  if(*abortnow)
    strcpy(result,"aborted");
  else
    sprintf(result,"%04lx %ld",start_seg,n_seg);

	return 1;
}


/*------------------------ Parameter File Parsing --------------------------*/

unsigned parse_file(const char *fname, ssl_block *ssl, byte *stream)
/* Read the parameter file fname, check for syntax and other errors, set    */
/* values of ssl fields and stream. Returns the file checksum.              */ 
{
	FILE *fparam;
	char buffer[100];
	char *bp, *keyword, *value;
	int line_no=1,got_ue_master=0, got_challenge=0, got_conn_id=0;
	int got_stream=0, errors=0, i;
	unsigned checksum=0;

	/* open file */
	if (strcmp(fname,"-")) 
	{
		if ((fparam=fopen(fname,"r"))==NULL)
    {
			fatal("cannot open parameter file \"%s\"",fname);
      return 0;
    }
	}
	else
	{
		fparam=stdin;
		fname="stdin";
	}

	/* parse each line of the parameter file */	
	while (fgets(buffer,100,fparam)!=NULL)
	{
		/* skip space, ignore lines which have only a comment */
		for (bp=buffer; *bp && isspace(*bp); bp++);
	 	if (*bp!='\0' && *bp!='#')
		{
			/* split line into keyword and value */
			keyword=bp;
			while (*bp && !isspace(*bp)) 
			{
				checksum += (unsigned)*bp;
				*bp = (char)tolower(*bp);
				bp++;
			}
			if (*bp=='\0') value=bp;
			else {
				*bp++='\0';
				if (!strcmp(keyword,"comment"))
				{
					/* hack to allow comments to include white space */
					while (*bp)
					{
						if (!isspace(*bp)) checksum += (unsigned)*bp;
						bp++;
					}
				} else {
					while (*bp && isspace(*bp)) bp++;
					if (*bp=='#') *bp='\0';
					value=bp;
					while (*bp && !isspace(*bp)) checksum += (unsigned)*bp++;
					if (*bp!='\0')
					{
						*bp++='\0';
						while (*bp && isspace(*bp)) bp++;
						if (*bp!='\0' && *bp!='#') 
							fatal("extra characters in parameter file line %d",line_no);
					}
				}
			}

			/* interpret keywords */
			if (!strcmp(keyword,"clear-master")) 
			{
				if (got_ue_master++) 
				{
					fprintf(stderr,"%s(%d): clear-master already specified\n",fname,line_no);
					errors++;
				} else {
					if (get_hex_bytes(value,ssl->master,11)!=11)
					{
						fprintf(stderr,"%s(%d): clear-master must be eleven bytes\n",fname,line_no);
						errors++;
					}
				}
			} else if (!strcmp(keyword,"challenge")) {
				if (got_challenge++) 
				{
					fprintf(stderr,"%s(%d): challenge already specified\n",fname,line_no);
					errors++;
				} else {
					if (get_hex_bytes(value,ssl->challenge,16)!=16)
					{
						fprintf(stderr,"%s(%d): challenge must be sixteen bytes\n",fname,line_no);
						errors++;
					}
				}
			} else if (!strcmp(keyword,"connection-id")) {
				if (got_conn_id++) 
				{
					fprintf(stderr,"%s(%d): connection-id already specified\n",fname,line_no);
					errors++;
				} else {
					if (get_hex_bytes(value,ssl->conn_id,16)!=16)
					{
						fprintf(stderr,"%s(%d): connection-id must be sixteen bytes\n",fname,line_no);
						errors++;
					}
				}
			} else if (!strcmp(keyword,"server-verify")) {
				if (got_stream++) 
				{
					fprintf(stderr,"%s(%d): server-verify or client-finished already specified\n",fname,line_no);
					errors++;
				} else {
					ssl->key_type='0';
					if (get_hex_bytes(value,stream,8)!=8)
					{
						fprintf(stderr,"server-verify must be eight bytes");
						errors++;
					}
				}
			} else if (!strcmp(keyword,"client-finished")) {
				if (got_stream++) 
				{
					fprintf(stderr,"%s(%d): server-verify or client-finished already specified\n",fname,line_no);
					errors++;
				} else {
					ssl->key_type='1';
					if (get_hex_bytes(value,stream,32)!=8)
					{
						fprintf(stderr,"%s(%d): client-finished must be eight bytes\n",fname,line_no);
						errors++;
					}
				}
			} else if (strcmp(keyword,"comment")) {
				fprintf(stderr,"%s(%d): unknown keyword \"%s\"\n",fname,line_no,keyword);
				errors++;
			}
		}

		line_no++;
	}

	/* this could take a while, so let's be good */
 	fclose(fparam);

	/* check that all necessary parameters have been defined */
	if (!got_ue_master) 
	{
		AfxMessageBox("%s: clear-master not specified\n");
		errors++;
	}
	if (!got_challenge)
	{
		AfxMessageBox("%s: challenge not specified\n");
		errors++;
	}
	if (!got_conn_id)
	{
		AfxMessageBox("%s: connection-id not specified\n");
		errors++;
	}
	if (!got_stream)
	{
		AfxMessageBox("%s: neither server-verify nor client-finished specified\n");
		errors++;
	}
	
	/* oops */
	if (errors)
	{
	  fatal("%d errors in parameter file\n",errors);
    return 0;
  }

	/* convert server-verify or client-finished to RC4 output stream */
	switch (ssl->key_type)
	{
		case '0':
			stream[0] ^= SSL_MT_SERVER_VERIFY;
			for (i=0; i<7; i++) stream[i+1] ^= ssl->challenge[i];
			break;
		case '1':
			stream[0] ^= SSL_MT_CLIENT_FINISHED;
			for (i=0; i<7; i++) stream[i+1] ^= ssl->conn_id[i];
			break;
		default:
			fatal("internal error: bad key_type value in parse_file()");
      return 0;
	}

	return checksum&0xFFFF;
}

/*-------------------------------- Key Search -------------------------------*/

int search_segment(ssl_block *ssl, word seg, const byte *stream,int *abortnow)
/* Search a segment. ssl contains the first 11 bytes of the master key, plus */
/* key_type, challenge and connection_id. seg is the segment to search,      */
/* stream is the RC4 generated stream we're looking for (at an offset of 16  */
/* bytes into the stream). Returns 1 if successful, in which case ssl has    */
/* the full master key, 0 otherwise.                                         */
{
	int i;

	/* initialize rest of ssl */
	ssl->master[11] = (byte)(seg>>8);
	ssl->master[12] = (byte)(seg&0xFF);
	ssl->master[13] = ssl->master[14] = ssl->master[15] = 0;
	ssl->pad[0] = 0x80;
	ssl->pad[1]=ssl->pad[2]=0;

	/* 64 iterations of 2^18 keys is one 24-bit segment */
	for (i=0; i<64; i++) 
	{
		if (search_18_bits(ssl,stream)) return 1;

    if(*abortnow)
      return 0;
	}
	return 0;
}

/*----------------------------- I/O Functions ------------------------------*/

int get_hex_bytes(const char *s, byte *dest, int max_bytes)
/* Convert hex string into bytes. Reads at most max_bytes and returns the  */
/* number of bytes read or -1 if error (including too many bytes).         */
{
	int i, x1, x2;

	for (i=0; i<max_bytes; i++)
	{
		if (*s=='\0') return i;
		if ((x1=hex_val(*s++))<0) return -1;
		if ((x2=hex_val(*s++))<0) return -1;
		*dest++ = (byte)(x1<<4|x2);
	}
	return (*s=='\0' ? max_bytes : -1);
}

int hex_val(char hex_digit)
/* Return the value of a hex digit, -1 if not a hex digit */
{
	if (hex_digit>='0' && hex_digit<='9') return hex_digit-'0';
	if (hex_digit>='a' && hex_digit<='f') return hex_digit-'a'+10;
	if (hex_digit>='A' && hex_digit<='F') return hex_digit-'A'+10;
	return -1;
}

void eureka(ssl_block *ssl)		
/* Output printed when the key is found in non-quiet mode */
{
	printf("\n\nEureka!\n\nEncrypted Master Key: ");
	print_bytes(ssl->master+11,5);
	printf("\n");
}			

void print_bytes(const byte *data, int n)
/* Print an array of bytes in hex */
{
	while (n--) printf("%02x",*data++);
}

void print_time(word secs, int print_secs)
/* Print a time value (seconds) in a user-friendly format. If print_secs    */
/* is zero, then the time is rounded to the nearest minute. Best to do this */
/* first, in case it carries into hours, days or years.                     */
{
	if (!print_secs) secs=((secs+30)/60)*60;
    if (secs>31536000) printf("%lu years ",secs/31536000);
    if (secs>86400) printf("%lu days ",(secs%31536000)/86400);
    if (secs>3600) printf("%lu hours ",(secs%86400)/3600);
    if (print_secs)
    {
    	if (secs>60) printf("%lu minutes and ",(secs%3600)/60);
		printf("%lu seconds",secs%60);
	} else {
		printf("%lu minutes",(secs%3600)/60);
	}
}

/*------------------------------ Diagnostics -------------------------------*/

int check_hardware()
/* Check that words are 32-bit unsigned integers and that the rotate_left   */
/* macro does rotate words. Returns 1 if hardware is big endian, else 0. 	*/
{
	word w = 0x87654321U;
	DECLARE_ROTATE_VARS

	if (sizeof(word)!=4) fatal("type \"word\" does not have 32 bits");
	if (w<0) fatal("type \"word\" must be unsigned");
	if (rotate_left(w,24)!=0x21876543)
		fatal("rotate_left() macro doesn't work");
	switch (*(byte*)&w)
	{
		case 0x21:
			return 0;
		case 0x87:
			return 1;
		default:
			fatal("your hardware has a weird byte order");
      return 0;
	}
}

int time_trial()
/* See how long it takes to search 2^17 keys, and calculate seconds needed  */
/* per segment (2^24) keys. Returns seconds per segment.                    */
{
	word t_wait, t_start, t_end, counter = TIMER_WAIT;
	int secs_per_seg, keys_per_sec, attempt=0, success;
	ssl_block ssl;

	/* banner etc */
	if (!quiet) 
	{
		printf("\n                                brutessl %s\n\n",version);
		printf("Self test... ");
		fflush(stdout);
	}

	/* Run the self test. Since the timer is platform-dependent, we get its */
	/* resolution by busy-waiting for it to change (this also ensures that  */
	/* we start at the beginning of a time period), and check to see if it  */
	/* clocks over during the trial.                                        */

	do {
		if (++attempt>2) fatal("time is going backwards");
		memcpy(&ssl,test_ssl_block,sizeof(ssl_block));
		t_wait = time_in_ms();
		do {
			if (!counter--) fatal("the time never changes");
			t_start = time_in_ms();
		} while (t_start==t_wait);
		success = search_18_bits(&ssl,test_stream);
		t_end=time_in_ms();	
		if (!success || memcmp(ssl.master,test_master,16)) 
			fatal("failed self test");
	} while (t_end<t_start || t_start<t_wait);
			
	/* Calculate seconds per segment and keys per second, attemptng to  */
	/* round things off correctly and take account of timer resolution. */
	secs_per_seg = (128*(t_end-t_start)+64*(t_start-t_wait)+500)/1000;
	keys_per_sec = (int)((16777216L + secs_per_seg*50)/(secs_per_seg*100))*100;
	if (!quiet) 
	{
		printf("OK\n\n");
		print_time(secs_per_seg,1);
		printf(" per segment, %d keys per second.\n",keys_per_sec);
	}

	return secs_per_seg;
}

/* We need a timer with a decent resolution. The function time_in_ms()   */
/* is a hook to platform-dependent timer routines. It should return a    */
/* time, in milli-seconds, relative to some arbitary frame of reference. */

#if defined(_CONSOLE)	|| defined(WIN32)		/* Windows NT Console Application */

	#include <sys/types.h>
	#include <sys/timeb.h>

	word time_in_ms()
	{
		struct _timeb tb;

		_ftime(&tb);
		return (word)(tb.time*1000+tb.millitm);
	}

#elif defined(_MSDOS) || defined(__OS2__) /* DOS or OS/2 */

	#include <time.h>

	word time_in_ms()
	{
		return (word)((1000*clock()+CLOCKS_PER_SEC/2)/CLOCKS_PER_SEC);
	}

#else /* assume its UNIX */

	#include <sys/time.h>

	word time_in_ms()
	{
		struct timeval tv;

		if (gettimeofday(&tv,NULL))
			fatal("error in gettimeofday()");
		return (word)(tv.tv_sec*1000+(tv.tv_usec+500)/1000);
	}

#endif

/*----------------------------- Miscellaneous ------------------------------*/

void set_low_priority()
/* make this a low priority process if necessary */
{

#ifdef __OS2__
  	/* set priority class to IDLETIME */
 	if (DosSetPriority(PRTYS_PROCESS, PRTYC_IDLETIME,0,0))
    	fatal("unable to set OS/2 priority");
#endif 

}

void usage()
/* Print a usage message and exit */
{
	fprintf(stderr,"usage: %s [-q] <filename> <checksum> <start segment> <no of segments>\n",
		myname);
	fprintf(stderr,"       %s [-q] -r <filename> [<no of segments>]\n",myname);
	fprintf(stderr,"       %s [-q] -t [<hours>[:<minutes>]]\n",myname);
	exit(2);
}

void fatal(const char *format,...)		
/* Print a formatted error message and exit */
{
	va_list marker;
  char buffer[256];

	fprintf(stderr,"\n%s: ",myname);
	va_start(marker,format);
	vsprintf(buffer,format,marker);
  AfxMessageBox(buffer);
	va_end(marker);
}
