/*----------------------------------------------------------------------------

                                brutessl 1.01

                        Copyright (C) 1995 Andrew Roos
                             All Rights Reserved

This program is licensed for academic and educational use only. It may not be
used  for  any commercial purpose. You may modify the program if you wish to,
provided that the original copyright  notice  and  license  restrictions  are
retained,  and  that  you include a notice stating that you have modified the
program and giving details of the changes that you have made. This program is
distributed  without  any warranty including, but not limited to, the implied 
warranty of merchantability or fitness for a particular purpose.  

synopsis:  A program for brute-force searching SSL data which has been 
           encrypted using 40-bit (export) RC4 keys.

usage:	   brutessl [-q] -t [hh:[mm]]
		   brutessl [-q] <file> <checksum> <start segment> <no of segments>
		   brutessl [-q] -r <start segment> [<no of segments>]

Andrew Roos <andrewr@vironix.co.za>
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>

#ifdef __OS2__
#define INCL_DOSPROCESS
#include <os2.h>
#endif 

#include "brutessl.h"

/*-------------------------------- Macros ----------------------------------*/

#define TIMER_WAIT 				200000U		/* iterations waiting for timer */
#define SSL_MT_CLIENT_FINISHED	3			/* client-finished message id */
#define SSL_MT_SERVER_VERIFY	5 			/* server-verify message id */

/*-------------------------------- Constants -------------------------------*/

const char *version = "1.01";

/* Test data. The test used requires 2^19 iterations to find the key. */

const byte test_ssl_block[52] = {
0x51,0xb7,0x63,0xf8,0xe0,0xe0,0xa6,0x33,0x5a,0x2d,0xcc,0xd5,0x7b,0x00,0x00,0x06,
0x30,0x1d,0x56,0x7b,0xce,0x00,0x9d,0x1d,0xc5,0xbd,0xb8,0x11,0x56,0x07,0x21,0x3e,
0xd9,0xa5,0x16,0x38,0x9f,0x72,0x8f,0x67,0x85,0xa9,0x73,0x66,0xc9,0x5f,0x0b,0xa9,
0x1c,0x80,0x00,0x00
};

const byte test_stream[8] = { 0xd2,0x19,0x75,0x76,0xb9,0x1c,0xf6,0xc1 };

const byte test_master[16] = {
0x51,0xb7,0x63,0xf8,0xe0,0xe0,0xa6,0x33,0x5a,0x2d,0xcc,0xd5,0x7b,0xff,0xff,0x07
};

/*---------------------------- Global Variables ----------------------------*/

int big_endian;						/* true if MSB of word stored first */
static int quiet;					/* terse output, set by the -q flag */
static char* myname = "brutessl";	/* program name taken from argv[0] */

/*--------------------------- Function Prototypes --------------------------*/

/* top level */
static int  search(const char *filename, const char *check_s, 
			const char *n_seg_s, const char *start_seg_s);
static int  search_random(const char *fname, char *n_seg_s);
static int  self_test(const char *time_s);

/* parameter file parsing */
static unsigned parse_file(const char *fname, ssl_block *ssl, byte *stream);
static void check_checksum(unsigned checksum, const char *check_s);

/* key search */
static int  search_segment(ssl_block *ssl, word seg, const byte *stream);

/* I/O functions */
static int  get_hex_bytes(const char *s, byte *dest, int max_bytes);
static int  hex_val(char hex_digit);
static void eureka(ssl_block *ssl);
static void print_bytes(FILE *f, const byte *data, int n);
static void print_time(word secs, int print_secs);

/* Diagnostics */
static int  check_hardware(void);
static int  time_trial(void);
static word time_in_ms(void);

/* Miscellaneous */
static void set_low_priority(void);
void print_cypher();
static void usage(void);
static void fatal(const char* format,...);

/*-------------------------------- Top Level -------------------------------*/

int main(int argc, char *argv[])
/* Interpret flags, check that the correct number of arguments have been    */
/* supplied, then call search(), search_random() or self_test(). The exit   */
/* status is 0 if a key is found, 1 for an unsuccessful search, 2 on error. */
{
	char **av, *cp;
	int ac;

	set_low_priority();
	big_endian = check_hardware();
	if (*argv && **argv)
	{
		myname=*argv;
		for (cp=*argv; *cp; cp++)
		{
			if (*cp=='/' || *cp=='\\') myname=cp;
		}
	}
	av=argv+1;
	ac=argc-1;
	if (ac==0) usage();
	if (!strcmp(av[0],"-q") || !strcmp(av[0],"-Q"))
	{
		quiet=1;
		if (--ac==0) 
		{
			printf("%s\n",version);
			return 0;
		}
		av++;
	}
	if (av[0][0]=='-' && (av[0][1]))
	{
		if (av[0][2]) usage();
		switch (tolower(av[0][1]))
		{
			case 't':
				if (ac>2) usage();
				return self_test(av[1]);
			case 'r':
				if (ac<2 || ac>3) usage();
				return search_random(av[1],av[2]);
			case '-':
				if (ac!=5) usage();
				return search(av[1],av[2],av[3],av[4]);
			default:
				fatal("bad option \"%s\"",av[0]);
		}
	} 
	if (ac!=4) usage();
	return search(av[0],av[1],av[2],av[3]);
}

int search(const char *fname, const char *check_s, const char *start_seg_s, 
	const char *n_seg_s)
/* Perform a key search, reading parameter file "fname", or standard input */
/* if fname is "-". start_seg_s and n_seg_s are string representations of  */
/* starting segment (in hex) and number of segments (decimal). Returns 0   */
/* if successful, 1 otherwise.                                             */
{
	word start_seg=0, n_seg, current_seg;
	int i,x,estimated_time;
	unsigned checksum, completed_sum, check_arg=0;
	byte stream[8];
	ssl_block ssl;

	/* get starting segment */
	for (i=0;i<4;i++)
	{
		if ((x=hex_val(*start_seg_s++))<0) 
			fatal("starting segment must be four hex digits");
		start_seg = start_seg<<4|x;
	}

	/* get number of segments */
	n_seg = (word)atol(n_seg_s);
	if (n_seg<1 || n_seg>65536) 
		fatal("number of segments must be a decimal number from 1 to 65536");

	/* read other parameters from parameter file */
	checksum = parse_file(fname,&ssl,stream);

	/* check the checksum */
	for (i=0;i<4;i++)
	{
		if ((x=hex_val(*check_s++))<0) 
			fatal("checksum must be four hex digits");
		check_arg = check_arg<<4|x;
	}
	if (checksum != check_arg)
		fatal("checksum doesn't match - should be %x",checksum);
	completed_sum = (checksum+start_seg+n_seg) & 0xFFFF;

	/* calculate how long this will take */
	if (!quiet) 
	{
		estimated_time = n_seg * time_trial();
		fprintf(stderr,"This will take about ");
		print_time(estimated_time,0);
		fprintf(stderr,"\n");
	}

	/* outermost search loop */	
	for (current_seg=start_seg; current_seg<start_seg+n_seg; current_seg++)
	{
		if (search_segment(&ssl,current_seg,stream))
		{
			if (quiet) printf("brutessl %s ",version);
			else {
				eureka(&ssl);		
				print_cypher();
			}
			printf("%04lx %04lx %04lx %ld ",checksum, completed_sum,
				start_seg,n_seg);
			print_bytes(stdout,ssl.master+11,5);
			printf("\n");
			return 0;
		}
	}

	if (quiet) printf("brutessl %s ",version);
	else {
		fprintf(stderr,"\n\nNo solution found.\n");
		print_cypher();
	}
	printf("%04lx %04lx %04lx %ld no\n",checksum,completed_sum,start_seg,n_seg);
	return 1;
}

int search_random(const char *fname, char *n_seg_s)
/* Search segments of the keyspace at random, invoked with the -r startup    */ 
/* option. n_seg_s is the string representation of the number of segments to */
/* search. If NULL, the search continues until a solution is found.          */
{
	ssl_block ssl;
	byte stream[32];
	word seg_no, n_segs=0, i;

	srand((unsigned int)time_in_ms());
	if (n_seg_s!=NULL)
	{
		n_segs = (word)atol(n_seg_s);
		if (n_segs<1) fatal("no of segments must be a decimal number");
	}		 
	parse_file(fname,&ssl,stream);
	if (!quiet) time_trial();
	for (i=0; n_segs==0 || i<n_segs; i++)
	{
		seg_no = (rand()&0xFF)<<8 | (rand()&0xFF);
		if (search_segment(&ssl,seg_no,stream))
		{
			if (quiet) 
			{
				print_bytes(stdout,ssl.master+11,5);
				printf("\n");
			} else {
				eureka(&ssl);
			}
			return 0;
		}				
	}

	if (!quiet) fprintf(stderr,"\n\nNo solution found.\n");

	return 1;
}	 

int self_test(const char *time_s)
/* Run a self test and report timing information. time_s specifies how long */
/* we would like the program to run for, in format HH[:MM]. If time_s is    */
/* specified then calculates how many segments we can search in this time.  */
/* Otherwise just reports seconds per segment and keys per sec.             */
{
	int mins=0, secs_per_seg, n_segs;

	/* parse time_s, setting mins to desired time in minutes */
	if (time_s!=NULL)
	{	
		if (!isdigit(*time_s)) fatal("-t argument must be in the form HH[:MM]");
		mins = (*time_s++ - '0')*60;
		if (isdigit(*time_s))
			mins = mins*10+(*time_s++ - '0')*60;
		if (*time_s!='\0')
		{
			if (time_s[0]!=':' || !isdigit(time_s[1]) || time_s[1]>'5' ||
				!isdigit(time_s[2]) || time_s[3]!='\0') 
				fatal("-t argument must be in the form HH[:MM]");
			mins += (time_s[1]-'0')*10+(time_s[2]-'0');
		}
	}

	secs_per_seg = time_trial();
	if (quiet) printf("%d",secs_per_seg);

	if (time_s) 
	{
		n_segs = (int)(((word)mins*60)/secs_per_seg);
		if (quiet) printf(" %d",n_segs);
		else {
			if (n_segs==0) fprintf(stderr,"The minimum is one segment.\n");
			else {
				fprintf(stderr,"%d segments will take about ",n_segs);
				print_time((word)n_segs*(word)secs_per_seg,0);
				fprintf(stderr,".\n");
			}
		}
	}

	if (quiet) printf("\n");

	return 0;
}

/*------------------------ Parameter File Parsing --------------------------*/

unsigned parse_file(const char *fname, ssl_block *ssl, byte *stream)
/* Read the parameter file fname, check for syntax and other errors, set    */
/* values of ssl fields and stream. Returns the file checksum.              */ 
{
	FILE *fparam;
	char buffer[100];
	char *bp, *keyword, *value;
	int line_no=1,got_ue_master=0, got_challenge=0, got_conn_id=0;
	int got_stream=0, errors=0, i;
	unsigned checksum=0;

	/* open file */
	if (strcmp(fname,"-")) 
	{
		if ((fparam=fopen(fname,"r"))==NULL)
			fatal("cannot open parameter file \"%s\"",fname);
	} else {
		fparam=stdin;
		fname="stdin";
	}

	/* parse each line of the parameter file */	
	while (fgets(buffer,100,fparam)!=NULL)
	{
		/* skip space, ignore lines which have only a comment */
		for (bp=buffer; *bp && isspace(*bp); bp++);
	 	if (*bp!='\0' && *bp!='#')
		{
			/* split line into keyword and value */
			keyword=bp;
			while (*bp && !isspace(*bp)) 
			{
				checksum += (unsigned)*bp;
				*bp = (char)tolower(*bp);
				bp++;
			}
			if (*bp=='\0') value=bp;
			else {
				*bp++='\0';
				if (!strcmp(keyword,"comment"))
				{
					/* hack to allow comments to include white space */
					while (*bp)
					{
						if (!isspace(*bp)) checksum += (unsigned)*bp;
						bp++;
					}
				} else {
					while (*bp && isspace(*bp)) bp++;
					if (*bp=='#') *bp='\0';
					value=bp;
					while (*bp && !isspace(*bp)) checksum += (unsigned)*bp++;
					if (*bp!='\0')
					{
						*bp++='\0';
						while (*bp && isspace(*bp)) bp++;
						if (*bp!='\0' && *bp!='#') 
							fatal("extra characters in parameter file line %d",line_no);
					}
				}
			}

			/* interpret keywords */
			if (!strcmp(keyword,"clear-master")) 
			{
				if (got_ue_master++) 
				{
					fprintf(stderr,"%s(%d): clear-master already specified\n",fname,line_no);
					errors++;
				} else {
					if (get_hex_bytes(value,ssl->master,11)!=11)
					{
						fprintf(stderr,"%s(%d): clear-master must be eleven bytes\n",fname,line_no);
						errors++;
					}
				}
			} else if (!strcmp(keyword,"challenge")) {
				if (got_challenge++) 
				{
					fprintf(stderr,"%s(%d): challenge already specified\n",fname,line_no);
					errors++;
				} else {
					if (get_hex_bytes(value,ssl->challenge,16)!=16)
					{
						fprintf(stderr,"%s(%d): challenge must be sixteen bytes\n",fname,line_no);
						errors++;
					}
				}
			} else if (!strcmp(keyword,"connection-id")) {
				if (got_conn_id++) 
				{
					fprintf(stderr,"%s(%d): connection-id already specified\n",fname,line_no);
					errors++;
				} else {
					if (get_hex_bytes(value,ssl->conn_id,16)!=16)
					{
						fprintf(stderr,"%s(%d): connection-id must be sixteen bytes\n",fname,line_no);
						errors++;
					}
				}
			} else if (!strcmp(keyword,"server-verify")) {
				if (got_stream++) 
				{
					fprintf(stderr,"%s(%d): server-verify or client-finished already specified\n",fname,line_no);
					errors++;
				} else {
					ssl->key_type='0';
					if (get_hex_bytes(value,stream,8)!=8)
					{
						fprintf(stderr,"server-verify must be eight bytes");
						errors++;
					}
				}
			} else if (!strcmp(keyword,"client-finished")) {
				if (got_stream++) 
				{
					fprintf(stderr,"%s(%d): server-verify or client-finished already specified\n",fname,line_no);
					errors++;
				} else {
					ssl->key_type='1';
					if (get_hex_bytes(value,stream,32)!=8)
					{
						fprintf(stderr,"%s(%d): client-finished must be eight bytes\n",fname,line_no);
						errors++;
					}
				}
			} else if (strcmp(keyword,"comment")) {
				fprintf(stderr,"%s(%d): unknown keyword \"%s\"\n",fname,line_no,keyword);
				errors++;
			}
		}

		line_no++;
	}

	/* this could take a while, so let's be good */
 	fclose(fparam);

	/* check that all necessary parameters have been defined */
	if (!got_ue_master) 
	{
		fprintf(stderr,"%s: clear-master not specified\n",fname);
		errors++;
	}
	if (!got_challenge)
	{
		fprintf(stderr,"%s: challenge not specified\n",fname);
		errors++;
	}
	if (!got_conn_id)
	{
		fprintf(stderr,"%s: connection-id not specified\n",fname);
		errors++;
	}
	if (!got_stream)
	{
		fprintf(stderr,"%s: neither server-verify nor client-finished specified\n",fname);
		errors++;
	}
	
	/* oops */
	if (errors) fatal("%d errors in parameter file\n",errors);

	/* convert server-verify or client-finished to RC4 output stream */
	switch (ssl->key_type)
	{
		case '0':
			stream[0] ^= SSL_MT_SERVER_VERIFY;
			for (i=0; i<7; i++) stream[i+1] ^= ssl->challenge[i];
			break;
		case '1':
			stream[0] ^= SSL_MT_CLIENT_FINISHED;
			for (i=0; i<7; i++) stream[i+1] ^= ssl->conn_id[i];
			break;
		default:
			fatal("internal error: bad key_type value in parse_file()");
	}

	return checksum&0xFFFF;
}

/*-------------------------------- Key Search -------------------------------*/

int search_segment(ssl_block *ssl, word seg, const byte *stream)
/* Search a segment. ssl contains the first 11 bytes of the master key, plus */
/* key_type, challenge and connection_id. seg is the segment to search,      */
/* stream is the RC4 generated stream we're looking for (at an offset of 16  */
/* bytes into the stream). Returns 1 if successful, in which case ssl has    */
/* the full master key, 0 otherwise.                                         */
{
	int i;

	/* initialize rest of ssl */
	ssl->master[11] = (byte)(seg>>8);
	ssl->master[12] = (byte)(seg&0xFF);
	ssl->master[13] = ssl->master[14] = ssl->master[15] = 0;
	ssl->pad[0] = 0x80;
	ssl->pad[1]=ssl->pad[2]=0;

	if (!quiet) fprintf(stderr,"\nSegment %04x: ",seg);

	/* 64 iterations of 2^18 keys is one 24-bit segment */
	for (i=0; i<64; i++) 
	{
		if (!quiet) 
		{
			putc('.',stderr);
			fflush(stdout);
		}
		if (search_18_bits(ssl,stream)) return 1;
	}
	return 0;
}

/*----------------------------- I/O Functions ------------------------------*/

int get_hex_bytes(const char *s, byte *dest, int max_bytes)
/* Convert hex string into bytes. Reads at most max_bytes and returns the  */
/* number of bytes read or -1 if error (including too many bytes).         */
{
	int i, x1, x2;

	for (i=0; i<max_bytes; i++)
	{
		if (*s=='\0') return i;
		if ((x1=hex_val(*s++))<0) return -1;
		if ((x2=hex_val(*s++))<0) return -1;
		*dest++ = (byte)(x1<<4|x2);
	}
	return (*s=='\0' ? max_bytes : -1);
}

int hex_val(char hex_digit)
/* Return the value of a hex digit, -1 if not a hex digit */
{
	if (hex_digit>='0' && hex_digit<='9') return hex_digit-'0';
	if (hex_digit>='a' && hex_digit<='f') return hex_digit-'a'+10;
	if (hex_digit>='A' && hex_digit<='F') return hex_digit-'A'+10;
	return -1;
}

void eureka(ssl_block *ssl)		
/* Output printed when the key is found in non-quiet mode */
{
	fprintf(stderr,"\n\nEureka!\n\nEncrypted Master Key: ");
	print_bytes(stderr,ssl->master+11,5);
	fprintf(stderr,"\n");
}			

void print_bytes(FILE *f, const byte *data, int n)
/* Print an array of bytes in hex */
{
	while (n--) fprintf(f,"%02x",*data++);
}

void print_time(word secs, int print_secs)
/* Print a time value (seconds) in a user-friendly format. If print_secs    */
/* is zero, then the time is rounded to the nearest minute. Best to do this */
/* first, in case it carries into hours, days or years.                     */
{
	if (!print_secs) secs=((secs+30)/60)*60;
    if (secs>31536000) fprintf(stderr,"%lu years ",secs/31536000);
    if (secs>86400) fprintf(stderr,"%lu days ",(secs%31536000)/86400);
    if (secs>3600) fprintf(stderr,"%lu hours ",(secs%86400)/3600);
    if (print_secs)
    {
    	if (secs>60) fprintf(stderr,"%lu minutes and ",(secs%3600)/60);
		fprintf(stderr,"%lu seconds",secs%60);
	} else {
		fprintf(stderr,"%lu minutes",(secs%3600)/60);
	}
}

/*------------------------------ Diagnostics -------------------------------*/

int check_hardware()
/* Check that words are 32-bit unsigned integers and that the rotate_left   */
/* macro does rotate words. Returns 1 if hardware is big endian, else 0. 	*/
{
	word w = 0x87654321U;
	DECLARE_ROTATE_VARS

	if (sizeof(word)!=4) fatal("type \"word\" does not have 32 bits");
	if (w<0) fatal("type \"word\" must be unsigned");
	if (rotate_left(w,24)!=0x21876543)
		fatal("rotate_left() macro doesn't work");
	switch (*(byte*)&w)
	{
		case 0x21:
			return 0;
		case 0x87:
			return 1;
		default:
			fatal("your hardware has a weird byte order");
	}
}

int time_trial()
/* See how long it takes to search 2^17 keys, and calculate seconds needed  */
/* per segment (2^24) keys. Returns seconds per segment.                    */
{
	word t_wait, t_start, t_end, counter = TIMER_WAIT;
	int secs_per_seg, keys_per_sec, attempt=0, success;
	ssl_block ssl;

	/* banner etc */
	if (!quiet) 
	{
		fprintf(stderr,"\n                                BruteSSL %s\n\n",version);
		fprintf(stderr,"Self test... ");
		fflush(stdout);
	}

	/* Run the self test. Since the timer is platform-dependent, we get its */
	/* resolution by busy-waiting for it to change (this also ensures that  */
	/* we start at the beginning of a time period), and check to see if it  */
	/* clocks over during the trial.                                        */

	do {
		if (++attempt>2) fatal("time is going backwards");
		memcpy(&ssl,test_ssl_block,sizeof(ssl_block));
		t_wait = time_in_ms();
		do {
			if (!counter--) fatal("the time never changes");
			t_start = time_in_ms();
		} while (t_start==t_wait);
		success = search_18_bits(&ssl,test_stream);
		t_end=time_in_ms();	
		if (!success || memcmp(ssl.master,test_master,16)) 
			fatal("failed self test");
	} while (t_end<t_start || t_start<t_wait);
			
	/* Calculate seconds per segment and keys per second, attemptng to  */
	/* round things off correctly and take account of timer resolution. */
	secs_per_seg = (128*(t_end-t_start)+64*(t_start-t_wait)+500)/1000;
	keys_per_sec = (int)((16777216L + secs_per_seg*50)/(secs_per_seg*100))*100;
	if (!quiet) 
	{
		fprintf(stderr,"OK\n\n");
		print_time(secs_per_seg,1);
		fprintf(stderr," per segment, %d keys per second.\n",keys_per_sec);
	}

	return secs_per_seg;
}

/* We need a timer with a decent resolution. The function time_in_ms()   */
/* is a hook to platform-dependent timer routines. It should return a    */
/* time, in milli-seconds, relative to some arbitary frame of reference. */

#if defined(_CONSOLE)			/* Windows NT Console Application */

#include <sys/types.h>
#include <sys/timeb.h>

	word time_in_ms()
	{
		struct _timeb tb;

		_ftime(&tb);
		return (word)(tb.time*1000+tb.millitm);
	}

#elif defined(_MSDOS) || defined(__OS2__) /* DOS or OS/2 */

#include <time.h>

	word time_in_ms()
	{
		return (word)((1000*clock()+CLOCKS_PER_SEC/2)/CLOCKS_PER_SEC);
	}

#else /* assume its UNIX */

#include <sys/time.h>

	word time_in_ms()
	{
		struct timeval tv;

		if (gettimeofday(&tv,NULL))
			fatal("error in gettimeofday()");
		return (word)(tv.tv_sec*1000+(tv.tv_usec+500)/1000);
	}

#endif

/*----------------------------- Miscellaneous ------------------------------*/

void set_low_priority()
/* make this a low priority process if necessary */
{

#ifdef __OS2__
  	/* set priority class to IDLETIME */
 	if (DosSetPriority(PRTYS_PROCESS, PRTYC_IDLETIME,0,0))
    	fatal("unable to set OS/2 priority");
#endif 

}

void print_cypher()
{
	fprintf(stderr,"\n----------------------------------------------------------\n");
	fprintf(stderr,"Please paste exactly what it says below this line into the\n");
	fprintf(stderr,"acknowledgment box on http://www.brute.cl.cam.ac.uk/brute/\n");
	fprintf(stderr,"----------------------------------------------------------\n\n");
}	

void usage()
/* Print a usage message and exit */
{
	fprintf(stderr,"usage: %s [-q] <filename> <checksum> <start segment> <no of segments>\n",
		myname);
	fprintf(stderr,"       %s [-q] -r <filename> [<no of segments>]\n",myname);
	fprintf(stderr,"       %s [-q] -t [<hours>[:<minutes>]]\n",myname);
	exit(2);
}

void fatal(const char *format,...)		
/* Print a formatted error message and exit */
{
	va_list marker;

	fprintf(stderr,"\n%s: ",myname);
	va_start(marker,format);
	vfprintf(stderr,format,marker);
	va_end(marker);
	fprintf(stderr,"\n");
	exit(2);
}
		


