/*
 *	msnl_usrreq.c
 *	-------------
 *
 * $Id: msnl_usrreq.c,v 1.28 1993/11/26 17:20:14 rjb17 Exp $
 *
 * Copyright (c) 1993 Cambridge University Computer Laboratory.
 *
 */

#include "../h/param.h"
#include "../h/protosw.h"
#include "../h/mbuf.h"
#include "../h/errno.h"
#include "../h/socket.h"
#include "../h/socketvar.h"
#include "../h/time.h"			/* needed for kernel.h */
#include "../h/kernel.h"		/* hz */

#include "msnl_manage.h"
#include "msnl_pcb.h"
#include "msnl_port.h"
#include "msnl_usrreq.h"

/*
 * move these elsewhere!
 */

typedef struct socket SOCK;
typedef struct mbuf MBUF;
typedef MBUF *MBUFP;

#define msnlspace (1024 * 32)
#define msnlconnecttimeout (hz * 30)

/*
 * These should be in socketvar.h
 */

extern void soisconnecting();
extern void soisdisconnected();
extern struct socket *sonewconn();
extern void soisconnected();
extern int sodisconnect();
extern int soabort();
extern void sofree();

/*
 * Prototypes.
 */

#ifdef __STDC__
int msnl_usrreq(SOCK *so, int req, MBUFP m, MBUFP nam, MBUFP rights);
int  msnl_attach	(SOCK *so);

void		msnl_connect_timeout	(caddr_t arg);
int  		msnl_operation		(struct manage_msg *msg);
static int	pcbrecv			(MSPCB *pcb, MBUFP m0);
static int	msnl_break_local	(MSPCB *pcb);

int		sorecv		(SOCK *so, MBUFP m);
void		mysoisconnected	(SOCK *so);
void		soisreallyconnected (SOCK *so);
MSPCB		*newconnwrapper	(SOCK *so, struct sockaddr *addr);

#endif

#ifndef __STDC__ /* Mips CC is so so broken */
	void	msnl_connect_timeout	();
static int	pcbrecv			();
static int	msnl_break_local	();

int		sorecv();
void		mysoisconnected	();
void		soisreallyconnected ();
MSPCB		*newconnwrapper();

#endif

/*
 * ----------------------------------------------------------------------
 */

struct msnl_methods msnl_socket_methods = {
    soisconnecting,
    soisdisconnected,
    newconnwrapper,
    mysoisconnected,
    sodisconnect,
    soabort,
    sofree,
    sorecv,
    soisreallyconnected,
} ;

/*
 * ----------------------------------------------------------------------
 */

int msnl_usrreq(so, req, m, nam, rights)
    SOCK		*so;
    int			req;
    MBUFP		m, nam, rights;
{
    int			s = splnet();
    int			error = 0;
    MSPCB		*temp;
    struct sockaddr	saddr;

    if ((so->so_pcb == 0) != (req == PRU_ATTACH))
    {
	splx(s);
	return EINVAL;
    }

    switch (req)
    {
    case PRU_ATTACH:
	error = msnl_attach(so);
	break;

    case PRU_ABORT:
	/* This only ever gets generated if the protocol should do
	 * a soabort on a socket. This can happen to the connections
	 * waiting to be accepted if the listener is closed. 
	 * This can be dealt with as if it was a detach.
	 */
    case PRU_DETACH:
	/* The socket is about to be deleted. 
	 * There is nothing we can do to stop it. 
	 * Note we should be disconnected by here.
	 */
	temp = sotomsnlpcb(so);
	so->so_pcb = 0;
	msnl_detach(temp);
	break;
	
    case PRU_BIND:
	/* Looks like state could be anything at this point! */
	/* This must complete inside the kernel */
	if (nam->m_len < sizeof(struct sockaddr))
	    error = EINVAL;
	else
	    error = msnl_bind(sotomsnlpcb(so), mtod(nam, struct sockaddr *));
	break;
	
    case PRU_DISCONNECT:
	/* This is only called if we are in the connected state 
	 * (as far as unix is concerned).
	 */
	msnl_disconnect(sotomsnlpcb(so));
	break;

    case PRU_SHUTDOWN:
	/* This is only ever generated by a shutdown(2) system call. */
	socantsendmore(so);
	break;
	
    case PRU_LISTEN:
	/* Looks like state could be anything at this point! */
	temp = sotomsnlpcb(so);
	if (temp->msp_state == MSPS_NEW)
	{
	    bzero ((char *)(& saddr), sizeof (struct sockaddr));
	    saddr.sa_family = AF_MSNL;
	    error = msnl_bind(temp, &saddr);
	    if (error) break;
	}
	msnl_listen(temp);
	break;
	
    case PRU_CONNECT:
	/* State already checked not SS_ISCONNECTING or SS_ISCONNECTED */
	if (nam->m_len < sizeof(struct sockaddr))
	    error = EINVAL;
	else 
	{
	    temp = sotomsnlpcb(so);
	    if (temp->msp_state == MSPS_NEW) 
	    {
		bzero ((char *)(& saddr), sizeof (struct sockaddr));
		saddr.sa_family = AF_MSNL;
		error = msnl_bind(temp, &saddr);
		if (error) break;
	    }
	    error = msnl_connect(temp, mtod(nam, struct sockaddr *));
	}
	break;
	
    case PRU_ACCEPT:
	if (nam->m_len < sizeof(struct sockaddr))
	    error = EINVAL;
	else
	    error = msnl_accept(sotomsnlpcb(so), mtod(nam, struct sockaddr *));
	break;
	
    case PRU_SEND:
	/* Send is done as a function to can be called from IP code
	 * like the others
	 */
	error = msnl_send(sotomsnlpcb(so), m);
	break;

    case PRU_SOCKADDR:
	error = msnl_sockaddr(sotomsnlpcb(so), nam);
	break;

    case PRU_PEERADDR:
	error = msnl_peeraddr(sotomsnlpcb(so), nam);
	break;
	
    case PRU_CONTROL:
	error = assoc_ioctl((int)m, (caddr_t)nam);
	break;
	
    default:
	error = EOPNOTSUPP;
	break;
    }

    splx(s);
    return error;
}

/*
 * ----------------------------------------------------------------------
 */

int msnl_sockaddr(pcb, nam)
    MSPCB		*pcb;
    MBUFP		nam;
{
    struct sockaddr		*msaddr = mtod(nam, struct sockaddr *);
    
    if (pcb->msp_state == MSPS_NEW)
	return EINVAL;
    
    *msaddr = pcb->msp_src;
    
    nam->m_len = sizeof(struct sockaddr);
    return 0;
}

int msnl_peeraddr(pcb, nam)
    MSPCB		*pcb;
    MBUFP		nam;
{
    struct sockaddr		*msaddr = mtod(nam, struct sockaddr *);
    
    *msaddr = pcb->msp_dst;
    
    nam->m_len = sizeof(struct sockaddr);
    return 0;
}

/*
 * ----------------------------------------------------------------------
 */

int msnl_send(pcb, m)
    MSPCB		*pcb;
    MBUFP		m;
{
    if ((pcb->msp_state != MSPS_CONNECTED)
	|| (!pcb->msp_tx_opaque)
	|| (!pcb->msp_tx_proc))
    {
	printf("msnl_send: unexpected %d, %08X, %08X\n",
	       pcb->msp_state, pcb->msp_tx_opaque, pcb->msp_tx_proc);
	
	/* This could be "connected" when not really etc so just bin it */
	m_freem(m);
	return ENOTTY;
    }
    return (pcb->msp_tx_proc)(pcb->msp_tx_opaque, m);
}
	
/*
 * ----------------------------------------------------------------------
 */

int msnl_attach(so)
    SOCK		*so;
{
    MSPCB		*pcb;
    int			error;
    MBUFP		m;

    if (!manage_running())
	return EPROTONOSUPPORT;

    if ((error = soreserve(so, msnlspace, msnlspace)))
	return error;
	
    if ((m = m_getclr(M_DONTWAIT, MT_PCB)) == 0)
	return ENOBUFS;
    
    pcb = mtod(m, MSPCB *);

    /* Initialise the PCB */
    so->so_pcb = (caddr_t) pcb;
    pcb->msp_so = (caddr_t) so;

    pcb->msp_methods = &msnl_socket_methods;
    
    pcb->msp_indirect_error = & (so->so_error);
    
    manage_register(&pcb->msp_man);
    
    return 0;
}

int msnl_bind(pcb, msaddr)
    MSPCB		*pcb;
    struct sockaddr	*msaddr;
{
    struct manage_msg		*man;
    MBUFP			m;
    int				error;

    if (pcb->msp_state != MSPS_NEW)
	return EISCONN;
    
    /* Ensure everything is going to work first */

    if (!manage_check(sizeof(struct manage_msg)))
	return ENOBUFS;

    m = m_get(M_DONTWAIT, MT_DATA);
    if (m == NULL)
	return ENOBUFS;

    if ((error = msnl_sap(msaddr)))
    {
	m_free(m);
	return error;
    }
    pcb->msp_src = *msaddr;

    pcb->msp_state = MSPS_BOUND;
    
    man = mtod(m, struct manage_msg *);
    
    man->mm_type = MMTYPE_SOCKET;
    man->mm_request = MMUP_BIND;
    man->mm_manage = &pcb->msp_man;
    man->mm_addr = pcb->msp_src;
    
    m->m_len = sizeof(struct manage_msg);
    
    manage_send(m);
    
    return 0;
}

int msnl_listen(pcb)
    MSPCB		*pcb;
{
    struct manage_msg		*man;
    MBUFP			m;
    
    if (pcb->msp_state != MSPS_BOUND)
	return EISCONN;

    if (!manage_check(sizeof(struct manage_msg)))
	return ENOBUFS;

    m = m_get(M_DONTWAIT, MT_DATA);
    if (m == NULL)
	return ENOBUFS;

    pcb->msp_state = MSPS_LISTENING;
    
    man = mtod(m, struct manage_msg *);
    
    man->mm_type = MMTYPE_SOCKET;
    man->mm_request = MMUP_LISTEN;
    man->mm_manage = &pcb->msp_man;

    m->m_len = sizeof(struct manage_msg);
    
    manage_send(m);
    return 0;
}

int msnl_connect(pcb, msaddr)
    MSPCB		*pcb;
    struct sockaddr	*msaddr;
{
    struct manage_msg		*man;
    MBUFP			m;

    if (pcb->msp_state != MSPS_BOUND)
	return EINVAL;

    if (!manage_check(sizeof(struct manage_msg)))
	return ENOBUFS;

    m = m_get(M_DONTWAIT, MT_DATA);
    if (m == NULL)
	return ENOBUFS;

    man = mtod(m, struct manage_msg *);
    
    man->mm_type = MMTYPE_SOCKET;
    man->mm_request = MMUP_CONNECT;
    man->mm_manage = &pcb->msp_man;
    man->mm_addr = *msaddr;

    m->m_len = sizeof(struct manage_msg);
    
    manage_send(m);

    pcb->msp_state = MSPS_CONNECTING;
    (pcb->msp_methods->meth_isconnecting)(pcb->msp_so);

    pcb->msp_timeoutset = 1;
    timeout(msnl_connect_timeout, pcb, msnlconnecttimeout);

    return 0;
}

int msnl_accept(pcb, msaddr)
    MSPCB		*pcb;
    struct sockaddr	*msaddr;
{
    struct manage_msg		*man;
    MBUFP			m;
    
    if (pcb->msp_state != MSPS_ACCEPTABLE)
	return EINVAL;
    
    if (!manage_check(sizeof(struct manage_msg)))
	return ENOBUFS;
    
    m = m_get(M_DONTWAIT, MT_DATA);
    if (m == NULL)
	return ENOBUFS;
    
    man = mtod(m, struct manage_msg *);
    
    man->mm_type = MMTYPE_SOCKET;
    man->mm_request = MMUP_ACCEPT;
    man->mm_manage = &pcb->msp_man;
    *msaddr = man->mm_addr = pcb->msp_dst;
    
    m->m_len = sizeof(struct manage_msg);
    
    manage_send(m);
    
    pcb->msp_state = MSPS_ACCEPTING;
    
    /* We don't need to worry about send here because the send code
     * will bin unless in MSPS_CONNECTED
     */
    
    return 0;
}

void msnl_disconnect(pcb)
    MSPCB		*pcb;
{
    struct manage_msg		*man;
    MBUFP			m;
    int				(* func)();
    caddr_t			arg;

    * (pcb->msp_indirect_error) = ECONNABORTED;
    (pcb->msp_methods->meth_isdisconnected)(pcb->msp_so);
    pcb->msp_state = MSPS_DISCONNECTED;
    
    func = pcb->msp_break_proc;
    arg = pcb->msp_tx_opaque;
    
    pcb->msp_break_proc = pcb->msp_tx_proc = (int (*)())0;
    pcb->msp_tx_opaque = (caddr_t)0;

    if (arg)
	(func)(arg);

    if (!manage_check(sizeof(struct manage_msg)))
	return ;

    m = m_get(M_DONTWAIT, MT_DATA);
    if (m == NULL)
	return ;

    man = mtod(m, struct manage_msg *);
    
    man->mm_type = MMTYPE_SOCKET;
    man->mm_request = MMUP_DISCONNECT;
    man->mm_manage = &pcb->msp_man;

    m->m_len = sizeof(struct manage_msg);
    
    manage_send(m);
}

void msnl_detach(pcb)
    MSPCB		*pcb;
{
    struct manage_msg		*man;
    MBUFP			m;
    int				(* func)();
    caddr_t			arg;

    if (pcb->msp_state == MSPS_NEW)
    {
	/* Special case. The upside has no knowledge so we can just bin it
	 */
	manage_deregister(&pcb->msp_man);

	(pcb->msp_methods->meth_free)(pcb->msp_so);
	
	m_free(dtom(pcb));

	return;
    }
    
    /* Here then we must unhook the PCB from the socket (which is going away)
     * and inform up above to do clean up
     */

    * (pcb->msp_indirect_error) = ECONNABORTED;
    (pcb->msp_methods->meth_isdisconnected)(pcb->msp_so);

    (pcb->msp_methods->meth_free)(pcb->msp_so);
    
    pcb->msp_so = 0;

    pcb->msp_state = MSPS_DEAD;

    func = pcb->msp_break_proc;
    arg = pcb->msp_tx_opaque;
    
    pcb->msp_break_proc = pcb->msp_tx_proc = (int (*)())0;
    pcb->msp_tx_opaque = (caddr_t)0;

    if (arg)
	(func)(arg);

    if (!manage_check(sizeof(struct manage_msg)))
	return ;

    m = m_get(M_DONTWAIT, MT_DATA);
    if (m == NULL)
	return ;

    man = mtod(m, struct manage_msg *);
    
    man->mm_type = MMTYPE_SOCKET;
    man->mm_request = MMUP_DETACH;
    man->mm_manage = &pcb->msp_man;

    m->m_len = sizeof(struct manage_msg);
    
    manage_send(m);
}

/*
 * ----------------------------------------------------------------------
 * This function gets called when on "timeout" of a connect request and
 * ensures that the connect request doesn't block. We have to be carefull to
 * add untimeouts so the rest of the code.
 */

void msnl_connect_timeout(arg)
    caddr_t		arg;
{
    MSPCB		*pcb = (MSPCB *)arg;
    int			s = splnet();
    MBUFP		m;
    struct manage_msg	*man;
    
    pcb->msp_timeoutset = 0;

    if (pcb->msp_state != MSPS_CONNECTING)
    {
	/* Something has happened */
	splx(s);
	return;
    }
    
    * (pcb->msp_indirect_error) = ETIMEDOUT;
    (pcb->msp_methods->meth_isdisconnected) (pcb->msp_so);
    
    pcb->msp_state = MSPS_DISCONNECTED;

    if (!manage_check(sizeof(struct manage_msg)))
    {
	splx(s);
	return;
    }

    m = m_get(M_DONTWAIT, MT_DATA);
    if (m == NULL)
    {
	splx(s);
	return ;
    }
    
    man = mtod(m, struct manage_msg *);
    
    man->mm_type = MMTYPE_SOCKET;
    man->mm_request = MMUP_CONNECTTIMEDOUT;
    man->mm_manage = &pcb->msp_man;

    m->m_len = sizeof(struct manage_msg);
    
    manage_send(m);
    splx(s);
}

/*
 * ----------------------------------------------------------------------
 * This function gets called when an asynchronous socket operation
 * is notified from user space. The protocol control block has
 * already been checked.
 */

int msnl_operation(msg)
    struct manage_msg		*msg;
{
    MSPCB			*pcb = (MSPCB *)(msg->mm_manage), *new_pcb;
    caddr_t			so = pcb->msp_so;
    int				error = 0;
    
    if ((!so) && (msg->mm_reply != MMDOWN_FINAL))
	return ENOTSOCK;

    switch (msg->mm_reply)
    {
    case MMDOWN_NEWCONN:
	if (pcb->msp_state != MSPS_LISTENING)
	    return EINVAL;
	
	new_pcb = (pcb->msp_methods->meth_newconn)(so, &msg->mm_addr);
	/* By now the PRU_ATTACH has been done */
	if (!new_pcb)
	{
	    error = ENOBUFS;
	    break;
	}
	new_pcb->msp_state = MSPS_ACCEPTABLE;
	new_pcb->msp_src = pcb->msp_src;
	new_pcb->msp_dst = msg->mm_addr;
	msnl_sap_again(&new_pcb->msp_src);

	msg->mm_manage = &new_pcb->msp_man;	/* gets back if ioctl */

	/* unix is so horrible yluch */
	(new_pcb->msp_methods->meth_isconnected)(new_pcb->msp_so);
	
	break;
	
    case MMDOWN_ACCEPTED:
	if (pcb->msp_state != MSPS_ACCEPTING)
	{
	    error = EINVAL;
	    break;
	}
	pcb->msp_state = MSPS_CONNECTED;
	(pcb->msp_methods->meth_isreallyconnected)(so);

	break;

    case MMDOWN_CONNECTED:
	if (pcb->msp_state != MSPS_CONNECTING)
	{
	    error = EINVAL;
	    break;
	}
	
	pcb->msp_dst = msg->mm_addr;

	(pcb->msp_methods->meth_isconnected)(so);
	pcb->msp_state = MSPS_CONNECTED;
	(pcb->msp_methods->meth_isreallyconnected)(so);

	break;

    case MMDOWN_CONNECTFAIL:
	if (pcb->msp_state != MSPS_CONNECTING)
	{
	    error = EINVAL;
	    break;
	}

	* (pcb->msp_indirect_error) = msg->mm_error;
	(pcb->msp_methods->meth_isdisconnected) (pcb->msp_so);

	pcb->msp_state = MSPS_DISCONNECTED;
	break;
	
    case MMDOWN_DISCONNECT:
	error = (pcb->msp_methods->meth_disconnect)(so);
	break;

    case MMDOWN_ABORT:
	error = (pcb->msp_methods->meth_abort)(so);
	break;

    case MMDOWN_FINAL:
	if (pcb->msp_state != MSPS_DEAD)
	{
	    error = EBUSY;
	    break;
	}
	
	if ((pcb->msp_so) || (pcb->msp_tx_opaque))
	    panic("msnl_operation: pcb not isolated\n");

	msnl_sap_free(&pcb->msp_src);
	manage_deregister(&pcb->msp_man);
	if (pcb->msp_timeoutset)
	    untimeout(msnl_connect_timeout, (caddr_t)pcb);
	
	m_free(dtom(pcb));
	break;
	
    case MMDOWN_SOCKETJOIN:
	if (!manage_lookup(msg->mm_pcb))
	{
	    error = EFAULT;
	    break;
	}
	new_pcb = (MSPCB *)msg->mm_pcb;
	
	if ((pcb->msp_state != MSPS_CONNECTING) 
	    || (new_pcb->msp_state != MSPS_ACCEPTING))
	{
	    error = EINVAL;
	    break;
	}
	
	if ((pcb->msp_tx_opaque) || (new_pcb->msp_tx_opaque))
	    panic("msnl_operation: MMDOWN_SOCKETJOIN\n");
	
	/* Everything OK so hook up */

	pcb->msp_state = MSPS_CONNECTED;
	new_pcb->msp_state = MSPS_CONNECTED;

	pcb->msp_tx_opaque = (caddr_t)new_pcb;
	new_pcb->msp_tx_opaque = (caddr_t)pcb;
	
	pcb->msp_tx_proc = new_pcb->msp_tx_proc = pcbrecv;
	
	pcb->msp_break_proc = new_pcb->msp_break_proc = msnl_break_local;
	
	/* The adresses for the pasive side will have been done when
	 * the new connection was informed. The active side needs to be
	 * given the targets address (although thats what it will have
	 * requested nominally in any case)
	 */
	pcb->msp_dst = new_pcb->msp_src;

	(pcb->msp_methods->meth_isconnected)(pcb->msp_so);

	(pcb->msp_methods->meth_isreallyconnected)(pcb->msp_so);
	(new_pcb->msp_methods->meth_isreallyconnected)(new_pcb->msp_so);

	error = 0;
	break;

    default:
	error = EINVAL;
	break;
    }
    return error;
}

/*
 * ----------------------------------------------------------------------
 * Code for local msnl
 */

static int pcbrecv(pcb, m0)
    MSPCB		*pcb;
    MBUFP		m0;
{
    if (pcb->msp_so)
	return (pcb->msp_methods->meth_recv)(pcb->msp_so, m0);
    
    printf("pcbrecv(msnl): This shouldn't happen (serious)\n");
    m_freem(m0);
    return 0;
}

static int msnl_break_local(pcb)
    MSPCB		*pcb;
{
    /* Need to think carefully to understand this. The argument PCB is
     * the "destination", ie the "other" one is already being dealt with
     * and this one needs dealt with.
     * All the callers have already cleared their fields, but we want
     * to clear ours so the disconnect on ours will not call us back for
     * the other one.
     */
    pcb->msp_break_proc = pcb->msp_tx_proc = (int (*)())0;
    pcb->msp_tx_opaque = (caddr_t)0;
    
    (pcb->msp_methods->meth_disconnect)(pcb->msp_so);

    return 0;
}

/*
 * ----------------------------------------------------------------------
 * This function is also used from msnl_manage.c
 */

int sorecv(so, m0)
    SOCK		*so;
    MBUFP		m0;
{
    int			len = 0;
    MBUFP		m;
    struct sockbuf	*sb = &so->so_rcv;

    /* Check space in socket receive buffer. */

    for (m = m0; m; m = m->m_next)
        len += m->m_len;

    if (len > sbspace(sb)) 
    {
        m_freem(m0);
        return ENOBUFS;
    }

    /* Pass ownership to socket */

    for (m = m0; m; m = m->m_next)
        sballoc(sb, m);

    /* Hook unto socket rx queue. This is faster than calls to sbappend
     * but could probably be improved by keeping the tail
     */

    m0->m_act = 0;
    if ((m = sb->sb_mb))
    {
        while (m->m_act)
            m = m->m_act;
        m->m_act = m0;
    }
    else 
        sb->sb_mb = m0;

    sorwakeup(so);

    return 0;
}

/*
 * ----------------------------------------------------------------------
 */

void mysoisconnected(so)
    SOCK		*so;
{
    soisconnected(so);
    so->so_snd.sb_hiwat = 0;
}

void soisreallyconnected(so)
    SOCK		*so;
{
    /*
     * This function is used for "soisreallyconnected"
     * for broken unix sockets.
     */
    so->so_snd.sb_hiwat = msnlspace;
    sowwakeup(so);
}

MSPCB *newconnwrapper(so, addr)
    SOCK		*so;
    struct sockaddr	*addr;
{
    SOCK		*new_so;
    
    new_so = sonewconn(so);
    if (!new_so)
	return 0;
    
    return sotomsnlpcb(new_so);
}
