/*
 *	msnl_tunnel.c
 *	-------------
 *
 * $Id: msnl_tunnel.c,v 1.14 1993/11/22 15:52:49 rjb17 Exp $
 *
 * Copyright (c) 1993 Cambridge University Computer Laboratory.
 *
 */

#include "../h/param.h"
#include "../h/protosw.h"
#include "../h/mbuf.h"
#include "../h/errno.h"
#include "../h/socket.h"
#include "../h/socketvar.h"
#include "../h/ioctl.h"
#include "../net/net/if.h"
#include "../net/net/netisr.h"
#include "../netinet/in.h"		/* Needed for if_ether.h */
#include "../netinet/if_ether.h"	/* Silly place for it */

#include "msnl.h"
#include "msnl_manage.h"
#include "msnl_pcb.h"
#include "msnl_usrreq.h"

/*
 * move these elsewhere!
 */

typedef struct mbuf MBUF;
typedef MBUF *MBUFP;

#define NMSIFS		(4)

#define MAXWAITBUFS	(6)

/*
 * ----------------------------------------------------------------------
 *
 * If you dont understand why these magic numbers are what they are
 * then youre not qualified to change them.
 */
/* 1344, 4608, 8160, 9180, 12192}; */
unsigned int tunnel_mtus[NMSIFS] = { 1344, 2592, 4032, 4608 };

/*
 * ----------------------------------------------------------------------
 */


/*
 * ----------------------------------------------------------------------
 */

typedef struct _tunnel {
    struct _tunnel	* tun_link;
    unsigned int	tun_dst;
    int			tun_waitcount;
    MBUFP		tun_waiting;
    unsigned short	tun_error;
    MSPCB		* tun_pcb;
    int			tun_valid;
    int			tun_passive;
    struct ifnet	* tun_ifp;
} TUNNEL;

/*
 * ----------------------------------------------------------------------
 */

#define TUNNEL_HASH_ARRAY 16
#define TUNNEL_HASH(x)	(((x >> 24) ^ (x)) & 15)

TUNNEL * tunnels[NMSIFS][TUNNEL_HASH_ARRAY] = { 0, };

static void tunnel_add(int ifn, TUNNEL *t);
static void tunnel_delete(int ifn, TUNNEL *t);
static TUNNEL *tunnel_lookup(int ifn, unsigned int dst);

static void tunnel_add(ifn, t)
    int			ifn;
    TUNNEL		*t;
{
    int			index;
    
    index = TUNNEL_HASH(t->tun_dst);
    t->tun_link = tunnels[ifn][index];
    tunnels[ifn][index] = t;
}

static void tunnel_delete(ifn, t)
    int			ifn;
    TUNNEL		*t;
{
    int			index;
    TUNNEL		**p;
    
    index = TUNNEL_HASH(t->tun_dst);
    p = &tunnels[ifn][index];
    while ((*p) != t)
	p = &((*p)->tun_link);
    *p = (*p)->tun_link;
}

static TUNNEL *tunnel_lookup(ifn, dst)
    int			ifn;
    unsigned int	dst;
{
    int			index;
    TUNNEL		*t;
    
    index = TUNNEL_HASH(dst);
    t = tunnels[ifn][index];
    while(t)
	if (t->tun_dst == dst)
	    return t;
	else
	    t = t->tun_link;
    return 0;
}

/*
 * ----------------------------------------------------------------------
 */

static TUNNEL *tunnel_alloc(unsigned int dst, struct ifnet *ifp);

static TUNNEL *tunnel_alloc(dst, ifp)
    unsigned int	dst;
    struct ifnet	*ifp;
{
    MBUFP		m;
    TUNNEL		*t;
    
    if ((m = m_getclr(M_DONTWAIT, MT_PCB)) == 0)
	return 0;
    
    m->m_len = sizeof(TUNNEL);
    t = mtod(m, TUNNEL *);
    t->tun_dst = dst;
    t->tun_ifp = ifp;
    tunnel_add(ifp->if_unit, t);
    return t;
}

/*
 * ----------------------------------------------------------------------
 * Tunnel Methods.
 *
 * We are not interested in:
 * 	isconnecting.
 *
 * We must check on
 *	isconnected
 * because it is a hack for accept.
 *
 * Both:
 *	disconnect, and abort
 * we just want to call detach.
 *
 * Have to be a little careful with soisdisconnected because the other
 * person is still using it. So mark as down and wait for the cleaner.
 *
 * Otherwise obvious.
 */

static void tunnel_nop(t)
    TUNNEL		*t;
{
}

static void tunnel_isdisconnected(t)
    TUNNEL		*t;
{
    t->tun_valid = 0;
}

static int tunnel_broken(t)
    TUNNEL		*t;
{
    t->tun_valid = 0;
    msnl_detach(t->tun_pcb);
    return 0;
}

/*
 * This function almost certainly shouldnt be called directly in this
 * file, but rather via the "sofree" method from msnl_usrreq.c.
 * what is probably wanted is a call to tunnel_broken.
 */

static void tunnel_free(t)
    TUNNEL		*t;
{
    MBUFP		m1,m2;
    
    m1 = t->tun_waiting;
    while (m1)
    {
	m2 = m1->m_act;
	m_freem(m1);
	m1 = m2;
	t->tun_ifp->if_oerrors++;
    }
    
    if (t->tun_passive)
    {
	/* This was the listening one. So we have to mark the interface as
	 * down.
	 */
	t->tun_ifp->if_flags &= ~IFF_UP;
    }
    tunnel_delete(t->tun_ifp->if_unit, t);
    m_free(dtom(t));
    /* No more references */
}

static void tunnel_isreallyconnected(t)
    TUNNEL		*t;
{
    MBUFP		m1;

    t->tun_valid = 1;

    /*
     * Clear possible backlog
     */
    while (t->tun_waiting)
    {
	m1 = t->tun_waiting->m_act;
	t->tun_ifp->if_opackets++;
	msnl_send(t->tun_pcb, t->tun_waiting);
	t->tun_waiting = m1;
    }
    t->tun_waitcount = 0;
}

static int tunnel_recv(t,m)
    TUNNEL		*t;
    MBUFP		m;
{
    struct ifqueue	*ipq = &ipintrq;
    int			s = splimp();
    
    if (IF_QFULL(ipq))
    {
	IF_DROP(ipq);
	m_freem(m);
	t->tun_ifp->if_ierrors++;
	splx(s);
	return ENOBUFS;
    }
    
    t->tun_ifp->if_ipackets++;
    
    IF_ENQUEUEIF(ipq, m, t->tun_ifp);
    schednetisr(NETISR_IP);
    splx(s);
    
    return 0;
}

static MSPCB *tunnel_newconn(passive, addr)
    TUNNEL		*passive;
    struct sockaddr_in	*addr;
{
    TUNNEL		*old, *new;
    
    if (addr->sin_family != AF_INET)
	return 0;
    
    if (!(addr->sin_addr.s_addr))
	return 0;

    old = tunnel_lookup(passive->tun_ifp->if_unit, addr->sin_addr.s_addr);
    if (old)
	tunnel_broken(old);
    
    new = tunnel_alloc(addr->sin_addr.s_addr, passive->tun_ifp);
    if (!new)
	return 0;
    
    if (ip_msnl_attach(new))
    {
	tunnel_free(new);
	return 0;
    }
    
    return new->tun_pcb;
}

static void tunnel_isconnected(t)
    TUNNEL		*t;
{
    struct sockaddr	dustbin;
    
    if (t->tun_pcb->msp_state == MSPS_ACCEPTABLE)
	msnl_accept(t->tun_pcb, &dustbin);
}

/*
 * ----------------------------------------------------------------------
 */

static void tunnel_trash_world(int unit)
{
    int		i;
    
    for (i=0; i<TUNNEL_HASH_ARRAY; i++)
	while(tunnels[unit][i])
	    tunnel_broken(tunnels[unit][i]);
}	

/*
 * ----------------------------------------------------------------------
 */

static struct msnl_methods msnl_tunnel_methods = {
    tunnel_nop,
    tunnel_isdisconnected,
    tunnel_newconn,
    tunnel_isconnected,
    tunnel_broken,
    tunnel_broken,
    tunnel_free,
    tunnel_recv,
    tunnel_isreallyconnected
};
    
/*
 * ----------------------------------------------------------------------
 */

static int ip_msnl_attach(t)
    TUNNEL		*t;
{
    MSPCB		*pcb;
    int			error;
    MBUFP		m;

    if (!manage_running())
	return EPFNOSUPPORT;

    if ((m = m_getclr(M_DONTWAIT, MT_PCB)) == 0)
	return ENOBUFS;
    
    pcb = mtod(m, MSPCB *);

    /* Initialise the PCB */
    t->tun_pcb = pcb;
    pcb->msp_so = (caddr_t) t;

    pcb->msp_methods = &msnl_tunnel_methods;
    
    pcb->msp_indirect_error = & (t->tun_error);
    
    manage_register(&pcb->msp_man);
    
    return 0;
}

/*
 * ----------------------------------------------------------------------
 */

int ip_msnl_output(ifp, m, dst)
    struct ifnet	*ifp;
    MBUFP		m;
    struct sockaddr	*dst;
{
    TUNNEL		*t;
    MBUFP		m1;
    
    if ((ifp->if_flags & (IFF_UP | IFF_RUNNING)) != (IFF_UP | IFF_RUNNING))
    {
	m_freem(m);
	return ENETDOWN;
    }
    
    if (dst->sa_family != AF_INET)
    {
	/* How could this happen ??? */
	m_freem(m);
	return EAFNOSUPPORT;
    }

    /* If this is to our address then we must do something special
     * because we arnt set up to have a connection that loops back
     * to the same socket, and we cant have two endpoints with the
     * same local address.
     * So in this case we just punt it back up.
     * Note that we DO have a local connection (if the manager permits)
     * for one interface talking to another
     */

    if ((((struct sockaddr_in *)dst)->sin_addr.s_addr) == 
	(((struct sockaddr_in *)(&ifp->if_addr))->sin_addr.s_addr))
    {
	int		s = splimp();
	struct ifqueue	*ipq = &ipintrq;

	if (IF_QFULL(ipq))
	{
	    IF_DROP(ipq);
	    m_freem(m);
	    ifp->if_oerrors++;
	    ifp->if_ierrors++;
	    splx(s);
	    return ENOBUFS;
	}
    
	ifp->if_opackets++;
	ifp->if_ipackets++;
    
	IF_ENQUEUEIF(ipq, m, ifp);
	schednetisr(NETISR_IP);
	splx(s);
	return 0;
    }
    
    /* Normal case */
    
    t = tunnel_lookup(ifp->if_unit, 
		      ((struct sockaddr_in *)dst)->sin_addr.s_addr);
    if (t)
    {
	/*
	 * A tunnel already exists for that destination
	 * we must check to see if it is really connected. If it is
	 * there should be no waiting mbufs as they will have gone when it
	 * was connected. If not we may already have an mbuf waiting to go on
	 * it. We limit the number of outstanding mbufs to MAXWAITBUFS
	 */
	if (t->tun_valid)
	{
	    if (t->tun_pcb->msp_state != MSPS_CONNECTED)
		panic("ip_msnl_output: Help! (%d)\n",2);
	    if (t->tun_waiting)
		panic("ip_msnl_output: Help! (%d)\n",3);
	    /*
	     * Send this one
	     */
	    ifp->if_opackets++;
	    msnl_send(t->tun_pcb, m);
	    return 0;
	}
	/*
	 * Here then not yet connected so queue it, possible dropping
	 * oldest one.
	 */
	if (t->tun_waitcount > MAXWAITBUFS)
	{
	    if (!t->tun_waiting)
		panic("ip_msnl_output: Help! (%d)\n",4);
	    m1 = t->tun_waiting;
	    t->tun_waiting = t->tun_waiting->m_act;
	    m_freem(m1);
	    ifp->if_oerrors++;
	}
	else
	{
	    t->tun_waitcount++;
	}

	m->m_act = 0;
	if ((m1 = t->tun_waiting))
	{
	    while (m1->m_act) m1 = m1->m_act;
	    m1->m_act = m;
	}
	else
	    t->tun_waiting = m;

	return 0;
    }
    else
    {
	/*
	 * No such tunnel exists - so go about creating one
	 */
	int		error;
	struct sockaddr	addr;
	
	t = tunnel_alloc(((struct sockaddr_in *)dst)->sin_addr.s_addr, ifp);
	if (!t)
	{
	    m_freem(m);
	    return ENOBUFS;
	}

	t->tun_waitcount = 1;
	t->tun_waiting = m;
	
	if ((error = ip_msnl_attach(t)))
	{
	    tunnel_delete(t->tun_ifp->if_unit, t);
	    m_free(dtom(t));
	    m_freem(m);
	    return error;
	}
     
	bzero(&addr, sizeof(struct sockaddr));
	addr.sa_family = AF_MSNL;
	
	if ((error = msnl_bind(t->tun_pcb, &addr)))
	{
	    tunnel_broken(t);
	    return error;
	}
	
	bcopy(dst, &addr, sizeof(struct sockaddr));
	addr.sa_data[6] = 0;
	addr.sa_data[7] = 0;
	addr.sa_data[8] = 0;
	addr.sa_data[9] = 56 - ifp->if_unit;

	if ((error = msnl_connect(t->tun_pcb, &addr)))
	{
	    tunnel_broken(t);
	    return error;
	}

	return 0;
    }
}

/*
 * ----------------------------------------------------------------------
 *	N O W   F O R   T H E   P A S S I V E   S I D E   !
 * ----------------------------------------------------------------------
 */

int address_ever[NMSIFS];
struct sockaddr_msnl my_address[NMSIFS];

int msnl_tunnel_ioctl(ifp, cmd, data)
    struct ifnet	*ifp;
    int			cmd;
    caddr_t		data;
{
    struct sockaddr_in	*addr;
    int			error = 0;
    TUNNEL		*t;
    int			unit = ifp->if_unit;

    /* We take every opportunity to check on the state of the manager
     */
    
    if (!manage_running())
    {
	ifp->if_flags &= ~IFF_UP;
    }

    switch (cmd)
    {
    case SIOCSIFADDR:
	printf("msnl_tunnel_ioctl: SIOCSIFADDR\n");

	addr = (struct sockaddr_in *)(&(((struct ifaddr *)data)->ifa_addr));

	ifp->if_addr = *((struct sockaddr *)addr);
	
	if (addr->sin_family != AF_INET)
	    return EAFNOSUPPORT;
	
	/* we use 0 for passive */
	if (!addr->sin_addr.s_addr)
	    return EADDRNOTAVAIL;
	
	if (address_ever[unit])
	{
	    if (my_address[unit].ms_ipa == addr->sin_addr.s_addr)
	    {
		/*
		 * Could be beinging back up???
		 */
		return 0;
	    }
	    /*
	     * The IP address of this interface has been changed. What a bore
	     */
	    tunnel_trash_world(unit);
	}
	else
	{
	    address_ever[unit] = 1;
	}
	my_address[unit].sa_family = AF_MSNL;
	my_address[unit].msnl_adr.m_host = 0;
	((unsigned char *)(&my_address[unit].msnl_adr.m_port))[0] = 0;
	((unsigned char *)(&my_address[unit].msnl_adr.m_port))[1] = 0;
	((unsigned char *)(&my_address[unit].msnl_adr.m_port))[2] = 0;
	((unsigned char *)(&my_address[unit].msnl_adr.m_port))[3] = 56 - unit;
	my_address[unit].ms_ipa = addr->sin_addr.s_addr;
	
	break;
	
    case SIOCSIFFLAGS:
	/* Changed already ! */
	printf("msnl_tunnel_ioctl: SIOCSIFFLAGS %x\n", ifp->if_flags);
	
	/* We dont actually need to do anything else if not comming up */
	if (!(ifp->if_flags & IFF_UP))
	    return 0;

	/* Set it down until it actually comes up */
	
	ifp->if_flags &= ~IFF_UP;

	/* Set flags gets called before set address (boggle) so
	 * if no address then dont do anything
	 */

	if (!address_ever[unit])
	    return EADDRNOTAVAIL;

	/* Need to make sure that there is a passive side here since things
	 * could have come back
	 */

	if (tunnel_lookup(unit, 0))
	{
	    ifp->if_flags |= IFF_UP;
	    return 0;
	}
	
	tunnel_trash_world(unit);
	
	/* We come to here if the interface is comming back */
	
	t = tunnel_alloc(0, ifp);
	if (!t)
	{
	    return ENOBUFS;
	}
	
	t->tun_passive = 1;
	
	if ((error = ip_msnl_attach(t)))
	{
	    tunnel_delete(unit, t);
	    m_free(dtom(t));
	    return error;
	}
	
	if ((error = msnl_bind(t->tun_pcb, &my_address[unit])))
	{
	    tunnel_broken(t);
	    return error;
	}
	
	if ((error = msnl_listen(t->tun_pcb)))
	{
	    tunnel_broken(t);
	    return error;
	}
	
	/*
	 * Things are starting to look up
	 */
	
	ifp->if_flags |= IFF_UP;
	
	break;
	
    default:
	printf("msnl_tunnel_ioctl: %d\n", cmd);
	return EOPNOTSUPP;
    }
    
    /*
     * ------
     */

    return 0;
}

/*
 * ----------------------------------------------------------------------
 */

static struct ifnet *tunnel_ifs[NMSIFS];

/*
 * ----------------------------------------------------------------------
 */

tunnel_init()
{
    struct ifnet		*ifp;
    int				i;
    
    /* Initialise local structures */
    bzero(address_ever, sizeof(address_ever));
    bzero(tunnels, sizeof(tunnels));

    for (i=0; i<NMSIFS; i++)
    {
	/* Init an interface */
	
	KM_ALLOC(ifp, struct ifnet *, sizeof(struct ifnet), KM_DEVBUF, 
		 KM_NOWAIT);
	
	if (!ifp)
	    panic("tunnel_init: memory exhausted during kernel init!\n");
	
	bzero(ifp, sizeof(struct ifnet));
	tunnel_ifs[i] = ifp;

	ifp->if_name = "ms";
	ifp->if_unit = i;
	
	ifp->if_mtu = tunnel_mtus[i];
	ifp->if_flags = IFF_NOTRAILERS | IFF_NOARP | IFF_RUNNING;
	
	ifp->if_output = ip_msnl_output;
	ifp->if_ioctl = msnl_tunnel_ioctl;
	
	if_attach(ifp);
    }
}    
