/*
 *	msnl_assoc.c
 *	------------
 *
 * $Id: msnl_assoc.c,v 1.16 1993/11/22 15:48:31 rjb17 Exp $
 *
 * Copyright (c) 1993 Cambridge University Computer Laboratory.
 *
 */

#include "../h/param.h"
#include "../h/mbuf.h"
#include "../h/errno.h"
#include "../h/socket.h"
#include "../h/socketvar.h"
#include "../net/net/if.h"
#include "../net/net/netisr.h"
#include "../netinet/in.h"		/* Needed for if_ether.h */
#include "../netinet/if_ether.h"	/* Silly place for it */

#include "msnl_sar.h"
#include "msnl_assoc.h"
#include "msnl_manage.h"
#include "msnl_pcb.h"

/*
 * move these elsewhere!
 */

typedef struct mbuf MBUF;
typedef MBUF *MBUFP;

/*
 * This file deals with things necessary for association management and
 * the hooks between the software interrupt code and the pcb code.
 */

#ifdef __STDC__

static void assoc_normal      (MBUFP m, ASSOC *assoc, caddr_t q);
static void assoc_promisc_edl (MBUFP m0, ASSOC *assoc, caddr_t q);
static int  assoc_alloc       (int associd, void (* rx_proc)(), 
			       struct ifnet *ifp, MSPCB *pcb);
static void assoc_destroy     (ASSOC *assoc);
int         assoc_operation   (struct manage_msg *msg);

#endif


/*
 * ---------------------------------------------------------------------
 */

ASSOC *assocTable[ASSOC_MAX+1];

u_char assocBad[ASSOC_MAX+1];

#define ASSOC_MAXPOD	(4)

struct assocPod {
    int				asp_valid;
    struct ifnet		*asp_ifp;
    int				(* asp_tx_proc)();
    int				(* asp_prepare)();
    int				(* asp_tx_promisc)();
} assocPods[ASSOC_MAXPOD+1];

/*
 * ---------------------------------------------------------------------
 */

static int ifp_to_pod(ifp)
    struct ifnet	*ifp;
{
    int		i;
    
    for (i=0; i<=ASSOC_MAXPOD; i++)
    {
	if ((assocPods[i].asp_valid) && (assocPods[i].asp_ifp == ifp))
	    return i;
    }
    panic("ifp_to_pod(msnl): ifp corrupt!\n");
}

/*
 * ---------------------------------------------------------------------
 */

static void assoc_normal(m, assoc, q)
    MBUFP		m;
    ASSOC		*assoc;
    caddr_t		q;		/* Not used */
{
    MSPCB		*pcb = (MSPCB *)assoc->as_pcb;
    
    assoc->as_idle = 0;

    if (pcb->msp_state != MSPS_CONNECTED)
    {
	/* Normally this would not happen except during the race
	 * between assoc management and connection management
	 */
	m_freem(m);
	return;
    }
    
    if (pcb->msp_so)
    {
	(pcb->msp_methods->meth_recv)(pcb->msp_so, m);
	return;
    }
    
    printf("assoc_normal(msnl): This shouldn't happen (serious)\n");
    m_freem(m);
    return;
}

/*
 * ---------------------------------------------------------------------
 */

static void assoc_promisc_edl(m0, assoc, q0)
    MBUFP		m0;
    ASSOC		*assoc;
    caddr_t		q0;
{
    struct manage_msg		*man;
    MBUFP			m;
    EDLQUEUE			*q = (EDLQUEUE *)q0;
    int				len;
    
    len=0;
    for (m=m0; m; m=m->m_next)
	len += m->m_len;

    if (!manage_check(sizeof(struct manage_msg) + len))
    {
	m_freem(m0);
	return ;
    }

    m = m_get(M_DONTWAIT, MT_DATA);
    if (m == NULL)
    {
	m_freem(m0);
	return ;
    }

    man = mtod(m, struct manage_msg *);
    
    man->mm_type = MMTYPE_ASSOC;
    man->mm_request = MMUP_EDL;
    bcopy(q->src, man->mm_src, 6);
    man->mm_pod = ifp_to_pod(q->ifp);

    m->m_len = sizeof(struct manage_msg);
    m->m_next = m0;
    
    manage_send(m);
}
    
/*
 * ---------------------------------------------------------------------
 */

static void assoc_promisc_fdl(m0, assoc, q0)
    MBUFP		m0;
    ASSOC		*assoc;
    caddr_t		q0;
{
    struct manage_msg		*man;
    MBUFP			m;
    FDLQUEUE			*q = (FDLQUEUE *)q0;
    int				len;
    
    len=0;
    for (m=m0; m; m=m->m_next)
	len += m->m_len;

    if (!manage_check(sizeof(struct manage_msg) + len))
    {
	m_freem(m0);
	return ;
    }

    m = m_get(M_DONTWAIT, MT_DATA);
    if (m == NULL)
    {
	m_freem(m0);
	return ;
    }

    man = mtod(m, struct manage_msg *);
    
    man->mm_type = MMTYPE_ASSOC;
    man->mm_request = MMUP_FDL;
    man->mm_pod = ifp_to_pod(q->ifp);

    m->m_len = sizeof(struct manage_msg);
    m->m_next = m0;
    
    manage_send(m);
}
    
/*
 * ---------------------------------------------------------------------
 * At this stage the pod has not been checked to be valid.
 * we must set the ifp back to 0 afterwards because the RX side uses it.
 */

static int assoc_tx_promisc_edl(msg)
    struct manage_msg		*msg;
{
    MBUFP			m0, mtx;
    ASSOC			*assoc;
    int				error;
    
    if ((msg->mm_pod > ASSOC_MAXPOD) || 
	(!assocPods[msg->mm_pod].asp_valid))
	return ENXIO;

    m0 = dtom(msg);
    mtx = m_copy(m0, sizeof(struct manage_msg), M_COPYALL);
    if ((mtx == 0) || (mtx->m_len == 0))
	return EMSGSIZE;

    /* Now send it */

    assoc = assocTable[ASSOC_EDL];
    
    (assocPods[msg->mm_pod].asp_prepare)(assoc, &msg->mm_addr, 
					 assocPods[msg->mm_pod].asp_ifp);

    assoc->as_ifp = assocPods[msg->mm_pod].asp_ifp;

    error = (assocPods[msg->mm_pod].asp_tx_proc)(assoc, mtx);
    
    assoc->as_ifp = 0;
    return error;
}

/*
 * ---------------------------------------------------------------------
 * At this stage the pod has not been checked to be valid.
 * we must set the ifp back to 0 afterwards because the RX side uses it.
 */

static int assoc_tx_promisc_fdl(msg)
    struct manage_msg		*msg;
{
    MBUFP			m0, mtx;
    ASSOC			*assoc;
    int				error;
    
    if ((msg->mm_pod > ASSOC_MAXPOD) || 
	(!assocPods[msg->mm_pod].asp_valid))
	return ENXIO;

    m0 = dtom(msg);
    mtx = m_copy(m0, sizeof(struct manage_msg), M_COPYALL);
    if ((mtx == 0) || (mtx->m_len == 0))
	return EMSGSIZE;

    /* Now send it */

    assoc = assocTable[ASSOC_FDL];
    
    (assocPods[msg->mm_pod].asp_prepare)(assoc, &msg->mm_addr, 
					 assocPods[msg->mm_pod].asp_ifp);

    assoc->as_ifp = assocPods[msg->mm_pod].asp_ifp;

    error = (assocPods[msg->mm_pod].asp_tx_proc)(assoc, mtx);
    
    assoc->as_ifp = 0;
    return error;
}

/*
 * ---------------------------------------------------------------------
 */

void assoc_got_bogus(associd, ifp)
    int			associd;
    struct ifnet	*ifp;
{
    MBUFP		m;
    struct manage_msg	*man;
    
    if ((associd < 1) || (associd > ASSOC_MAX))
    {
	printf("assoc_got_bogus(msnl): corrupt %04X\n",associd);
	return;
    }

    if (assocBad[associd]++ != 1)
	return;
    
    if (!manage_check(sizeof(struct manage_msg)))
	return;

    m = m_get(M_DONTWAIT, MT_DATA);
    if (m == NULL)
	return;

    man = mtod(m, struct manage_msg *);
    
    man->mm_type = MMTYPE_ASSOC;
    man->mm_request = MMUP_ASSOCBAD;
    man->mm_associd = associd;
    man->mm_pod = ifp_to_pod(ifp);

    m->m_len = sizeof(struct manage_msg);
    
    manage_send(m);
}

/*
 * ---------------------------------------------------------------------
 */

static int assoc_alloc(associd, rx_proc, ifp, pcb)
    int			associd;
    void		(* rx_proc)();
    struct ifnet	*ifp;
    MSPCB		*pcb;
{
    ASSOC		*assoc;

    if ((associd < 1) || (associd > ASSOC_MAX))
	return EINVAL;
    
    if (assocTable[associd])
	return EEXIST;

    assocBad[associd] = 0;

    KM_ALLOC(assoc, ASSOC *, sizeof(ASSOC), KM_DEVBUF, KM_NOWAIT);
    if (!assoc)
	return ENOBUFS;
    
    bzero(assoc, sizeof(ASSOC));
    assoc->as_id = associd;
    assoc->as_ifp = ifp;
    assoc->as_rx_proc = rx_proc;
    assoc->as_pcb = (caddr_t)pcb;
    assoc->as_idle = 1;
    
    assocTable[associd] = assoc;
    return 0;
}

/*
 * ---------------------------------------------------------------------
 * This is called when the pcb is deallocated to destroy the underlying
 * association.
 */

static void assoc_destroy(assoc)
    ASSOC		*assoc;
{
    int			associd = assoc->as_id;

    assocTable[associd] = (ASSOC *)0;
    /* That atomically removed the assoc */

    if (assoc->as_m_start)
	m_freem(assoc->as_m_start);
    KM_FREE(assoc, KM_DEVBUF);
}

/*
 * ---------------------------------------------------------------------
 */

int fdl_tx_prepare(ASSOC *assoc, struct sockaddr *ptr, struct ifnet *ifp)
{
    assoc->as_txadr = *ptr;
    assoc->as_txsar.fdlsar.all = 0;

    return 0;
}

/*
 * ---------------------------------------------------------------------
 */

int assoc_operation(msg)
    struct manage_msg		*msg;
{
    int				error = 0;
    MSPCB			*pcb;
    ASSOC			*assoc;
    
    switch (msg->mm_reply)
    {
    case MMDOWN_ASSOCCREATE:
	if (!manage_lookup(msg->mm_pcb))
	    return EFAULT;
	/* Check that the stated pcb does not already have a data path.
	 * Whether its local or not the tx_opaque field will be filled.
	 */
	pcb = (MSPCB *)msg->mm_pcb;
	/*
	 * First check that PCB is in a sensible state (ie not dead!!)
	 */
	if ((pcb->msp_state != MSPS_ACCEPTING)
	    && (pcb->msp_state != MSPS_CONNECTING))
	    return EINVAL;
	if (pcb->msp_tx_opaque)
	    return EINVAL;
	if ((msg->mm_pod > ASSOC_MAXPOD) || 
	    (!assocPods[msg->mm_pod].asp_valid))
	    return ENXIO;
	if ((error = assoc_alloc(msg->mm_associd, assoc_normal,
				assocPods[msg->mm_pod].asp_ifp, 
				(MSPCB *)msg->mm_pcb)))
	    break;
	/* Somewhere here should check with the driver that its OK etc.
	 */
	pcb->msp_tx_proc = (int (*)())assocPods[msg->mm_pod].asp_tx_proc;
	pcb->msp_break_proc = (int (*)(caddr_t))assoc_destroy;
	pcb->msp_tx_opaque = (caddr_t)assocTable[msg->mm_associd];

	(assocPods[msg->mm_pod].asp_prepare)(assocTable[msg->mm_associd], 
					     &msg->mm_addr, 
					     assocPods[msg->mm_pod].asp_ifp);
	break;

    case MMDOWN_ASSOCSEND:
	if ((msg->mm_pod > ASSOC_MAXPOD) || 
	    (!assocPods[msg->mm_pod].asp_valid))
	    return ENXIO;
	if (!(assocPods[msg->mm_pod].asp_tx_promisc))
	    panic("assoc_operation(msnl): No tx_promisc!\n");
	
	error = (assocPods[msg->mm_pod].asp_tx_promisc)(msg);
	break;
	
    case MMDOWN_ASSOCIDLE:
	if (!(assoc = assoc_check(msg->mm_associd)))
	    return EINVAL;
	msg->mm_reply = assoc->as_idle;
	assoc->as_idle = 1;
	break;

    default:
	error = EINVAL;
	break;
    }
    return error;
}

/*
 * ---------------------------------------------------------------------
 */

int assoc_ioctl(cmd, data)
    int		cmd;
    caddr_t	data;
{
    int		pod = *(u_int *)data;
    
    if ((pod < 0) || (pod > ASSOC_MAXPOD) || (!(assocPods[pod].asp_valid)))
	return EINVAL;
    
    return (assocPods[pod].asp_ifp->if_ioctl)(assocPods[pod].asp_ifp,
					      cmd, data+sizeof(pod));
}

/*
 * ---------------------------------------------------------------------
 */

void assoc_init()
{
    struct ifnet		*ifp;
    int				pod = 0;
    int				gotedlpod = 0;
    int				gotfdlpod = 0;
    struct assocPod		*ap;
    
    bzero(assocTable, sizeof(assocTable));
    bzero(assocPods, sizeof(assocPods));
    bzero(assocBad, sizeof(assocBad));
    
    /* Search for interfaces */

    for( ifp=ifnet; ifp; ifp = ifp->if_next, pod++ )
    {
	if (
#ifdef __arm
	    ((ifp->if_name[0] == 'e') & (ifp->if_name[1] == 't'))
#endif
#ifdef __mips
	    (ifp->if_type == IFT_ETHER)
#endif
	    )
	{
	    if (pod > ASSOC_MAXPOD)
	    {
		printf("assoc_init: too many network interfaces!\n");
		return ;
	    }
	    ap = &assocPods[pod];
	    
	    ap->asp_valid = 1;
	    ap->asp_ifp = ifp;
	    ap->asp_tx_proc = edl_tx;
	    ap->asp_prepare = edl_tx_prepare;
	    ap->asp_tx_promisc = assoc_tx_promisc_edl;
	    
	    if (!gotedlpod++)
		if (assoc_alloc(ASSOC_EDL, assoc_promisc_edl, 
				 (struct ifnet *)0, (MSPCB *)0))
		    panic("assoc_init: could not init alloc\n");

	    printf("assoc_init: Using (ether) driver %s on pod %d\n",
		   ifp->if_name, pod);
	}
	if (
#ifdef __arm
	    0
#endif
#ifdef __mips
	    (ifp->if_type == IFT_OTHER)
	    && (ifp->if_name[0]=='y')
	    && (ifp->if_name[1]=='e')
	    && (ifp->if_name[2]=='s')
#endif
	    )
	{
	    if (pod > ASSOC_MAXPOD)
	    {
		printf("assoc_init: too many network interfaces!\n");
		return ;
	    }
	    ap = &assocPods[pod];
	    
	    ap->asp_valid = 1;
	    ap->asp_ifp = ifp;
	    ap->asp_tx_proc = ifp->if_output;
	    ap->asp_prepare = fdl_tx_prepare;
	    ap->asp_tx_promisc = assoc_tx_promisc_fdl;
	    
	    if (!gotfdlpod++)
		if (assoc_alloc(ASSOC_FDL, assoc_promisc_fdl, 
				 (struct ifnet *)0, (MSPCB *)0))
		    panic("assoc_init: could not init alloc\n");

	    printf("assoc_init: Using (fdl) driver %s on pod %d\n",
		   ifp->if_name, pod);
	}

    }
}
