;;; Add drule construct to GDE. 

;<drule declaration > ::= <drule-name> : <value> <pattern-category>+ .

;eg

;drule d1 : 100 [N +, V -, BAR 0, CONJ -] [N +, V, -, BAR 0, CONJ -].
;drule d2: 50 [TXTCAT, UNIT].

;drules (domain rules) are intended to be unordered lists of categories (that 
;match in the same way that lp rules match), and the idea is that should all 
;the pattern categories in a drule match categories in a  (learnt) fully 
;specified object rule, then that rule will suffer the numeric penalty.  The 
;higher the penalty, the worse the rule.  d1 will therefore match against 
;binary and ternary rules that involve conjunctless coordination; d2 
;will match against any rule using a text category.  Object rules are 
;matched against all drules, and the penalties are summed.  Need to ensure that
;when matching, all cats in a drule (a dterm) match separate cats in an object
;rule.

(defstruct drule name value dterms file comment)
(defvar *drules nil)  ;;; holds all drules defined.
(defvar *max-dvalue* 10) ;;; maximum value
(defvar *min-dvalue* -10) ;;; maximum value 

(defun parse-drule-declaration
   (input-list)
   (let
      ((*input-text input-list)
       (highest-binding-no 0)
       (*input-comments nil) (*current-item nil)
       (drule-name nil) (dterms nil) (dvalue nil))

      (setf *current-item (pop *input-text))
      (setf drule-name
         (parse-word "DRULE rule name"))
      (parse-literal '\: ": expected")

      (setf dvalue (string-to-integer 
		    (symbol-name (parse-word "DRULE value"))))

      (when (> dvalue *max-dvalue*)
	    (gde-ferror dvalue "is too high a value " "of a D rule"))

      (when (< dvalue *min-dvalue*)
	    (gde-ferror dvalue "is too low a value " "of a D rule"))

      (setf highest-binding-no 0)
      (loop
       (push (parse-category-binding t nil)
	     dterms)
       (cond
	((eq *current-item '\.) (return nil))))
      (list drule-name
         (make-drule :dterms  dterms :name drule-name :value dvalue
            :file
            (cond
               ((get drule-name 'drule)
                  (drule-file (get drule-name 'drule)))
               (t (get-new-construct-file)))
            :comment *input-comments))))


(defun string-to-integer (str)

;;; map str to an integer eg "10" = 10. 

  (let ((num 0) (neg nil))
    (dotimes (x (length str) (if neg (* -1 num) num))
	    (let ((digit (aref str x)))
	      (cond
	       ((and (= x 0) (eq digit '#\-)) (setf neg t))
	       ((eq digit '#\0) (setf num (* 10 num)))
	       ((eq digit '#\1) (setf num (+ 1 (* 10 num))))
	       ((eq digit '#\2) (setf num (+ 2 (* 10 num))))
	       ((eq digit '#\3) (setf num (+ 3 (* 10 num))))
	       ((eq digit '#\4) (setf num (+ 4 (* 10 num))))
	       ((eq digit '#\5) (setf num (+ 5 (* 10 num))))
	       ((eq digit '#\6) (setf num (+ 6 (* 10 num))))
	       ((eq digit '#\7) (setf num (+ 7 (* 10 num))))
	       ((eq digit '#\8) (setf num (+ 8 (* 10 num))))
	       ((eq digit '#\9) (setf num (+ 9 (* 10 num))))
	       (t 
		(gde-ferror digit 
			    " is not allowed as part of a " 
			    "D rule value")))))))

;;; The command handler called by top loop for inputting a DRULE
;;; rule declaration (the "Input Drule" command).

(defun input-drule nil
   (let
      ((input
          (prompt-if-necessary
             "Drule declaration? ")))
      (when input
         (insert-drule-declaration input))))


(defun insert-drule-declaration
   (input-list)
   (let
      ((parsed-rule
          (parse-drule-declaration input-list)))
      (when t         
         (output-defining-message "D rule"
            (car parsed-rule))
         (setf (get (car parsed-rule) 'drule)
            (cadr parsed-rule))
         (setf *drules
            (insert-declaration-at-end *drules
               (car parsed-rule)))
         (remprop (car parsed-rule)
            'normalised-drule))))


;;; add new command to gde command table

(push (make-command-entry :shortest 1 :name 'drule :action '(input-drule))
      *input-commands)

;;; add new command for drule editing.

(push (make-command-entry :shortest 2 :name
         'drule :action '(edit-drule)) *edit-commands)

;;; add new command for drule viewing

(push (make-command-entry :shortest 2 :name 'drule :action 
			  '(view-drule 'normalised))
      *view-commands)
      

;;; add command for deleting drules

(push (make-command-entry :shortest 2 :name 'drule :action 
			  '(delete-drule))
      *delete-commands)

;;; The command handler called by top loop for displaying a
;;; drule declaration (the "View drule" command).

(defun view-drule (format)
   (let
      ((pattern
          (prompt-if-necessary "D rule? ")))
      (when pattern
         (let
            ((drule-names
                (get-items *drules pattern)))
            (cond
               (drule-names
                  (dolist (name drule-names)
                     (progn
                        (terpri)
                        (cond
                           ((eq format 'aliased)
                              (print-drule-definition name
                                 (get name 'drule) nil))
                           (t
                              (print-drule-definition name
                                 (normalise-drule-definition
                                    name)
                                 nil))))))
               (t (gde-cerror "D rule not found")))))))

(defun edit-drule nil
   (let
      ((name
          (get-names-for-editing "D rule name? "
             *drules "D rules" nil)))
      (when name
         (let
            ((text
                (get-edited-definition
                   'print-drule-definition
                   (list name (get name 'drule) nil))))
            (when text
               (insert-drule-declaration text))))))


;;; Print the definition of a drule.

;;; need to change this.

(defun print-drule-definition
   (drule-name drule-definition identify)
   (pprint drule-definition))

(defun print-drule-definition
  (drule-name drule-definition identify)
    (let
      ((ppfree (gde-linelength))
         (pplevel 1) (pplowest 1) (ppindex 0)
         (dterms
            (drule-dterms drule-definition)))
      (gde-pp-start)
      (when identify (gde-pp-put-string "DRULE "))
      (gde-pp-put-text drule-name)
      (gde-pp-put-string " : ")
      (gde-pp-put-string (format nil "~A " (drule-value drule-definition)))
      (gde-pp-comment
         (drule-comment drule-definition))
      (pp-print-category
         (category-binding-category (caar dterms))
         (cdar dterms))
      (loop
         (setf dterms (cdr dterms))
         (cond
            ((null dterms) (return nil)))         
	 (gde-pp-put-string " ")
         (pp-print-category
            (category-binding-category (caar dterms))
            (cdar dterms)))
      (gde-pp-put-string ".")
      (gde-pp-put-marker 0)
      (gde-pp-flush)))


;;; The command handler called by top loop for deleting an
;;; drule declaration (the "Delete drule" command).

(defun delete-drule nil
   (let
      ((pattern
          (prompt-if-necessary "D rule? ")))
      (when pattern
         (let
            ((drule-names
                (get-items *drules pattern)))
            (cond
               (drule-names
                  (delete-drule-list drule-names))
               (t (gde-cerror "D rule not found")))))))


(defun delete-drule-list (drule-names)
   (dolist (drule drule-names)
      (progn
         (print-drule-definition drule
            (get drule 'drule) nil)
         (when
            (yes-for-question
               "Delete drule definition")
            (delete-drule-invalidations drule))
         (terpri))))


(defun delete-drule-invalidations (drule)
   (setf *drules
      (remove-list-1 drule *drules))
   (remprop drule 'perms)
   (remprop drule 'normalised-drule))

(defun delete-all-constructs nil
   (let
      ((pattern
          (prompt-if-necessary "Pattern? ")))
      (unless
         (nconc
            (delete-all-constructs1
               (get-items *features pattern)
               (get-items *sets pattern)
               (get-items *aliases pattern)
               (get-items *categories pattern)
               (get-items *extensions pattern)
               (get-items *top pattern)
               (get-items *id-rules pattern))
            (delete-all-constructs2
               (get-items *meta-rules pattern)
               (get-items *prop-rules pattern)
               (get-items *default-rules pattern)
               (get-items *lp-rules pattern)
	       (get-items *drules pattern)
               (get-items *ec-rules pattern)
               (get-items *multiply-rules pattern)
               (get-items *cc-rules pattern)
               (get-items (append *words *cached-words)
                  pattern)))
         (gde-cerror "no constructs found"))))

(defun delete-all-constructs2
   (metarules proprules defrules lprules drules
      ecrules multrules ccrules words)
   (nconc
      (when metarules
         (delete-metarule-list metarules) (ncons t))
      (when proprules
         (delete-proprule-list proprules) (ncons t))
      (when defrules
         (delete-defrule-list defrules) (ncons t))
      (when lprules (delete-lprule-list lprules)
         (ncons t))
      (when drules (delete-drule-list drules)
         (ncons t))
      (when ecrules (delete-ecr-list ecrules)
         (ncons t))
      (when multrules (delete-mr-list multrules)
         (ncons t))
      (when ccrules (delete-cc-list ccrules)
         (ncons t))
      (when words (delete-word-list words)
         (ncons t))))


;;; Clear grammar, asking for confirmation first.

(defun clear-whole-grammar nil
   (when
      (yes-for-question
         "Do you really want to clear the grammar")
      (d-markunload 'di) (d-markunload 'gr)
      (dolist (feature *features)
         (dolist
            (prop
               '(feature feature-order variable-list
                   variable-pointer proper-value-list
                   altered))
            (remprop feature prop)))
      (setf *features nil) (setf d-features nil)
      (setf d-variables nil)
      (dolist (set *sets)
         (progn
            (remprop set 'set)
            (remprop set 'altered)))
      (setf *sets nil)
      (dolist (alias *aliases)
         (dolist
            (prop '(alias normalised-alias altered))
            (remprop alias prop)))
      (setf *aliases nil)
      (setf *sorted-aliases nil)
      (setf d-aliases nil)
      (dolist (category *categories)
         (dolist
            (prop
               '(category normalised-category altered))
            (remprop category prop)))
      (setf *categories nil)
      (dolist (extension *extensions)
         (dolist
            (prop
               '(extension normalised-extension altered))
            (remprop extension prop)))
      (setf *extensions nil)
      (dolist (top *top)
         (dolist
            (prop
               '(top normalised-top altered))
            (remprop top prop)))
      (setf *top nil)
      (dolist (idrule *id-rules)
         (dolist
            (prop
               '(idrule normalised-idrules
                   expanded-idrules compiled-idrules altered))
            (remprop idrule prop)))
      (setf *id-rules nil)
      (dolist (lprule *lp-rules)
         (progn
            (remprop lprule 'lprule)
            (remprop lprule 'altered)))
      (setf *lp-rules nil)

      (dolist (drule *drules)
         (progn
            (remprop drule 'drule)
            (remprop drule 'perms)))
      (setf *drules nil)

      (dolist (metarule *meta-rules)
         (dolist
            (prop
               '(metarule normalised-metarule altered))
            (remprop metarule prop)))
      (setf *meta-rules nil)
      (clear-whole-grammar1)))



(defun forget-file2 (file)
   (dolist (metarule *meta-rules)
      (cond
         ((equal
             (meta-rule-file (get metarule 'metarule))
             file)
            (delete-metarule-invalidations metarule))))
   (dolist (defrule *default-rules)
      (cond
         ((equal
             (default-rule-file (get defrule 'defrule))
             file)
            (delete-defrule-invalidations defrule))))
   (dolist (proprule *prop-rules)
      (cond
         ((equal
             (prop-rule-file (get proprule 'proprule))
             file)
            (delete-proprule-invalidations proprule))))
   (dolist (lprule *lp-rules)
      (cond
         ((equal (lp-rule-file (get lprule 'lprule))
             file)
            (delete-lprule-invalidations lprule))))
   (dolist (drule *drules)
      (cond
         ((equal (drule-file (get drule 'drule))
             file)
            (delete-drule-invalidations drule))))
   (dolist (word *words)
      (cond
         ((equal
             (word-definition-file (get word 'word))
             file)
            (delete-word-invalidations word))))
   (dolist (ecr *ec-rules)
      (cond
         ((equal (ec-rule-file (get ecr 'ecr)) file)
            (delete-ecr-invalidations ecr))))
   (dolist (mr *multiply-rules)
      (cond
         ((equal (multiply-rule-file (get mr 'mr))
             file)
            (delete-mr-invalidations mr))))
   (dolist (cc *cc-rules)
      (cond
         ((equal (cc-rule-file (get cc 'cc)) file)
            (delete-cc-invalidations cc))))
   (forget-grammar-file file)
   (format t "File forgotten~%"))


;;; Normalise a D rule definition.

(defun normalise-drule-definition
   (drule-name)
   (let
      ((grammar-errors nil)
         (normalised-drule
            (get drule-name 'normalised-drule)))
      (cond
         (normalised-drule normalised-drule)
         (t
            (setf normalised-drule
               (let
                  ((structure-146
                      (copy-drule
                         (get drule-name 'drule))))
                  (setf (drule-dterms structure-146)
                     (mapcar
                        #'(lambda (dterm)
                             (normalise-category-bindings
                                dterm))
                        (drule-dterms structure-146)))
                  structure-146))
            (report-normalisation-errors "D rule"
               drule-name)
            (setf (get drule-name 'normalised-drule)
               normalised-drule) normalised-drule))))


(defun drule-match (idrule)

;;; return sum of all drule values that match object rule idrule.

   (let
      ((idrule-lhs
            (car (id-rule-binding-nos idrule)))
         (idrule-rhs
            (cdr (id-rule-binding-nos idrule)))
         (binding-list
            (id-rule-binding-list idrule))
         (name (id-rule-name idrule)))
      (let
         ((result               
	   (match-drules  idrule-rhs
			  binding-list *drules)))
	result)))
         

(defun match-drules
   (rhs-binding-nos binding-list
      drules)
;   (format t " Checking drules ")
   (let ((sum 0))
     (dolist (r drules sum)
	     ;;; no point checking if we know we are going to reject rule
	     (when (> (* 10 sum) *exception-threshold*)
		   (return-from match-drules sum))
	     (setf sum (+ sum 
			  (match-drule1 r 
					rhs-binding-nos binding-list))))))


(defun permutations (bag)

;;; generate all permutations of a list.  Code taken from Russel and Norvig.

  (if (null bag) '(())
    (mapcan #'(lambda (e)
		(mapcar #'(lambda (p) (cons e p))
			(permutations 
			 (remove e bag :count 1 :test #'eq))))
	    bag)))

(defun match-drule1 (drule rhs-binding-nos binding-list)

  (let* ((normalised (let ((res (get drule 'normalised-drule))
			   (res2 nil))
		      (if res res
			(progn
			  (normalise-drule-definition drule)

			  (dolist (c (drule-dterms 
				      (get drule 'normalised-drule)))
				  (push (car c) res2))

			  (setf (drule-dterms 
				 (get drule 'normalised-drule)) res2)
			  (get drule 'normalised-drule)))))
	(perms (let ((res (get drule 'perms)))
		 (if res res
		   (progn
		     (setf (get drule 'perms) (permutations 
					       (drule-dterms normalised)))
		     (get drule 'perms))))))

    (setf binding-list (cdr binding-list)) ;;; don't bother with mother

    (dolist 
     (p perms 0)	    	    
     (let* ((dterms p)
	    (dterms-copy dterms)
	    (object-cats binding-list)
	    (object-cats-copy object-cats))
       (loop	 
	  ;;; all dterm cats matched
	(when (null dterms) (return-from match-drule1 
					 (drule-value normalised)))
	
	  ;;; ran out of object cats, so try next permuation.
	  (when (null object-cats) (return nil))

	  (let* ((dcat (car dterms))
		 (rhs-cat (car object-cats))
		 (dcat-binding-no 
		  (category-binding-number dcat))
		 (rhs-binding-no (category-binding-number 
				   rhs-cat)))

	      (if (match-category dcat-binding-no dterms-copy
				  rhs-binding-no
				  object-cats-copy)
		  (progn
		    (pop dterms)
		    (pop object-cats))
		(pop object-cats))))))))
    
	
	      
;;; End of file






