;;
;;      Name: Morphological Analyser Common Lisp conpatability
;;         functions and macros
;;
;;      Author:  Alan W Black  November 1986
;;
;;      Copyright Graeme Ritchie, Alan Black,
;;                Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;               
;;   Description:
;;     This defines a number of macros and functions that allow
;;     the franz lisp version of the morphological analyser to be
;;     run (and compiled) in Common Lisp
;;
;;     >>> WARNING <<<
;;     Note that although this defines some Franz Routines it only
;;     does so for the cases reuqired and there is NO guarantee 
;;     that all general cases are coped with.
;;     >>> WARNING <<<
;;
;;     I would like to thank Jeff Dalton of AIAI, Edinburgh for advice
;;     about the finer points of Common Lisp.
;;

;;; Problems with shadowing catch and throw so had to edit source - JAC

(eval-when (compile load eval)
   (shadow
      '(include allocate concat copy neq ncons
          declare assoc assq delq eq 
          error gensym load member memq remove tyi zerop)))


(defmacro assoc (key alist)
;;  has to use equal test
   `(cl:assoc ,key ,alist :test #'equal))

(defmacro member (element set)
;;  has to use equal test
   `(cl:member ,element ,set :test #'equal))


(defvar D-LOADEDFILES nil)


(defmacro include (filename)
;;
;;  This is difficult becasue of the way kyoto common lisp compiles
;;  files.  Include includes things only the first time they are
;;  called, macros are not compiled.  They are evaluated during loading.
;;  If the file has been included before it is not included again.
;;
;;  note mafuncs should therefore be compiled first so that it gets the
;;  actaul load commands embedded in it
;;
   (setq filename
      (concatenate 'string *morph-source-dir* filename))
   (cond
      ((member filename D-LOADEDFILES)
	 nil)        ;; do nothing
      (t
	 (let ( (stream (open filename)) )
	    (fresh-line)
            (princ "Including: ") (princ filename) (terpri)
	    (setq D-LOADEDFILES (cons filename D-LOADEDFILES))
	    (cons
	       'progn
	       (D-ReadInAllS-Expressions
		  (read stream nil 'EOF)
		  stream)))
      )
   )
)


(defun D-ReadInAllS-Expressions (expr istream)
;;
;;  keeps reading s-expressions until eof is found
;;
   (let ((result nil))
      (loop
         (cond
            ((eql expr 'EOF)
               (close istream)
               (return (nreverse result)))
            ((and (listp expr) (eql (car expr) 'defmacro))
               (eval expr))    ;; eval macros at load time
            (t
               (push expr result)))
         (setf expr (read istream nil 'EOF)))))


(defmacro declare (&rest junk)
   ;;
   ;;  Shadows the Common Lisp declare and just ingnores
   ;;  the localf function declarations and changes the special
   ;;  declarations to proclaims
   ;;
   (let ((specials (assoc 'special junk)))
      (cond
         ((null specials) nil)
         (t
            `(eval-when (compile cl:load eval)
               (proclaim (quote ,specials)))))))


(defmacro add (&rest nums) `(+ ,@nums))

(defmacro add1 (num) `(+ 1 ,num))

(defmacro aexplodec (word)
;;
;;  This is what is used to split a string into a list of its
;;  characters
;;  It does not do the same thing as the franz thing as this returns
;;  a list of characters
;;
   `(mapcar
      #'(lambda (thing) (intern (string thing)))
      (coerce (string ,word) 'list))
)

(defmacro aexploden (word)
   `(mapcar
      #'(lambda (thing)	(intern (string thing)))
      (coerce (string ,word) 'list))
)

(defmacro allocate (&rest rest)
;; throw this declaration away at present
   `(progn nil)
)

(defmacro alphalessp (c1 c2)
   `(string< (string ,c1) (string ,c2)))

(defmacro assq (akey alist)
;;  simple one using eql
   `(cl:assoc ,akey ,alist))

(defun attach (newcar oldlist)
;;  returns a list with newcar as the car and oldlist as the cdr
;;  but it has the same cons cell as the oldlist
   (rplaca
      (rplacd oldlist (cons (car oldlist) (cdr oldlist)))
      newcar)
)

(defun concatl (things)
;;  
   (intern
      (apply
         #'concatenate 'string
         (mapcar
            #'(lambda (x) 
                 (cond
                    ((numberp x) (princ-to-string x))
                    (t (string x))))
            things))))


(defun concat (&rest rest)
   (concatl rest))

(defmacro copy (tree)
   `(copy-tree ,tree))

(defmacro delq (item list count)
   `(delete ,item ,list :test #'eql :count ,count))

(defmacro drain ()
   `(finish-output))

(defmacro eq (a b) `(eql ,a ,b))

(defmacro error (message)
;;   Makes the mssage a string then passes it on the CL error
;;   handler.
   `(cl:error (string ,message))
)
  
;;; *** May need changing***. Evaluate the form and return the result
;;; - do not allow any error to unwind the stack further. Return
;;; nil if an error occurs.

(defmacro errset (form)
   #+(and HP T) 
     `(extn:break-on-errors
         (eval (list 'quote ,form)))
   #+KCL 
      (let ((tag (cl:gensym)) (result (cl:gensym)))
         `(multiple-value-bind (,tag ,result)
             (si:error-set (list 'quote ,form))
             (if ,tag nil ,result)))
   #+(and ALLEGRO (not (or cltl2 x3j13)))
      (let ((no-error-p (cl:gensym)) (result (cl:gensym)))
         `(multiple-value-bind (,no-error-p ,result)
             (errorset ,form t)
             (if ,no-error-p ,result nil)))
   #+(and XEROX (not (or cltl2 x3j13)))
      (let ((result (cl:gensym)))
         `(let 
             ((,result
                 (il:errorset (list 'quote ,form))))
             (if ,result (car ,result))))
   #+(and PROCYON (not (or cltl2 x3j13)))
      (let ((result (cl:gensym)))
         `(let
             ((,result
                 (procyon:trap-exits (eval (list 'quote ,form)))))
             (if ,result (car ,result))))
   #+(and :CORAL (not (or cltl2 x3j13)))
      `(ccl:catch-error
          (ccl:catch-abort (ccl:catch-cancel (eval (list 'quote ,form)))))
   #+(or cltl2 x3j13)
      `(with-simple-restart
         (abort "Return to morphological analyser command loop.")
         (eval (list 'quote ,form)))
   #-(or (and HP T) KCL ALLEGRO XEROX PROCYON :CORAL (or cltl2 x3j13))
      `(eval (list 'quote ,form))
   )


;;;#-POPLOG
(defmacro filepos (&rest args)
   `(file-position ,@args))

;;; For pre-version 1 POPLOG in which file-position was not implemented
;;; #+POPLOG
;;; (defun filepos (stream &optional position)
;;;    (let
;;;       ((device
;;; 	    (funcall
;;; 	       (pop11::class_access
;;; 		     (if (input-stream-p stream) 3 4)
;;; 		     '#.(pop11::datakey *terminal-io*))
;;;             stream)))
;;;       (cond
;;;          (position
;;;             (pop11::sysseek device position 0)
;;;             t)
;;;          (t
;;;             (pop11::sysfilesize device)))))


(defmacro gensym (&optional prefix)
;;
   (cond
      ((null prefix) `(cl:gensym))
      (t `(cl:gensym (string ,prefix)))
   )
)

(defmacro implode (charlist)
;;  only defined for lists of characters NOTE numbers
   `(intern (concatenate 'string ,charlist)))

(defmacro infile (filename)
;;
;;  Franz infile function.  returns a file stream
;;
   `(open (string ,filename) :direction :input)
)

(defmacro lessp (&rest nums)
   `(< ,@nums)
)

;;; *** May need changing ***. Load the object file corresponding
;;; to the given filename. Need to add the default object file type
;;; if load in the implementation does not look for the object
;;; file by default.

(defmacro load (filename)
   `(cl:load
       (concatenate 'string (string ,filename)
          #+POPLOG ".lsp"
          #+XEROX ".dfasl"
          #+CMU ".sparcf")
       :verbose nil :print nil))


(defmacro memq (element set)
;;  simple one using eql
   `(cl:member ,element ,set))

(defmacro minus (num)
;;
   `(- 0 ,num))

(defmacro ncons (item)
   `(cons ,item nil))

(defmacro neq (a b)
   `(not (eql ,a ,b)))

(defmacro newsym (&optional prefix)
;;
   (cond
      ((null prefix) `(intern (string (cl:gensym))))
      (t `(intern (string (cl:gensym (string ,prefix)))))
   )
)

(defmacro outfile (filename)
;;
;;  Franz outfile function, open a file for writing
;;
   `(open (string ,filename) :direction :output
       :if-exists :supersede :if-does-not-exist :create))

(defmacro pp-form (arg)
;; pretty printer
   `(pprint ,arg))

(defun ptime ()
;; this hacky solution will do the trick for the way I use ptime
   (list
      (get-internal-real-time)
      0))

(defmacro plus (&rest args)
   `(+ ,@args))

(defmacro readc (&rest args)
   `(intern (string (read-char ,@args))))

(defmacro remove (element set)
;;  has to use equal test
   `(cl:remove ,element ,set :test #'equal))

(defmacro remq (element set)
;;  simple one using eql
   `(cl:remove ,element ,set :test #'eql))

(defmacro sstatus (&rest rest)
;; ignored
   `(progn nil)
)

(defmacro times (&rest args)
   `(* ,@args))

(defun tyi (&optional port)
;;  have to check if standard input
   (cond
      ((or (null port) (eql port 't))
	 (read-char *standard-input* nil 'EOF))
      (t (read-char port nil 'EOF)))
)

(defun tyipeek (&optional port)
;;  have to check if standard input
   (cond
      ((or (null port) (eql port 't))
	 (peek-char nil *standard-input* nil 'EOF))
      (t (peek-char nil port nil 'EOF)))
)

(defun zerop (thing)
;;   cl zerop doesn't allow symbols only numbers
   (and (numberp thing) (cl:zerop thing)))

;;
;;
;;      Title : maload
;;
;;      Function : This defines the bootstrap function for the morphological
;;                 analyser system
;;
;;      Author :  Alan W Black   28th Oct 1985
;;                Dept of A.I.   University of Edinburgh
;;
;;      Copyright Graeme Ritchie, Alan Black,
;;                Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;      Description :
;;          This function when called loads the rest of the analyser and
;;          dictionary access code from the given directory.  Also the
;;          system is initialised by calling the D-Initialise function.
;;
;;      Parameters :
;;        directory :  optional, the name of the directory where the
;;                     code is.
;;      Returns :
;;                    t  if code loaded, otherwise an error is signalled
;;      Side Effect :
;;          loads the functions for the dictionary and analyser system
;;          and initialises the global variables.
;;
;;      External references :
;;
;;      System functions used :
;;        Name     type      Args #  type                 Comment
;;
;;      Restrictions :
;;
;;

(defun d-maload  (&rest args)
;;
;;
;;  this function loads the basic analyser files from the given directory
;;  if directory is not given the current directory (or in the 
;;  status variable load-search-path).  This also initialises the 
;;  dictionary system.
;;
   (cond
      ((and args (atom (car args)))
	 (load (concat (car args) "makesp"))
;;; ***
	 (load (concat (car args) "smkwgram"))
	 (load (concat (car args) "makelex"))
;;; ***
	 (load (concat (car args) "smafuncs"))
	 (load (concat (car args) "analyse"))
	 (load (concat (car args) "autorun"))
	 (load (concat (car args) "mconcat"))
	 (load (concat (car args) "spdebug"))
	 (load (concat (car args) "debug"))
      )
      ((null args)              ;; no directory given
         (load "makesp")        ;; compile spelling rules
;;; ***
         (load "smkwgram")       ;; compile word grammar
         (load "makelex")       ;; compile dictionary
;;; ***
	 (load "smafuncs")       ;; some of the basic files
         (load "analyse")       ;; morpho-parser
         (load "autorun")       ;; morphographemic-analyser
	 (load "mconcat")       ;; surface form generator
	 (load "spdebug")       ;; spelling rule debugger
	 (load "debug")         ;; debugger for parses   
      )
      (t
         (error "directory not specified as atom")
      )
   )
   (D-Initialise)    ;; initialise the system
)

(defun d-maloadcomp (&rest args)
;;
;;   Loads only those files necessary for the compilation of user files
;;   and not those used in the analyser
;;
   (cond
      ((and args (atom (car args)))
	 (load (concat (car args) "makesp"))
;;; ***
	 (load (concat (car args) "smkwgram"))
	 (load (concat (car args) "makelex"))
;;; ***
	 (load (concat (car args) "smafuncs"))
      )
      ((null args)         ;; no directory given
         (load "makesp")   ;; compile spelling rules
;;; ***
         (load "smkwgram")     ;; compile word grammar
         (load "makelex")   ;; compile dictionary
;;; ***
	 (load "smafuncs")       ;; some of the basic files
      )
      (t
         (error "directory not specified as atom")
      )
   )
   (D-Initialise)    ;; initialise the system
)


(defun d-maloadmap (&rest args)
;;
;;   Loads only those files necessary for the loading of compiled 
;;   user files and the analyser.
;;
   (cond
      ((and args (atom (car args)))
;;; ***
	 (load (concat (car args) "smafuncs"))
	 (load (concat (car args) "analyse"))
	 (load (concat (car args) "autorun"))
	 (load (concat (car args) "mconcat"))
      )
      ((null args)         ;; no directory given
;;; ***
	 (load "smafuncs")       ;; some of the basic files
         (load "analyse")       ;; morpho-parser
         (load "autorun")       ;; morphographemic-analyser
	 (load "mconcat")       ;; surface form generator
      )
      (t
         (error "directory not specified as atom")
      )
   )
   (D-Initialise)    ;; initialise the system
)


