;;
;;
;;      Title : Analyse
;;
;;      Function : does morphological analysis on a given word
;;                 with repsect to a currently loaded lexicon,
;;                 word and spelling rules.
;;
;;      Author : Alan W Black  3rd April 1986
;;     
;;      Copyright : Graeme Ritchie, Alan Black,
;;                  Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;      Description :
;;          this is basic analysis, it calls the chart parser to
;;          get the analyses then processes the output from the 
;;          chart depending on the lookup flag. (in D-LOOKUPFORMAT)
;;
;;      Parameters :
;;      Returns :
;;      Side Effect :
;;
;;      External references :
;;
;;      System functions used :
;;        Name     type      Args #  type                 Comment
;;
;;      Restrictions :
;;         If the returned category (or word structure) involves variables
;;         it is possible that some are uninstantiated.  The variables are
;;         typed and are gensym'ed atoms so it will probably not be possible
;;         for user to know what the range of an uninstantiated variable
;;         is.  All I can say is this is really the users responsible - sorry
;;
;;

(declare
   (special
      D-LCATEGORIES
      D-MORPHOLOGYONLY
      D-LCATEGORIES
      D-ALLVERTICES
   )
   (localf
      D-GenerateEdgeWordStructure     
      D-GenerateEdgeCat               
      D-CheckMORPHONLYFeats
      D-ApplyLCategoryTree
      D-ApplyLCategories
      D-AddandCheck
      D-NameVertices
      D-RemoveDuplicateEdges          
      D-FindAllParsePaths             
      D-FindWordEdges                 
      D-FindFullParses                
      D-HasEndProperty                
   )
)

(include "macros")
(include "subrout")
(include "catrouts")
(include "parser")


(defun D-GetCategoryForm (word)
;;
;;   simplest form of lookup which returns only the 
;;   to symtactic category of all edges spanning the chart 
;;   that are extensions of D-DISTINGUISHEDCATEGORY
;;
;;   removes any MorphologyOnly features from the returned categories
;;   if any are declared
;;
   (let ( (categories 
	       (mapcan
		  #'(lambda (edge)
		      (ncons
			 (D-ApplyLCategories
			    (D-ApplySubstitutions
			       (D-getedgeLABEL edge)
			       (ncons (D-getedgeBIND edge)))
                            D-LCATEGORIES))
		    )
		  (D-FindFullParses
		     (D-getvertexEDGEOUTC (D-Parse word)))
	       )) )
      (D-CheckMORPHONLYFeats categories)
   )
)

(defun D-CheckMORPHONLYFeats (categories)
;;
;;   Removes any features declated as D-MORPHOLOGYONLY from the
;;   given list of categories
;;
   (cond
      ((null D-MORPHOLOGYONLY) categories)  ;; no features to remove
      (t
	 (mapcar
	    #'(lambda (category)
	       (mapcan 
		  #'(lambda (feature)
		     (cond
			((memq (car feature) D-MORPHOLOGYONLY)
			   nil
			)
			(t          ;; not to be removed
			   (ncons feature))))
		  category))
	    categories)
      )
   )
)

(defun D-ApplyLCategoryTree (wordtree)
;;
;;   Applies the LCategory definitions to the root category of the 
;;   given wordstructure.  LCategories consist of a category and a list
;;   of feature names.  If the root category of a structure is an extension
;;   of an LCategory condition then each feature in the LCategory list
;;   is added with the value @(gensym 'D) unless that feature is already
;;   in the category.
;;
   (cons
      (D-ApplyLCategories (car wordtree) D-LCATEGORIES)
      (cdr wordtree)
   )
)

(defun D-ApplyLCategories (category lcats)
;;
;;   applies the lcats to the given category, returns the 
;;   new category
;;
;;   for feature lcategories if the feature has a variable value than simply
;;   leave it as a variable.  Maybe this shouyld flag an error ?
;;
;;
   (cond
      ((null lcats) category)
      ((eq (length (car lcats)) 3)   ;; a feature lcategory
	 (cond
	    ((and (D-HasFeature (caar lcats) category)
		  (D-VariableP (D-GetFeatureValue (caar lcats) category)))
              ;; don't change the value just leave it as a variable
	       (D-ApplyLCategories
	          category (cdr lcats)))
	    ((and (D-HasFeature (caar lcats) category)
		  (D-IsExtensionOf (cadar lcats) '((t t)) 
			      (D-GetFeatureValue (caar lcats) category)))
               (D-ApplyLCategories
		  (cons   ;; check value of category-valued feature
		     (list (caar lcats)
			(D-AddandCheck 
			   (D-GetFeatureValue (caar lcats) category)
			   (caddr (car lcats))))
		     (D-RemoveFeature (caar lcats) category))
                  (cdr lcats))
            )
	    (t   ;; no change required
	       (D-ApplyLCategories
	          category (cdr lcats)))))
      ((D-IsExtensionOf (caar lcats) '((t t)) category)
	 (D-ApplyLCategories
	    (D-AddandCheck category (cadar lcats))
	    (cdr lcats))
      )
      (t      ;; current lcats doesn't apply
         (D-ApplyLCategories
	    category (cdr lcats)))
   )
)

(defun D-AddandCheck (category featlist)
;;
;;   adds each feature in featlist to category is not already there
;;   new feat value is gensymed starting with @
;;
   (cond
      ((null featlist)
	 category    ;; finished
      )
      ((D-HasFeature (car featlist) category)
	 (D-AddandCheck category (cdr featlist))
      )
      (t
	 (D-AddandCheck
	    (cons
	       (list (car featlist) (D-BlankVariable))
	       category) (cdr featlist))
      )
   )
)
 
(defun D-GetWordStructure (word)
;;
;;  returns a list of words structures for all edges spanning
;;  the chart and have a lebel that is an extension of 
;;  D-DISTINGUISHEDCATEGORY
;;
   (mapcar
      #'(lambda (edge)
	 (D-ApplyLCategoryTree
	    (D-FindParseTree  ;; make the spanning edges into word trees 
	       (D-getedgeLABEL edge)
	       edge
	       (ncons (D-getedgeBIND edge)) ;; bindings to apply
	    )))
      (D-FindFullParses    ;; only check complete edges that start from
         (D-getvertexEDGEOUTC (D-Parse word))  ;; from the initial vertex
      )
   )
)

(defun D-StringSegmentCat (word)
;;
;;  This takes the given word (or better thought of as character
;;  string).  And find all DISTINGUISHEDCATEGORY * segmentations
;;  of it.  Labels of returned edges are CATEGORIES
;;
   (let ( (initvertex (D-Parse word)) )
      (cons
	 (list (D-getvertexNAME initvertex)
	       (D-getvertexNAME D-ENDVERTEX))
         (mapcar
            #'D-GenerateEdgeCat
            (D-RemoveDuplicateEdges
	       (D-FindAllParsePaths initvertex))
         )
      )
   )
)

(defun D-StringSegmentWStructure (word)
;;
;;  This takes the given word (or better thought of as character
;;  string).  And find all DISTINGUISHEDCATEGORY * segmentations
;;  of it.  Labels of returned edges are Word Structure
;;
   (let ( (initvertex (D-Parse word)) )
      (D-NameVertices D-ALLVERTICES)
      (cons
	 (list (D-getvertexNAME initvertex)
	       (D-getvertexNAME D-ENDVERTEX))
         (mapcar
            #'D-GenerateEdgeWordStructure
            (D-RemoveDuplicateEdges
	       (D-FindAllParsePaths initvertex))
         )
      )
   )
)

(defun D-NameVertices (vertices)
;;
;;  Names all the vertices
;;
   (mapc
      #'(lambda (vertex)
	 (D-putvertexNAME vertex (gensym 'D)))
      vertices)
)

(defun D-AnalyseWordWithDebug (word)
;;
;;  This takes the given word and anlyses it is a way similar to 
;;  when the format is set to D-STRINGSEGMENTWS.  Returns the 
;;  simplified chart and initvertex as returned from the actual
;;  parse.
;;
   (let ( (initvertex (D-Parse word)) )
      (list
	 initvertex
	 (list (D-getvertexNAME initvertex)
	       (D-getvertexNAME D-ENDVERTEX))
         (mapcar
            #'D-GenerateEdgeWordStructure
            (D-RemoveDuplicateEdges
	       (D-FindAllParsePaths initvertex))
         )
      )
   )
)

(defun D-GenerateEdgeWordStructure (edge)
;;
;;  returns the edge in a normal form to be returned
;;  to the caller outside the system. The form is
;;     (STARTVERTEX ENDVERTEX LABEL)
;;
;;  where STARTVERTEX and ENDVERTEX are newsym'ed atomic names and
;;  LABEL is a word tree 
;;
   (list
      (D-getvertexNAME (D-getedgeSTART edge))
      (D-getvertexNAME (D-getedgeEND edge))
      (D-ApplyLCategoryTree
	 (D-FindParseTree     ;; make the spanning edges into word trees 
	   (D-getedgeLABEL edge)
	   edge
	   (ncons (D-getedgeBIND edge)) ;; bindings to apply
	 )
      )
   )
)

(defun D-GenerateEdgeCat (edge)
;;
;;  returns the edge in a normal form to be returned
;;  to the caller outside the system. The form is
;;     (STARTVERTEX ENDVERTEX LABEL)
;;
;;  where STARTVERTEX and ENDVERTEX are newsym'ed atomic names and
;;  LABEL is a syntactic category of a whole word tree depending
;;  on the value of D-LOOKUPFORMAT.
;;
   (list
      (D-getvertexNAME (D-getedgeSTART edge))
      (D-getvertexNAME (D-getedgeEND edge))
      (car
	 (D-CheckMORPHONLYFeats
	    (ncons
	       (D-ApplyLCategories
	          (D-ApplySubstitutions
	             (D-getedgeLABEL edge)
	             (ncons (D-getedgeBIND edge)))
                  D-LCATEGORIES))))
   )
)

(defun D-RemoveDuplicateEdges (edges)
;;
;;  this is a simple remove duplicates using eq to make it faster
;;  (and possible) rather than using the D-RemoveDuplicates in 
;;  subrout - this is also destructive
;;
;;  This is inefficient you should be checking membership in the 
;;  setified list not the unsetified one.
;;
   (cond
      ((null edges)
	 nil
      )
      ((memq (car edges) (cdr edges)) ;; is a duplicate so drop it
	 (D-RemoveDuplicateEdges (cdr edges))
      )
      (t    
	 (rplacd edges (D-RemoveDuplicateEdges (cdr edges)))
      )
   )
)
	 
(defun D-FindAllParsePaths (vertex)
;;
;;   Returns a list of edges that go from given vertex to 
;;   end of chart (and are each extensions of the DISTINGUISHEDCATEGORY)
;;   or nil if there are non
;;
   (mapcan
      #'(lambda (edge)
         (cond
            ((eq 'END (D-getvertexSTATUS (D-getedgeEND edge)))
               (ncons edge))
            (t
	       (let ( (subedges (D-FindAllParsePaths (D-getedgeEND edge))) )
		  (cond
		     ((null subedges) nil)  ;; failed
		     (t
			(cons edge subedges)))))
         ))
      (D-FindWordEdges vertex)
   )
)

(defun D-FindWordEdges (vertex)
;;
;;  returns a list of all edges that have labels that are extensions
;;  of the DISTINGUISHEDCATEGORY
;;
   (mapcan
      #'D-CheckWordProp
      (D-getvertexEDGEOUTC vertex)
   )
)

(defun D-FindFullParses (edges)
;;
;;  This checks the given list of edges and returns any that END at
;;  a vertex marked with END. (i.e. has a valid final configuration)
;;
   (mapcan
      #'D-HasEndProperty
      edges
   )
)

(defun D-HasEndProperty (edge)
;;
;;  returns the edge if it ends at a vertex which has 'END as its status
;;  and its label is an extension of the category declared as top
;;  (held in D-DISTINGUISHEDCATEGORY). Note the edge is returned is
;;  a list.
;;  otherwise it returns nil
;;
   (cond
      ((eq 'END (D-getvertexSTATUS (D-getedgeEND edge)))
	 (D-CheckWordProp edge)  ;; is it a valid parse
      )
      (t
	 nil         ;; not a valid parse
      )
   )
)

