/* outcomp.c
   (Replaces an earlier program of the same name.)

  Process the output from a parsing run, to give an approximate evaluation.

  Data structure: have an array with one slot for each bracket read. Label
  each bracket as either T (output from tagger) or O (original), and link
  each bracket to its corresponding close bracket. Also have walls, as
  described below.

  Building the structure (have buffers for T tags and O tags).
   read a line
   if it comes from the original:
       if it contains any brackets:
           find brackets in this line to O set
       if T set or O set is non empty
         add brackets to array from T set then O set
         clear T set and O set
   else
     find brackets in this line, appending to T set

   When adding brackets to array, if at the end of it, there are no open
   brackets waiting to be closed, then we analyse the array, as follows.
   First, for a call with either an empty T set or an empty O set, we add a
   wall; for adding the arrays, we first all all closing brackets from O, then
   all closing ones from T, then all opening ones from T, then all opening ones
   from O. Raise an error if we find a closing one when processing opening
   ones.
   Then: starting from LH end of the array
    take outermost bracket
    if a close bracket, flag an error in the code!
    if a T bracket, scan forward to next O bracket, not crossing walls
      if the O bracket is found and matches
	find the closing T bracket
        scan backwards for the nearest closing O bracket, not crossing walls
      if both now match
        delete the T brackets and the O brackets
        record a success
      else
        delete the T bracket and the corresponding close one
        record it as unmatched
    else (O bracket)
      delete the O bracket and the corresponding close one
      record it as unmatched
    else (wall)
      skip
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Line buffer length */
#define MaxLine (200)

/* Array for set of T brackets and O brackets, and their sizes */
#define MaxTag  (10)
#define MaxTags (300)
static char t_set[MaxTags][MaxTag];
static char o_set[MaxTags][MaxTag];
static int  t_set_max = 0;
static int  o_set_max = 0;

/* Structure for analysing brackets */
typedef enum {T, O, wall, null} b_type;
typedef struct
{
    b_type type;
    char   tag[MaxTag];
    int    link;		/* Other end of the bracket */
} bracket_st;
static bracket_st bracket[MaxTags];
static int max_bracket = 0;

static int missed = 0;
static int added  = 0;
static int exact  = 0;

static int fline = 0;

static void output_results(FILE *out)
{
    int total_old = exact + missed;
    int total_new = exact + added;

    fprintf(out,
	"%d exact matches (%5.2f%%)\n", exact, (100.0 * exact) / total_old);
    fprintf(out,
	"%d extra brackets (%5.2f%%)\n", added, (100.0 * added) / total_new);
    fprintf(out,
	"%d missed brackets (%5.2f%%)\n", missed, (100.0 * missed) /total_old);
}

static int bracket_match(char *open, char *close)
{
    int match;
    int len = strlen(close)-1;
    close[len] = 0;
    match = (strcmp(open+1, close) == 0);
    close[len] = ']';
    return match;
}

static void add_member(char *member, b_type type)
{
    if (max_bracket >= MaxTags)
    {
	fprintf(stderr, "Array overflow (MaxTags), line %d\n", fline);
	exit(1);
    }
    bracket[max_bracket].type = type;
    strcpy(bracket[max_bracket].tag, member);
    bracket[max_bracket].link = -1;

    /* If the bracket was a closing one, scan back for a link */
    if (member[0] != '[')
    {
	int j;

	for (j = max_bracket-1 ; j >= 0 ; j--)
	{
	    if (bracket[j].type == type && bracket[j].link == -1 &&
		bracket_match(bracket[j].tag, member))
	    {
		bracket[j].link = max_bracket;
		bracket[max_bracket].link = j;
		break;
	    }
	}

	if (j < 0)
	{
	    fprintf(stderr, "Unmatched bracket '%s', line %d\n",
			member, fline);

	    for (j = 0 ; j <= max_bracket ; j++)
	    {
		if (bracket[j].type == T)
		    fprintf(stderr, "T %s\n", bracket[j].tag);
		if (bracket[j].type == O)
		    fprintf(stderr, "O %s\n", bracket[j].tag);
	    }
	    exit(1);
	}
    }

    max_bracket += 1;
}

static void add_set(void)
{
    int o_index = 0, t_index = 0;
    int state = 0;
	/* 0: add closing from O, 1: add closing from T, 2: add opening from
	   T, 3: add opening from O */

    if (o_set_max == 0 || t_set_max == 0)
	bracket[max_bracket++].type = wall;

    do
    {
	switch (state)
	{
	    case 0:
		if (o_index < o_set_max && o_set[o_index][0] != '[')
		    add_member(o_set[o_index++], O);
		else
		    state = 1;
		break;
	    case 1:
		if (t_index < t_set_max && t_set[t_index][0] != '[')
		    add_member(t_set[t_index++], T);
		else
		    state = 2;
		break;
	    case 2:
		if (t_index < t_set_max)
		{
		    if (t_set[t_index][0] != '[')
		    {
			fprintf(stderr,
				"Unexpected close bracket %s, line %d\n",
				t_set[t_index], fline);
			exit(1);
		    }
		    add_member(t_set[t_index++], T);
		}
		else
		    state = 3;
		break;
	    case 3:
		if (o_index < o_set_max)
		{
		    if (o_set[o_index][0] != '[')
		    {
			fprintf(stderr,
				"Unexpected close bracket %s, line %d\n",
				o_set[o_index], fline);
			exit(1);
		    }
		    add_member(o_set[o_index++], O);
		}
		else
		    state = 4;
		break;
	}
    } while (state != 4);
}

static void add_brackets(FILE *out)
{
    int i;

    /* Add the sets to the brackets array */
    add_set();

    /* Scan the array to see if there are any open brackets */
    for (i = 0 ; i < max_bracket ; i++)
    {
	if (bracket[i].type != wall && bracket[i].type != null
	    && bracket[i].link == -1)
	{
	    return;
	}
    }

    /* Now process the array */
    for (i = 0 ; i < max_bracket ; i++)
    {
	switch (bracket[i].type)
	{
	    case T:
	    {
		int j;
		int matched = 0;

		if (bracket[i].tag[0] != '[')
		{
		    fprintf(stderr, "Consistency fail!\n");
		    exit(1);
		}

		/* Scan forward to next O bracket */
		for (j = i+1 ; j < max_bracket ; j++)
		{
		    if (bracket[j].type == wall) break;
		    if (bracket[j].type == O) break;
		}

		if (bracket[j].type == O &&
		    strcmp(bracket[j].tag, bracket[i].tag) == 0)
		{
		    int k, link = bracket[j].link;

		    /* Scan backward for closing O from closing T */
		    for (k = bracket[i].link-1 ; k >= link ; k--)
		    {
			if (bracket[k].type == O) break;
		    }

		    if (k == link)
		    {
			exact += 1;

			/* Erase O bracket */
			bracket[j].type = null;
			bracket[k].type = null;

			matched = 1;
		    }
		}

		if (matched == 0)
		    added += 1;

		/* Erase T bracket */
		bracket[i].type = null;
		bracket[bracket[i].link].type = null;

		break;
	    }
	    case O:
		if (bracket[i].tag[0] != '[')
		{
		    fprintf(stderr, "Consistency fail!\n");
		    exit(1);
		}

		/* Delete the bracket and its corresponding one */
		bracket[i].type = null;
		bracket[bracket[i].link].type = null;
		missed += 1;
		break;
	    case wall:
	    case null:
		break;
	}
    }

    max_bracket = 0;
}

static void find_brackets(char *buffer, char set[MaxTags][MaxTag], int *max)
{
    char *b = buffer;
    char *o = set[*max];

    while (*b != '\n' && *b != 0)
    {
	if (*b == '[')
	{
	    if (*(++b) != 0 && strchr(" \n[", *b) == NULL)
	    {
		*o++ = '[';
		do *o++ = *b++; while (*b != 0 && strchr(" \n[", *b) == NULL);
		*o = 0;
		if (set[*max][1] != 0)
		{
		    *max += 1;
		    o = set[*max];
		}
	    }
	}
	else if (*b == ']')
	{
	    char *b1 = b;
	    if (--b >= buffer && strchr(" ]", *b) == NULL)
	    {
		while (--b >= buffer && strchr(" ]", *b) == NULL) ;
		b += 1;
		while (b != b1) *o++ = *b++;
		*o++ = ']';
		*o = 0;
		if (set[*max][1] != 0)
		{
		    *max += 1;
		    o = set[*max];
		}
	    }
	    b = b1+1;
	}
	else
	    b += 1;
    }
}

static void process_line(char *buffer, FILE *out)
{
    int wordline = (buffer[0] == ' ' || buffer[0] == 'a');

    if (wordline)
    {
	if (strpbrk(buffer, "[]") == NULL)
	    o_set_max = 0; /* Clear O set */
	else /* Look for tags */
	    find_brackets(buffer, o_set, &o_set_max);

	/* Do the output */
	if (o_set_max != 0 || t_set_max != 0)
	{
	    add_brackets(out);
	    o_set_max = t_set_max = 0;
	}
    }
    else /* Accumulate brackets */
    {
	if (strpbrk(buffer, "[]") != NULL)
	    find_brackets(buffer, t_set, &t_set_max);
    }
}

int main(void)
{
    char buffer[MaxLine];

    while (fgets(buffer, MaxLine, stdin) != NULL)
    {
	fline += 1;
	process_line(buffer, stdout);
    }
    fputc('\n', stdout);
    output_results(stdout);
    return 0;
}
