/* Statistical labeller - finite state machines.

   25-04-93	Created
   19-05-93	Allow phrases in FSM match lists

   Copyright (C) David Elworthy 1995

   Contains the simplified finite state machine package, replacing the earlier
   (and more complex) version.

   Principal external functions:
	fsm_read, fsm_read_named
	fsm_advance
	fsm_dump_state
	fsm_init_node, fsm_free_node
	fsm_may_tag

   The syntax of FSMs is rather ugly at present, and is poorly checked.
    Brief syntax and semantics:
    <fsm>    ::= <name> <state>* 'end'
    <state>  ::= <id> <item>* ';'
    <item>   ::= [<tags>] '_' <action> 
    <action> ::= <id> | <tagSc> | 'back' <tagSc>
    <tags>   ::= <tag> | <tag> <tags>
    <tagSc>  ::= <tag> | <tag>_<score>

    Spaces, tabs and newlines separate any objects. FSMs must end with 'end'.
    The list of items in a state must end with ";". To include ";" as a tag,
    escape it as "\;". There must be a space before the ";" and the "_" in an
    <item>. FSM names and state ids are any text which is not a tag. In items,
    '_' acts as a separator. The actions are: 
    <id>	jump to this item
    <tagSc>	create a phrase with this tag, and run the tagger on it. The
		score defaults to 1.0 and is used as the base score of the
		phrase. 
    back <tag>	as <tag>, but without this final match.
    All of the tags that match are created as hypotheses within a phrase. An
    empty tag list matches everything not specified in another tag list on the
    item.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "common.h"
#include "list.h"
#include "map.h"

#include "label.h"
#include "stack.h"
#include "phrase.h"
#include "fsm.h"

/*
==============================================================================
FSM reading
*/

/* Main data structure for FSMs. The fsms are held in a linked list, each
entry containing the name of the machine and a pointer to its initial state.
The state definitions consist of a list of items, each being one of the
possible things that can happen at the state. Each item is a list of the tags
that may match, and what to do.
*/

typedef struct fsm_st    FSMSt, *FSM;
typedef struct state_st  StateSt, *State;
typedef struct item_st   ItemSt, *Item;

struct fsm_st
{
    FSM   next;		/* Next FSM definition */
    uchar *name;	/* Name of this FSM */
    State start;	/* Initial state */
};

struct state_st
{
    State next;		/* Next state definition for this FSM */
    uchar *id;		/* ID of this state */
    FSM   fsm;		/* Link back to FSM (for tracing) */
    Item  item;		/* List of items at this state */
};

typedef enum {JumpAct, TagAct, BackAct} Action;

struct item_st
{
    Item   next;	/* Next item for this state */
    Tags   tags;	/* Tags which may match */
    Action action;	/* Action to take */
    union
    {
	struct
	{
	    Tag    tag;	/* Argument for Tag and Back actions */
	    Score  score;
	} t;
	struct
	{
	     uchar *id;	/* Argument for Jump action */
	     State  to;	/* Argument for Jump after resolving */
	} j;
    } u;
};

/* List of known FSMs */
static FSM fsm_head = NULL;

/* Input buffer and line count */
#define MaxLine (200)
static uchar buffer[MaxLine];
static int   line_count = 0;

/*-----------------------------------------------------------------------------
    get_token

    Get a token, reading a line if necessary. Returns NULL at eof.
-----------------------------------------------------------------------------*/

static uchar *get_token(FILE *in)
{
    uchar *token;
    uchar *term = " \t\n";

    do
    {
	/* Get a line if necessary */
	if (buffer[0] == 0)
	{
            if (feof(in) || fgets(buffer, MaxLine, in) == NULL)
		return NULL;
	    else
	    {
		int len = strlen(buffer);
		line_count += 1;

		if (len != 0 && len != 1)	/* Empty line if not */
		{
		    if (len >= MaxLine-1)
		    {
			fprintf(stderr, "Line overflows buffer:\n%s\n",
				buffer);
			get_out();
		    }
		}

		/* Get first token */
		token = strtok(buffer, term);
	    }
	}
	else /* Get next token */
	{
	    token = strtok(NULL, term);
	    if (token == NULL) buffer[0] = 0;
	}
    } while (token == NULL);

    return token;
}

/*-----------------------------------------------------------------------------
    fsm_error

    Report error when reading FSMs.
-----------------------------------------------------------------------------*/

static void fsm_error(char *text)
{
    fprintf(stderr, "%s at line %d\n", text, line_count);
}

/*-----------------------------------------------------------------------------
    fsm_error_token

    Report an error and what token caused it when reading FSMs.
-----------------------------------------------------------------------------*/

static void fsm_error_token(char *text, uchar *token)
{
    fprintf(stderr, "%s at line %d ('%s')\n", text, line_count, token);
}

/*-----------------------------------------------------------------------------
    check_token

    Return FALSE for a NULL token, reporting an error if not allowed; also
    check whether token is a tag and report an error (and return FALSE) if it
    not allowed to be.
-----------------------------------------------------------------------------*/

static BOOL check_token(uchar *token, BOOL null_allowed, BOOL tag_allowed)
{
    if (token == NULL)
    {
	if (!null_allowed)
	    fsm_error("Unexpected end of file");
	return FALSE;
    }
    else if (!tag_allowed && map_tag_quiet(token) != NOTAG)
    {
	fsm_error_token("Token coincides with a tag", token);
	return FALSE;
    }
    else
	return TRUE;
}

/*-----------------------------------------------------------------------------
    find_fsm

    Search for an FSM and report an error if it already exists; otherwise
    create it.
-----------------------------------------------------------------------------*/

static int fsm_cmp(Key *c1, Key *c2)
{
    uchar **s1 = (uchar **)c1;
    uchar **s2 = (uchar **)c2;
    return (strcmp(*s1, *s2));
}

static FSM find_fsm(uchar *name, FSM *head)
{
    FSM  fsm;
    BOOL match;

    fsm = list_search_and_add((void **)head, (Key *)&name, fsm_cmp,
				sizeof(FSMSt), "FSM", &match);

    if (match)
    {
	fsm_error_token("Duplicate FSM name", name);
	return NULL;
    }
    else
    {
	fsm->name  = allocate_string(name, "FSM name");
	fsm->start = NULL;

	return fsm;
    }
}

/*-----------------------------------------------------------------------------
    create_state
-----------------------------------------------------------------------------*/

static State create_state(uchar *id, FSM f)
{
    State new;
    Allocate(new, sizeof(StateSt), "state");
    new->next = NULL;
    new->id   = allocate_string(id, "State ID");
    new->fsm  = f;
    new->item = NULL;

    return new;
}

/*-----------------------------------------------------------------------------
    create_item
-----------------------------------------------------------------------------*/

static Item create_item(void)
{
    Item new;

    Allocate(new, sizeof(ItemSt), "item");
    new->next = NULL;
    new->tags = NULL;
    return new;
}

/*-----------------------------------------------------------------------------
    read_tags_list

    Read a list of tags, given the first token in it, stopping on a non-tag or
    on '_'. TRUE if no error.
-----------------------------------------------------------------------------*/

static BOOL read_tags_list(uchar *token, Item item, FILE *in)
{
    while (strcmp(token, "_") != 0)
    {
	Tag  tag;

	/* Convert the token to a tag */
	if ((tag = map_tag_quiet(token)) == NOTAG)
	{
	    fsm_error_token("Unknown tag", token);
	    return FALSE;
	}

	/* Add to sorted Tags list */
	item->tags = add_tag_to_list(item->tags, tag);

	/* Get the next token */
	if (!check_token(token = get_token(in), FALSE, TRUE))
	    return FALSE;
    }

    return TRUE;
}

/*-----------------------------------------------------------------------------
    set_action_arg

    Decode the token into either a tag or a tag and a score. Return TRUE if
    OK, FALSE (but without error unless flag says so) otherwise.
-----------------------------------------------------------------------------*/

static BOOL set_action_arg(uchar *token, Item item, BOOL tag_required)
{
    Tag   tag;
    uchar *sep;

    /* Token may be a tag, tag_score or non-tag */
    if ((tag = map_tag_quiet(token)) != NOTAG)
    {
	/* Simple tag */
	item->u.t.tag   = tag;
	item->u.t.score = 1.0;
	return TRUE;
    }
    else if ((sep = strchr(token, '_')) != NULL)
    {
	uchar s = *sep;

	/* Blank out separator */
	*sep = 0;

	if ((tag = map_tag_quiet(token)) == NOTAG)
	{
	    if (tag_required)
	    {
		*sep = s;
		fsm_error_token("Tag required", token);
		return FALSE;
	    }
	    return FALSE;
	}
	else /* Set tag and score */
	{
	    item->u.t.tag = tag;
	    *sep = s;
	    if (sscanf(sep+1, score_format, &(item->u.t.score)) != 1)
	    {
		fsm_error_token("Bad score", token);
		return FALSE;
	    }
	    else
	    {
		return TRUE;
	    }
	}
    }
    else if (tag_required)
    {
	fsm_error_token("Tag required", token);
    }

    return FALSE;
}

/*-----------------------------------------------------------------------------
    set_item_action

    Set up the action arguments in an item; this is for the case where no
    keyword is specified, i.e. a Jump or Tag action.
-----------------------------------------------------------------------------*/

static void set_item_action(uchar *token, Item item)
{
    /* If token is a tag, then this is a Tag action */
    if (set_action_arg(token, item, FALSE))
    {
	item->action = TagAct;
    }
    else /* Action must be a jump */
    {
	item->action = JumpAct;
	item->u.j.id = allocate_string(token, "Jump ID");
    }
}

/*-----------------------------------------------------------------------------
    read_items
-----------------------------------------------------------------------------*/

static BOOL read_items(State s, FILE *in)
{
    uchar *token;
    Item  *item_at = &(s->item);

    /* Check token was available */
    while (check_token(token = get_token(in), FALSE, TRUE))
    {
	Item item;

	/* Check for presence of tag list and for end marker */
	if (strcmp(token, ";") == 0)
	    return TRUE;
	else if (strcmp(token, "\\;") == 0)
	    strcpy(token, ";");

	/* Create the item */
	*item_at = item = create_item();
	item_at  = &(item->next);

	/* Get its tags list */
	if (strcmp(token, "_") != 0)
	    if (!read_tags_list(token, item, in)) return FALSE;

	/* Get action token */
	if (check_token(token = get_token(in), FALSE, TRUE))
	{
	    /* Check for reserved words */
	    if (strcmp(token, "back") == 0)
	    {
		item->action = BackAct;

		/* Get argument for it */
		if (check_token(token = get_token(in), FALSE, TRUE))
		{
		    if (!set_action_arg(token, item, FALSE)) return FALSE;
		}
		else return FALSE;
	    }
	    else
	    {
		set_item_action(token, item);
	    }
	}
	else return FALSE;
    }

    fsm_error("Unexpected end of item definition");
    return FALSE;
}

/*-----------------------------------------------------------------------------
    build_default_tags_list

    Build a list containing all tags not specified in any item.
-----------------------------------------------------------------------------*/

static Tags build_default_tags_list(State state)
{
    Item item;
    Tag  tag;
    Tags head = NULL, *new_at = &head;

    /* Test each defined tag */
    for (tag = 0 ; tag < tags_max ; tag++)
    {
	BOOL wanted = TRUE;

	/* Scan each item */
	for (item = state->item ; item != NULL ; item = item->next)
	{
	    Tags t;

	    /* Test each tag in the list */
	    for (t = item->tags ; t != NULL ; t = t->next)
	    {
		if (t->tag == tag)
		{
		    wanted = FALSE;
		    break;
		}
	    }
	    if (!wanted) break;
	}

	if (wanted)
	{
	    Tags new;
	    Allocate(new, sizeof(TagsSt), "tag list");
	    new->tag  = tag;
	    new->next = NULL;
	    *new_at   = new;
	    new_at    = &(new->next);
	}
    }

    return head;
}

/*-----------------------------------------------------------------------------
    resolve_states

    Check all IDs specified in jumps are defined and replace them by pointers.
    Checks for duplicated IDs.
    Also builds a tags list for any items that have an empty one.
-----------------------------------------------------------------------------*/

static BOOL resolve_states(FSM f)
{
    State state;

    /* Check each state */
    for (state = f->start ; state != NULL ; state = state->next)
    {
	State s;
	Item  item;
	BOOL  any_default = FALSE;

	/* Compare ID against all others */
	for (s = state->next ; s != NULL ; s = s->next)
	{
	    if (strcmp(state->id, s->id) == 0)
	    {
		fsm_error_token("Duplicate state ID", s->id);
		return FALSE;
	    }
	}

	/* Check each item of state */
	for (item = state->item ; item != NULL ; item = item->next)
	{
	    /* Resolve jump destination */
	    if (item->action == JumpAct)
	    {
		uchar *id = item->u.j.id;

		for (s = f->start ; s != NULL ; s = s->next)
		{
		    if (strcmp(s->id, id) == 0)
		    {
			free(item->u.j.id);
			item->u.j.id = NULL;
			item->u.j.to = s;
			break;
		    }
		}
		if (s == NULL)	/* Didn't find the ID */
		{
		    fsm_error_token("Jump destination missing", id);
		    return FALSE;
		}
	    }
	    else if (state == f->start && item->action == BackAct)
	    {
		fsm_error_token("Back action on initial state", f->name);
		return FALSE;
	    }

	    /* Check for empty tags lists */
	    if (item->tags == NULL) any_default = TRUE;
	}

	/* Second pass to record default lists */
	for (item = state->item ; item != NULL ; item = item->next)
		if (item->tags == NULL)
		    item->tags = build_default_tags_list(state);
    }

    return TRUE;
}

/*-----------------------------------------------------------------------------
    free_fsm

    Free a wholly or partly constructed FSM. In doing so, the global fsms list
    must be scanned, and this fsm removed; this list is at fsm_head.
-----------------------------------------------------------------------------*/

static void free_fsm(FSM f, FSM *head)
{
    State state, nextstate;

    /* Search the global list and remove this FSM */
    while (*head != f) head = &((*head)->next);
    *head = f->next;

    /* Free the name */
    if (f->name) free(f->name);

    /* Free each state */
    for (state = f->start ; state != NULL ; state = nextstate)
    {
	Item item, nextitem;

	/* Free each item of the state */
	for (item = state->item ; item != NULL ; item = nextitem)
	{
	    nextitem = item->next;
	    if (item->action == JumpAct && item->u.j.id != NULL)
		free(item->u.j.id);
	    free_tags_list(item->tags);
	    free(item);
	}

	nextstate = state->next;
	free(state->id);
	free(state);
    }
}

/*-----------------------------------------------------------------------------
    read_fsm_states

    Read the state definitions of a FSM.
-----------------------------------------------------------------------------*/

static BOOL read_fsm_states(FSM f, FSM *head, FILE *in)
{
    uchar *id;
    State *state_at = &(f->start);

    /* Get a state and test for end */
    id = get_token(in);
    while (check_token(id, FALSE, FALSE))
    {
	State state;

	if (strcmp(id, "end") == 0)
	{
	    BOOL ok = resolve_states(f);
	    if (!ok) free_fsm(f, head);
	    return ok;
	}

	for (state = f->start ; state ; state = state->next)
	{
	    if (strcmp(id, state->id) == 0)
	    {
		fsm_error_token("Duplicate state ID", id);
		free_fsm(f, head);
		return FALSE;
	    }
	}

	/* Create a state */
	*state_at = state = create_state(id, f);
	state_at  = &(state->next);

	/* Read the items of the state */
	if (!read_items(state, in))
	{
	    free_fsm(f, head);
	    return FALSE;
	}

	/* Get start of next state */
	id = get_token(in);
    }

    fsm_error("'end' of FSM missing");
    free_fsm(f, head);
    return FALSE;
}

/*-----------------------------------------------------------------------------
    fsm_read

    Read fsm definitions from a given file, merging them with an existing
    list.
-----------------------------------------------------------------------------*/

void fsm_read(FILE *in)
{
    buffer[0]  = 0;
    line_count = 0;

    do
    {
	uchar *token;
	FSM  f;

	/* Get FSM name */
	token = get_token(in);
	if (!check_token(token, TRUE, FALSE))
	    return;
		
	/* Find or create a machine */
	if ((f = find_fsm(token, &fsm_head)) == NULL)
	    return;

	/* Read states */
	if (!read_fsm_states(f, &fsm_head, in))
	    return;
    } while (forever);
}

/*---------------------------------------------------------------------------
    fsm_read_named
----------------------------------------------------------------------------*/

void fsm_read_named(char *name)
{
    FILE *file = open_file(name, "r");
    fsm_read(file);
    fclose(file);
}

/*
==============================================================================
FSM execution
*/

/* Main data structure for FSM execution. This is placed into a node
structure, and consists of a list of FSMStates. Each FSMState consists of a
list of things that matched at this node, and contains a Link list of unscored
hypotheses that matched, the next state to go to, and a pointer back to the
previous FSMState, so that we can create the contents of phrases. In doing so,
it will be necessary to make a copy of the Link list, since (a) we need to add
scores and (b) there can be more than one successor to a state. Where the
FSMState represents a match to a phrasal hypothesis, it is attached to the
node for the end of the hypothesis. It has back pointer to both the start and
end nodes for the hypothesis. This implies that all entries in the match list
must have the same start and end.

Note that the match list will have a single item in it for phrasal matches; it
only contains multiple entries for base hypotheses. This is a consequence of
the phrase building mechanism and could be changed.
*/

struct fsmstate_st
{
   FSMState next;	/* Next active state in the list */
   Link     match;	/* List of unscored hypotheses that matched */
   State    jump;	/* State to jump to */
   FSMState prev;	/* Predecessor to this state */
   Node     start, end;	/* Back pointers to nodes */
   State    state;	/* State recognised (for tracing) */
};

/* Forward declaration */
static void fsm_advance_on_list(Link hyps, Node start, Node end,
				BOOL lex_only, Trans *trans, Trans *new);

/*-----------------------------------------------------------------------------
    fsm_dump_state

    Print out the IDs of all active states.
-----------------------------------------------------------------------------*/

void fsm_dump_state(FILE *out, int depth, FSMState state)
{
    while (state)
    {
	State s = state->state;

	indent(out, depth);
	printf("%s:%s:%s\n", s->fsm->name, s->id, state->jump->id);

	state = state->next;
    }
}

/*-----------------------------------------------------------------------------
    fsm_init_node
-----------------------------------------------------------------------------*/

void fsm_init_node(Node node)
{
    node->fsm_state = NULL;
}

/*-----------------------------------------------------------------------------
    fsm_free_node
-----------------------------------------------------------------------------*/

void fsm_free_node(Node node)
{
    FSMState s, next;

    for (s = node->fsm_state ; s != NULL ; s = next)
    {
	/* Free match list */
	free_links(s->match, FALSE);

	/* Free the state itself */
	next = s->next;
	free(s);
    }

    node->fsm_state = NULL;
}

/*-----------------------------------------------------------------------------
    match_item

    Attempt to match a list of hypotheses to an item. On a match return a new
    unscored list consisting of those things that matched; otherwise return
    NULL. hyp is a scored list. 'lex_only' limits matching to lexical tags.
-----------------------------------------------------------------------------*/

static Link match_item(Link hyps, Item item, BOOL lex_only)
{
    if (item == NULL)
    {
	fprintf(stderr, "Consistency fail (match_item)\n");
	get_out();
    }

    /* Form the list of things that match */
    return match_tags_list(item->tags, hyps, lex_only);
}

/*-----------------------------------------------------------------------------
    create_phrase

    Create a phrase. This involves tracking back through the states to the
    start, creating a node at each stage, and creating a scored hypothesis
    list on each node. Finally, an overall phrasal hypothesis created and the
    whole lot is tagged. The last state which matches is at 'state'.
-----------------------------------------------------------------------------*/

static void create_phrase(Tag tag, Score score, FSMState state,
			  Trans *trans, Trans *new)
{
    Node new_node, last_new;
    Hyp  phrase;
    SHyp p_shyp;
    FSMState last;
    LinkSt p_link;

    if (state == NULL) return; /* Nothing to create */

    /* Create the phrasal hypothesis */
    phrase = create_hyp(tag, score, PhraseHyp, NULL, state->end->lex);
    p_shyp = create_shyp(phrase, NULL, state->end);

    /* Create the final node */
    last_new = make_node(NULL, state->match, state->start, state->end);
    phrase->p.phrase.end = last_new;

    /* Track back through states */
    while (last = state, (state = state->prev) != NULL)
    {
	new_node = make_node(last_new, state->match, state->start, state->end);
	last_new->pred = new_node;
	last_new = new_node;
    }

    /* Set start of phrase links */
    phrase->lex_start      = last->start->lex;
    phrase->p.phrase.start = last_new;
    last_new->pred = NULL;
    p_shyp->start  = last->start;
    link_to_base(phrase, p_shyp->start);

    /* Link the phrase to its start and end nodes */
    insert_link(&(p_shyp->start->start),
				create_link(NULL, TRUE, NULL, p_shyp, NULL));
    insert_link(&(p_shyp->end->end),
				create_link(NULL, TRUE, NULL, p_shyp, NULL));

    /* Tag within the phrase */
    if (trans != NULL) tag_phrase(phrase, trans, new);

    /* Having built the phrase, we now try to advance on it, placing it in a
       one-member list */
    p_link.next   = NULL;
    p_link.u.shyp = p_shyp;
    fsm_advance_on_list(&p_link, p_shyp->start, p_shyp->end, FALSE,
			trans, new);
}

/*-----------------------------------------------------------------------------
    create_fsm_state

    Create a new fsm state using the given match and existing context. Place a
    pointer to it at *at, and return a pointer to its next field.
-----------------------------------------------------------------------------*/

static FSMState *create_fsm_state(Link match, State to, FSMState old,
					Node start, Node end,
					State state, FSMState *at)
{
    FSMState new;
    Allocate(new, sizeof(FSMStateSt), "FSM state");
    new->next  = NULL;
    new->match = match;
    new->jump  = to;
    new->prev  = old;
    new->start = start;
    new->end   = end;
    new->state = state;
    if (at != NULL) *at = new;
    return &(new->next);
}

/*-----------------------------------------------------------------------------
    do_actions

    Do the action that matches at the given item. 'hyps' is the list of
    hypotheses to match, 'old' the previous state (which may be NULL), 'start'
    and 'end' the nodes spanned by the hyps, 'state' the state in the FSM
    definition from which 'item' comes and 'new_at' a pointer to the next
    field of the last state. 'lex_only' tells us whether only lexical
    hypotheses are to be matched.
-----------------------------------------------------------------------------*/

static FSMState *do_actions(Item item, Link hyps, FSMState old_state,
			    Node start, Node end, State state, BOOL lex_only,
			    FSMState *new_at, Trans *trans, Trans *new)
{
    /* Get a list of hypotheses that match */
    Link match;
    match = match_item(hyps, item, lex_only);

    if (match != NULL)	/* Match made */
    {
	switch (item->action)
	{
	    case JumpAct:
		/* Create a new FSMState */
		new_at = create_fsm_state(match, item->u.j.to, old_state,
					  start, end, 
					  state, new_at);
		break;
	    case TagAct:
	    {
		/* Create a final state */
		FSMState new_state;
		create_fsm_state(match, NULL, old_state, start, end, state,
				 &new_state);

		/* Create a phrase using this item */
		create_phrase(item->u.t.tag, item->u.t.score, new_state,
			      trans, new);

		/* Kill the match list and fsm state */
		free_links(match, FALSE);
		free(new_state);
		break;
	    }
	    case BackAct:
		/* Create a phrase using the previous state */
		create_phrase(item->u.t.tag, item->u.t.score, old_state,
				trans, new);

		/* Kill the match list */
		free_links(match, FALSE);
		break;
	    default:	/* Do nothing */
		break;
	}
    }

    return new_at;
}

/*-----------------------------------------------------------------------------
    fsm_advance_on_list

    Use the list of hypotheses at 'hyps', which spans node 'start' to 'end' to
    advance any FSMs we can. This involves looking at the state list at the
    predecessor of 'start'. Note that if we fail to match a state, we do NOT
    delete it: this will happen at the stage of tidying up.
-----------------------------------------------------------------------------*/

static void fsm_advance_on_list(Link hyps, Node start, Node end,
				BOOL lex_only, Trans *trans, Trans *new)
{
    FSM fsm;

    /* Find preceding state list */
    FSMState old_state = (start->pred == NULL) ? NULL : start->pred->fsm_state;
    FSMState *new_at = &(end->fsm_state);

    /* Continue execution where possible */
    for ( ; old_state != NULL ; old_state = old_state->next)
    {
	Item item;

	/* Test each item of the jump destination */
	for (item = old_state->jump->item ; item != NULL ; item = item->next)
	{
	    /* Do actions on each item of the jump destination */
	    new_at = do_actions(item, hyps, old_state, start, end,
				old_state->jump, lex_only, new_at, trans, new);
	}
    }

    /* Start new machines where possible */
    for (fsm = fsm_head ; fsm != NULL ; fsm = fsm->next)
    {
	Item item;

	/* Test each item at start state */
	for (item = fsm->start->item ; item != NULL ; item = item->next)
	{
	    new_at = do_actions(item, hyps, NULL, start, end,
				fsm->start, lex_only, new_at, trans, new);
	}
    }
}

/*-----------------------------------------------------------------------------
    fsm_advance

    Advance FSMs based on the base hypotheses at 'node'. This must be called
    before any phrasal hypotheses have been created.
-----------------------------------------------------------------------------*/

void fsm_advance(Node node, Trans *trans, Trans *new)
{
    /* Find hypotheses list */
    Link hyps = node->start;

    /* Detach the hyp list from the node */
    node->start = NULL;

    /* Do the advancing on this list */
    fsm_advance_on_list(hyps, node, node, TRUE, trans, new);

    /* Paste basic hyp list back on to node */
    insert_link(&(node->start), hyps);
}

/*-----------------------------------------------------------------------------
    fsm_may_tag

    Test whether tagging is permitted: we may do so if the state list on the
    node is NULL.
-----------------------------------------------------------------------------*/

BOOL fsm_may_tag(Lexeme top)
{
    return (top->node->fsm_state == NULL);
}
