;;; weight prior using a gamma density function.

(defvar *q* 0.1)

(defvar *alpha* 100) ;;; shape parameter
(defvar *lambda* 0.45) ;;; scale parameter
(defvar *g* nil) ;;; constant
(defvar *e* 2.7)
(defconstant +log-pi+ (log pi))
(defconstant +euler-constant+ 0.577215664901532860606512d0)

(defvar *cached-weight* -1) 
(defvar *cached-value* 0)

(defun prior-weight (n)

;;; given sample n, work out weighting of prior.
;;; returns number in range 0 .. 1.

  (if (= n *cached-value*) 
      *cached-weight*
    (let ((raw-gamma (gamma-density ( / n 1000))))
      (setf *cached-weight* (- 1.0 (/ raw-gamma -700)))
      (setf *cached-value* n)
      *cached-weight*)))
      

(defun prior-weight (n)
  (/ n (+ n *q*)))

(defun prior-weight (n)
  *q*)

;;; Based upon code by:

;
;  6/3/93
;
;  SAPA, Version 1.0; Copyright 1993, Donald B. Percival, All Rights Reserved
;
;  Use and copying of this software and preparation of derivative works
;  based upon this software are permitted.  Any distribution of this
;  software or derivative works must comply with all applicable United
;  States export control laws.
; 
;  This software is made available AS IS, and no warranty -- about the
;  software, its performance, or its conformity to any
;  specification -- is given or implied. 
;
;  Comments about this software can be addressed to dbp@apl.washington.edu
;-------------------------------------------------------------------------------
;;; used by the next set of functions ...
(defparameter +coefficents-for-log-of-gamma+
  (list 76.18009172947146d0  -86.50532032941677d0      24.01409824083091d0
        -1.231739572450155d0   0.1208650973866179d-2   -0.5395239384953d-5))



(defun gamma-test (alpha lambda n)
  (setf *alpha* alpha)
  (setf *lambda* lambda)
  (setf *g* nil)
  (shell "rm /tmp/gamma")
  (with-open-file (out "/tmp/gamma" :direction :output)
  (dotimes (x n)
	   (format out "~A ~A ~%" x (prior-weight (+ 1 x))))))

(defun test (n q)
  (dotimes (x n)
	   (format t "~A ~A ~%" x (/ x (+ n q 0.0)))))

(defun gamma-density (x)

"compute gamma density function of x"

  (unless *g*
	  (setf *g* (- (log (expt *lambda* *alpha*))
		       (log-of-gamma *alpha*))))

  (+ *g* (log (expt x (- *alpha* 1))) (* -1.0 *lambda* x)))

(defun log-of-gamma (xx)
  "given xx (a real or complex valued number
whose real part is greater than 0),
returns the log (base e) of gamma(xx)
---
Note: based upon the discussion in Section 6.1,
Numerical Recipes, Second Edition"
  (assert (plusp (realpart xx)))
  (if (< (realpart xx) 1)
    ;;; use reflection formula (6.1.4)
    (let ((1-xx (- 1.0 xx)))
      (- (+ +log-pi+ (log 1-xx ))
         (+ (log-of-gamma (1+ 1-xx)) (log (sin (* pi 1-xx)))))))
    ;;; since Re(xx) > 1, use approximation due to Lanczos
    (let* ((x (1- xx))
           (tmp-1 (+ x 5.5))
           (tmp-2 (- (* (+ x 0.5) (log tmp-1 )) tmp-1))
           (ser 1.000000000190015d0))
      (dolist (a-coefficient +coefficents-for-log-of-gamma+)
        (incf x)
        (incf ser (/ a-coefficient x)))
      (+ tmp-2 (log (* 2.5066282746310005 ser)))))

;-------------------------------------------------------------------------------
(defun factorial (k)
  "given an integer k, returns k!"
  (assert (and (integerp k) (not (minusp k))))
  (cond
   ((> k 15)  ; arbitrary choice ...
    (exp (log-of-gamma (1+ k))))
   ((or (= k 1) (= k 0))
    1)
   (t
    (* k (factorial (1- k))))))
