;;
;;
;;      Title : D-MakeLexicon
;;
;;      Function : This takes the file <name>.le and processes the
;;           entries with the given completion rules and multiplication
;;           rules and consistency checks (also in file <name>.le).  The
;;           result is a lexicon tree in <name>.le.ma and the actual lexical
;;           entries in <name>.en.ma
;;
;;      Author : Alan W Black   Oct 7th 1985
;;               Dept. of A.I.  University of Edinburgh
;;
;;      Copyright Graeme Ritchie, Alan Black,
;;                Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;     
;;      1.14 January 15th 1986
;;           Changed the parsing and processing of entries to
;;           add in new rule types and formats.  This should
;;           remove some of the bugs, and also increasing the
;;           speed.
;;      1.15 February 24th 1986
;;           Changed an entry to include a phonological field
;;      1.16 March 14th 1986
;;           Allow multiple dictionaries at once.  This means you
;;           need to have a list of dictionaries rather than just
;;           one.
;;      1.18 May 4th 1986
;;           Changed the pretty printer of the lexicon tree to make it 
;;           faster.  It no longer uses the lisp pretty printer which
;;           is very slow.  This means the tree is not very readable
;;      2.1  30th July 1986
;;           Removed closures and replaced with simpler system of actually
;;           saving and restoring globals.  Changed variable declaration
;;           syntax.  Chnaged order of lexical rules to Multiplication
;;           then completion.
;;      2.2  6th October 1986
;;           Made the order of completion rules and multiplication 
;;           rules dependent on the order they are speified.  Made 
;;           keywords be macros so they are defined only onc.
;;      2.4  23rd January 1987
;;           Changed the syntax of lexical rules so that they have an
;;           associated name which gets used in debugging.  Added debug
;;           flag so information is shown during compilation.  
;;      2.6  15th April 1987
;;           Added NonInflectable category definitions
;;
;;      Description :
;;           This is made up from the functions in entryconv and 
;;           lruleconv that were standard in versions up to 1.11.
;;           The reason for the changes is so the lexicon can be 
;;           built separately from the grammar expansion phase and
;;           compiling of spelling rules. (Though now all the 
;;           original code has been replaced)
;;
;;      Parameters :
;;        name :  atom  where <name>.le is the name of the entries
;;           data file.  This file should contain feature and alias
;;           declarations, CRs, MRs, CCs  and the actual entries.  The
;;           complete syntax is declared in the User Manual 1.14 or 
;;           later.
;;      Returns :
;;           the lexicon tree 
;;      Side Effect :
;;           creates file <name>.en.ma and fills it with the expanded
;;           lexical entries.  Also <name>.le.ma which contains the 
;;           lexicon tree, the leaves of which are integers which point
;;           to the corresponding entries in <name>.en.ma
;;
;;
;;      External references :
;;
;;      System functions used :
;;        Name     type      Args #  type                 Comment
;;
;;      Restrictions :
;;
;;

(declare
   (special
      D-fileid
      D-INCLUDEFILES
      D-SIMPLELEXICON
      D-LASTSYMBOL
      D-NONINFLECTS
      D-CurrentSym
      D-LEFTPAREN
      D-RIGHTPAREN
      D-ENTRYFILEID
      D-INCLUDESTACK
      D-FEATURES
      D-DISTINGUISHEDCATEGORY
      D-ALIASES
      D-CATVALFEAT
      D-LEXICON
      D-VARIABLES
      D-RULESORDER
      D-TRACEFLAG
      D-TRACEFLAGSWITCH
      D-INCOREFLAG
      D-INCOREFLAGSWITCH
      D-NONINFLECTFLAG
      D-NONINFLECTFLAGSWITCH
      D-CATEGORYDEFINITIONS
   )  
   (localf
      D-PrettyPrintLexicon
      D-ParseLexiconFile              
      D-ParseCRs                     
      D-ParseMRs                     
      D-ParseSkeletons                
      D-ParseSkeleton                 
      D-ParseCCs                    
      D-ParsePrecondition             
      D-ParsePattern                  
      D-ParseCitationForm             
      D-ParseSyntaxPatt               
      D-ParseFeaturePattern           
      D-ParseSemanticPatt             
      D-ParseUserPatt                 
      D-ParseLispS-Expression         
      D-ParseToEndOfList
      D-ParseEntry                    
      D-ParsePhonoForm             
      D-ParseSemanticField            
      D-ParseUserField                
      D-ParseSyntaxField              
      D-InitLexGlobals                
   )
)

(include "macros")
(include "keywords")
(include "subrout")
(include "catrouts")
(include "unify")
(include "parserouts")
(include "specrouts")   ;; routines that are specific to unification type
(include "dclsconv")
(include "entryconv")
(include "lruleconv")

(defun D-MakeLexicon (name)
;;
;;  main routine.  reads in <name>.le and produces the lexicon tree
;;  and declarations in file <name>.le.ma and the entries themselves
;;  (expanded using the lexical rules) 
;;
   (let ( lexicon )
      (D-SaveGlobalEnvironment); so as the currently loaded stuff is unaffected
      (D-InitLexGlobals)
      (setq D-LEXICON-STATE 'D-CREATE)
      (setq D-ENTRYFILEID (outfile (concat name (DK-EN-MA))))
      (setq D-fileid (infile (concat name (DK-LE))))    ;; open dictionary file
      (setq D-INCLUDEFILES (ncons (concat name (DK-LE))))

      (D-GetNextSym)

      (D-ParseLexiconFile)

      (close D-fileid)             ;; be nice to the number of free file ids

      (setq lexicon 
	 (cons (concat name (DK-EN-MA)) D-SIMPLELEXICON))
      (setq D-fileid (outfile (concat name (DK-LE-MA))))
      (print (DK-VERSION-STAMP) D-fileid) (terpri D-fileid)
      (D-PrettyPrintLexicon lexicon)
      (print D-FEATURES  D-fileid) (terpri D-fileid)
      (print D-CATVALFEAT  D-fileid) (terpri D-fileid)
      (print D-CATEGORYDEFINITIONS D-fileid) (terpri D-fileid)

      (close D-fileid)
      (close D-ENTRYFILEID)

      (D-RestoreGlobalEnvironment)
      lexicon         ;; return lexicon
   )
)

(defun D-PrettyPrintLexicon (lexicon)
;;
;;   Pretty prints the lexicon on D-fileid.  The result is not
;;   very pretty but runs faster than the lisp pretty printer
;;
   (terpri D-fileid)
   (princ "(" D-fileid)
   (print (car lexicon) D-fileid)
   (terpri D-fileid)
   (mapcar
      #'(lambda (chunk)
	 (terpri D-fileid) (princ " (" D-fileid)
	 (print (car chunk) D-fileid)
	 (mapcar
	    #'(lambda (part)
	       (terpri D-fileid) (princ "   (" D-fileid)
	       (print (car part) D-fileid)
	       (mapcar 
		  #'(lambda (bit)
	             (terpri D-fileid)
		     (princ "      " D-fileid)
		     (print bit D-fileid))
                  (cdr part))
               (princ ")" D-fileid))
            (cdr chunk)
        )
	(princ ")" D-fileid))
      (cdr lexicon)
   )
   (princ ")" D-fileid)
   (terpri D-fileid)
)

(defun D-ParseLexiconFile ()
;;
;;   parses the lexicon file and builds the lexicon itself
;;
   (let ( (crs nil) (mrs nil) (ccs nil) (entry nil) )
      (D-MustHave (DK-Declarations))
         (D-ParseDeclarations)   
      (D-MustHave (DK-Rules))
	 (cond
	    ((eq D-CurrentSym (DK-Completion))
	       (setq D-RULESORDER 'COMP)
               (setq crs (D-ParseCRs))
               (setq mrs (D-ParseMRs)))
	    (t
	       (setq D-RULESORDER 'MULT)
               (setq mrs (D-ParseMRs))
               (setq crs (D-ParseCRs))))
	 (setq ccs (D-ParseCCs))
      (D-MustHave (DK-Entries))
	 (D-while (not (eq D-CurrentSym 'EOF))
	    (setq entry (D-ParseEntry))
	    (setq D-SIMPLELEXICON ;; add these expanded entries to the lexicon
	       (D-EntryConv
		  (D-ApplyRules 
		     mrs crs ccs entry)
                  D-SIMPLELEXICON
               )
            )
         )
   )
)

(defun D-ParseCRs ()
;;
;;  parses the completion rules and returns them normalised
;;  to caller.
;;  Not sure what terminates the rules but must be one of
;;      '(Completion Consistency Multiplication)
;;
   (let ( (name nil) (precond nil) (action nil) (crs nil) )
      (D-MustHave (DK-Completion))
      (D-MustHave (DK-Rules))
      (D-while (not (memq D-CurrentSym (list (DK-Completion) (DK-Consistency)
					   (DK-Multiplication))))
         (setq name D-CurrentSym)
	 (D-CheckAtom name)    ;; should be a sensible name
	 (D-GetNextSym)
	 (D-MustHave (DK-SYM-COLON))
	 (setq precond (D-ParsePrecondition))
	 (D-MustHave '=)
	 (D-MustHave '>)
	 (setq action (D-ParseSkeleton)) ;; new entry    
	 (setq crs
	    (cons
	       (list name precond action)  ;; new cr
	       crs))
      )
      (nreverse crs)     ;; in the order the user specified
   )
)

(defun D-ParseMRs ()
;;
;;  parses the multiplication rules and returns them normalised
;;  to caller.
;;
   (let ( (name nil) (precond nil) (skellys nil) (mrs nil) )
      (D-MustHave (DK-Multiplication))
      (D-MustHave (DK-Rules))
      (D-while (not (memq D-CurrentSym (list (DK-Completion) (DK-Consistency)
					   (DK-Multiplication))))
         (setq name D-CurrentSym)
	 (D-CheckAtom name)    ;; should be a sensible name
	 (D-GetNextSym)
	 (D-MustHave (DK-SYM-COLON))
	 (setq precond (D-ParsePrecondition))
	 (D-MustHave '=)
	 (D-MustHave '>)
	 (D-MustHave '>)
	 (setq skellys (D-ParseSkeletons))   ;; list of skeletons
	 (setq mrs
	    (cons
	       (list name precond skellys)  ;; skeletons
	       mrs))
      )
      mrs    ;; this is the reverse order of specification
   )         ;; this should not be significant
)

(defun D-ParseSkeletons ()
;;
;;   parses the skeletons of MRs 
;;
   (let ( (skellys nil) )
      (D-MustHave D-LEFTPAREN)
      (D-while (not (eq D-CurrentSym D-RIGHTPAREN))
         (setq skellys 
	    (cons (D-ParseSkeleton) skellys))
      )
      (D-MustHave D-RIGHTPAREN)
      skellys
   )
)

(defun D-ParseSkeleton ()
;;
;;   parses an entry skeleton 
;;
   (D-ParsePattern)    ;; same as pattern but some extra checks maybe
)

(defun D-ParseCCs ()
;;
;;  parses the consistency checks and returns them normalised
;;  to caller.
;;
   (let ( (name nil) (precond nil) (postcond nil) (ccs nil) )
      (D-MustHave (DK-Consistency))
      (D-MustHave (DK-Checks))
      (D-while (not (eq D-CurrentSym (DK-Entries)))
         (setq name D-CurrentSym)
	 (D-CheckAtom name)    ;; should be a sensible name
	 (D-GetNextSym)
	 (D-MustHave (DK-SYM-COLON))
	 (setq precond (D-ParsePrecondition))
	 (D-MustHave (DK-demands))
	 (setq postcond (D-ParsePrecondition))   ;; conditions demanded
	 (setq ccs
	    (cons
	       (list name precond postcond)  ;; demandeds
	       ccs))
      )
      ccs       ;; return list of consistency checks
   )
)

(defun D-ParsePrecondition ()
;;
;;   parses a precondition and returns it in normalised form
;;
   (let ( (precond nil) )
      (cond                         ;; Read first pattern
         ((eq D-CurrentSym '~)
            (D-MustHave '~)  ;; skip not sign
            (setq precond
               (cons
                  (list '~ (D-ParsePattern))
                  precond))
         )
         (t
            (setq precond
               (cons (D-ParsePattern) precond))
         )
      )
      (D-while (eq D-CurrentSym (DK-and))
	 (D-MustHave (DK-and))  ;; skip the conjunction
	 (cond
	    ((eq D-CurrentSym '~)
	       (D-MustHave '~)  ;; skip not sign
	       (setq precond
		  (cons
		     (list '~ (D-ParsePattern))
		     precond))
            )
	    (t
	       (setq precond
		  (cons (D-ParsePattern) precond))
            )
         )
      )
      (reverse precond)
   )
)

(defun D-ParsePattern ()
;;
;;  reads a pattern and returns it as a four item list
;;
   (let ( (patt nil) )
      (D-MustHave D-LEFTPAREN)
      (setq patt
         (list
            (D-ParseCitationForm)   ;; any atomic thingy
            (D-ParsePhonoForm)      ;; any atomic thingy
            (D-ParseSyntaxPatt)
            (D-ParseSemanticPatt)
            (D-ParseUserPatt)
         )
      )
      (D-MustHave D-RIGHTPAREN)
      patt        ;; return pattern
   )
)

(defun D-ParseCitationForm ()
;;
;;  parses a citation form which is the current symbol what ever it is
;;
   (let ( (citform D-CurrentSym) )
      (D-GetNextSym)
      citform
   )
)

(defun D-ParseSyntaxPatt ()
;;
;;  parses a syntax field
;;
   (let ( (cat nil) )
      (cond
	 ((eq D-CurrentSym D-LEFTPAREN)
            (D-MustHave D-LEFTPAREN)
            (D-while (not (eq D-CurrentSym D-RIGHTPAREN))
	       (cond
		  ((eq D-CurrentSym '~)   ;; negated feature spec
		     (D-MustHave '~)
		     (setq cat
			(cons
			   (list '~ (D-ParseFeaturePattern))
			   cat)))
                  (t       ;; simple pattern
	             (setq cat (cons (D-ParseFeaturePattern) cat)))
               )
            )
            (D-MustHave D-RIGHTPAREN)
            (reverse cat)
         )
	 ((D-IsPattVariable D-CurrentSym)   ;; a variable
	    (setq cat D-CurrentSym)
	    (D-GetNextSym)
	    cat
         )
	 (t   ;; an atomic value, this is an error
	    (D-FindCurrentLine)
	    (error (concat "Atomic value " D-CurrentSym
	       " found but syntax field expected "))
         )
      )
   )
)

(defun D-ParseFeaturePattern ()
;;
;;  parses a feature pair or a pattern variable
;;
   (let ( (fpair nil) )
      (cond
         ((eq D-CurrentSym D-LEFTPAREN)
            (let ( (featname nil) (featvalue nil) )
               (D-MustHave D-LEFTPAREN)
               (setq featname D-CurrentSym)
	       (D-CheckAtom featname) ;; check its not a bracket
               (D-GetNextSym)
               (cond
		  ((and (D-CatValFeatP featname)
	                (eq D-CurrentSym D-LEFTPAREN))   ;; category valued feat
	             (setq featvalue
			(D-ParseSyntaxPatt))  ;; category
                  )
		  ((and (D-CatValFeatP featname)   ;; category variable
			(D-IsPattVariable D-CurrentSym))
                     (setq featvalue D-CurrentSym)
		     (D-GetNextSym)
                  )
		  ((D-CatValFeatP featname)    ;; atomic value for catvalfeat
		     (D-FindCurrentLine)
		     (error (concat "Atomic value for category-valued feature "
				     featname)))
	          (t         ;; atomic valued
	             (setq featvalue D-CurrentSym)
	             (D-CheckAtom featvalue) ;; check its not a bracket
	             (D-GetNextSym)
                  )
               )
	       (D-MustHave D-RIGHTPAREN)
	       (setq fpair (list featname featvalue))
            )
         )
         ((D-IsPattVariable D-CurrentSym)
	    (setq fpair D-CurrentSym)
	    (D-GetNextSym)
         )
	 (t   ;; atomic value that is not fpair or variable 
	    (D-FindCurrentLine)
	    (error "Atomic value found where variable expected")
         )
      )
      fpair      ;; return feature pair
   )
)

(defun D-ParseSemanticPatt ()
;;
;;  parsers the semantic pattern
;;
   (D-ParseLispS-Expression)
)

(defun D-ParseUserPatt ()
;;
;;  parsers the user field pattern
;;
   (D-ParseLispS-Expression)
)

(defun D-ParseLispS-Expression ()
;;
;;  This uses my reader to read a lisp s-expression.  Not that
;;  This only deals with lists and atoms and not all the fancy junk
;;  of read macros.  This could be a problem.
;;
   (let (part)
      (cond
	 ((eq D-CurrentSym D-LEFTPAREN)
	    (D-GetNextSym)
	    (setq part (D-ParseToEndOfList))
	 )
	 ((eq D-CurrentSym D-RIGHTPAREN)
	    (D-FindCurrentLine)
	    (error "Right Parenthesis found when s-expression expected")
	 )
	 (t         ;; must be simple atom
	    (setq part D-CurrentSym)
	    (D-GetNextSym)
         )
      )
      part
   )
)

(defun D-ParseToEndOfList ()
;;
;;  This parses down an list s-expression.
;;
   (cond
      ((eq D-CurrentSym D-RIGHTPAREN)
	 (D-GetNextSym)
	 nil               
      )
      (t
	 (cons
	    (D-ParseLispS-Expression)
	    (D-ParseToEndOfList))
      )
   )
)

(defun D-ParseEntry ()
;;
;;  reads an entry and returns it as a four item list
;;
   (let ( (entry nil) )
      (setq D-TRACEFLAG D-TRACEFLAGSWITCH) ;; switch tracing to current setting
      (setq D-INCOREFLAG D-INCOREFLAGSWITCH)
      (setq D-NONINFLECTFLAG D-NONINFLECTFLAGSWITCH)
      (D-MustHave D-LEFTPAREN)
      (let*   ;; so that the code doesn't depend of Left to Right eval of
	      ;; arguments (Cambridge Lisp is the awkward system)
	   ( (cite (D-ParseCitationForm))   ;; any atomic thingy
             (phono (D-ParsePhonoForm))      ;; any atomic thingy
             (syntax (D-ParseSyntaxField))
             (semantics (D-ParseSemanticField))
             (user (D-ParseUserField)))
         (setq entry
            (list cite phono syntax semantics user))
      )
      (D-MustHave D-RIGHTPAREN)
      entry
   )
)

(defun D-ParsePhonoForm ()
;;
;;  parses the phonological form of of any entry.  this will
;;  be restricted to the Alvey Speech Club phonetic notation
;;  at present its just any atomic form
;;  This may be a list of atomic forms or one single atomic form
;;  
;;
   (let ( (phonoform nil) )
      (cond
	 ((eq D-CurrentSym D-LEFTPAREN)   ;; its a list
	    (setq phonoform (D-ParseSimpleList))
         )
	 (t             ;; simple atom
	    (setq phonoform D-CurrentSym)
            (D-GetNextSym)
         )
      )
      phonoform   ;; return phonoform
   )
)

(defun D-ParseSemanticField ()
;;
;;  parsers the semantic field of an entry
;;
   (D-ParseLispS-Expression)
)

(defun D-ParseUserField ()
;;
;;  parsers the user field of an entry
;;
   (D-ParseLispS-Expression)
)

(defun D-ParseSyntaxField ()
;;
;;  parses a syntax field of a lexical entry
;;
   (D-SubsAliasCategory (D-ParseCategory))
)

(defun D-InitLexGlobals ()
;;
;;  This initialises the global variables so that no previous 
;;  junk can appear in the analyser
;;
   (D-InitReader)         ;; initialise the globals for the reader
   (setq D-ALIASES nil) 
   (setq D-FEATURES nil)
   (setq D-CATVALFEAT nil)      ;; category-valued features
   (setq D-INCLUDESTACK nil)     ;; initialise list of include fileids
   (setq D-SIMPLELEXICON nil)         ;; lexicon index tree
   (setq D-DISTINGUISHEDCATEGORY nil)
		   ;; this has to be done to catch the idiots who 
		   ;; declare Top in a lexicon file - I will have to
		   ;; think of a better thing to do about this problem
   (setq D-LASTSYMBOL nil)       ;; no previous symbol 
   (setq D-RULESORDER 'COMP)     ;; completion rules order
   (setq D-LCATEGORIES nil)      ;; not used in this compilation
   (setq D-NONINFLECTS nil)
   (setq D-NONINFLECTFLAG (DK-OFF))
   (setq D-INCOREFLAG (DK-OFF))
   (setq D-NONINFLECTFLAGSWITCH (DK-OFF))
   (setq D-INCOREFLAGSWITCH (DK-OFF))
   (setq D-CATEGORYDEFINITIONS nil)
)

