(*---------------------------------------------------------------------------
                An ML script for building HOL
 ---------------------------------------------------------------------------*)

(* The following lines are written at configuration time. *)

val LN_S = "ln -s";
val HOLDIR = "/local/scratch/kxs/utah";
val DEPDIR = ".HOLMK";
val SRCDIRS = 
    ["/local/scratch/kxs/utah/src/portableML",
     "/local/scratch/kxs/utah/src/0",
     "/local/scratch/kxs/utah/src/parse",
     "/local/scratch/kxs/utah/src/bool",
     "/local/scratch/kxs/utah/src/goalstack",
     "/local/scratch/kxs/utah/src/taut",
     "/local/scratch/kxs/utah/src/compute/src",
     "/local/scratch/kxs/utah/src/q",
     "/local/scratch/kxs/utah/src/combin",
     "/local/scratch/kxs/utah/src/lite",
     "/local/scratch/kxs/utah/src/refute",
     "/local/scratch/kxs/utah/src/simp/src",
     "/local/scratch/kxs/utah/src/meson/src",
     "/local/scratch/kxs/utah/src/basicProof",
     "/local/scratch/kxs/utah/src/relation",
     "/local/scratch/kxs/utah/src/pair/src",
     "/local/scratch/kxs/utah/src/sum",
     "/local/scratch/kxs/utah/src/one",
     "/local/scratch/kxs/utah/src/option",
     "/local/scratch/kxs/utah/src/num/theories",
     "/local/scratch/kxs/utah/src/num/reduce/src",
     "/local/scratch/kxs/utah/src/num/arith/src",
     "/local/scratch/kxs/utah/src/num",
     "/local/scratch/kxs/utah/src/IndDef",
     "/local/scratch/kxs/utah/src/datatype/parse",
     "/local/scratch/kxs/utah/src/datatype/equiv",
     "/local/scratch/kxs/utah/src/datatype/record",
     "/local/scratch/kxs/utah/src/datatype",
     "/local/scratch/kxs/utah/src/list/src",
     "/local/scratch/kxs/utah/src/tfl/src",
     "/local/scratch/kxs/utah/src/unwind",
     "/local/scratch/kxs/utah/src/boss",
     "/local/scratch/kxs/utah/src/string",
     "/local/scratch/kxs/utah/src/llist",
     "/local/scratch/kxs/utah/src/integer",
     "/local/scratch/kxs/utah/src/res_quan/src",
     "/local/scratch/kxs/utah/src/pred_set/src",
     "/local/scratch/kxs/utah/src/word/theories",
     "/local/scratch/kxs/utah/src/word/src",
     "/local/scratch/kxs/utah/src/finite_map",
     "/local/scratch/kxs/utah/src/hol88",
     "/local/scratch/kxs/utah/src/real",
     "/local/scratch/kxs/utah/src/bag",
     "/local/scratch/kxs/utah/src/ring/src",
     "/local/scratch/kxs/utah/src/temporal/src",
     "/local/scratch/kxs/utah/src/temporal/smv.2.4.3",
     "/local/scratch/kxs/utah/src/prob",
     "/local/scratch/kxs/utah/src/muddy/muddyC",
     "/local/scratch/kxs/utah/src/muddy",
     "/local/scratch/kxs/utah/src/HolBdd",
     "/local/scratch/kxs/utah/help/src"];
val GNUMAKE = "gnumake";

val SIGOBJ = Path.concat(HOLDIR, "sigobj");

fun normPath s = Path.toString(Path.fromString s)
fun itstrings f [] = raise Fail "itstrings: empty list"
  | itstrings f [x] = x
  | itstrings f (h::t) = f h (itstrings f t);
fun fullPath slist = normPath
   (itstrings (fn chunk => fn path => Path.concat (chunk,path)) slist);

fun quote s = String.concat["\"", s, "\""];

local val Holmake_path = fullPath [HOLDIR, "bin/Holmake --qof"]
      open Process
in
fun Holmake dir =
  if system Holmake_path = success
  then ()
  else (print ("Build failed in directory "^dir^"\n");
        raise Fail "Couldn't make directory")
end;


fun gnumake dir =
  if Process.system GNUMAKE = Process.success
  then ()
  else (print ("Build failed in directory "^dir
                ^" ("^GNUMAKE^" failed).\n");
        raise Fail "Couldn't make directory");

(*---------------------------------------------------------------------------
           Compile a HOL directory in place. Some libraries,
           e.g., the robdd libraries, need special treatment because
           they come with external tools or C libraries.
 ---------------------------------------------------------------------------*)

fun build_dir dir =
  let val _ = FileSys.chDir dir
      val _ = print ("Working in directory "^dir^"\n")
  in
   case #file(Path.splitDirFile dir)
    of "muddyC"    => (gnumake dir handle _ => print(String.concat
                         ["\nmuddyLib has NOT been built!! ",
                          "(continuing anyway).\n\n"]))
     | "smv.2.4.3" => (gnumake dir handle _ => print(String.concat
                         ["\nCompilation of SMV fails!!",
                          " temporal Lib has NOT been built!! ",
                          "(continuing anyway).\n\n"]))
     | _   => Holmake dir
  end
  handle OS.SysErr(s, erropt)
    => (print ("OS error: "^s^" - "^
            (case erropt of SOME s' => OS.errorMsg s' | _ => "") ^ "\n");
        Process.exit Process.failure);

(*---------------------------------------------------------------------------
      Arrange for all executables and relevant .sig files
      to reside in a common directory.
 ---------------------------------------------------------------------------*)

fun map_dir f dir =  (* map a function over the files in a directory *)
  let val dstrm = FileSys.openDir dir
      fun loop() =
        case FileSys.readDir dstrm
         of NONE => FileSys.closeDir dstrm
          | SOME file => (f (dir,file) ; loop())
  in loop()
  end handle OS.SysErr(s, erropt) =>
    (print ("OS error: "^s^" - "^
            (case erropt of SOME s' => OS.errorMsg s' | _ => "") ^ "\n");
     Process.exit Process.failure);


fun copy file path =  (* Dead simple file copy *)
 let open TextIO
     val (istrm,ostrm) = (openIn file, openOut path)
     fun loop() =
       case input1 istrm
        of SOME ch => (output1(ostrm,ch) ; loop())
         | NONE    => (closeIn istrm; flushOut ostrm; closeOut ostrm)
  in loop()
  end;

fun link s1 s2 =
  let open Process
  in if system (String.concat[LN_S," ",s1," ",s2]) = success then ()
     else (print ("Unable to link file "^quote s1^" to file "^quote s2^".\n");
           raise Fail "link")
  end;

fun transfer_file symlink targetdir (df as (dir,file)) =
  let val cp = if symlink then link else copy
      fun mv s1 s2 =
        let val s1' = normPath s1
            val s2' = normPath s2
        in
          if symlink then link s1' s2' else FileSys.rename {old=s1', new=s2'}
        end
      fun transfer f (dir,file1,file2) = 
               f (fullPath [dir,file1])
                 (fullPath [targetdir,file2])
      fun idtransfer f (dir,file) = transfer f (dir,file,file)
      fun digest file =
        let val b = Path.base file
        in if (String.extract(b,String.size b -4,NONE) = "-sig" 
               handle _ => false)
           then SOME (String.extract(b,0,SOME (String.size b - 4)) ^".sig")
           else NONE
        end
  in
   case Path.ext file
    of SOME"ui"     => idtransfer mv df
     | SOME"uo"     => idtransfer mv df
     | SOME"so"     => idtransfer mv df   (* for dynlibs *)
     | SOME"xable"  => idtransfer mv df   (* for executables *)
     | SOME"sig"    => idtransfer cp df
     | SOME"sml"    => (case digest file
                         of NONE => ()
                          | SOME file' => transfer cp (dir,file,file'))
     |    _         => ()
  end;

(*---------------------------------------------------------------------------
        Transport a compiled directory to another location. The
        symlink argument says whether this is via a symbolic link,
        or by copying. The ".uo", ".ui", ".so", ".xable" and ".sig"
        files are transported.
 ---------------------------------------------------------------------------*)

fun upload (src,target,symlink) =
  (print ("Uploading files to "^target^"\n");
   map_dir (transfer_file symlink target) src)
        handle OS.SysErr(s, erropt) =>
          (print ("OS error: "^s^" - "^
                  (case erropt of SOME s' => OS.errorMsg s' | _ => "") ^ "\n");
           Process.exit Process.failure)


(*---------------------------------------------------------------------------
    For each element in SRCDIRS, build it, then upload it to SIGOBJ.
    This allows us to have the build process only occur w.r.t. SIGOBJ
    (thus requiring only a single place to look for things).
 ---------------------------------------------------------------------------*)

fun build_src symlink =
  List.app (fn s => (build_dir s; upload(s,SIGOBJ,symlink))) SRCDIRS;

fun rem_file f =
 FileSys.remove f
   handle _ => (print ("Trouble with removing file "^f^"?\n"); ());


fun clean_sigobj() =
  let val _ = print ("Cleaning out "^SIGOBJ^"\n")
  in 
     map_dir (rem_file o normPath o Path.concat) SIGOBJ
  end;


fun build_help () =
 let val dir = Path.concat(Path.concat (HOLDIR,"help"),"src")
     val _ = FileSys.chDir dir
     val _ = print ("Working in directory "^dir^"\n");
     val _ = build_dir dir
     val doc2html = fullPath [dir,"Doc2Html"]
     val docpath  = fullPath [HOLDIR, "help", "Docfiles"]
     val htmlpath = fullPath [docpath, "html"]
     val _        = if (FileSys.isDir htmlpath handle _ => false) then ()
                    else (print ("Creating directory "^htmlpath^"\n");
                          FileSys.mkDir htmlpath)
     val cmd1     = String.concat [doc2html, " ",docpath, " ", htmlpath]
     val cmd2     = fullPath [dir,"makebase"]
 in
   if Process.system cmd1 = Process.success then ()
   else (print ("Build failed in directory "^dir^"\n");
        raise Fail "Couldn't make html versions of Docfiles")
 ;
   if Process.system cmd2 = Process.success then ()
   else (print ("Build failed in directory "^dir^"\n");
        raise Fail "Couldn't make help database")
 end;


fun build_hol symlink =
  let val _ = clean_sigobj()
      val _ = build_src symlink
      val _ = build_help()
  in
    print "\nHol built successfully.\n"
  end;


(*---------------------------------------------------------------------------
       Get rid of compiled code and dependency information.
 ---------------------------------------------------------------------------*)

local val lenScript = String.size "Script"
      val lenTheory_ext = String.size "Theory.sig"
in
fun suffixCheck s =
 let val len = String.size s
 in (("Script" = String.extract(s,len-lenScript,NONE)) orelse raise Subscript)
    handle Subscript
    =>  let val suffix = String.extract(s,len - lenTheory_ext, NONE)
        in (len > 10
            andalso ((suffix = "Theory.sig") orelse (suffix = "Theory.sml")))
           orelse raise Subscript
         end
        handle Subscript => false
  end
end;

(*---------------------------------------------------------------------------
    "cleandir" could be extended to do a better job of cleaning up
    directories where external tools have been built.
 ---------------------------------------------------------------------------*)

fun cleandir dir =
  let fun efile s =
        case Path.ext s
         of SOME "ui" => true
          | SOME "uo" => true
          | SOME "so" => true         (* for dynlibs, like muddyLib *)
          | SOME "o"  => true         (* for C libraries *)
          | SOME "xable"  => true     (* for executables *)
          |    _          => suffixCheck s
      fun del (d,f) = if efile f then rem_file (fullPath [d,f]) else ()
  in
    map_dir del dir
  end;

fun cleanAlldir dir = (* clean directory d and also remove d/DEPDIR *)
  let val _ = cleandir dir
      val depdir = Path.concat(dir,DEPDIR)
  in
    if (FileSys.isDir depdir handle _ => false)
    then (map_dir (rem_file o normPath o Path.concat) depdir;
          FileSys.rmDir depdir handle e
           => (print ("Unable to remove directory "^depdir^".\n"); ()))
    else ()
  end;

fun clean_dirs f = clean_sigobj() before List.app f SRCDIRS;

fun errmsg s = TextIO.output(TextIO.stdErr, s ^ "\n");
val help_mesg = "Usage: build\n\
                                \  or: build -symlink\n\
                                \  or: build -clean\n\
                                \  or: build -cleanAll.";
val _ =
  case Mosml.argv ()
   of [_]             => build_hol false (* no symbolic linking *)
    | [_,"-symlink"]  => build_hol true  (*    symbolic linking *)
    | [_,"-clean"]    => clean_dirs cleandir
    | [_,"-cleanAll"] => clean_dirs cleanAlldir
    | [_,"clean"]     => clean_dirs cleandir
    | [_,"cleanAll"]  => clean_dirs cleanAlldir
    | [_,"help"]      => build_help()
    |     _           => errmsg help_mesg
