(* ---------------------------------------------------------------------*)
(* CONTENTS: functions on the syntax of paired abstractions and         *)
(*           quantifications.                                           *)
(* ---------------------------------------------------------------------*)
(*$Id: Pair_syn.sml,v 1.2 2000/11/16 12:50:22 kxs Exp $*)

(* =====================================================================*)
(* Constructors for paired HOL syntax.                                  *)
(* =====================================================================*)

structure Pair_syn :> Pair_syn =
struct

local open pairTheory in end;

open HolKernel Parse boolLib;

val PAIR_ERR = mk_HOL_ERR "pair lib";

fun dest_pair M = 
  let val (p1,p2) = pairSyntax.dest_pair in {fst=p1,snd=p2} end;


(* ===================================================================== *)
(* All the elements in a pair struture.                                  *)
(* ===================================================================== *)

fun rip_pair p = 
    let val (fst,snd) = dest_pair p 
    in rip_pair fst @ rip_pair snd
    end
    handle HOL_ERR _ => [p];

(* ===================================================================== *)
(* Check if a term is a pair structure of variables.                     *)
(* ===================================================================== *)

val is_pvar = pairSyntax.is_vstruct

(* (all is_var) o rip_pair ; *)

(* ===================================================================== *)
(* Paired version of variant.                                            *)
(* ===================================================================== *)

val pvariant =
    let fun uniq [] = []
	  | uniq (h::t) = h::uniq (filter (not o equal h) t)
	fun variantl avl [] = []
	  | variantl avl (h::t) =
	    let val h' = variant (avl@(filter is_var t)) h
	    in {residue=h',redex=h}::(variantl (h'::avl) t)
	    end
    in
  fn pl => 
  fn p =>
   let val avoid = (flatten (map ((map (assert is_var)) o rip_pair) pl)) 
       val originals = uniq (map (assert (fn p => is_var p orelse is_const p)) 
                                 (rip_pair p))
       val subl = variantl avoid originals 
   in
     subst subl p
  end end;

(* ===================================================================== *)
(* Generates a pair structure of variable with the same structure as     *)
(* its parameter.                                                        *)
(* ===================================================================== *)

fun genlike p = pairSyntax.gen_vstruct (type_of p);

(* ===================================================================== *)
(* Paired bound variable and body.                                       *)
(* ===================================================================== *)

fun bndpair tm = 
      fst (pairSyntax.dest_abs tm) handle HOL_ERR _ => failwith "bndpair"
and pbody tm = 
      snd (pairSyntax.dest_abs tm) handle HOL_ERR _ => failwith "pbody";

(* ===================================================================== *)
(* Occurence check for bound pairs.                                      *)
(* occs_in p t    true iff any of the variables in p occur free in t     *)
(* ===================================================================== *)

val occs_in =
 let fun occs_check vl t =
      case dest_term t
       of CONST _ => false
        | VAR _   => mem t vl
        | COMB(Rator,Rand) => occs_check vl Rator orelse occs_check vl Rand
        | LAMB(Bvar,Body)  => occs_check (filter (not o equal Bvar) vl) Body
 in
  fn p => fn t =>
    if is_pvar p 
       then let val vs = free_vars p 
            in occs_check vs t
            end
       else failwith "occs_in: not a pvar"
 end;

end;
