(*===========================================================================*)
(* Autopilot Specification Example                                           *)
(*                                                                           *)
(* The Hol98 part of an intended comparison between PVS, ACL2 and HOL98.     *)
(* Copyright 1998, Mark Staples                                              *)
(* Modifications (April,June 1998) Konrad Slind.                             *)
(*===========================================================================*)

app load ["HolBddLib", "bossLib"] ; open HolBddLib bossLib;

(*---------------------------------------------------------------------------*
 * We'll track information on inferences.                                    *
 *---------------------------------------------------------------------------*)

val meter = Count.mk_meter();


(*---------------------------------------------------------------------------*
 * Start the theory.                                                         *
 *---------------------------------------------------------------------------*)

val _ = new_theory "Autopilot";


Hol_datatype `events = press_att_cws 
                     | press_cas_eng 
                     | press_alt_eng 
                     | press_fpa_sel
                     | input_alt     
                     | input_fpa
                     | input_cas
                     | alt_reached   
                     | fpa_reached   
                     | alt_gets_near`;


Hol_datatype `off_eng 
                     = off 
                     | engaged`;


(*****************************************************************************)
(* Hol_datatype `mode_status                                                 *)
(*                      = armed                                              *)
(*                      | Mode of off_eng`;                                  *)
(*****************************************************************************)


Hol_datatype `mode_status 
                     = armed 
                     | Mode_off
                     | Mode_engaged`;

val Mode_def = Define
 `Mode oe = ((oe=off) => Mode_off | (oe=engaged) => Mode_engaged | ARB)`;


val Mode =
 prove
  (``(Mode off = Mode_off) /\ (Mode engaged = Mode_engaged)``,
   RW_TAC std_ss [Mode_def]);

Hol_datatype `disp_status 
                     = pre_selected 
                     | current`;

Hol_datatype `altitude_vals 
                     = away 
                     | near_pre_selected 
                     | at_pre_selected`;

(*---------------------------------------------------------------------------*
 * State-type projection and update functions.                               *
 *---------------------------------------------------------------------------*)

(*---------------------------------------------------------------------------*
 * Define state-type projection and update functions.                        *
 *---------------------------------------------------------------------------*)

Hol_datatype `states = <| att_cws  : off_eng;
                          cas_eng  : off_eng;
                          fpa_sel  : off_eng;
                          alt_eng  : mode_status;
                          alt_disp : disp_status;
                          fpa_disp : disp_status;
                          cas_disp : disp_status;
                          altitude : altitude_vals |>`;

(*---------------------------------------------------------------------------*
 * State predicates.                                                         *
 *---------------------------------------------------------------------------*)

val tran_att_cws_def = 
 Define 
     `tran_att_cws st =
         if st.att_cws = off 
         then st with
                 <| att_cws := engaged; 
                    fpa_sel := off; 
                    alt_eng := Mode off;
                    fpa_disp := current; 
                    alt_disp := current|>
         else st`;


val tran_cas_eng_def = 
 Define 
    `tran_cas_eng st =
        if st.cas_eng = off 
          then st with cas_eng := engaged
          else (st with <|cas_disp := current; cas_eng := off|>)`;


val tran_fpa_sel_def = 
 Define 
    `tran_fpa_sel st =
       if st.fpa_sel = off 
       then st with
               <| fpa_sel := engaged; 
                  att_cws := off; 
                  alt_eng := Mode off;
                 alt_disp := current|>
       else (st with
                <| fpa_sel := off; 
                  fpa_disp := current; 
                   att_cws := engaged;
                   alt_eng := Mode off; 
                   alt_disp := current|>)`;


val tran_alt_eng_def = 
 Define 
     `tran_alt_eng st =
         if (st.alt_eng = Mode off) /\ 
            (st.alt_disp = pre_selected) 
         then (if ~(st.altitude = away) 
               then st with
                       <| att_cws := off; 
                          fpa_sel := off; 
                          alt_eng := Mode engaged;
                         fpa_disp := current|>
               else (st with
                        <|att_cws := off; 
                          fpa_sel := engaged; 
                          alt_eng := armed|>))
         else st`;

val tran_input_alt_def = 
 Define 
     `tran_input_alt st =
         if st.alt_eng = Mode off 
         then st with alt_disp := pre_selected
         else if (st.alt_eng = armed) \/ (st.alt_eng = Mode engaged) 
              then st with
                    <|alt_eng := Mode off; 
                      alt_disp := pre_selected;
                      att_cws := engaged; 
                      fpa_sel := off; 
                      fpa_disp := current|>
              else st`;


val tran_input_fpa_def = 
 Define 
     `tran_input_fpa st =
         if st.fpa_sel = off 
         then st with fpa_disp := pre_selected 
         else st`;


val tran_input_cas_def = 
 Define 
     `tran_input_cas st =
         if st.cas_eng = off 
         then st with cas_disp := pre_selected 
         else st`;


val tran_alt_gets_near_def = 
 Define 
     `tran_alt_gets_near st =
         if st.alt_eng = armed 
         then st with
                <|altitude := near_pre_selected; 
                  alt_eng := Mode engaged;
                  fpa_sel := off; 
                  fpa_disp := current|>
         else
           (st with altitude := near_pre_selected)`;

val tran_alt_reached_def = 
 Define 
     `tran_alt_reached st =
        if st.alt_eng = armed 
        then st with
              <|altitude := at_pre_selected; 
                alt_disp := current;
                alt_eng := Mode engaged; 
                fpa_sel := off; 
                fpa_disp := current|>
        else (st with <|altitude := at_pre_selected; alt_disp := current|>)`;

val tran_fpa_reached_def = 
 Define 
     `tran_fpa_reached st = (st with fpa_disp := current)`;

val tran_defs = 
  [ tran_att_cws_def, tran_alt_eng_def, tran_fpa_sel_def, tran_cas_eng_def,
    tran_input_alt_def, tran_input_fpa_def, tran_input_cas_def,
    tran_alt_reached_def, tran_fpa_reached_def, tran_alt_gets_near_def];


(*---------------------------------------------------------------------------*
 * The transition function.                                                  *
 *---------------------------------------------------------------------------*)

val nextstate_def = Define
  `(nextstate st press_att_cws = tran_att_cws st)     /\
   (nextstate st press_alt_eng = tran_alt_eng st)     /\
   (nextstate st press_fpa_sel = tran_fpa_sel st)     /\
   (nextstate st press_cas_eng = tran_cas_eng st)     /\
   (nextstate st input_alt     = tran_input_alt st)   /\
   (nextstate st input_fpa     = tran_input_fpa st)   /\
   (nextstate st input_cas     = tran_input_cas st)   /\
   (nextstate st alt_reached   = tran_alt_reached st) /\
   (nextstate st fpa_reached   = tran_fpa_reached st) /\
   (nextstate st alt_gets_near = tran_alt_gets_near st)`;

(*---------------------------------------------------------------------------*
 * Initial state.                                                            *
 *---------------------------------------------------------------------------*)

val st0_def = 
 Define 
    `st0 = <|att_cws  := engaged;
             cas_eng  := off;
             fpa_sel  := off;
             alt_eng  := Mode off;
             alt_disp := current;
             fpa_disp := current;
             cas_disp := current;
             altitude := away|>`;


val is_initial_def = 
 Define 
     `is_initial st =
         (st.att_cws = engaged)  /\ 
         (st.cas_eng = off)      /\
         (st.fpa_sel = off)      /\
         (st.alt_eng = Mode off) /\ 
         (st.alt_disp = current) /\
         (st.fpa_disp = current) /\ 
         (st.cas_disp = current)`;

val valid_state_def = 
 Define 
     `valid_state st =
         ((st.att_cws = engaged) \/ (st.fpa_sel = engaged) \/ 
          (st.alt_eng = Mode engaged)) 
     /\  (~(st.alt_eng = Mode engaged) \/ ~(st.fpa_sel = engaged)) 
     /\  ((st.att_cws = engaged) 
           ==> ~(st.alt_eng = Mode engaged) /\ 
               ~(st.fpa_sel = engaged)) 
     /\  ((st.alt_eng = armed) ==> (st.fpa_sel = engaged))`;

(*

rep_engaged = rep_off_eng engaged
rep_att_cws = rep_off_eng o att_cws o abs_off_eng
rep_fpa_sel = rep_off_eng o fpa_sel o abs_off_eng
rep_alt_eng = rep_mode_status o alt_eng o abs_mode_status


val valid_state_def = Define 
  `valid_state st =
      ((rep_att_cws st = rep_engaged) 
      \/ (rep_fpa_sel st = rep_engaged) \/ (rep_alt_eng st = rep_Mode_engaged))
   /\ (~(rep_alt_eng st = rep_Mode_engaged) \/ ~(fpa_sel st = rep_engaged)) 
   /\ ((rep_att_cws st = rep_engaged)
   ==> ~(rep_alt_eng st = rep_Mode_engaged) /\ ~(rep_fpa_sel st = rep_engaged))
   /\ ((rep_alt_eng st = rep_armed) ==> (rep_fpa_sel st = rep_engaged))`;
*)

(*---------------------------------------------------------------------------*
 * Proofs. First we build the simplification set.                            *
 *---------------------------------------------------------------------------*)

val ap_ss = std_ss && [nextstate_def, valid_state_def, Mode];

val st0_initial = prove (Term`is_initial st0`,
 ZAP_TAC 
    (ap_ss && [is_initial_def,st0_def]) []);


val is_initial_valid_state = prove(Term`is_initial st ==> valid_state st`,
  ZAP_TAC 
    (ap_ss && [is_initial_def]) []);


val st0_valid_state = prove (Term`valid_state st0`,
  ZAP_TAC 
    (ap_ss && [is_initial_valid_state,st0_initial]) []);

(*---------------------------------------------------------------------------*
 * nextstate preserves valid_stateness.                                      *
 * It takes approx. 16 seconds of runtime on 80Meg Pentium 133Mhz.           *
 * Memory consumption is steady at about 9 Mbyte.                            *
 *---------------------------------------------------------------------------*)

val nextstate_valid_state = 
Count.apply prove  
  (Term`!event. valid_state st ==> valid_state (nextstate st event)`,
   Induct 
     THEN ZAP_TAC (ap_ss && tran_defs) []);

(*---------------------------------------------------------------------------*
 * Reachability. This could also be given as an inductive definition.        *
 *---------------------------------------------------------------------------*)

val reachable_in_def = Define
  `(reachable_in 0 st = is_initial st) /\
   (reachable_in (SUC n) st =
     ?pst ev. (st = nextstate pst ev) /\ reachable_in n pst)`;

(*---------------------------------------------------------------------------*
 * Every reachable state is valid_state.                                     *
 *---------------------------------------------------------------------------*)

val reachable_valid_state = prove
 (Term`!n st. reachable_in n st ==> valid_state st`,
  Induct THEN 
   ZAP_TAC (std_ss && [reachable_in_def])
     [is_initial_valid_state,nextstate_valid_state]);


(*---------------------------------------------------------------------------*
 * A state is reachable if it is reachable in a finite number of steps.      *
 *---------------------------------------------------------------------------*)

val is_reachable_def = Define 
  `is_reachable st = ?n. reachable_in n st`;

val is_reachable_valid_state = prove
(Term`!st. is_reachable st ==> valid_state st`,
  PROVE_TAC[is_reachable_def,reachable_valid_state]);

(*---------------------------------------------------------------------------*
 * A couple of safety properties.                                            *
 *---------------------------------------------------------------------------*)

val safety1 = prove
(Term`!event st. 
       valid_state st 
       /\ (fpa_sel st = engaged) 
       /\ (fpa_sel (nextstate st event) = off)
         ==> 
          (fpa_disp (nextstate st event) = current)`,
Induct 
  THEN ZAP_TAC (ap_ss && tran_defs)  (tl (type_rws "off_eng")));

val safety2 = prove
(Term`!event st. 
       valid_state st 
       /\ (alt_eng st = Mode_engaged)
       /\ ~(event = input_alt)
       /\ (alt_eng(nextstate st event) = Mode_off)
         ==>
          (alt_disp (nextstate st event) = current)`,
Induct 
  THEN ZAP_TAC (ap_ss && tran_defs) 
             (tl(type_rws "off_eng") @ tl(type_rws "mode_status")));

val _ = Count.report (Count.read meter);

(*---------------------------------------------------------------------------*
 * What now probably ought to be proved is some sort of induction theorem.   *
 * Tobias Nipkow and I proved such a thing for IOA (see our paper). The      *
 * analogue for this system would be:                                        *
 *                                                                           *
 * !P. (!st. is_initial st ==> P st) /\                                      *
 *    (!st st' e. reachable st /\ P st /\ (st' = nextstate st e) ==> P st')  *
 *      ==>                                                                  *
 *      !st. reachable st ==> P st                                           *
 *                                                                           *
 *---------------------------------------------------------------------------*)

(*****************************************************************************)
(* Start of BDD analysis                                                     *)
(*****************************************************************************)

val {abs_spec  = abs_events_def,
     rep_spec  = rep_events_def,
     range_def = range_events_def} = define_rep(theorem "events_Axiom");

val rep_events_forall = MATCH_MP FORALL_REP abs_events_def;
val abs_events_one_one = MATCH_MP ABS_ONE_ONE abs_events_def;

val {abs_spec  = abs_off_eng_def,
     rep_spec  = rep_off_eng_def,
     range_def = range_off_eng_def} = define_rep(theorem "off_eng_Axiom");

val rep_off_eng_forall = MATCH_MP FORALL_REP abs_off_eng_def;
val abs_off_eng_one_one = MATCH_MP ABS_ONE_ONE abs_off_eng_def;

val {abs_spec  = abs_mode_status_def,
     rep_spec  = rep_mode_status_def,
     range_def = range_mode_status_def} = 
    define_rep(theorem "mode_status_Axiom");

val rep_mode_status_forall = MATCH_MP FORALL_REP abs_mode_status_def;
val abs_mode_status_one_one = MATCH_MP ABS_ONE_ONE abs_mode_status_def;

val {abs_spec  = abs_disp_status_def,
     rep_spec  = rep_disp_status_def,
     range_def = range_disp_status_def} = 
    define_rep(theorem "disp_status_Axiom");

val rep_disp_status_forall = MATCH_MP FORALL_REP abs_disp_status_def;
val abs_disp_status_one_one = MATCH_MP ABS_ONE_ONE abs_disp_status_def;

val {abs_spec  = abs_altitude_vals_def,
     rep_spec  = rep_altitude_vals_def,
     range_def = range_altitude_vals_def} = 
    define_rep(theorem "altitude_vals_Axiom");

val rep_altitude_vals_forall = MATCH_MP FORALL_REP abs_altitude_vals_def;
val abs_altitude_vals_one_one = MATCH_MP ABS_ONE_ONE abs_altitude_vals_def;

val abs_states_aux_def = Define
 `abs_states(o0r, o1r, o2r, mr, d0r, d1r, d2r, ar) =
   states (abs_off_eng o0r)
          (abs_off_eng o1r)
          (abs_off_eng o2r)
          (abs_mode_status mr)
          (abs_disp_status d0r)
          (abs_disp_status d1r)
          (abs_disp_status d2r)
          (abs_altitude_vals ar)`;

val rep_states_aux_def = Define
 `rep_states_aux o0 o1 o2 m d0 d1 d2 a =
   (rep_off_eng o0,
    rep_off_eng o1,
    rep_off_eng o2,
    rep_mode_status m,
    rep_disp_status d0,
    rep_disp_status d1,
    rep_disp_status d2,
    rep_altitude_vals a)`;

val range_states_def = Define
 `range_states(o0r, o1r, o2r, mr, d0r, d1r, d2r, ar) =
    (range_off_eng o0r)       /\
    (range_off_eng o1r)       /\
    (range_off_eng o2r)       /\
    (range_mode_status mr)    /\
    (range_disp_status d0r)   /\
    (range_disp_status d1r)   /\
    (range_disp_status d2r)   /\
    (range_altitude_vals ar)`;

local val th1 = ISPEC ``rep_states_aux`` (theorem "states_Axiom")
      val th2 = CONJUNCT1
                 (Ho_rewrite.PURE_REWRITE_RULE 
                   [Ho_theorems.EXISTS_UNIQUE_THM] 
                   th1)
in
val rep_states_def = 
 REWRITE_RULE
  [rep_states_aux_def]
  (Rsyntax.new_specification
    {name    = "rep_states_def", 
     sat_thm = th2,
     consts  = [{const_name = "rep_states", fixity = Prefix}]})
end;


val v0  = ``v0:bool``  and v0'  = ``v0':bool``
and v1  = ``v1:bool``  and v1'  = ``v1':bool``
and v2  = ``v2:bool``  and v2'  = ``v2':bool``
and v30 = ``v30:bool`` and v30' = ``v30':bool``
and v31 = ``v31:bool`` and v31' = ``v31':bool``
and v4  = ``v4:bool``  and v4'  = ``v4':bool``
and v5  = ``v5:bool``  and v5'  = ``v5':bool``
and v6  = ``v6:bool``  and v6'  = ``v6':bool``
and v70 = ``v70:bool`` and v70' = ``v70':bool``
and v71 = ``v71:bool`` and v71' = ``v71':bool``;

val st =
 ``(^v0,               (* ("att_cws",  Type`:off_eng`)       *)
    ^v1,               (* ("cas_eng",  Type`:off_eng`)       *)
    ^v2,               (* ("fpa_sel",  Type`:off_eng`)       *)
    (^v30,^v31),       (* ("alt_eng",  Type`:mode_status`)   *)
    ^v4,               (* ("alt_disp", Type`:disp_status`)   *)
    ^v5,               (* ("fpa_disp", Type`:disp_status`)   *)
    ^v6,               (* ("cas_disp", Type`:disp_status`)   *)
    (^v70,^v71))       (* ("altitude", Type`:altitude_vals`) *)``;

val st' =
 ``(^v0',              (* ("att_cws",  Type`:off_eng`)       *)
    ^v1',              (* ("cas_eng",  Type`:off_eng`)       *)
    ^v2',              (* ("fpa_sel",  Type`:off_eng`)       *)
    (^v30',^v31'),     (* ("alt_eng",  Type`:mode_status`)   *)
    ^v4',              (* ("alt_disp", Type`:disp_status`)   *)
    ^v5',              (* ("fpa_disp", Type`:disp_status`)   *)
    ^v6',              (* ("cas_disp", Type`:disp_status`)   *)
    (^v70',^v71'))     (* ("altitude", Type`:altitude_vals`) *)``;

val ev = ``(e0, e1, e2, e3):bool#bool#bool#bool``;;


val abs_rep_range_thms =
 [abs_events_def,abs_off_eng_def,abs_mode_status_def,
  abs_disp_status_def,abs_altitude_vals_def,
  rep_events_def,rep_off_eng_def,rep_mode_status_def,
  rep_disp_status_def,rep_altitude_vals_def,
  range_events_def,range_off_eng_def,range_mode_status_def,
  range_disp_status_def,range_altitude_vals_def];

val abs_thms =
 map
  (GSYM o CONJUNCT2)
  [abs_events_def,abs_off_eng_def,abs_mode_status_def,
   abs_disp_status_def,abs_altitude_vals_def];

val abs_states_def = 
 prove
  (``(!a. abs_states (rep_states a) = a)
     /\ 
     (!r. range_states r = rep_states (abs_states r) = r)``,
   CONJ_TAC
    THENL
     [INDUCT_THEN (theorem"states_induction") ASSUME_TAC
       THEN REWRITE_TAC[abs_states_aux_def,rep_states_def,range_states_def]
       THEN REWRITE_TAC abs_rep_range_thms,
      PGEN_TAC st
       THEN REWRITE_TAC[abs_states_aux_def,rep_states_def,range_states_def]
       THEN REWRITE_TAC (PAIR_EQ::abs_thms)]);

val rep_states_forall = MATCH_MP FORALL_REP abs_states_def;
val abs_states_one_one = MATCH_MP ABS_ONE_ONE abs_states_def;

(* Link with BDD fixedpoint *)

val trans_def =
 Def ``trans (st,st') = ?ev. st' = nextstate st ev``;

val rep_trans_def =
 Def ``rep_trans (rst,rst') = 
        range_states rst /\ range_states rst' /\ 
        trans(abs_states rst, abs_states rst')``;

val trans_rep_trans =
 prove
  (``rep_trans(rep_states st, rep_states st') = trans(st,st')``,
   PROVE_TAC[rep_trans_def,rep_states_def,abs_states_def,range_states_def]);

val rep_is_initial_def =
 Def ``rep_is_initial rst = 
        range_states rst /\ is_initial(abs_states rst)``;

val is_initial_rep_is_initial =
 prove
  (``rep_is_initial(rep_states st) = is_initial st``,
   PROVE_TAC[rep_is_initial_def,rep_states_def,abs_states_def,range_states_def]);

val rep_is_reachable_def =
 Def ``rep_is_reachable rst = 
        range_states rst /\ is_reachable(abs_states rst)``;

val is_reachable_rep_is_reachable =
 prove
  (``rep_is_reachable(rep_states st) = is_reachable st``,
   PROVE_TAC
    [rep_is_reachable_def,rep_states_def,abs_states_def,range_states_def]);

val rep_reachable_in_def =
 Def ``rep_reachable_in n rst = 
        range_states rst /\ reachable_in n (abs_states rst)``;

val reachable_in_rep_reachable_in =
 prove
  (``rep_reachable_in n (rep_states st) = reachable_in n st``,
   PROVE_TAC[rep_reachable_in_def,rep_states_def,abs_states_def,range_states_def]);

val rep_valid_state_def =
 Def ``rep_valid_state rst = 
        range_states rst /\ valid_state(abs_states rst)``;

val valid_state_rep_valid_state =
 prove
  (``rep_valid_state(rep_states st) = valid_state st``,
   PROVE_TAC
    [rep_valid_state_def,rep_states_def,abs_states_def,range_states_def]);


(* Not needed
val reachable_in_trans =
 prove
  (``!n st. reachable_in n st = ITER n trans is_initial st``,
   Induct
    THENL
     [PROVE_TAC[reachable_in_def,ITER,is_initial_def],
      REPEAT STRIP_TAC
       THEN ASM_REWRITE_TAC
             [reachable_in_def,ITER,NEXT,trans_def]
       THEN EQ_TAC
       THEN REPEAT STRIP_TAC
       THEN ASM_REWRITE_TAC[]
       THEN PROVE_TAC[]]);
*)

val rep_reachable_in_trans =
 prove
  (``!n st. reachable_in n st = 
             ITER n rep_trans rep_is_initial (rep_states st)``,
   ONCE_REWRITE_TAC[GSYM reachable_in_rep_reachable_in]
    THEN REWRITE_TAC[rep_reachable_in_def]
    THEN Induct
    THENL
     [PROVE_TAC[reachable_in_def,ITER,rep_is_initial_def],
      REPEAT STRIP_TAC
       THEN REWRITE_TAC
             [reachable_in_def,ITER,NEXT,trans_def,rep_trans_def]
       THEN EQ_TAC
       THEN REPEAT STRIP_TAC
       THEN ASM_REWRITE_TAC[]
       THEN PROVE_TAC[abs_states_def]]);

(* Link with BDD fixedpoint *)

val is_reachable_REACHABLE =
 prove
  (``!st. is_reachable st = 
           REACHABLE rep_trans rep_is_initial (rep_states st)``,
   PROVE_TAC[is_reachable_def,rep_reachable_in_trans,REACHABLE]);

val rep_is_reachable_REACHABLE =
 REWRITE_RULE
  [SPEC ``abs_states ^st`` is_reachable_REACHABLE]
  (SPEC st rep_is_reachable_def);

val rep_events_exists = MATCH_MP EXISTS_REP abs_events_def;

val nextstate_cond =
 prove
  (``!ev st.
       nextstate st ev =
        ((ev = press_att_cws) => tran_att_cws st
        |(ev = press_alt_eng) => tran_alt_eng st
        |(ev = press_fpa_sel) => tran_fpa_sel st
        |(ev = press_cas_eng) => tran_cas_eng st
        |(ev = input_alt)     => tran_input_alt st
        |(ev = input_fpa)     => tran_input_fpa st
        |(ev = input_cas)     => tran_input_cas st
        |(ev = alt_reached)   => tran_alt_reached st
        |(ev = fpa_reached)   => tran_fpa_reached st
        | tran_alt_gets_near st)``,
   Induct THEN RW_TAC ap_ss []);

val events_exists =
 prove
  (``(?ev. P ev) =
     (P press_att_cws) \/
     (P press_cas_eng) \/
     (P press_alt_eng) \/
     (P press_fpa_sel) \/
     (P input_alt) \/
     (P input_fpa) \/
     (P input_cas) \/
     (P alt_reached) \/
     (P fpa_reached) \/
     (P alt_gets_near)``,
   EQ_TAC
    THENL[REWRITE_TAC[EXISTS_IMP_LEMMA] THEN Induct,ALL_TAC]
    THEN ZAP_TAC ap_ss []);

val abs_one_one_thms =
 [abs_events_one_one,abs_off_eng_one_one,abs_mode_status_one_one,
  abs_disp_status_one_one,abs_altitude_vals_one_one,abs_states_one_one];

val range_thms =
 [range_events_def,range_off_eng_def,range_mode_status_def,
  range_disp_status_def,range_altitude_vals_def,range_states_def];

use "bespoke.ml";

fun REP_REWRITE spec_list def rep_def =
 let val th1 = REWRITE_RULE
                [events_exists,nextstate_def,
                 tran_att_cws_def,tran_cas_eng_def,tran_alt_eng_def,
                 tran_fpa_sel_def,
                 tran_input_alt_def,tran_input_fpa_def,tran_input_cas_def,
                 tran_alt_reached_def,tran_fpa_reached_def,
                 tran_alt_gets_near_def]
                (SPECL (map fst spec_list) def)
     val th2 = REWRITE_RULE[theorem"states_accessors"]th1
     val th3 = REWRITE_RULE[theorem"states_updates"]th2
(*     val th4 = REWRITE_RULE[EQ_COND]th3 *)
     val th4 = RIGHT_CONV_RULE(REWRITE_CONV[EQ_COND])th3 
     val th5 = REWRITE_RULE[theorem"states_one_one"]th4
     val th6 = REWRITE_RULE
                [PROVE_ABS_THMS abs_events_def rep_events_def,
                 PROVE_ABS_THMS abs_off_eng_def rep_off_eng_def,
                 PROVE_ABS_THMS abs_mode_status_def rep_mode_status_def,
                 PROVE_ABS_THMS abs_disp_status_def rep_disp_status_def,
                 PROVE_ABS_THMS abs_altitude_vals_def rep_altitude_vals_def]
                th5
     val th7 = REWRITE_RULE[GSYM abs_states_aux_def]th6
     val th8 = REWRITE_RULE
                [th7,range_states_def]
                (SPECL (map snd spec_list) rep_def)
     val th9 = RIGHT_CONV_RULE 
                (bespoke
                 (pairTheory.CLOSED_PAIR_EQ::(range_thms@abs_one_one_thms)))
                th8
 in
  th9
 end;

val abs_st = 
 ``states (abs_off_eng v0) (abs_off_eng v1) (abs_off_eng v2)
          (abs_mode_status(v30,v31)) (abs_disp_status v4) 
          (abs_disp_status v5) (abs_disp_status v6) 
          (abs_altitude_vals(v70,v71))``;;

val abs_st' =
 ``states (abs_off_eng v0') (abs_off_eng v1') (abs_off_eng v2')
          (abs_mode_status(v30',v31')) (abs_disp_status v4') 
          (abs_disp_status v5') (abs_disp_status v6') 
          (abs_altitude_vals(v70',v71'))``;;

val trans_rep_eqn = 
 REP_REWRITE [(abs_st,st),(abs_st',st')] trans_def rep_trans_def;

val is_initial_rep_eqn = 
 REP_REWRITE [(abs_st,st)] is_initial_def rep_is_initial_def;

val valid_state_rep_eqn = 
 REP_REWRITE [(abs_st,st)] valid_state_def rep_valid_state_def;

(* fail; *)

time addEquation trans_rep_eqn;

time addEquation is_initial_rep_eqn;

time MAKE_REACHABLE_BDD1(trans_rep_eqn, is_initial_rep_eqn);

map (addEquation o REWRITE_RULE[FORALL_PROD]) range_thms;

addEquation
 (REWRITE_RULE
   [GSYM(CONJUNCT2 abs_states_def)]
   (RIGHT_CONV_RULE 
    (bespoke[CONJUNCT2 abs_states_def])
    rep_is_reachable_REACHABLE));

val rep_is_reachable_simp = bdd_to_cond(termToBdd ``rep_is_reachable ^st``);

val COND_SIMP =
 DECIDE `((b => F | F) = F)  /\
         ((b => F | T) = ~b) /\
         ((b => T | F) = b)  /\
         ((b => T | T) = T)`;

val simplified_rep_is_reachable_simp =
 rhs(concl(REWRITE_CONV[COND_SIMP]rep_is_reachable_simp));

val rep_is_reachable_lemma =
 prove
  (``rep_is_reachable ^st = ^simplified_rep_is_reachable_simp``,
   BDD_TAC);

val bdd_is_reachable_def =
 Def  
  ``bdd_is_reachable st =
     ((att_cws st = engaged)
       => ((fpa_sel st = off) /\ (alt_eng st = Mode_off))
       | ((fpa_sel st = engaged)
          => (alt_eng st = Mode_off) \/
             ((alt_eng st = armed) /\
              (alt_disp st = pre_selected) /\ (altitude st = away))
          | (alt_eng st = Mode_engaged) /\
            ((altitude st = near_pre_selected)\/
             (altitude st = at_pre_selected))))``;

val rep_bdd_is_reachable_def =
 Def
  ``rep_bdd_is_reachable rst = 
     range_states rst /\ bdd_is_reachable(abs_states rst)``;

val bdd_is_reachable_eqn = 
 REP_REWRITE [(abs_st,st)] bdd_is_reachable_def rep_bdd_is_reachable_def;

addEquation bdd_is_reachable_eqn;

val rep_bdd_is_reachable_thm =
 prove
  (``rep_is_reachable ^st =  rep_bdd_is_reachable ^st``,
   BDD_TAC);

val is_reachable_thm =
 REWRITE_RULE
  [abs_states_def]
  (SPEC 
   ``rep_states st``
   (PGEN ``st:^(type_of st)`` st
    (REWRITE_RULE
      [rep_bdd_is_reachable_def,bdd_is_reachable_def,rep_is_reachable_def]
       rep_bdd_is_reachable_thm)));

addEquation valid_state_rep_eqn;

tautCheck ``rep_is_reachable ^st ==> rep_valid_state ^st``;
mk_bdd_thm ``rep_is_reachable ^st ==> rep_valid_state ^st``;

tautCheck ``rep_valid_state ^st ==> rep_is_reachable ^st``;

val counterex = 
 find_refutation ``rep_valid_state ^st ==> rep_is_reachable ^st``;

(*
val Counterex_def =
 Def ``Counterex ^st = ~(rep_valid_state ^st ==> rep_is_reachable ^st)``;

FIND_REFUTATION_TRACE(trans_rep_eqn, is_initial_rep_eqn,Counterex_def);

Need to fix termToBdd so that

  termToBdd ``Conjecture (T, F, F, (F, T), T, T, T, F, F)``;

works, given a BDD for

  ``Conjecture (v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71)``

*)

addEquation (GSYM(SPEC st rep_is_reachable_def));
addEquation (GSYM(SPEC st rep_valid_state_def));


val CONJ_IMP = DECIDE `(A = B /\ C) ==> (B ==> (C = A))`;

val rep_flip_lemmas =
 [MATCH_MP CONJ_IMP (SPEC_ALL rep_valid_state_def),
  MATCH_MP CONJ_IMP (SPEC_ALL rep_is_reachable_def)];

val counterex_thm =
 prove
  (``!st.
      (altitude st = away)        /\
      (alt_eng st = Mode_engaged) /\
      (fpa_sel st = off)          /\
      (att_cws st = off)
      ==>
      (valid_state st /\ ~is_reachable st)``,
   REWRITE_TAC[rep_states_forall]
    THEN CONV_TAC(bespoke(pairTheory.CLOSED_PAIR_EQ::rep_flip_lemmas))
    THEN REWRITE_TAC
          [rep_states_forall,rep_events_forall,rep_off_eng_forall,
           rep_mode_status_forall,rep_disp_status_forall,
           rep_altitude_vals_forall,FORALL_PROD,abs_states_aux_def,
           theorem"states_accessors"]
    THEN REWRITE_TAC
          [theorem"states_accessors",
           rep_states_forall,rep_events_forall,rep_off_eng_forall,
           rep_mode_status_forall,rep_disp_status_forall,
           rep_altitude_vals_forall,
           valid_state_def,is_reachable_thm]
    THEN REWRITE_TAC
          [PROVE_ABS_THMS abs_events_def rep_events_def,
           PROVE_ABS_THMS abs_off_eng_def rep_off_eng_def,
           PROVE_ABS_THMS abs_mode_status_def rep_mode_status_def,
           PROVE_ABS_THMS abs_disp_status_def rep_disp_status_def,
           PROVE_ABS_THMS abs_altitude_vals_def rep_altitude_vals_def]
    THEN REWRITE_TAC[range_states_def]
    THEN REPEAT GEN_TAC
    THEN CONV_TAC
          (bespoke(pairTheory.CLOSED_PAIR_EQ::(range_thms@abs_one_one_thms)))
    THEN BDD_TAC);
 
val range_states_eqn =
 REWRITE_RULE
  [FORALL_PROD,PAIR_EQ,
   range_off_eng_def,range_mode_status_def,
   range_disp_status_def,range_altitude_vals_def]
  range_states_def;

addEquation range_states_eqn;

val valid_and_not_reachable_simp = 
 bdd_to_cond
  (termToBdd ``range_states ^st /\ rep_valid_state ^st /\ ~rep_is_reachable ^st``);

val simplified_valid_and_not_reach_simp =
 rhs(concl(REWRITE_CONV[COND_SIMP]valid_and_not_reachable_simp));

(* 


(att_cws st = off)
/\
((fpa_sel st = engaged) 
 => (alt_eng = armed) 
    /\ 
    ((alt_disp st = current) => (
 | 

*)

(* Not needed
val abs_events_thms =
 prove
  (``(press_att_cws = abs_events(F, F, F, F))  /\ 
     (press_cas_eng = abs_events(F, F, F, T))  /\ 
     (press_alt_eng = abs_events(F, F, T, F))  /\ 
     (press_fpa_sel = abs_events(F, F, T, T))  /\ 
     (input_alt = abs_events(F, T, F, F))      /\ 
     (input_fpa = abs_events(F, T, F, T))      /\ 
     (input_cas = abs_events(F, T, T, F))      /\ 
     (alt_reached = abs_events(F, T, T, T))    /\ 
     (fpa_reached = abs_events(T, F, F, F))    /\ 
     (alt_gets_near = abs_events(T, F, F, T))``,
   PROVE_TAC[rep_events_def,abs_events_def]);
*)


val is_reachable_valid_state_lemma =
 prove
  (``(!s. is_reachable s ==> valid_state s) = 
     (!r. range_states r ==> rep_is_reachable r ==> rep_valid_state r)``,
   PROVE_TAC[rep_states_forall, abs_states_def,
             SYM is_reachable_rep_is_reachable,
             SYM valid_state_rep_valid_state]);

val is_reachable_valid_state_bdd_thm =
 prove
  (``!s. is_reachable s ==> valid_state s``,
   REWRITE_TAC[is_reachable_valid_state_lemma,FORALL_PROD]
    THEN BDD_TAC);

(*

Encoding in bool^n

val is_reachable_valid_state_lemma =
 prove
  (``(!s. is_reachable s ==> valid_state s) = 
     (!r. range_states r ==> rep_is_reachable r ==> rep_valid_state r)``,
   PROVE_TAC[rep_states_forall, abs_states_def,
             SYM is_reachable_rep_is_reachable,
             SYM valid_state_rep_valid_state]);

Meson search level: ........................
runtime: 0.930s,    gctime: 0.050s,     systime: 0.000s.
> val is_reachable_valid_state_lemma =
    [oracles: #] [axioms: ] []
    |- (!s. is_reachable s ==> valid_state s)
       = (!r. range_states r ==> rep_is_reachable r ==> rep_valid_state r)
    : Thm.thm

Computing BDDs:


- initHolBdd[];
> val it = () : unit
- time addEquation trans_rep_eqn;
runtime: 0.170s,    gctime: 0.000s,     systime: 0.000s.
> val it =
    (``rep_trans
         ((v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71), v0', v1', v2'
          , (v30', v31'), v4', v5', v6', v70', v71')``, <robdd>)
    : Term.term * Robdd.robdd
- time addEquation is_initial_rep_eqn;
runtime: 0.000s,    gctime: 0.000s,     systime: 0.000s.
> val it =
    (``rep_is_initial (v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71)``,
     <robdd>)
    : Term.term * Robdd.robdd
- time MAKE_REACHABLE_BDD1(trans_rep_eqn, is_initial_rep_eqn);
........runtime: 0.830s,    gctime: 0.010s,     systime: 0.000s.
> val it =
    {Thm =
       [oracles: ##] [axioms: ] []
       |- REACHABLE rep_trans rep_is_initial
            (v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71)
          = CUMULATE_ITER 7 rep_trans rep_is_initial
              (v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71), iterations = 7}
    : {Thm : Thm.thm, iterations : int}
- 


Proving theorem using BDDs

- time (map (addEquation o REWRITE_RULE[FORALL_PROD])) range_thms;
runtime: 0.130s,    gctime: 0.010s,     systime: 0.000s.
> val it =
    [(``range_events (p_1, p_1', p_1'', p_2)``, <robdd>),
     (``range_off_eng x``, <robdd>),
     (``range_mode_status (p_1, p_2)``, <robdd>),
     (``range_disp_status x``, <robdd>),
     (``range_altitude_vals (p_1, p_2)``, <robdd>),
     (``range_states (o0r, o1r, o2r, (p_1, p_2), d0r, d1r, d2r, p_1', p_2')``,
      <robdd>)]
    : (Term.term * Robdd.robdd) list


- time addEquation
 (REWRITE_RULE
   [GSYM(CONJUNCT2 abs_states_def)]
   (RIGHT_CONV_RULE 
    (bespoke[CONJUNCT2 abs_states_def])
    rep_is_reachable_REACHABLE));

runtime: 0.010s,    gctime: 0.000s,     systime: 0.000s.
> val it =
    (``rep_is_reachable (v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71)``,
     <robdd>)
    : Term.term * Robdd.robdd

- time addEquation valid_state_rep_eqn;
runtime: 0.010s,    gctime: 0.000s,     systime: 0.000s.
> val it =
    (``rep_valid_state (v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71)``,
     <robdd>)
    : Term.term * Robdd.robdd

- val is_reachable_valid_state_bdd_thm =
 prove
  (``!s. is_reachable s ==> valid_state s``,
   REWRITE_TAC[is_reachable_valid_state_lemma,FORALL_PROD]
    THEN BDD_TAC);

runtime: 0.080s,    gctime: 0.030s,     systime: 0.000s.
> val is_reachable_valid_state_bdd_thm =
    [oracles: ##] [axioms: ] [] |- !s. is_reachable s ==> valid_state s
    : Thm.thm

BDD map

-show_bdd_map();
> val it =
    [(``REACHABLE rep_trans rep_is_initial
          (v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71)``, <robdd>),
     (``range_mode_status (p_1, p_2)``, <robdd>),
     (``CUMULATE_ITER 7 rep_trans rep_is_initial
          (v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71)``, <robdd>),
     (``CUMULATE_ITER 8 rep_trans rep_is_initial
          (v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71)``, <robdd>),
     (``range_events (p_1, p_1', p_1'', p_2)``, <robdd>),
     (``rep_is_initial (v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71)``,
      <robdd>),
     (``range_states (o0r, o1r, o2r, (p_1, p_2), d0r, d1r, d2r, p_1', p_2')``,
      <robdd>),
     (``rep_is_reachable (v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71)``,
      <robdd>),
     (``rep_valid_state (v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71)``,
      <robdd>), (``range_disp_status x``, <robdd>),
     (``range_altitude_vals (p_1, p_2)``, <robdd>),
     (``range_off_eng x``, <robdd>),
     (``rep_trans
          ((v0, v1, v2, (v30, v31), v4, v5, v6, v70, v71), v0', v1', v2'
           , (v30', v31'), v4', v5', v6', v70', v71')``, <robdd>)]
    : (Term.term * Robdd.robdd) list
- 
*)


(*
Loading time with old definitions
[closing file "/tmp/03694aaa.hol98"]
runtime: 394.760s,    gctime: 119.720s,     systime: 0.650s.
*)
