(* ------------------------------------------------------------------------- *
 *     Automatic generation of clickable image maps of the                   *
 *     HOL theory hierarchy.                                                 *
 *                                                                           *
 * The algorithm works as follows:                                           *
 *                                                                           *
 *   1. Generate the dependencies from the current theory hierarchy          *
 *   2. Write them to a file ready for processing by "dot"                   *
 *   3. Invoke dot once to generate the "gif"-format image                   *
 *   4. Invoke dot again to generate the theory coodinates                   *
 *   5. Parse the results of (4) and generate the desired html file.         *
 *   6. Generate the html files for each segment in the current theory       *
 *      hierarchy.                                                           *
 *                                                                           *
 * Before loading this file, you might need to tell "dot" where the          *
 * TrueType fonts that it needs are. On my system, this is:                  *
 *                                                                           *
 *     DOTFONTPATH=/dosc/windows/fonts; export DOTFONTPATH                   *
 *                                                                           *
 * ------------------------------------------------------------------------- *)

app load ["Substring","Int","FileSys", "Process"];

val HTML_DIR   = Path.concat(Globals.HOLDIR,"html");
val SIGOBJ_DIR = Path.concat(Globals.HOLDIR,"sigobj");
(* val BASE_URL   = "http://www.cl.cam.ac.uk/ftp/hvg/hol98/HTML"; *)

(*---------------------------------------------------------------------------
     Extract dot-friendly digraph from HOL theory graph.
 ---------------------------------------------------------------------------*)

fun node thy = map (fn p => (p,thy)) (parents thy);

local fun leq (s:string,_) (t,_) = s <= t
in
fun ancestor_nodes thy = 
  let val thys = ancestry thy
      val pairs = sort leq (flatten (map node thys))
  in (thys, pairs)
  end
end;

fun pp_dot_file {size,ranksep,nodesep} (dom,pairs) ppstrm =
 let open Portable 
     val {add_string,add_break,begin_block,end_block,
          add_newline,flush_ppstream,...} = with_ppstream ppstrm
     fun pp_thy s =
       (begin_block CONSISTENT 2;
        add_string s; add_break (1,0);
        add_string "[URL ="; add_break(1,0); 
        add_string (quote (s^"Theory.html"));
        add_string "]";
        end_block())
     fun pp_pair (x,y) =
       (begin_block CONSISTENT 0;
        add_string x;
        add_string " -> ";
        add_string y;
        end_block())
 in
  begin_block CONSISTENT 0;
  begin_block CONSISTENT 5;
  add_string "digraph G {";
  add_break (1,0);
  add_string "size = ";    add_string (quote size); add_break(1,0);
  add_string "ranksep = "; add_string ranksep;      add_break(1,0);
  add_string "nodesep = "; add_string nodesep;      add_break(1,0);
  add_string "node [fontcolor = darkgreen fontsize=20]"; add_break(1,0);
  add_newline();
  begin_block CONSISTENT 0;
   pr_list pp_thy (fn () => ()) add_newline dom;
  end_block();
  add_newline(); add_newline();
  begin_block CONSISTENT 0;
   pr_list pp_pair (fn () => ()) add_newline pairs;
  end_block();
  end_block();
  add_newline(); add_string "}"; add_newline();
  end_block()
 end;

fun gen_dotfile file node_info =
 let open TextIO
     val ostrm = openOut file
 in  PP.with_pp {consumer=fn s => output(ostrm,s),linewidth=75,
                 flush=fn () => flushOut ostrm}
        (pp_dot_file {size="8.5,11",ranksep="0.4",nodesep="0.20"} node_info);
     closeOut ostrm
 end;

(*---------------------------------------------------------------------------
       Parse theory coordinates generated by dot -Tismap. Note that
       these come (i,j) (p,q), but it seems that they should be 
       transformed to (i,q) (p,j) ... and so that's what we do!
 ---------------------------------------------------------------------------*)

val comma_sp = ", " and qdelim = "\"";

val splitter = 
  let val test = Char.contains "(,.)"
  in fn c => Char.isSpace c orelse test c
  end;

fun translate s = 
  case String.tokens splitter s
   of "rectangle"::n1::n2::n3::n4::nm::_ => String.concat
        ["<AREA SHAPE=\"RECT\" COORDS= ",
         qdelim, n1,comma_sp, n4,comma_sp, n3,comma_sp, n2,qdelim,
         " HREF=", qdelim, nm,".html",qdelim, ">"]
    | other => raise Fail 
      "wanted : rectangle (<num>,<num>) (<num>,<num>) <str>.html";

fun parse_ismap_file file = 
  let val strm = TextIO.openIn file
      fun loop() =
         case TextIO.inputLine strm 
          of "" => []
           | line => translate line :: loop()
  in 
       rev (loop()) before TextIO.closeIn strm
  end;


fun gen_map_file dot node_info name =
 let open TextIO
     val dotfile   = name^".dot"
     val giffile   = name^".gif"
     val ismapfile = name^".ismap"
     val mapfile   = name^".html"
     val gifurl    = name^".gif"
     val  _        = gen_dotfile dotfile node_info
     val cmd1 = String.concat[dot," -Tgif ",  dotfile, " > ", giffile]
     val cmd2 = String.concat[dot," -Tismap ",dotfile, " > ", ismapfile]
 in if Process.system cmd1 = Process.success andalso
       Process.system cmd2 = Process.success
    then let val areas = parse_ismap_file ismapfile
             val ostrm = openOut mapfile
             fun out s = output(ostrm, s^"\n")
         in
             out "<HTML>";
             out "<HEAD><TITLE>HOL Theory Hierarchy</TITLE></HEAD>";
             out "<BODY bgcolor=linen text=crimson>";
             out "<H1>HOL Theory Hierarchy</H1>";
             out "<MAP NAME = \"HOL_theories\">";
             map out areas;
             out "</MAP>";
             out (String.concat["<a href= \"Default.html\"> ",
                  "<img src=\"",gifurl,"\" ",
                  "usemap=\"#HOL_theories\" ",
                  "border=\"0\"> </a>"]);
             out "</BODY>";
             out "</HTML>";
             closeOut ostrm
         end
    else raise Fail "gen_map_file: failed "
 end;

fun dir_theories dir = 
 let open Substring FileSys
     val dstrm = openDir dir
     fun thys () =
       case readDir dstrm
        of NONE => []
         | SOME file => 
            let val (ss1,ss2) = position "Theory.sig" (all file)
            in if isEmpty ss1 orelse isEmpty ss2
               then thys()
               else string ss1 :: thys()
            end
 in thys()
 end;


fun load_theories thyl = app (load o C concat "Theory") thyl;

fun isThm (_,(_,DB.Thm)) = true | isThm _ = false;
fun isAxm (_,(_,DB.Axm)) = true | isAxm _ = false;
fun isDef (_,(_,DB.Def)) = true | isDef _ = false;

fun bind ((thy,s),(th,cl)) = (s,th);

fun dest_theory s =
 let val name = if s="-" then current_theory() else s
     val thms = DB.thy name
 in {parents = parents name,
     type_constants = types name,
     term_constants = constants name,
     axioms      = map bind (gather isAxm thms),
     definitions = map bind (gather isDef thms),
     theorems    = map bind (gather isThm thms)}
 end;

fun theory_to_html ppstrm theory_name =
 let open Portable
     val {parents,type_constants, term_constants, 
          axioms, definitions, theorems} = dest_theory theory_name
       val {add_string,add_break,begin_block,end_block,
            add_newline,flush_ppstream,...} = with_ppstream ppstrm
       val pp_thm  = pp_thm ppstrm
       val pp_type = pp_type ppstrm
       fun strong s = 
        (add_string"<STRONG>"; 
         add_string (String.concat["<font color=\"black\">",s,"</font>"]); 
         add_string"</STRONG>")
       fun title s = add_string(String.concat
                      ["<H1><font color=\"black\">",s,"</font></H1>"]);
       fun link (l,s) = 
        (add_string("<A HREF = "^Lib.quote l^">");
                         strong s;
                         add_string"</A>")
       fun HR() = (add_newline();add_string"<HR>";add_newline());

       fun vblock(header, ob_pr, obs) =
             ( begin_block CONSISTENT 4;
               title header;
               add_newline();
               add_string"<UL>"; add_newline();
               pr_list (fn x => (add_string"<LI>"; ob_pr x))
                          (fn () => ()) add_newline obs;
               add_newline();
               add_string"</UL>";
               end_block();
               add_newline();
               add_newline())

       fun dl_block(header, ob_pr, obs) =
             ( begin_block CONSISTENT 0;
               title header;
               add_newline();
               add_string"<DL>"; add_newline();
               pr_list 
                (fn (x,ob) => 
                     (begin_block CONSISTENT 0;
                      add_string"<DT>"; strong x; add_newline();
                      add_string"<DD>"; add_newline();
                      ob_pr ob;
                      end_block()))
                (fn () => ()) add_newline obs;
               add_newline();
               add_string"</DL>";
               end_block();
               add_newline();
               add_newline())

       (* fun fixity_to_string(Infix i)  = "Infix "^(Lib.int_to_string i)
         | fixity_to_string Prefix = "Prefix"
         | fixity_to_string Binder = "Binder"
       *)
       fun fixity_to_string _ = "ERROR"

       fun pr_thm (heading, ths) = dl_block(heading, 
           (fn th => (begin_block CONSISTENT 0;
                      add_string"<PRE>";
                      add_newline();
                      pp_thm th;
                      add_newline();
                      add_string"</PRE>";
                      add_newline();
                      end_block())),    ths)
   in begin_block CONSISTENT 0;
         add_string "<HTML>"; add_newline();
         add_string("<HEAD><TITLE>Theory: "^theory_name^"</TITLE></HEAD>");
         add_newline();
         add_string "<BODY bgcolor=linen text=midnightblue>";
         add_newline();
         add_string("<H1>Theory: "^theory_name^"</H1>");
         add_newline()
         ;
         vblock ("Parents", (fn n => link(n^"Theory.html",n)), parents)
         ; HR();
         vblock ("Types", (fn (Name,Arity) => 
                   (strong Name; 
                    add_string("(Arity = "^ Lib.int_to_string Arity^")"))),
                (rev type_constants))
         ; HR();
         let val consts = rev term_constants
(*             val infixes = gather (is_infix  o #Name o dest_const) consts *)
               val infixes = []   (* temporary *)
         in vblock("Constants", (fn const => 
                let val {Name,Thy,Ty} = Term.dest_thy_const const
                in begin_block CONSISTENT 0;
                   strong Name;
                   add_break(1,0); 
                   add_string"<EM>";
                   add_string"&nbsp"; 
                   pp_type Ty;
                   add_string"</EM>";
                   end_block()
                end),     consts)
            ;
            if null infixes then () else ()
              (* else vblock("Infixes", (fn const => 
                  let val Name = fst (dest_const const)
                      val infix_str = 
                         (case fixity Name
                           of Infix i => Lib.int_to_string i
                            | _ => raise ERR "theory_to_html"
                                             "Non-infix in infix block")
                  in begin_block CONSISTENT 0;
                     strong (Name^"{");
                     add_string("fixity = "^infix_str);
                     strong"}";
                     end_block()
                  end),     infixes)
              *)
         end
         ; HR();
         if null axioms then () else (pr_thm ("Axioms", axioms) ; HR())
         ;
         if null definitions then () 
         else (pr_thm ("Definitions", definitions); HR())
         ;
         if null theorems then () 
         else (pr_thm ("Theorems", theorems); HR());

         add_string "</BODY>"; add_newline();
         add_string "</HTML>"; add_newline();
       end_block()
   end;

fun html_theory path s = 
   let val name = (case s of "-" => current_theory() | other => s)
       val ostrm = Portable.open_out (Path.concat(path,name)^"Theory.html")
   in
     PP.with_pp {consumer = Portable.outputc ostrm, linewidth = 78,
                 flush = fn () => Portable.flush_out ostrm}
        (Lib.C theory_to_html name)
     handle e => (Portable.close_out ostrm; raise e);
     Portable.close_out ostrm
   end;

(* 
  val sys_theories = dir_theories SIGOBJ_DIR;
  load_theories sys_theories;
  map (html_theory HTML_DIR) sys_theories;
  gen_map_file "dot" (ancestor_nodes "-") "theories";
*)
